package Lire::Config::TypeSpec;

use strict;

use Lire::I18N qw/ dgettext dgettext_para /;
use Lire::DataTypes qw/:xml /;
use Lire::ReportParser::DocBookFormatter qw/dbk2txt/;
use Lire::Utils qw/ check_param check_object_param /;

use Carp;

BEGIN {
    no strict 'refs';

    foreach my $pkg (qw/Scalar Compound Object List Config String File
                        Command Executable Directory Boolean Integer
                        Select Option DlfConverter DlfSchema Record Plugin
                        / )
    {
        eval "use Lire\::Config\::${pkg}Spec";
        die if $@;
    }
}

=pod

=head1 NAME

Lire::Config::TypeSpec - Base class for object which specifies configuration types.

=head1 DESCRIPTION

Abstract superclass for the other type classes. Implements basic methods but
croaks when those are invoked.

=head2 new( 'name' => $name )

The constructor takes one mandatory argument which is the parameter's
name.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;
    my %args = @_;

    check_param ( $args{'name'}, 'name', \&check_xml_name,
                  "name should only contain digits, numbers or '.', '-' or '_'" );

    my $self = bless { 'name' => $args{'name'},
                       'section' => undef,
                       'summary' => $args{'summary'},
                       'description' => undef,
                       'i18n_domain' => $args{'i18n_domain'} || 'lire',
                       'contexts' => []
                     }, $class;

    $self->section($args{'section'});
    $self->contexts($args{'contexts'});
    $self->description( $args{'description'} )
      if exists $args{'description'};

    return $self;
}

=pod

=head2 name()

Returns the parameter's name.

=cut

sub name {
    return $_[0]{'name'};
}

=pod

=head2 has_context()

Returns if this parameter is interesting for the given context.

=cut

sub has_context {
    my $self = shift;

    return 1 if !%{$self->{'contexts'}};
    foreach(@_) {
        s/^\s+|\s+$//g;
        return exists $self->{'contexts'}{lc $_};
    }

    return;
}

=pod

=head2 contexts()

Returns a list of contexts of this parameter. If an argument is passed, it is
used to change the list of contexts. It's ok to pass undef as a parameter
(treated as '').

=cut

sub contexts {
    my $self = shift;

    return keys %{$self->{'contexts'}}
      unless @_;

    my %contexts;

    @contexts{split(' ', lc)} = ()
      foreach @_;

    $self->{'contexts'} = \%contexts;

    return keys %contexts;
}

=pod

=head2 has_section()

Returns if this parameter is interesting for the given section.

=cut

sub has_section {
    my $self = shift;

    foreach(@_) {
        if(defined) {
            s/^\s+|\s+$//g;
            return lc eq $self->section;
        } else {
            return !defined $self->section;
        }
    }

    return;
}

=pod

=head2 section()

Returns a the section of this parameter. If an argument is passed, it is
used to change the section. It's ok to pass undef as a parameter
(treated as "not a member of any section").

=cut

sub section {
    my $self = shift;

    foreach(@_) {
        if(defined) {
            s/^\s+|\s+$//g;
            $self->{'section'} = lc;
        } else {
            $self->{'section'} = undef;
        }
    }

    return $self->{'section'};
}

=pod

=head2 summary( [$summary] )

Returns this parameter summary description. It can also be used
to change the parameter's summary.

=cut

sub summary {
    my $self = $_[0];

    if (@_ == 2) {
        $self->{'summary'} = $_[1];
    }

    return ( defined $self->{'summary'}
             ? dgettext( $self->{'i18n_domain'}, $self->{'summary'} )
             : $self->{'name'} );
}

=pod

=head2 description( [$description] )

Returns this parameter's description. This is a DocBook string. It can
also be used to change the parameter's description.

=cut

sub description {
    my $self = $_[0];

    if (@_ == 2) {
        croak "'description' parameter should be a DocBook string"
          unless $_[1] =~ /^\s*<[\w.-]+>/;

        $self->{'description'} = $_[1];
    }

    return dgettext_para( $self->{'i18n_domain'}, $self->{'description'} );
}

=pod

=head2 text_description()

Returns the specification's description formatted in plain text,
one line per paragraph.

=cut

sub text_description {
    my $self = $_[0];

    my $description = $self->description();
    return undef unless defined $description;

    my $text = dbk2txt( $description, undef, 32_766 );
    $text =~ s/^ +//gm;
    $text =~ s/ +$//gm;
    chomp $text while ( substr( $text, -1, 1 ) eq "\n" );

    return $text;
}

=pod

=head2 instance()

This method is a factory type function which returns
a new Lire::Config::Value object of the correct type.

=cut

sub instance {
    croak( "instance() method not implemented in ", ref $_[0] );
}


=pod

=head2 is_instance( $value_object )

Returns true if $value_object is a valid instance of this parameter.

=cut

sub is_instance {
    my ( $self, $instance ) = @_;

    check_object_param( $instance, 'instance', 'Lire::Config::Value' );

    return $instance->spec() eq $self;
}

sub instance_class {
    croak( "instance_class_() method not implemented in ", ref $_[0] );
}

1;
__END__

=pod

=head1 VERSION

$Id: TypeSpec.pm,v 1.5 2004/03/26 00:27:33 wsourdeau Exp $

=head1 AUTHORS

Francis J. Lacoste <flacoste@logreport.org>
Wessel Dankers <wsl@logreport.org>
Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002-2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
