package Lire::OutputFormat;

use strict;

use Lire::Config;
use Lire::Utils qw/check_param tempdir shell_quote/;

use Carp;

=pod

=head1 NAME

Lire::OutputFormat - Objects used to encapsulate format specific details.

=head1 SYNOPSIS

  use Lire::OutputFormat;

  my $html = new Lire;:OutputFormat( 'html', 'charts' => 1, 'xhtml' => 1 );
  print $html->xml2report_options(); # => -i -o xhtml
  $html->output_to_file( 'report.xml', 'output_filename' );

=head1 DESCRIPTION

The Lire::OutputFormat objects are used to encapsulate format details.

=head1 new( $format, %format_params );

Create a new Lire::OutputFormat object. This is really a factory
method which will return the appropriate subclass. Available
parameters depends on the actual type.

=cut

sub new {
    my ( $pkg,  $format, %args ) = @_;

    check_param( $format, 'format', qr/^(docbook|excel95|html|pdf|txt|xml)$/,
                 "'format' parameter should be one of 'docbook', 'excel95', 'html', 'pdf', 'txt' or 'xml'" );

    if ( $format eq 'docbook' ) {
        return new Lire::OutputFormat::DocBook( %args );
    } elsif ( $format eq 'excel95' ) {
        return new Lire::OutputFormat::Excel95( %args );
    } elsif ( $format eq 'html' ) {
        return new Lire::OutputFormat::HTML( %args );
    } elsif ( $format eq 'pdf' ) {
        return new Lire::OutputFormat::PDF( %args );
    } elsif ( $format eq 'txt' ) {
        return new Lire::OutputFormat::Text( %args );
    } elsif ( $format eq 'xml' ) {
        return new Lire::OutputFormat::XML( %args );
    }
}

=pod

=head2 xml2mail_params()

This method is used by Lire::OutputJob to find the required parameters
to lr_xml2mail to send a report in that format by email.

=cut

sub xml2mail_params {
    croak ref( $_[0] ) . "::xml2mail_params() unimplemented";
}

=pod

=head2 output_to_file( $xml_file, $output_file )

This method is used by Lire;:OutputJob to generate the Lire XML Report
$xml_file into the $output_file destination file (or directory
depending on output format) 

=cut

sub output_to_file {
    croak ref( $_[0] ) . "::output_to_file() unimplemented";
}

sub _output_tar_report {
    my ( $self, $report_file, $output_file ) = @_;

    my $tar = Lire::Config->get( 'tar_path' );
    my $tmpdir = tempdir( "report_XXXXXX", CLEANUP => 1 );
    my $param = $self->xml2mail_params();
    system( "cd $tmpdir; lr_xml2report $param " . shell_quote( $report_file )
            . " | tar xf -" );
    return if $?;
    system( "mv $tmpdir/report " . shell_quote( $output_file ) );

    return;
}

=pod

=head1 Lire::OutputFormat::DocBook

Formats the report into DocBook XML. This object accepts a 'charts'
boolean parameter which control whether charts are generated. There is
a chart() accessor to query the status of this attribute.

=cut

package Lire::OutputFormat::DocBook;

use base qw/Lire::OutputFormat/;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless { '_charts' => $args{'charts'} }, $pkg;

    return $self;
}

sub charts {
    return $_[0]{'_charts'};
}

sub xml2mail_params {
    my $self = $_[0];

    my $i_flag = $self->{'_charts'} ? '-i ' :  '';
    return "$i_flag-o docbookx";
}

sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    $self->_output_tar_report( $report_file, $output_file );

    return;
}

=pod

=head1 Lire::OutputFormat::Excel95

Formats the report into an Excel 95 spreadsheet. This object doesn't
have any configurable parameters.

=cut

package Lire::OutputFormat::Excel95;

use base qw/Lire::OutputFormat/;

# Since that package is part of the CORE, we don't want Lire
# to fail because Spreadsheet::WriteExcel isn't installed
# Loading Lire::ReportParser::ExcelWriter that way prevents
# this.
eval "use Lire::ReportParser::ExcelWriter;";
use Carp;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless {}, $pkg;

    return $self;
}

sub xml2mail_params {
    return "-o excel95";
}

sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    my $parser =
      Lire::ReportParser::ExcelWriter->new( 'output_file' => $output_file );
    $parser->parsefile( $report_file );

    return;
}

=pod

=head1 Lire::OutputFormat::HTML

Formats the report into HTML 4.0 or XHTML 1.0. This object accepts
several parameters:

=over

=item charts

Boolean which determines if charts are generated in the report.
Defaults to false.

=item one_page

Boolean which controls whether the report is generated into one big
file, or one by section. Defaults to false.

=item xhtml

Boolean which controls if HTML 4.0 or XHTML 1.0 is generated. Defaults
to true.

=back

There is an accessor for each of these attributes: xhtml(),
one_page() and charts().

=cut

package Lire::OutputFormat::HTML;

use base qw/Lire::OutputFormat/;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless { '_xhtml' => exists $args{'xhtml'} ? $args{'xhtml'} : 1,
                       '_one_page' => $args{'one_page'},
                       '_charts' => $args{'charts'},
                     }, $pkg;

    return $self;
}

sub charts {
    return $_[0]{'_charts'};
}

sub one_page {
    return $_[0]{'_one_page'};
}

sub xhtml {
    return $_[0]{'_xhtml'};
}

sub xml2mail_params {
    my $self = $_[0];

    my $i_flag = $self->{'_charts'} ? '-i ' :  '';
    my $format = $self->{'_xhtml'} ? 'xhtml' : 'html';
    my $page = $self->{'_one_page'} ? '_page' : '';
    return "$i_flag-o ${format}$page";
}

sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    $self->_output_tar_report( $report_file, $output_file );

    return;
}

=pod

=head1 Lire::OutputFormat::PDF

Formats the report into PDF. This object accepts a 'charts' boolean
parameter which control whether charts are generated. There is a
chart() accessor to query the status of this attribute.

=cut

package Lire::OutputFormat::PDF;

use base qw/Lire::OutputFormat/;

use Lire::Utils qw/shell_quote/;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless { '_charts' => $args{'charts'} }, $pkg;

    return $self;
}

sub charts {
    return $_[0]{'_charts'};
}

sub xml2mail_params {
    my $self = $_[0];

    my $i_flag = $self->{'_charts'} ? '-i ' :  '';
    return "$i_flag-o pdf";
}

sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    my $param = $self->xml2mail_params();
    system( "lr_xml2report $param " . shell_quote( $report_file ) 
            . " > " . shell_quote( $output_file ) );

    return;
}

=pod

=head1 Lire::OutputFormat::Text

Formats the report into plain text. This object accepts an 'encoding'
parameter which sets the charset in which the report will be written.
There is an encoding() accessor which can be used to query this
attribute. The default is to use the system encoding.

=cut

package Lire::OutputFormat::Text;

use base qw/Lire::OutputFormat/;

use Lire::ReportParser::AsciiWriter;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless { '_encoding' => $args{'encoding'} }, $pkg;

    return $self;
}

sub encoding {
    return $_[0]{'_encoding'};
}

sub xml2mail_params {
    return "-o txt";
}

sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    open my $fh, "> $output_file"
      or die "can't open '$output_file' for writing: $!\n";
    my $parser = 
      new Lire::ReportParser::AsciiWriter( 'output' => $fh,
                                           'encoding' => $self->{'_encoding'});
    $parser->parsefile( $report_file );
    close $fh;

    return;
}

=pod

=head1 Lire::OutputFormat::XML

Formats the report in Lire XML native format. This is basically a
no-op. This format doesn't have any configurable properties.

=cut

package Lire::OutputFormat::XML;

use base qw/Lire::OutputFormat/;

use File::Copy qw/copy/;

sub new {
    my ( $pkg, %args ) = @_;

    my $self = bless {}, $pkg;

    return $self;
}

sub xml2mail_params {
    return "-o xml";
}


sub output_to_file {
    my ( $self, $report_file, $output_file ) = @_;

    copy( $report_file, $output_file );

    return;
}

1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportJob(3pm) Lire::OutputJob(3pm)

=head1 VERSION

$Id: OutputFormat.pm,v 1.2 2004/04/07 18:13:47 flacoste Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
