package Lire::OutputJob;

use strict;

use Lire::OutputFormat;
use Lire::Utils qw/check_param check_object_param/;

use Carp;

=pod

=head1 NAME

Lire::OutputJob - Object used to represent a configured formatting of a report

=head1 SYNOPSIS

  use Lire::OutputJob;
  use Lire::OutputFormat;

  my $email = new Lire::OutputJob( 'email',
                                    new Lire::OutputFormat( 'txt',
                                                            'output_encoding' => 'UTF-8' ),
                                   'emails' => [ 'email@domain.org' ] );

  $email->run( 'report.xml' );

  my $file = new Lire::OutputJob( 'file', 'html',
                                  'charts' => 1,
                                  'xhtml' => 1,
                                  'filename' => '/var/www/reports/daily-%Y%m%d'  );
  $file->run( 'report.xml' );

=head1 DESCRIPTION

The Lire::OutputJob objects are used to represent how a report should
formatted. These objects are most often created from the Lire
configuration. (The 'output_jobs' specification.)

=head1 new( $destination_type, $format, ... );

Create a new Lire::OutputJob object. Two parameters are required:

=over

=item $destination_type

Can be either 'email' or 'file'.

=item $format

A Lire::OutputFormat object.

=back

Other parameters can be set depending on the 'type :

=over

=item emails

An array reference to the emails that will receive the formatted report.

=item file

A file path with possible strftime(3pm) interpolation. This will be
were the formatted report will be saved.

=item

=back

The method will returns an instance of Lire::EmailOutputJob or
Lire::FileOutputJob.

=cut

sub new {
    my ( $pkg, $type, $format, %args ) = @_;

    check_param( $type, 'type', qr/^(email|file)$/,
                 "'type' parameter should be one of 'email' or 'file'" );
    check_object_param( $format, 'format', 'Lire::OutputFormat' );

    my $self = { '_format' => $format };
    if ( $type eq 'email' ) {
        bless $self, 'Lire::EmailOutputJob';
    } else {
        bless $self, 'Lire::FileOutputJob';
    }
    $self->init( %args );

    return $self;
}

=pod

=head2 format

Returns the Lire::OutputFormat associated with this OutputJob.

=cut

sub format {
    return $_[0]{'_format'};
}

=pod

=head2 run( $report_file )

Formats and sends or save the XML report $report_file according the
OutputJob parameters.

=cut

sub run {
    croak ref shift, "::run unimplemented";
}

sub new_from_config {
    my ( $pkg, $value ) = @_;

    my $format = new Lire::OutputFormat( $value->{'format'}{'plugin'},
                                         %{$value->{'format'}{'properties'}} );
    return new Lire::OutputJob( $value->{'destination'}{'plugin'},
                                $format,
                                %{$value->{'destination'}{'properties'}} );
}

package Lire::EmailOutputJob;

use base qw/Lire::OutputJob/;

use Lire::Utils qw/check_object_param shell_quote/;

sub init {
    my ( $self,  %args ) = @_;

    check_object_param( $args{'emails'}, 'emails', 'ARRAY' );
    $self->{'_emails'} = $args{'emails'};
    $self->{'_subject'} = $args{'subject'} ? $args{'subject'} : '';

    return;
}

=pod

=head1 Lire::EmailOutputJob

=head2 emails()

Returns as an array the emails to which the report will be sent.

=cut

sub emails {
    return @{$_[0]{'_emails'}};
}

=pod

=head2 subject()

Returns the subject of the email.

=cut

sub subject {
    return $_[0]{'_subject'};
}

sub _shell_cmd {
    my ( $self, $report_file ) = @_;

    my $emails = join " ", map { shell_quote( $_ ) } $self->emails();
    my $subject = shell_quote( $self->subject() );
    my $param = $self->format()->xml2mail_params();
    return "lr_xml2mail $param -s $subject -x " . shell_quote( $report_file ) .
      " $emails";
}

sub run {
    my ( $self, $report_file ) = @_;

    system( $self->_shell_cmd( $report_file ) );

    return;
}

package Lire::FileOutputJob;

use base qw/Lire::OutputJob/;

use Lire::Utils qw/check_param/;

use POSIX qw/strftime/;

sub init {
    my ( $self,  %args ) = @_;

    check_param( $args{'file'}, 'file' );
    $self->{'_file'} = $args{'file'};

    return;
}

=pod

=head1 Lire::FileOutputJob

=head2 file()

Returns this OutputJob destination file.

=cut

sub file {
    return $_[0]{'_file'};
}

=pod

=head2 output_file( [$time] )

Returns the  destination file with strftime(3) specifier interpolated.

=cut

sub output_file {
    my ( $self, $time ) = @_;

    $time ||= time();
    return strftime( $self->{'_file'}, localtime $time );
}

sub run {
    my ( $self, $report, $time ) = @_;

    $time ||= time();

    $self->format()->output_to_file( $report, $self->output_file( $time ) );
    return;
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportJob(3pm) Lire::OutputFormat(3pm)

=head1 VERSION

$Id: OutputJob.pm,v 1.2 2004/04/07 18:13:47 flacoste Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
