package Lire::Report::Subreport;

use strict;

use base qw/ Lire::Report::Group /;

use Carp;
use POSIX qw/ strftime /;

use Lire::DataTypes qw/ :special :xml /;
use Lire::Utils qw/ xml_encode /;

=pod

=head1 NAME

Lire::Subreport - Interface to a Lire subreport.

=head1 SYNOPSIS

    use Lire::ReportParser::ReportBuilder;
    my $parser = new Lire::ReportParser::ReportBuilder;
    my $report = $parser->parse( "report.xml" );

    foreach my $s ( $report->sections ) {
        print "Section: '", $s->title, "\n\n";
        foreach my $r ( $s->subreports ) {
            print "Subreport ", $r->title, " has ", scalar $r->entries, 
                " entries\n";
        }
    }

=head1 DESCRIPTION

This module offers an API to the subreports included in the sections
of a Lire's report.

This object has all the methods that the Lire::Report::Group(3pm)
object offers. You'll find more information on to access the data
contained in a subreport in that man page.

=head1 CONSTRUCTORS

=head2 new( $report, $type)

Creates a new Lire::Report::Subreport object. The $type parameter is
the id of the report specification that is used to generate this
subreport's data. The $report parameter is of type Lire::Report and is
used to fill in the timespan, superservice and type attributes of the
subreport.

=cut

sub new {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $report, $type ) = @_;

    croak "report isn't of type Lire::Report ($report)"
      unless UNIVERSAL::isa( $report, "Lire::Report" );

    croak "invalid type: $type"
      unless check_xml_name( $type );

    bless { 'superservice' => $report->superservice,
            'type'         => $type,
            'entries'      => [],
            'missing'      => 0,
            'report'       => $report,
            'notes'        => [],
            'summary'      => {},
            'parent_entry' => undef,
            'nrecords'     => 0,
            'missing_cases' => 0,
            'row_idx'      => undef,
            'title'        => undef,
            'description'  => undef,
            'show'         => undef,
          }, $class;
}

=pod

=head2 new_missing( $report, $type, [$reason])

Creates a new Lire::Report::Subreport object marked as missing. The
$type parameter is the id of the report specification that is used to
generate this subreport's data. The $report parameter is of type
Lire::Report and is used to fill in the timespan, superservice and
type attributes of the subreport. The $reason parameter sets the
reason why this subreport is missing.

=cut

sub new_missing {
    my $proto = shift;
    my $class = ref $proto || $proto;

    my ( $report, $type, $reason ) = @_;

    croak "report isn't of type Lire::Report ($report)"
      unless UNIVERSAL::isa( $report, "Lire::Report" );

    croak "invalid type: $type"
      unless check_xml_name( $type );

    bless { 'superservice' => $report->superservice,
            'type'         => $type,
            'missing'      => 1,
            'entries'      => [],
            'report'       => $report,
            'reason'       => $reason,
          }, $class;
}

=pod

=head1 OBJECT METHODS

=head2 superservice()

Returns the superservice of this subreport. This is usually identical
to the report's value.

=cut

sub superservice {
    $_[0]{'superservice'};
}

=head2 is_missing()

Returns true if this subreport was marked as missing in the XML file.
A missing subreport is a subreport which was present in the report
configure file but which was marked as missing in the XML report. This
happens when the report specification requires fields that weren't
available in the DLF files generated. It can also happen when an error
occurs in the report generation process.

=cut

sub is_missing {
    return $_[0]{'missing'};
}

=pod

=head2 missing_reason()

Reports why the subreport is missing. This will be undefined when the
subreport is not missing.

=cut

sub missing_reason {
    $_[0]{'reason'};
}

=head2 type()

Returns the type of this subreport. This is the ID of the report specification that was used to generate this subreport.

=cut

sub type {
    $_[0]{'type'};
}

=pod

=head2 charttype( [$new_type] )

Returns the suggested chart's type for this subreport. If the
$new_type parameter is set, this subreport's charttype attribute will
be modified accordingly.

=cut

sub charttype {
    if ( defined $_[1]) {
        croak "invalid charttype values: $_[1]"
          unless check_chart( $_[1]);
        $_[0]{'charttype'} = $_[1];
    }
    $_[0]{'charttype'};
}

=pod

=head2 hostname([$hostname])

Returns the hostname of this subreport. This is usually identical to
the report's value.

If the $hostname is parameter is set, the subreport's hostname will be
set to this new value.

=cut

sub hostname {
    $_[0]{'hostname'} = $_[1] if @_ == 2;
    $_[0]{'hostname'} || $_[0]{'report'}->hostname;
}

=pod

=head2 date()

Returns the date in seconds since epoch on which this subreport was
generated. This is usually identical to the report's value.

=cut

sub date {
    $_[0]{'date'} || $_[0]{'report'}->date;
}

=pod

=head2 timespan_start()

Returns the start of the timespan covered by this subreport in seconds
since epoch. This is usually identical to the report's value.

=cut

sub timespan_start {
    $_[0]{'timespan_start'} || $_[0]{'report'}->timespan_start;
}

=pod

=head2 timespan_end()

Returns the end of the timespan covered by this subreport in seconds
since epoch. This is usually identical to the report's value.

=cut

sub timespan_end {
    $_[0]{'timespan_end'} || $_[0]{'report'}->timespan_end;
}

=pod

=head2 title( [$title] )

Returns the subreport's title.

If the $title is parameter is set, the subreport's title will be set to
this new value.

=cut

sub title {
    $_[0]{'title'} = $_[1] if defined $_[1];
    $_[0]{'title'};
}

=pod

=head2 description( [$new_desc] )

Returns this subreport's description. The description is encoded in
DocBook XML.

If the $description parameter is set, this method will set this
subreport's description to this new value. If the $description
parameter is undef, that description will be removed.

=cut

sub description {
    my ( $self, $new_desc ) = @_;

    $self->{'description'} = $new_desc
      if @_ == 2;

    $self->{'description'};
}

=head2 notes

Returns this subreport's notes as an array. This will be an array of
Lire::Report::Note objects.

=cut

sub notes {
    @{$_[0]{'notes'}};
}

=pod

=head2 add_note( $note )

Add a note to this subreport. The $note parameter must be of the
Lire::Report::note type.

=cut

sub add_note {
    my ( $self, $note ) = @_;

    croak ( "note should be of type Lire::Report::Note (not $note)" )
      unless UNIVERSAL::isa( $note, "Lire::Report::Note" );

    push @{$self->{'notes'}}, $note;
}

=pod

=head2 table_info( [$table_info] )

Returns the top-level Lire::Report::TableInfo object describing this
table layout. If $table_info parameter is set, the subreport table
info will be set to this object.

=cut

sub table_info {
    my ( $self, $info ) = @_;

    if ( @_ == 2 ) {
        croak ( "can't change table_info once entries were added" )
          if @{$self->{'entries'}};

        croak ( "info must be of type Lire::Report::TableInfo (not $info)" )
          unless UNIVERSAL::isa( $info, "Lire::Report::TableInfo" );

        $self->{'table_info'} = $info;
        $self->{'group_info'} = $info;
    }

    $self->{'table_info'};
}

=pod

=head2 finalize()

=cut

sub finalize {
    my $self = $_[0];

    $self->assign_row_indices();
    $self->compute_column_info_stats();
    $self->{'table_info'}->compute_columns_width();

    return;
}

sub assign_row_indices {
    my ( $self, $start ) = @_;

    # Prevent infinite recursion as parent call back this
    # implementation
    return $self->SUPER::assign_row_indices( $start )
      if defined $start;

    my $idx = $self->assign_row_indices( -1 );
    $self->{'row_idx'} = undef;

    return $idx;
}

=pod

=head2 image( [$image] )

If an image was generated for this subreport, this methods returns the
Lire::Report::Image that contains the image's information.

If the $image parameter is set, the subreport's image will be set to
this new value.

=cut

sub image {
    if ( @_ == 2 ) {
        croak( "invalid image parameter. Should be of type Lire::Image and not $_[1]" )
          unless UNIVERSAL::isa( $_[1], "Lire::Image" );
        $_[0]{'image'} = $_[1];
    }
    $_[0]{'image'};
}

# Overrides Lire::Report::Group::subreport
sub subreport {
    return $_[0];
}

sub write_report {
    my ( $self, $fh, $indent ) = @_;

    $fh ||= *STDOUT;
    my $pfx = ' ' x $indent;

    if ( $self->{'missing'} ) {
        print $fh qq!$pfx<lire:missing-subreport type="$self->{'type'}" !,
          'reason="', xml_encode( $self->{'reason'} ), qq!"/>\n\n!;    

    } else {
        print $fh qq!$pfx<lire:subreport type="$self->{'type'}" superservice="$self->{'superservice'}">\n!;

        print $fh "$pfx <lire:title>", xml_encode( $self->{'title'} ), 
          "</lire:title>\n\n";

        if ( $self->{'date'} ) {
            my $time  = strftime '%Y-%m-%d %H:%M:%S %Z', localtime $self->date;
            print $fh qq!$pfx <lire:date time="$self->{'date'}">$time</lire:date>\n!
        }

        if ( $self->{'timespan_start'} ) {
            my $stime = strftime '%Y-%m-%d %H:%M:%S %Z', localtime $self->timespan_start;
            my $etime = strftime '%Y-%m-%d %H:%M:%S %Z', localtime $self->timespan_end;
            print $fh qq!$pfx <lire:timespan start="$self->{'timespan_start'}" end="$self->{'timespan_end'}">$stime - $etime</lire:timespan>\n!
        }

        print $fh "$pfx <lire:hostname>", xml_encode( $self->{'hostname'} ),
          "</lire:hostname>\n"
          if defined $self->{'hostname'};

        if ( $self->description ) {
            print $fh "$pfx <lire:description>";
            print $fh $self->description, " </lire:description>\n\n";
        }

        if ( $self->notes ) {
            print $fh "$pfx <lire:notes>\n";
            foreach my $n ( $self->notes ) {
                $n->write_report( $fh, $indent + 2 );
            }
            print $fh "$pfx </lire:notes>\n\n"
        }

        print $fh "$pfx <lire:table";
        print $fh qq! charttype="$self->{'charttype'}"!
          if defined $self->{'charttype'};
        print $fh qq! show="$self->{'show'}"!
          if defined $self->{'show'};
        print $fh ">\n";

        $self->table_info->write_report( $fh, $indent + 2 );

        $self->write_group_summary( $fh, $indent + 2 );

        foreach my $e ( $self->entries ) {
            $e->write_report( $fh, $indent + 2 );
        }

        print $fh "$pfx </lire:table>\n";

        $self->{'image'}->write_report( $fh, $indent + 1 )
          if $self->{'image'};

        print $fh "$pfx</lire:subreport>\n\n";
    }
}

#------------------------------------------------------------------------
# Method delete()
#
# Remove circular references
sub delete {
    my ( $self ) = @_;

    $self->{'table_info'}->delete
      if $self->{'table_info'};
    foreach my $e ( $self->entries ) {
        $e->delete;
    }
    delete $self->{'report'};
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportParser::ReportBuilder(3pm) Lire::Report::Section(3pm)
Lire::Report::Report(3pm) Lire::Report::Entry(3pm),
Lire::Report::Group(3pm), Lire::Report::Image(3pm)

=head1 VERSION

$Id: Subreport.pm,v 1.24 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
