package Lire::ReportMerger;

use strict;

use Carp;
use POSIX qw/ strftime /;

use Lire::Logger qw/ lr_warn lr_info /;
use Lire::ReportConfig;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/ check_object_param /;

=pod

=head1 NAME

Lire::ReportMerger - Merge XML reports

=head1 SYNOPSIS

    use Lire::ReportMerger;

    my $merger = new Lire::ReportMerger( $report_cfg, "report1.xml",
                                         "report2.xml" );
    my $report = $merger->merge_reports();

=head1 DESCRIPTION

This method object will generate a new Lire::Report from a report
configuration and one or more XML reports.

=cut

sub new {
    my ( $class, $report_cfg, @files ) = @_;

    check_object_param( $report_cfg, 'report_cfg', 'Lire::ReportConfig' );

    croak "new() requires one or more 'file' parameters"
      unless @files;

    croak "'report_cfg' parameter should have been initialized with a Lire::AsciiDlf::AsciiDlfFactory"
      unless $report_cfg->factory()->isa( 'Lire::AsciiDlf::AsciiDlfFactory' );

    my $self = bless {
                      'report_cfg'   => $report_cfg,
                      'reports'      => [ @files ],
                     }, $class;

    return $self;
}

sub sort_and_check_reports {
    my $self = $_[0];

    my @files = @{$self->{'reports'}};

    my %start = ();
    my @reports = ();

    # Check all reports for well-formedness and valid starting 
    # and ending period.
    my $parser = new Lire::ReportParser::ReportBuilder( 'only_head' => 1 );
    foreach my $r ( @files ) {
        unless (open REPORT, $r) {
            lr_warn( "can't open XML report $r: $!. Skipping" );
            next;
        }

        lr_info( "checking XML report '$r'" );
        my $report = eval { $parser->parse( \*REPORT ) };
        if ( $@ ) {
            lr_warn( "error parsing XML report $r: $@. Skipping" );
            next;
        }
        close REPORT;

        # Start and end of merged report is the union of the range of
        # all the reports
        # timespan_start and timespan_end are in seconds since epoch
        unless ( defined $self->{'report_start'} ) {
            $self->{'report_start'} = $report->timespan_start();
            $self->{'report_end'}   = $report->timespan_end();
        }

        # It is possible for a report to have
        # an unknown period
        if ( $report->timespan_start ) {
            $self->{'report_start'} = $report->timespan_start()
              if $report->timespan_start() < $self->{'report_start'};
            $self->{'report_end'}   = $report->timespan_end()
              if $report->timespan_end() > $self->{'report_end'};
        }

        push @reports, $r;
        $start{$r} = $report->timespan_start() || 0;

        $report->delete();
    }

    lr_info( "Will merge ", scalar @reports, " files. " );
    if ( $self->{'report_start'} ) {
        lr_info( "Merged period starts ",
                 strftime( "%Y-%m-%d %H:%M:%S", localtime $self->{'report_start'} ),
                 "; ends on ",
                 strftime( "%Y-%m-%d %H:%M:%S", localtime $self->{'report_end'} ));
    } else {
        lr_info( "Merging reports from unknown period." );
    }

    $self->{'reports'} = [ sort { $start{$a} <=> $start{$b} } @reports ];

    return;
}

sub init_merge {
    my ( $self, $cfg, $subreport_idx ) = @_;

    my %type_count = ();
    foreach my $section ( $cfg->sections() ) {
        foreach my $spec ( $section->reports() ) {
            my $id = $spec->id();
            my $key = $spec->key();

            $type_count{$id} = 0 unless exists $type_count{$id};
            $subreport_idx->{$key} = $type_count{$id}++;

            eval { $spec->calc_spec->init_merge( $self->{'report_start'},
                                                 $self->{'report_end'} ) };
            if ( $@ ) {
                lr_warn( "$@\nreport '$key' will be skipped" );
                $spec->mark_missing( "init_merge() failed: $@" );
            }
        }
    }
}

sub end_merge {
    my ( $self, $cfg, $non_missing ) = @_;

    foreach my $section ( $cfg->sections() ) {
        foreach my $spec ( $section->reports() ) {
            my $key = $spec->key();

            eval { $spec->calc_spec->end_merge() };
            if ( $@ ) {
                lr_warn( "$@\nreport '$key' will be skipped" );
                $spec->mark_missing( "end_merge() failed: $@" );
            }

            # Mark missing, if it was also missing in all reports
            $spec->mark_missing( "missing in all merged reports" )
              unless $non_missing->{$key};
        }
    }
}

sub merge_report {
    my ( $self, $cfg, $report, $subreport_idx, $non_missing ) = @_;

    foreach my $section ( $cfg->sections() ) {
        foreach my $spec ( $section->reports() ) {
            my $type = $spec->id();

            # Skip failed reports
            next if $spec->is_missing();

            my $idx = $subreport_idx->{$spec->key()};
            die "assertion failed: can't find index of subreport ",
              $spec->key()
                unless defined $idx;

            # Find the matching subreport in this report
            my @subreports = $report->subreports_by_type( $type );

            if (  $idx > $#subreports ) {
                lr_warn( "there's only ", scalar @subreports,
                         " subreports of type $type while trying to merge index $idx" );
                next;
            }

            my $subreport = $subreports[$idx];
            if ( $subreport->is_missing() ) {
                lr_info( "skipping merge of missing $type index $idx" );
                next;
            }

            eval { $spec->calc_spec->merge_subreport( $subreport ); };
            if ($@) {
                lr_warn( $@ );
                lr_warn( "Merge of $type index $idx failed" );
                next;
            }

            $non_missing->{$spec->key} = 1;
        }
    }
}

sub merge_reports {
    my $self = $_[0];

    my $cfg = $self->{'report_cfg'};
    $self->sort_and_check_reports();

    my $subreport_idx = {};
    $self->init_merge( $cfg, $subreport_idx );

    my $non_missing = {};
    my $parser = new Lire::ReportParser::ReportBuilder();
    foreach my $r ( @{$self->{'reports'}} ) {
        lr_info( "parsing XML report '$r'" );
        my $report = eval { $parser->parsefile( $r ) };
        if ( $@ ) {
            lr_warn( "error parsing XML report $r: $@. Skipping" );
            next;
        }

        lr_info( "merging XML report '$r'" );
        $self->merge_report( $cfg, $report, $subreport_idx, $non_missing );
        $report->delete();
    }

    $self->end_merge($cfg, $non_missing );

    return $cfg->create_report( $self->{'report_start'}, $self->{'report_end'} );
}


# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::ReportConfig(3pm), Lire::ReportGenerator(3pm), lr_xml_merge(1)

=head1 VERSION

$Id: ReportMerger.pm,v 1.10 2004/03/30 17:38:53 wsourdeau Exp $

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2003 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

