package Lire::ReportOperator;

use strict;

use Carp;

use Lire::DataTypes qw/ :special /;
use Lire::I18N qw/ dgettext /;
use Lire::Utils qw/ check_param check_object_param /;

=pod

=head1 NAME

Lire::ReportOperator - Base class for all operators that can be used to compute reports

=head1 SYNOPSIS

    print $op->name(), "\n";
    print $op->op(), "\n";
    print $op->label(), "\n";

=head1 DESCRIPTION

The Lire::ReportOperator is the base class for all operators that can
be used to compute the data appearing in Lire's reports. This class
only provides the behavior common to all operators.

Currently, the operators defined in Lire come into two main varieties:
first, there are aggregators which are operators that can split DLF
records into multiple groups, second there are group operators that can
compute data based on a group of DLF records. Aggregators are
subclasses of Lire::Aggregator and group operations are subclasses of
Lire::Aggregate. There are other special operators like records which
don't fit in these two categories.

=head1 METHODS

=head2 op()

Returns the XML element's name which refers to this operator.

=cut

sub op {
    return $_[0]{'op'};
}

=pod

=head2 report_spec()

Returns the Lire::ReportSpec object in which this operator is.

=cut

sub report_spec {
    return $_[0]{'report_spec'};
}

=pod

=head2 parent()

Returns the parent of this operator. This will be undefined for the
top-level aggregator in the report specification. Only
Lire::Aggregator can have children and as such be a parent.

=cut

sub parent {
    return $_[0]{'parent'};
}

=pod

=head2 last_parent()

Returns our top-level parent.

=cut

sub last_parent {
    my ( $self ) = @_;

    my $parent = $self;
    while ( $parent->parent() ) {
        $parent = $parent->parent()
    }

    return $parent;
}

=pod

=head2 name( [$new_name] )

Returns the name of this operator in the current report specification.
The operator's name is an identifier that must be unique in a given
report specification. This name will be used to identify the values
generated by this operator in the generated report.

If $new_name is set, this operator's name will be changed to this new
value.

=cut

sub name {
    croak "Unimplemented name() method in ", ref $_[0];
}

=pod

=head2 has_label()

Returns true if this operator has an explicit label assigned.

=cut

sub has_label {
    return defined $_[0]->{'label'};
}

=pod

=head2 label( [$new_label] )

Returns what should be used as column's label for the data generated
by this operator.

If the second argument is set, the column label will be set to this
new value.

=cut

sub label {
    my ( $self, $label ) = @_;

    $self->{'label'} = $label 
      if @_ == 2;

    return dgettext( 'lire-' . $self->{'report_spec'}->superservice(),
                     ( $self->{'label'} ? $self->{'label'} : $self->name() ) );
}

=pod

=head1 METHODS FOR SUBCLASSES

Subclasses must use a hash to represent their instance data. They
should call init() from their constructor method.

=head2 init( %params )

Lire::Report is an abstract class which shouldn't be instanciated
directly. Subclasses should call the init() method in their
constructor. This method initializes the attributes common to all
operators. The %params keys that should be defined are:

=over

=item op

The operator's kind. That's the XML element's name. It's a mandatory
parameter.

=item parent

This operator's parent. This must be present unless the element is the
top-level aggregator.

=item report_spec

The report specification in which this operator is added.

=item label

This operator's label. This is an optional information.

=back

This method returns the object.

=cut

sub init {
    my ( $self, %args ) = @_;

    check_param( $args{'op'}, 'op' );
    check_object_param( $args{'report_spec'}, 'report_spec',
                        'Lire::ReportSpec' );
    check_object_param( $args{'parent'}, 'parent', 'Lire::Aggregator' )
      if ( defined $args{'parent'} );

    $self->{'report_spec'} = $args{'report_spec'};
    $self->{'op'}          = $args{'op'};
    $self->{'parent'}      = $args{'parent'};
    $self->label( $args{'label'} )
      if defined $args{'label'};

    return;
}

=pod

=head2 print( $fh, $prefix )

This is the method which is called to write an XML representation of
this operator.

$fh is the file handle onto which to output the XML representation.
$indent is an integer which gives the number of spaces which should be
used as indentation when writing the XML.

=cut

sub print {
    croak( "unimplemented print() method" );
}

=pod

=head2 build_query( $query )

=cut

sub build_query {
    croak( "unimplemented build_query() method in ", ref $_[0] );
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::Aggregate(3pm), Lire::Aggregator(3pm).

=head1 VERSION

$Id: ReportOperator.pm,v 1.16 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
