package Lire::ReportParser::PloticusChartWriter;

use strict;

use base qw/ Lire::ReportParser /;

use File::Basename;
use Carp;

use Lire::Config;
use Lire::ReportParser;
use Lire::Logger;
use Lire::Utils qw/ check_param /;

sub new {
        my $proto = shift;
        my $class = ref $proto || $proto;
        my $self  = $class->SUPER::new( @_ );

        my %args  = @_;

        $args{'format'}    ||= Lire::Config->get( 'lr_image_format' );
        $args{'outputdir'} ||= ".";

        check_param( $args{'format'}, 'format',
                     qr/^(png|gif|jpe?g|svg|eps|pdf)$/i,
                     'supported file formats are png, gif, jpeg, svg, eps and pdf' );
        check_param( "$args{'outputdir'}", 'outputdir',
                     sub { -d $_[0] },
                     "'outputdir' is not a directory" );

        croak "can't write in $args{'outputdir'}\n"
          unless -w $args{'outputdir'};

        $self->{'format'}                         = lc $args{'format'};
        $self->{'outputdir'}                      = $args{'outputdir'};
        $self->{'graph_resolution'}       = $args{'graph_resolution'} || 128;
        $self->{'max_label_len'}          = $args{'max_label_length'} || 20;
        $self->{'high_quality'}           = $args{'high_quality'} || 0;

        return $self;
}

sub parse_start {
        my ($self ) = @_;
        # Register handler for XMLDcl and DocType event, so that 
        # we output them as is

        my $print_it = sub {
                print $_[0]->original_string(), "\n";
        };

        $self->expat->setHandlers( 'XMLDecl'        => $print_it,
                                                           'Comment'        => $print_it,
                                                         );
        1;
}

# Print an identical copy of the XML report.
# We will add image and file element when we create graphics.
sub element_start {
        my ( $self, $name, %attr ) = @_;

        print $self->original_string;

        return 0; # Continue normal processing
}

sub element_end {
        my ( $self, $name ) = @_;

        print $self->original_string;

        return 0; # Continue normal processing
}

sub pcdata {
        my ( $self, $str ) = @_;

        print $self->original_string;
        return 0; # Continue normal processing
}

sub ignorable_ws {
        my ( $self, $str ) = @_;

        print $self->original_string;
        return 0; # Continue normal processing
}

sub _ploticus_cmdline {
    my ( $self, $charttype, $filename, $format ) = @_;

    check_param( $charttype, 'charttype' );
    check_param( $filename, 'filename' );
    check_param( $format, 'format' );

    my @cmd = ( Lire::Config->get( 'ploticus_path' ) );

    my $font = Lire::Config->get( 'lr_chart_font' );
    push @cmd, "-font '$font'" if $font;

    push @cmd, ( "-$format", "-o", $filename, "delim=tab",
                 "comment=__COMMENT__", "data=-" );

    if ($charttype eq 'histogram') {
        push @cmd, ( "-prefab", "vbars", "x=3", "y=2", "stubvert=yes",
                     "curve=width=.7", "color='rgb(.4,0,.4)'" );
    } elsif ($charttype eq 'bars') {
        push @cmd, ( '-prefab', 'vbars', 'x=3', 'y=2', 'stubvert=yes',
                     "color='rgb(.4,0,.4)'" );
    } elsif ($charttype eq 'pie' ) {
        push @cmd, ( '-prefab', 'pie', 'values=2', 'labels=3',
                     "colors='rgb(.75,0,0) rgb(1,.75,0) rgb(0,.75,0) rgb(0,0,1) rgb(.75,0,1) rgb(.25,.6,.9) rgb(.9,.6,.25) rgb(.5,.8,.6) rgb(.8,.5,.7) rgb(.5,.5,.5)'" );
    } else {
        croak "invalid 'charttype' parameter: '$charttype'";
    }

    push @cmd, '2>/dev/null';
    return join( " ", @cmd );
}

sub table_start {
    my ( $self ) = shift;

    $self->SUPER::table_start( @_ );

    $self->{'write_chart'} = defined $self->current_charttype;
    $self->{'keep_only'}       = $self->current_group_entry_show;

    $self->{'count'} = 0;
    $self->{'label'} = undef;
    $self->{'value'} = undef;

    return unless $self->{'write_chart'};

    my $basename = $self->current_subreport_count().'-'.$self->current_type();
    $basename =~ s/\./_/g;

    my $hq = defined($self->{'high_quality'}) && $self->{'high_quality'};
    my $ext = lc $self->{'format'};
    my $img_file = $self->{'outputdir'}."/$basename.$ext";

    $self->{'need_gv'} = ($ext eq 'pdf') || ($ext eq 'png' && $hq);

    my ($fmt, $out);

    if ($self->{'need_gv'}) {
        $fmt = 'eps';
        $out = $self->{'outputdir'}."/$basename.$fmt";
        $self->{'temp'} = $out;
    } else {
        $fmt = $ext eq 'jpg' ? 'jpeg' : $ext;
        $out = $img_file;
        $self->{'temp'} = undef;
    }

    my $pl = $self->_ploticus_cmdline( $self->current_charttype(), $out, $fmt);
    open( FH, "|" . $pl )
      or die "error forking ploticus: $!\n";

    $self->{'pid'} = *FH{'IO'};
    $self->{'file'} = $img_file;
}

sub epscat {
        my $bbfound = 0;
        my $nested = 0;
        my $bbox = '%%(?:HiRes|Exact)?BoundingBox:\s*';
        my $bval = '[0-9eE\.\-]';
        $bval = "($bval+)\\s+($bval+)\\s+($bval+)\\s+($bval+)";

        sub printfixedbbox {
                my ($llx, $lly, $urx, $ury) = @_;
                my ($xoffset, $yoffset) = (-$llx, -$lly);
                my ($width, $height) = ($urx - $llx, $ury - $lly);

                print OUTFILE "%%BoundingBox: 0 0 $width $height\n";
                print OUTFILE "<< /PageSize [$width $height] >> setpagedevice\n";
                print OUTFILE "gsave $xoffset $yoffset translate\n";

                return 1;
        }

        open(INFILE, $_[0])
                or croak "can't open EPS input file";
        open(OUTFILE, $_[1])
                or croak "can't open output stream $_[1]";

        ### scan first line
        $_ = <INFILE>;
        # only act on PostScript files
        if(/%!/) {
                # throw away binary junk before %!
                s/(.*)%!/%!/o;
                print OUTFILE;

                while (<INFILE>) {
                        $nested++ if /^%%BeginDocument/;
                        $nested-- if /^%%EndDocument/;

                        if($nested) {
                                print OUTFILE;
                                next;
                        }

                        ### BoundingBox with values
                        if(/^$bbox$bval/io) {
                                if($bbfound) {
                                        next;
                                } else {
                                        $bbfound = printfixedbbox $1, $2, $3, $4;
                                }
                        }

                        ### BoundingBox with (atend)
                        if(/^$bbox\(atend\)/io && !$bbfound) {
                                my $pos = tell(INFILE);

                                while(<INFILE>) {
                                        $nested++ if /^%%BeginDocument/;
                                        $nested-- if /^%%EndDocument/;

                                        # skip over included documents
                                        if(!$nested && /^$bbox$bval/io) {
                                                $bbfound = printfixedbbox $1, $2, $3, $4;
                                                last;
                                        }
                                }

                                # go back
                                seek(INFILE, $pos, 0)
                                        or croak "Cannot go back to line 'BoundingBox (atend)'";
                                next;
                        }

                        # print header line
                        print OUTFILE;
                }
        } else {
                print OUTFILE while(<INFILE>);
        }

        close(INFILE);

        if($bbfound) {
                print OUTFILE "\ngrestore\n";
        } else {
                croak "BoundingBox not found in $_[0]";
        }
        close(OUTFILE);
}

sub table_info_end {
    my ( $self ) = @_;

    $self->SUPER::table_info_end;

    # We only use the first categorical and numerical columns
    my @cols = $self->current_table_info->column_infos;
    my ($cat) = grep { $_->class eq 'categorical' } @cols;
    my ($num) = grep { $_->class eq 'numerical' } @cols;
    $self->{'x_name'} = $cat->name;
    $self->{'y_name'} = $num->name;
}

sub table_end {
        my ( $self ) = shift;

        $self->SUPER::table_end( @_ );

        return unless $self->{'write_chart'};

        close($self->{'pid'});

        unless($self->{'count'}) {
                lr_info("skipping chart for ", $self->current_type , " because table is empty");
                unlink($self->{'file'});
                return;
        }

        if($self->{'need_gv'}) {
                my $fmt = $self->{'temp'};
                my $out = $self->{'file'};
                my $res = $self->{'graph_resolution'};
                my $driver
                        = $self->{'format'} eq 'png' ? "png16m -dTextAlphaBits=4 -dGraphicsAlphaBits=4 -r$res"
                        : $self->{'format'} eq 'pdf' ? 'pdfwrite'
                        : $self->{'format'};
                epscat("<$fmt", "|gs -q -dAutoRotatePages=/None -dBATCH -dNOPAUSE -sDEVICE=$driver -sOutputFile=$out -");
                unlink($self->{'temp'});
        }

        my $format = uc $self->{'format'}; # DBX Notation are in uppercase
        my $file = basename ( $self->{'file'} );
        print qq{\n      <lire:image>\n};
        print qq{       <lire:file format="$format">$file</lire:file>\n};
        print qq{  </lire:image>};
}

sub handle_name {
        my ( $self, $name_rec ) = @_;

        return unless $self->{'write_chart'};
        if ($self->{'keep_only'}) {
                return unless $self->{'count'} < $self->{'keep_only'};
        }

        return unless $name_rec->{'col_info'}->name eq $self->{'x_name'};

        my $name = $name_rec->{'content'};
        # Work around strange ploticus "issue"
        $name =~ s/[^a-z0-9]$/$& /i;
        # Crop $name if necessary
        $name = substr( $name, 0, $self->{'max_label_len'} ) . "..."
          if length $name > $self->{'max_label_len'};

        if(defined $self->{'value'}) {
                my $val = $self->{'value'};
                my $fh = $self->{'pid'};
                print $fh ":\t$val\t$name\n";
                $self->{'value'} = undef;
                $self->{'count'}++;
        } else {
                $self->{'label'} = $name;
        }
}

sub handle_value {
        my ( $self, $val_rec ) = @_;

        return unless $self->{'write_chart'};
        if ($self->{'keep_only'}) {
                return unless $self->{'count'} < $self->{'keep_only'};
        }

        return unless $val_rec->{'col_info'}->name eq $self->{'y_name'};

        my $val = $val_rec->{'value'};

        if(defined $self->{'label'}) {
                my $name = $self->{'label'};
                my $fh = $self->{'pid'};
                print $fh ":\t$val\t$name\n";
                $self->{'label'} = undef;
                $self->{'count'}++;
        } else {
                $self->{'value'} = $val;
        }
}

# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::ReportParser::PloticusChartWriter - Lire::ReportParser processor that generate charts for all available subreports.

=head1 SYNOPSIS

        use Lire::ReportParser::PloticusChartWriter;

        my $parser = new Lire::ReportParser::PloticusChartWriter( 'format' => "png" );

        eval { $parser->parsefile( "report.xml" ) };
        print "Parse failed: $@" if $@;

=head1 DESCRIPTION

This is a Lire::ReportParser processor which will generate charts for
every table that has the C<charttype> attribute set. It outputs on STDOUT
a copy of the XML report with the appropriate C<image> and C<file> elements added.

A version of ploticus more recent than 2.0.4 is required.

Its constructor takes the following parameters:

=over 4

=item format

Selects the format of the charts. This can be one of C<png>, C<jpg>,
C<gif>, C<eps>, C<pdf>, C<svg>.
Defaults to C<png>. The ploticus program must support the requested format.

=item outputdir

The directory where the charts will be created. Defaults to F<.>.

=item graph_resolution

The resolution of the "high quality" graph. Defaults to 128.

=item max_label_length

The number of characters that should be kept from C<name> elements.
Defaults to 20.

=back

=head1 SEE ALSO

Lire::ReportParser(3pm)
ploticus(1)

=head1 VERSION

$Id: PloticusChartWriter.pm,v 1.20 2004/03/26 00:27:33 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHORS

Francis J. Lacoste <flacoste@logreport.org>
Wessel Dankers <wsl@logreport.org>

=cut
