package Lire::Timegroup;

use strict;

use base qw/ Lire::Aggregator /;

use Carp;
use POSIX qw/ setlocale strftime LC_TIME /;

use Lire::DataTypes qw( :special :time );
use Lire::Utils qw/ sql_quote_name check_param /;
use Lire::WeekCalculator;

=pod

=head1 NAME

Lire::Timegroup - Base class for implementation of the timegroup aggregator

=head1 SYNOPSIS

    use Lire::Timegroup;

=head1 DESCRIPTION

This module is the base class for implementation of the timegroup
aggregator. This aggregator will split the DLF records based on a time
period controlled throught the period attribute. For example, using 1d
as the period value, this aggregator will group all records in the
same day period together.

=head1 CONSTRUCTOR

=head2 new( %params )

Creates a new instance of a timegroup aggregator. In addition to the
normal report operator parameters, the timegroup aggregator can take
several parameters:

=over

=item field

This optional parameter contains the DLF field which contains the time
value used to group the DLF records together. See the field() method
for more information.

=item period

This mandatory parameter should contains the period's length that will
be used to group the records. See the period() method for more
information.

=back

=cut

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %params = @_;

    check_param( $params{'period'}, 'period' );

    my $self = bless {}, $class;
    $self->SUPER::init( %params, 'op' => "timegroup" );

    $self->field( $params{'field'} );
    $self->period( $params{'period'} );

    return $self;
}

=pod

=head1 METHODS

=head2 field( [$new_field] )

Returns the DLF field's name that is used to group the DLF records.
This should be a valid timestamp DLF field in the current schema. By
default, the default timestamp field of the DLF schema is used.

You can change the field by passing a $new_field parameter.

=cut

sub field {
    my ( $self, $field ) = @_;

    if (@_ == 2 ) {
	if ( defined $field ) {
	    croak "$field isn't a valid field for the ",
	      $self->report_spec->schema->id, " schema"
		unless $self->report_spec->schema->has_field( $field );

	    croak "$field isn't a timestamp field"
	      unless $self->report_spec->schema->field( $field )->type()
		eq "timestamp";

	} else {
	    $field = $self->report_spec->schema->timestamp_field->name();
	}
	$self->{'field'} = $field;
    }

    return $self->{'field'};
}

=pod

=head2 period( [$new_period])

Returns the period's length in which the records are grouped. This can
either be a duration value or the name of a report specification's
parameter containing a duration value.

The period's length can be changed by using the $new_period parameter.

=cut

sub period {
    my ( $self, $period ) = @_;

    if (@_ == 2 ) {
        check_param( $period, 'period' );

	if ( $period =~ /^\$/ ) {
	    my $name = substr $period, 1;
	    croak "parameter '$name' isn't defined"
	      unless $self->report_spec->has_param( $name );

            my $type = $self->report_spec->param( $name )->type();
	    croak "parameter '$name' isn't a 'duration' parameter: '$type'"
	      unless $type eq "duration";

	} else {
	    croak "'period' parameter isn't a valid duration: '$period'"
	      unless check_duration( $period );
	}

        my $value = $self->report_spec()->resolve_param_ref( $period );
        croak "can't use multiple with period of type 'd'"
          unless ( !daily_duration( $value ) || $value =~ m/^\s*1\s*d/ ); #1

	$self->{'period'} = $period;
    }

    return $self->{'period'};
}

# Implements Lire::ReportOperator::name
sub name {
    return "timegroup:" . $_[0]->{'field'};
}

# ------------------------------------------------------------------------
# Method create_categorical_info( $info )
#
# Implementation of the method required by Lire::Aggregator
sub create_categorical_info {
    my ( $self, $info ) = @_;

    my $dlf_field = $self->report_spec->schema->field( $self->field() );
    $info->create_column_info( $self->name(), 'categorical',
			       $dlf_field->type(), $self->label() );
}

# ------------------------------------------------------------------------
# Method xml_attrs( $info )
#
# Implementation of the method required by Lire::Aggregator
sub xml_attrs {
    my ( $self ) = @_;

    return qq{field="$self->{'field'}" period="$self->{'period'}"};
}

sub build_query {
    my ( $self, $query ) = @_;

    $self->SUPER::build_query( $query );

    my $period = $self->report_spec()->resolve_param_ref( $self->{'period'} );
    my ( $mult, $unit ) = $period =~ /(\d+)\s*(\w)/;
    my ($func, $param);
    if ( $unit eq 'y' ) {
        ( $func, $param ) = ('lr_timegroup_year', ",$mult" );
    } elsif ( $unit eq 'M' ) {
        ( $func, $param ) = ('lr_timegroup_month', ",$mult" );
    } elsif ( $unit eq 'w' ) {
        ( $func, $param ) = ('lr_timegroup_week', ",$mult" );
    } elsif ( $unit eq 'd' ) {
        ( $func, $param ) = ( 'lr_timegroup_day', '' );
    } else {
        $mult = duration2sec( $period );
        ( $func, $param ) = ('lr_timegroup_sec', ",$mult" );
    }

    $query->add_group_field( $self->name(),
                             sprintf( '%s(%s%s)', $func, 
                                      sql_quote_name( $self->{'field'} ),
                                      $param) );
    $query->set_sort_spec( $self->name() );

    return;
}

sub create_entry {
    my ( $self, $group, $row ) = @_;

    my $entry = $group->create_entry();
    my $value = $row->{ $self->name() };

    unless ( defined $value ) {
        $group->missing_cases( $row->{'_lr_nrecords'} );
        return undef;
    }

    my $period = $self->report_spec()->resolve_param_ref( $self->{'period'} );
    my $period_sec = duration2sec( $period );
    my ( $mult, $unit ) = $period =~ /(\d+)\s*(\w)/;

    my $fmt;
    if ( $unit eq 'y' ) {     $fmt = '%Y'; }
    elsif ( $unit eq 'M' ) {  $fmt = '%B %Y'; }
    elsif ( $unit eq 'w' ) {  $fmt = new Lire::WeekCalculator()->strformat(); }
    elsif ( $unit eq 'd' ) {  $fmt = '%Y-%m-%d'; }
    elsif ($self->_is_day_change( $group, $value ) ){ $fmt = '%Y-%m-%d %H:%M';}
    else {                    $fmt = '           %H:%M'; }

    my $old_locale = setlocale( LC_TIME, 'C' );
    $entry->add_name( strftime( $fmt, localtime $value), $value, $period_sec );
    setlocale( LC_TIME, $old_locale );

    return $entry;
}

sub _is_day_change {
    my ( $self, $group, $value ) = @_;

    my @entries = $group->entries();
    return 1 unless @entries > 1;

    # -1 is the entry currently being created
    my $name = $entries[-2]->data_by_name( $self->name() );

    my $new_day = (localtime $value)[3];
    my $old_day = (localtime $name->{'value'})[3];

    return $new_day != $old_day;
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::Group(3pm), Lire::ReportOperator(3pm),
Lire::Timeslot(3pm)

=head1 VERSION

$Id: Timegroup.pm,v 1.28 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <Wolfgang.Sourdeau@Contre.COM>

=cut
