package Lire::Timeslot;

use strict;

use base qw/ Lire::Aggregator /;

use Carp;
use POSIX qw/ setlocale strftime LC_TIME /;

use Lire::Aggregator;
use Lire::DataTypes qw/ :special :time /;
use Lire::Utils qw/ sql_quote_name /;
use Lire::WeekCalculator;

use vars qw/ @MONTHS @DAYS  %unit_index /;

BEGIN {
    @MONTHS = qw/January February March
                 April   May      June
                 July    August   September
                 October November December/;

    @DAYS =  qw/Sunday   Monday Tuesday  Wednesday
                Thursday Friday Saturday/;

    %unit_index = ( 'h' => 2,
                    'd' => 6,
                    'M' => 4,
                    'm' => 1,
                    's' => 0 );
}

=pod

=head1 NAME

Lire::Timegroup - Base class for implementation of the timeslot
aggregator

=head1 SYNOPSIS

    use Lire::Timeslot;

=head1 DESCRIPTION

This module is the base class for implementation of the timeslot
aggregator. This aggregator will split the DLF records based on a unit
of time. For example, if the unit is 1h, all DLF records occuring on
the same hour of day (independant of the day on which it occurs) are
grouped together.

=head1 CONSTRUCTOR

=head2 new( %params )

Creates a new instance of a timegroup aggregator. In addition to the
normal report operator parameters, the timegroup aggregator can take
several parameters:

=over

=item field

This optional parameter contains the DLF field which contains the time
value used to group the DLF records together. See the field() method
for more information.

=item unit

This mandatory parameter should contains the time unit that will be
used to group the records. See the unit() method for more information.

=back

=cut

sub new {
    my $proto = shift;
    my $class = ref( $proto) || $proto;

    my %params = @_;

    croak( "Missing unit parameter" )
      unless exists $params{'unit'};

    my $self = bless {}, $class;
    $self->SUPER::init( %params, 'op' => "timeslot" );

    $self->field( $params{'field'} );
    $self->unit( $params{'unit'} );

    return $self;
}

=pod

=head1 METHODS

=head2 field( [$new_field] )

Returns the DLF field's name that is used to group the DLF records.
This should be a valid timestamp DLF field in the current schema. By
default, the default timestamp field of the DLF schema is used.

You can change the field by passing a $new_field parameter.

=cut

sub field {
    my ( $self, $field ) = @_;

    if (@_ == 2 ) {
	if ( defined $field ) {
	    croak "$field isn't a valid field for the ",
	      $self->report_spec->schema->id, " schema"
		unless $self->report_spec->schema->has_field( $field );

	    croak "$field isn't a timestamp field"
	      unless $self->report_spec->schema()->field( $field )->type()
		eq "timestamp";

	} else {
	    $field = $self->report_spec->schema()->timestamp_field->name();
	}
	$self->{'field'} = $field;
    }

    return $self->{'field'};
}

=pod

=head2 unit( [$new_unit])

Returns the time unit that will be used to group the DLF records. The
unit is expressed as a duration value (for example 1s), but it should
be noted that this will group the records based on the second unit of
the DLF record's timestamp value. It could also be the name of a
report specification's parameter containing a duration value.

The time unit can be changed by using the $new_unit parameter.

=cut

sub unit {
    my ( $self, $unit ) = @_;

    if (@_ == 2 ) {
	my $unit_sec;
	if ( $unit =~ /^\$/ ) {
	    my $name = substr $unit, 1;
	    croak "$name isn't a defined parameter"
	      unless $self->report_spec->has_param( $name );

	    croak "$name parameter isn't of type duration"
	      unless $self->report_spec->param( $name )->type()
		eq "duration";
	} else {
	    croak "$unit isn't a valid duration"
	      unless check_duration( $unit );
	}

	$self->{'unit'} = $unit;
    }

    return $self->{'unit'};
}

# ------------------------------------------------------------------------
# Method xml_attrs()
#
# Implementation of the method required by Lire::Aggregator
sub xml_attrs {
    my ( $self ) = @_;

    return qq{field="$self->{'field'}" unit="$self->{'unit'}"};
}

# Implements Lire::ReportOperator::name()
sub name {
    return 'timeslot:' . $_[0]->{'field'};
}

# ------------------------------------------------------------------------
# Method create_categorical_info( $info )
#
# Implementation of the method required by Lire::Aggregator
sub create_categorical_info {
    my ( $self, $info ) = @_;

    my $dlf_field = $self->report_spec->schema->field( $self->field );
    $info->create_column_info( $self->name(), 'categorical',
                               $dlf_field->type(), $self->label() );
}

sub build_query {
    my ( $self, $query ) = @_;

    $self->SUPER::build_query( $query );

    my $unit = $self->report_spec()->resolve_param_ref( $self->{'unit'} );
    my ( $range, $unit_char ) = $unit =~ /(\d+)\s*(\w)/;
    my $func = ( $unit_char eq 'w' ) ? 'lr_timeslot_week' : 'lr_timeslot';
    my $param = ",$range";
    $param .= ",'$unit_char'" if ( $unit_char ne 'w' );

    $query->add_group_field( $self->name(),
                             sprintf( '%s(%s%s)', $func,
                                      sql_quote_name( $self->{'field'} ),
                                      $param) );
    $query->set_sort_spec(  $self->name() );

    return;
}

sub create_entry {
    my ( $self, $group, $row ) = @_;

    my $entry = $group->create_entry();
    my $slot = $row->{ $self->name() };

    unless ( defined $slot ) {
        $group->missing_cases( $row->{'_lr_nrecords'} );
        return undef;
    }

    my $unit = $self->report_spec->resolve_param_ref($self->{'unit'});

    my ( $range, $unit_char ) = $unit =~ /(\d+)\s*(\w)/;

    my $idx  = int( $slot / $range );
    my ($content, $value);
    if ( $unit_char eq 'M' ) {
        $value = "M$idx";
        $content = $MONTHS[$slot];
    } elsif ( $unit_char eq 'd' ) {
        $value = "D$idx";
        $content = $DAYS[$slot];
    } elsif ( $unit_char eq 'w' ) {
        $value = "W". int ( ( ($slot || 1) -1) / $range );
        my $fmt = Lire::Config->get( 'lr_week_numbering' ) eq 'ISO' ?
          '-W%02d' : 'Week %02d';
        $content = sprintf( $fmt, $slot );
    } else {
        $value = $unit_index{$unit_char} .'-' . $idx;
        my %unit_fmts = ( 'h' => '%.2d:00',
                          'm' => '00:%.2d',
                          's' => '00:00:%.2d' );
        $content = sprintf( $unit_fmts{$unit_char}, $slot );
    }

    $entry->add_name( $content, $value, $range );

    return $entry;
}

# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::Group(3pm), Lire::ReportOperator(3pm),
Lire::Timegroup(3pm)

=head1 VERSION

$Id: Timeslot.pm,v 1.22 2004/03/26 00:27:34 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <Wolfgang.Sourdeau@Contre.COM>

=cut
