package Lire::UI::ListWidget;

use strict;

use base qw/ Curses::UI::Container Lire::UI::Widget /;

use Carp;

use Curses::UI::Common;
use Locale::TextDomain 'lire';

use Lire::Utils qw/ check_object_param /;
use Lire::UI::Utils qw/ button_box_width text_for_width /;

use vars qw/@CARP_NOT/;

@CARP_NOT = qw/Curses::UI::Container/;

sub new {
    my $class = shift;
    my %userargs = @_;
    keys_to_lowercase(\%userargs);

    check_object_param( $userargs{'value'}, 'value', 'Lire::Config::List' );

    my $spec = $userargs{'value'}->spec();
    my @components = $spec->components();
    croak "'" . $spec->name() . "' contains more than one component"
      if ( scalar @components > 1 );

    my $self = $class->Curses::UI::Container::new( '-height' => 9,
                                                   %userargs,
                                                   '-releasefocus' => 1,
                                                 );
    my $component = $components[0];
    $self->{'component'} = $component;

    my $gap = ( $component->isa( 'Lire::Config::ScalarSpec' ) ? 3 : 1 );
    my $list_widget = $self->add( 'list', 'Listbox',
                                  '-onchange' => \&_value_change_cb,
                                  '-onselchange' => \&_sel_change_cb,
                                  '-vscrollbar' => 1,
                                  '-padright' => 1,
                                  '-border' => 1,
                                  '-selected' => 0,
                                  '-width' => $self->width(),
                                  '-height' => $self->height() - $gap);
    $self->refresh_view();

    my $selected = $list_widget->{'-selected'};
    my $new_value = ( defined $selected
                      ? $self->{'value'}->get( $selected )->clone()
                      : $component->instance() );
    $self->_add_buttons( $new_value );

    return $self;
}

sub _add_buttons {
    my ( $self, $value ) = @_;

    my $component = $self->{'component'};
    my $button_width = int( $self->canvaswidth() / 3 - 2 );
    my $add_label = text_for_width( __( '[Add]' ), $button_width );
    my $delete_label = text_for_width( __( '[Delete]' ),
                                       $button_width );

    my $middle_button;
    if ( $component->isa( 'Lire::Config::ScalarSpec' ) ) {
        $self->add( 'value_widget', 'Lire::UI::Widget',
                    'value' => $value,
                    '-width' => $self->width() - 3,
                    '-x' => 1, '-y' => -2, '-height' => 2 );
        my $label = text_for_width( __( '[Apply]' ), $button_width );
        $middle_button =  { '-label' => $label,
                            '-onpress' =>\&_apply_cb };
    } else {
        my $label = text_for_width( __( '[Edit]' ), $button_width );
        $middle_button =  { '-label' => $label,
                            '-onpress' =>\&_edit_cb };
    }

    $self->add( 'buttons', 'Buttonbox',
                '-releasefocus' => 1,
                '-buttons' =>  [ { '-label' => $add_label,
                                   '-onpress' =>\&_add_cb },
                                 $middle_button,
                                 { '-label' => $delete_label,
                                   '-onpress' =>\&_delete_cb }, ],
                '-x' => 1,
                '-y' => $self->height() - 1 );
}

sub refresh_view {
    my $self = $_[0];

    my $list_widget = $self->getobj( 'list' );
    my $component = $self->{'component'};

    my %labels = ();
    my $values = [];

    if ( $component->isa( 'Lire::Config::ScalarSpec' ) ) {
        $values = [ $self->{'value'}->elements() ];
        foreach my $scalar ( $self->{'value'}->elements() ) {
            my $scalar_value = $scalar->get();
            $labels{ $scalar } = ( defined $scalar_value
                                   ? $scalar_value
                                   : '' );
        }
    } elsif ( $component->isa( 'Lire::Config::CompoundSpec' ) ) {
        foreach my $scalar ( $self->{'value'}->elements() ) {
            push @$values, $scalar;
            $labels{ $scalar } = $scalar->as_label();
        }
    } else {
        croak "This code should never be reached.\n";
    }

    if ( scalar @$values == 0 ) {
        $values = [ __( '-- empty list --' ) ];
        $list_widget->{'-focusable'} = 0;
        $list_widget->{'-selected'} = undef;
    } else {
        $list_widget->{'-focusable'} = 1;
        $list_widget->{'-selected'} = $list_widget->{'-ypos'};
    }

    $list_widget->{'-values'} = $values;
    $list_widget->{'-labels'} = \%labels;
    $list_widget->draw();

    return;
}

sub _editing_title {
    my $self = $_[0];

    my $name = $self->{'component'}->name();
    $name =~ s/_/ /;

    return __x( 'Editing a {component}', component => $name );
}

sub _edit_compound {
    my ( $self, $compound ) = @_;

    my $win = $self->root();

    my $value = 0;

    my $name = "properties_dialog_$self";
    my $dlg = $win->add( $name, 'Curses::UI::Window',
                         '-title' => $self->_editing_title(),
                         '-border' => 1, '-pad' => 2, '-ipad' => 1 );
    my $widget = $dlg->add( 'widget', 'Lire::UI::CompoundWidget',
                            '-releasefocus' => 1,
                            'value' => $compound,
                            '-width' => $dlg->canvaswidth() - 1,
                            '-height' => $dlg->canvasheight() - 2 );

    my $buttons = [ {'-label' => __( '< Cancel >' ),
                     '-onpress' => sub { $dlg->loose_focus() } },
                    {'-label' => __( '< OK >' ),
                     '-onpress' => sub { $value = 1;
                                         $dlg->loose_focus() } } ];
    my $bb_width = button_box_width( $buttons );
    $dlg->add( 'buttons', 'Curses::UI::Buttonbox',
               '-buttons' => $buttons,
               '-x' => $dlg->canvaswidth() - $bb_width - 10,
               '-y' => $dlg->canvasheight() - 1,
               '-width' => $bb_width, '-height' => 1 );

    $dlg->modalfocus();
    $win->delete( $name );

    $self->root()->draw( 1 );
    $self->parent()->draw( 1 );

    return $value;
}

# callback functions
sub _add_cb {
    my $buttons = $_[0];

    my $self = $buttons->parent();
    my $list_widget = $self->getobj( 'list' );
    my $component = $self->{'component'};

    if ( $component->isa( 'Lire::Config::ScalarSpec' ) ) {
        my $value = $self->getobj( 'value_widget' )->{'value'};

        my $component = $self->{'component'};
        my $instance = $component->instance('value' => $value->get());
        $self->{'value'}->append( $instance );
        $list_widget->{'-ypos'} = scalar $self->{'value'}->elements() - 1;
        $self->refresh_view();
    } else {
        my $new_instance = $component->instance();
        if ( $self->_edit_compound( $new_instance ) ) {
            $self->{'value'}->append( $new_instance );
            $list_widget->{'-ypos'} = scalar $self->{'value'}->elements() - 1;
            $self->refresh_view();
        }
    }

    return;
}

sub _edit_cb {
    my $buttons = $_[0];

    my $self = $buttons->parent();
    my $list_widget = $self->getobj( 'list' );

    return unless $list_widget->{'-focusable'};

    my $instance = $list_widget->{'-values'}->[$list_widget->{'-selected'}];
    my $new_instance = $instance->clone();
    if ( $self->_edit_compound( $new_instance ) ) {
         foreach my $elem ( $instance->spec()->components() ) {
             $instance->set( $new_instance->get( $elem->name() ) );
         }

        $self->refresh_view();
    }

    return;
}

sub _delete_cb {
    my $buttons = $_[0];

    my $self = $buttons->parent();
    my $list_widget = $self->getobj( 'list' );
    return unless $list_widget->{'-focusable'};

    $self->{'value'}->remove( $list_widget->{'-ypos'} );

    my $nbr = scalar $self->{'value'}->elements() - 1;
    if ( $list_widget->{'-selected'} > $nbr ) {
        $list_widget->{'-ypos'} = $nbr;
    }

    $self->refresh_view();

    return;
}

sub _sel_change_cb {
    my $list_widget = $_[0];

    $list_widget->{'-selected'} = $list_widget->{'-ypos'};
    $list_widget->run_event( '-onchange' );

    return;
}

sub _value_change_cb {
    my $list_widget = $_[0];

    croak "'-selected' can never be undefined"
      unless defined $list_widget->{'-selected'};

    my $self = $list_widget->parent();
    return unless ( $self->{'component'}->isa('Lire::Config::ScalarSpec') );

    my $new_value = $self->{'value'}->get( $list_widget->{'-selected'} );
    my $value_widget = $list_widget->parent()->getobj( 'value_widget' );
    $value_widget->{'value'}->set( $new_value->get() );
    $value_widget->refresh_view();

    return;
}

sub _apply_cb {
    my $buttons = $_[0];

    my $self = $buttons->parent();
    my $list_widget = $self->getobj( 'list' );
    my $value_widget = $self->getobj( 'value_widget' );
    return unless defined $list_widget->{'-selected'};

    my $value = $self->{'value'}->get( $list_widget->{'-selected'} );
    $value->set( $value_widget->{'value'}->get() );

    $self->refresh_view();

    return;
}

1;
