package tests::ConfigSpecTest;

use strict;

use base qw/Test::Unit::TestSuite/;

use Lire::Config::TypeSpec;
use Carp;

sub name {
    return "Configuration Specification Tests";
}

sub include_tests {
    return qw/tests::ConfigListSpecTest
              tests::ConfigFileSpecTest tests::ConfigExecutableSpecTest
              tests::ConfigDirectorySpecTest tests::ConfigBooleanSpecTest
              tests::ConfigStringSpecTest tests::ConfigIntegerSpecTest
              tests::ConfigSelectSpecTest tests::ConfigOptionSpecTest
              tests::ConfigConfigSpecTest tests::ConfigCommandSpecTest
            /;
}

package tests::ConfigTypeSpecTest;

use base qw/Lire::Test::TestCase tests::TestStoreFixture /;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();
    $self->init_i18n();

    return $self;
}

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up();
    $self->set_up_locale();

    $self->{'spec'} = $self->type->new( 'name' => "test_spec",
                                        $self->additional_new_params(),
                                      );
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down();
    $self->tear_down_locale();

    # Since we aren't calling our parent's set_up, we shouldn't
    # call our parent's tear_down().
}

sub additional_new_params { return () }

sub test_new {
    my $self = $_[0];

    my $type = $self->type();
    my $spec = $type->new( 'name' => "parameter",
                           'summary' => "Parameter Summary",
                           'description' => "<para>Parameter Description</para>",
                           $self->additional_new_params(),
                         );
    $self->assert_not_null( $spec, "$type->new returned undef" );
    $self->assert( UNIVERSAL::isa( $spec, $type ),
                   "$type->new returned value of wrong type: $spec"
                 );
    $self->assert_str_equals( "parameter", $spec->name() );
    $self->assert_str_equals( "Parameter Summary", $spec->summary() );
    $self->assert_str_equals( "<para>Parameter Description</para>",
                              $spec->description() );

    $self->assert_died( sub { $type->new( $self->additional_new_params() ) },
                        qr/missing 'name' parameter/ );
    $self->assert_died( sub { $type->new( 'name' => 'wrong%name',
                                          $self->additional_new_params() ) },
                        qr/name should only contain/ );
}

sub test_instance {
    my $self = $_[0];

    my $value = $self->{'spec'}->instance();
    $self->assert_not_null( $value, $self->type() .
                            "->instance() returned undef" );
    $self->assert( UNIVERSAL::isa( $value, $self->value_type() ),
                   $self->type() . "->instance() returned value of wrong type: $value" );
}

sub test_is_instance {
    my $self = $_[0];

    my $value = $self->{'spec'}->instance();

    $self->assert( $self->{'spec'}->is_instance( $value ),
                   "is_instance() returned false",
                 );
    $self->assert_died( sub { $self->{'spec'}->is_instance() },
                        qr/missing \'instance\' parameter/ );
    $self->assert_died( sub { $self->{'spec'}->is_instance( $self->{'spec'}) },
                        qr/\'instance\' parameter should be a \'Lire::Config::Value\' instance, not/ );

    my $bad_value = $self->type->new( 'name'=> "bad_name",
                                      $self->additional_new_params(),
                                    )->instance();
    $self->assert( ! $self->{'spec'}->is_instance( $bad_value ),
                   "is_instance() with value with different name should return false"
                 );

    my $clone = $self->type->new( 'name' => $self->{'spec'}->name(),
                                  $self->additional_new_params(),
                                )->instance();
    $self->assert( ! $self->{'spec'}->is_instance( $clone ),
                   "is_instance() with different spec should die" );
}


sub test_summary {
    my $self = $_[0];

    $self->assert_str_equals( "test_spec", $self->{'spec'}->summary() );
    $self->{'spec'}->summary( 'Test Summary' );
    $self->assert_str_equals( "Test Summary", $self->{'spec'}{'summary'} );
    $self->assert_str_equals("Test Summary", $self->{'spec'}->summary(), );
}

sub test_description {
    my $self = $_[0];

    $self->assert_null( $self->{'spec'}->description,
                        "uninitialized description should be empty" );

    my $desc = "<para>Description</para>";
    $self->{'spec'}->description( $desc );
    $self->assert_equals( $desc, $self->{'spec'}->description );

    $self->assert_died( sub {$self->{'spec'}->description( "Description" ) },
                        qr/'description' parameter should be a DocBook/ );
}

sub test_summary_i18n {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    $spec->{'i18n_domain'} = 'lire-test';
    $spec->{'summary'} = 'JPEG Files';
    $self->assert_str_equals( 'JPEG Files', $spec->summary() );
    $self->set_locale( 'fr_CA' );
    $self->assert_str_equals( 'Fichiers JPEG', $spec->summary() );

}

sub test_description_i18n {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    $spec->{'description'} = '<para>JPEG Files</para>';
    $spec->{'i18n_domain'} = 'lire-test';
    $self->assert_str_equals( '<para>JPEG Files</para>',
                              $spec->description() );
    $self->set_locale( 'fr_CA' );
    $self->assert_str_equals( '<para>Fichiers JPEG</para>',
                              $spec->description() );
}

sub test_text_description {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    $spec->{'description'} = "<para>A parameter's description</para>";
    $self->assert_str_equals( "A parameter's description",
                              $spec->text_description() );

    $spec->{'description'} = <<EOT;
<para>A very very long multi-paragraphs text description for
this parameter. It is really long for a filled paragraph.</para>

      <para>This is yet another paragraph. This description never
ends.</para>

EOT
    my $expected = "A very very long multi-paragraphs text description for this parameter. It is really long for a filled paragraph.\n\nThis is yet another paragraph. This description never ends.";
    $self->assert_str_equals( $expected, $spec->text_description() );
}

package tests::ConfigScalarSpecTest;

use base qw/tests::ConfigTypeSpecTest/;

sub test_normalize {
    my $self = $_[0];

    $self->assert_null( $self->{'spec'}->normalize( undef ),
                        "normalize( undef ) should return undef",
                      );

    my %normalize_test_data = $self->normalize_test_data;
    while ( my ( $value, $expected ) = each %normalize_test_data ) {
        my $normalized = $self->{'spec'}->normalize( $value );
        if ( ! defined $expected ) {
            $self->assert_null( $normalized,
                                'expected undef from ' . ref( $self->{'spec'}) . "->normalize( $value ), got " . ($normalized || '<undef>') );
        } else {
            $self->assert_str_equals( $expected, $normalized );
        }
    }
}

sub test_is_valid {
    my $self = $_[0];

    my $warning = '';
    local $SIG{'__WARN__'} = sub { $warning .= join "", @_ };
    $self->assert( ! $self->{'spec'}->is_valid( undef ),
                   "is_valid( undef ) returned true" );
    $self->assert( ! $warning, "is_valid( undef ) had warnings: $warning" );
    foreach my $v ( $self->valid_test_data ) {
        $self->assert( $self->{'spec'}->is_valid( $v ),
                       $self->type . "->is_valid should return true for $v" );
        $self->assert( ! $warning, "is_valid( '$v' ) had warnings: $warning" );
    }

    foreach my $v ( $self->invalid_test_data ) {
        $self->assert( !$self->{'spec'}->is_valid( $v ),
                       $self->type . "->is_valid should return false for $v" );
        $self->assert( ! $warning, "is_valid( '$v' ) had warnings: $warning" );
    }
}

package tests::ConfigFileSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub type {
    return 'Lire::Config::FileSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            "/etc/passwd" => "/etc/passwd",
            "~/tmp" => "$ENV{'HOME'}/tmp",
           );
}

sub valid_test_data {
    return ( "/etc/passwd", __FILE__ );
}

sub invalid_test_data {
    return ( "~", "/no/such/file", "/dev/null" );
}

package tests::ConfigExecutableSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'old_PATH'} = $ENV{'PATH'};
    $ENV{'PATH'} = '/bin:/sbin';
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();


    $ENV{'PATH'} = defined $self->{'old_PATH'} ? $self->{'old_PATH'} : '';
}

sub type {
    return 'Lire::Config::ExecutableSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            "sh" => "/bin/sh",
            "/bin/sh" => "/bin/sh",
            "echo" => "/bin/echo",
            "bin/echo" => "bin/echo",
            "/./bin/echo" => "/bin/echo",
           );
}

sub valid_test_data {
    return ( "sh", "/bin/sh", "/bin//cat" );
}

sub invalid_test_data {
    return ( "/bin/nosuchfile", "/etc", "/dev/null", "/etc/passwd" );
}

package tests::ConfigDirectorySpecTest;

use base qw/tests::ConfigScalarSpecTest/;

use File::Basename;

sub type {
    return 'Lire::Config::DirectorySpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            "/etc/passwd" => "/etc/passwd",
            "/bin/./../" => "/bin/..",
            "~" => "$ENV{'HOME'}",
           );
}

sub valid_test_data {
    return ( ".", "/bin", dirname( __FILE__ ) . "/../../" );
}

sub invalid_test_data {
    return ( "/bin/nosuchfile", __FILE__, "/dev/null", "/etc/passwd" );
}

package tests::ConfigListSpecTest;

use base qw/tests::ConfigCompoundSpecTest/;

sub type {
    return 'Lire::Config::ListSpec';
}

sub value_type {
    return 'Lire::Config::List';
}

sub test_instance {
    shift->tests::ConfigTypeSpecTest::test_instance();
}

sub test_is_instance {
    shift->tests::ConfigTypeSpecTest::test_is_instance();
}

package tests::ConfigConfigSpecTest;

use base qw/tests::ConfigTypeSpecTest/;


sub test_components_by_section {
    my $self = $_[0];

    my $spec = $self->{'spec'};
    my $test1_spec = new Lire::Config::StringSpec( 'name' => 'test1',
                                                   'section' => 'section1' );
    my $test2_spec = new Lire::Config::StringSpec( 'name' => 'test2',
                                                   'section' => 'section2' );
    my $test3_spec = new Lire::Config::StringSpec( 'name' => 'test3',
                                                   'section' => 'section1' );
    $spec->add( $test1_spec );
    $spec->add( $test2_spec );
    $spec->add( $test3_spec );

    $self->assert_died( sub { $spec->components_by_section() },
                        qr/missing 'section' parameter/ );
    $self->assert_deep_equals( [ $test1_spec, $test3_spec ],
                               $spec->components_by_section( 'section1' ) );
    $self->assert_deep_equals( [ $test2_spec ],
                               $spec->components_by_section( 'section2' ) );
    $self->assert_deep_equals( [],
                               $spec->components_by_section( 'section3' ) );
}

sub type {
    return 'Lire::Config::ConfigSpec';
}

sub value_type {
    return 'Lire::Config::Dictionary';
}

sub test_new {
    my $self = $_[0];

    my $spec = $self->type->new();
    $self->assert_not_null( $spec, "new() returned undef" );
    $self->assert( UNIVERSAL::isa( $spec, $self->type ),
                   "new() returned value of wrong type: $spec"
                 );
}

package tests::ConfigStringSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub type {
    return 'Lire::Config::StringSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return ( 'test' => "test",
             "another string" => "another string",
           );
}

sub valid_test_data {
    return ( "test", "string", 10 );
}

sub invalid_test_data {
    return ();
}

package tests::ConfigIntegerSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub type {
    return 'Lire::Config::IntegerSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            0 => 0,
            -1_100_200 => -1100200,
            1 => 1,
           );
}

sub valid_test_data {
    return ( 0, 1, -1, 1_200_300, -1_100, "-10", "-1_000_000" );
}

sub invalid_test_data {
    return ( 0.1, "two" );
}

package tests::ConfigBooleanSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub type {
    return 'Lire::Config::BooleanSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            '' => '',
            '0' => '',
            '1' => 1,
            'enabled' => 1,
            'disabled' => '',
            'on' => 1,
            'off' => '',
            'Yes' => 1,
            'yes' => 1,
            'no' => '',
            'NO' => '',
            'F' => '',
            'FALSE' => '',
            'TRUE' => 1,
            't' => 1,
           );
}

sub valid_test_data {
    return ('', qw/0 1 enabled DISABLED Off oN Yes no True false t f/);
}

sub invalid_test_data {
    return qw/y n boolean/;
}

package tests::ConfigSelectSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

sub set_up {
    my $self = $_[0];

    $self->SUPER::set_up;

    $self->{'spec'}->add( new Lire::Config::OptionSpec( 'name' => "option_1" ) );
    $self->{'spec'}->add( new Lire::Config::OptionSpec( 'name' => "option_2" ) );
    $self->{'spec'}->add( new Lire::Config::OptionSpec( 'name' => "option_3" ) );
}

sub type {
    return "Lire::Config::SelectSpec";
}

sub value_type {
    return "Lire::Config::Scalar";
}

sub normalize_test_data {
    return (
            'option_1' => "option_1",
            'option_2' => "option_2",
            'option_3' => "option_3",
            'option_4' => undef,
            'OPTION_1' => "option_1",
            'OpTiOn_2' => "option_2",
           );
}

sub valid_test_data {
    return qw/option_1 Option_2 OPTION_3 option_3 option_2/;
}

sub invalid_test_data {
    return qw/option_4/;
}

sub test_add {
    my $self = $_[0];

    my $select = new Lire::Config::SelectSpec( 'name' => "select" );
    $self->assert_equals( 0, scalar $select->options );

    my $option_1 = new Lire::Config::OptionSpec( 'name' => "option_1" );
    $select->add( $option_1 );
    $self->assert_deep_equals( [ $option_1 ], [ $select->options ] );

    my $option_2 = new Lire::Config::OptionSpec( 'name' => "option_2" );
    $select->add( $option_2 );
    $self->assert_deep_equals( [ sort ($option_1, $option_2) ],
                               [ sort $select->options ] );

    $self->assert_died( sub { $select->add( undef ) },
                        qr/missing 'option' parameter/ );
#     $self->assert_died( sub { $select->add( new Lire::Config::OptionSpec( 'name' => "OPTION_1" ) ) },
#                         qr/already contains an option named 'option_1'/ );

    $self->assert_died( sub { $select->add( new Lire::Config::ListSpec( 'name' => "OPTION_1" ) ) },
                              qr/\'option\' parameter should be a \'Lire::Config::OptionSpec\' instance, not \'Lire::Config::ListSpec/ );
}

package tests::ConfigOptionSpecTest;

use base qw/tests::ConfigTypeSpecTest/;

sub type {
    return "Lire::Config::OptionSpec";
}

sub test_instance  {
    my $self = $_[0];

    $self->assert_died( sub { $self->{'spec'}->instance() },
                        qr/not implemented/ );
}

sub test_is_instance {
    my $self = $_[0];

    my $scalar = new Lire::Config::Scalar( 'spec' => new Lire::Config::ScalarSpec( name => "test" ) );
    $self->assert( !$self->{'spec'}->is_instance( $scalar ),
                   "is_instance() should return false" );
}

package tests::ConfigCommandSpecTest;

use base qw/tests::ConfigScalarSpecTest/;

use File::Basename;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'old_PATH'} = $ENV{'PATH'};
    $ENV{'PATH'} = '/bin:/sbin';
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();


    $ENV{'PATH'} = defined $self->{'old_PATH'} ? $self->{'old_PATH'} : '';
}

sub type {
    return 'Lire::Config::CommandSpec';
}

sub value_type {
    return 'Lire::Config::Scalar';
}

sub normalize_test_data {
    return (
            "//bin//sh" => "/bin/sh",
            "sh" => "/bin/sh",
           );
}

sub valid_test_data {
    return ( "sh", "/bin/sh", "ls", "cat" );
}

sub invalid_test_data {
    return ( "/bin/nosuchfile", __FILE__, "/dev/null" );
}

1;
