package tests::FilterExprTest;

use strict;

use base qw/Test::Unit::TestSuite/;

sub name { "Lire::FilterExpr Test Suite" }

sub include_tests {
    qw/ tests::FilterExprTest::Base
        tests::FilterExprTest::Binary
        tests::FilterExprTest::Eq
        tests::FilterExprTest::Ne
        tests::FilterExprTest::SQL
      /;
}

package tests::FilterExprTest::Base;

use base qw/Lire::Test::TestCase/;

use Lire::FilterExpr;
use Lire::ReportSpec;

use File::Basename;
use Cwd qw/realpath/;
use Carp;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->{'schemasdir'} = realpath( dirname(__FILE__) . "/schemas" );

    $self->{'container'} =  new Lire::ReportSpec();
    $self->{'container'}->id( 'test-filter' );
    $self->{'container'}->superservice( 'test' );
    $self->{'container'}->title( 'Test FilterExpr Report Specification' );
    $self->{'container'}->display_title( 'Test FilterExpr Report' );

    $self->{'container'}->param( "num_param", 
                                 new Lire::Param( 'name' => "num_param",
                                                  'type' => "number" ));

    $self->{'container'}->param( "str_param",
                                 new Lire::Param( 'name' => "str_param",
                                                  'type' => "string" ));

    $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->{'cfg'}{'lr_schemas_path'} = [ $self->{'schemasdir'} ];

    no strict 'refs';
    $self->{'expr'} = $self->filter_class->new( 'container'=> $self->{'container'},
                                                $self->good_new_params );
}

sub test_new {
    my $self = $_[0];

    no strict 'refs';

    my $pkg = $self->filter_class;

    $self->assert_died( sub { $pkg->new() },
                        qr/missing 'container' parameter/ );

    $self->assert_died( sub { $pkg->new( 'container'=> $self ) },
                        qr/\'container\' parameter should be a \'Lire::XMLSpecContainer\' instance, not \'tests\:\:/ );

    foreach my $bad_spec ( $self->bad_new_params() ) {
        $self->assert_died( sub { $pkg->new( 'container'=> $self->{'container'},
                                             @{$bad_spec->[0]} )},
                            $bad_spec->[1],
                          )
    }
    my $expr = $pkg->new( 'container'=> $self->{'container'},
                          $self->good_new_params(),
                        );

    $self->assert_not_null( $expr, "new() returned undef" );
    $self->assert( UNIVERSAL::isa( $expr, $pkg ),
                   "new() didn't returned a $pkg instance, $expr");
}

sub test_is_value_a_ref {
    my $self = $_[0];

    $self->assert( $self->{'expr'}->_is_value_a_ref( '$field' ),
                   "_is_value_a_ref( '\$field' ) returned false" );
    $self->assert( ! $self->{'expr'}->_is_value_a_ref( 'field' ),
                   "_is_value_a_ref( 'field' ) returned true" );
}

sub test_validate_ref_value {
    my $self = $_[0];

    $self->assert_died( sub { $self->{'expr'}->_validate_ref_value( "field" ) },
                        qr/value 'field' isn't a parameter or field reference/ );

    $self->assert_died( sub { $self->{'expr'}->_validate_ref_value( '$do_not_exists' ) },
                        qr/'do_not_exists' isn't a defined parameter or field name/ );

    $self->assert( $self->{'expr'}->_validate_ref_value( '$str_param' ),
                   '$str_param is a valid parameter'
                 );
    $self->assert( $self->{'expr'}->_validate_ref_value( '$time_start' ),
                   '$str_param is a valid schema field'
                 );
}

sub test_validate_value {
    my $self = $_[0];

    foreach my $good_value ( $self->good_values() ) {
        $self->assert( $self->{'expr'}->_validate_value( $good_value ),
                       "_validate_value() failed for '$good_value'" );
    }

    foreach my $bad_value ( $self->bad_values() ) {
        $self->assert_died( sub { $self->{'expr'}->_validate_value( $bad_value->[0] ) },
                            $bad_value->[1] );
    }

}

sub filter_class {
    return "Lire::FilterExpr";
}

sub good_new_params {
    return ();
}

sub bad_new_params {
    return ();
}

sub good_values {
    return ( "" );
}

sub bad_values {
    return ( [ '$do_not_exists', qr/defined parameter or field name/ ] );
}

package tests::FilterExprTest::Binary;

use base qw/tests::FilterExprTest::Base/;

sub filter_class {
    return "Lire::FilterExpr::BinaryExpr";
}

sub bad_new_params {
    return (
            [ [], qr/missing 'arg1' parameter/ ], #']
            [ [ 'arg1' => "", ], qr/missing 'arg2' parameter/ ],
            [ [ 'arg1' => "", 'arg2' => "", ], qr/missing 'op' parameter/ ],
            [ [ 'arg1' => "", 'arg2' => "", 'op' => 'op' ],
              qr/missing 'sql_op' parameter/ ],
           )
}

sub good_new_params {
    return (
            'arg1' => "",
            'arg2' => "",
            'sql_op' => 'op',
            'op' => "op",
           );
}

sub test_arg12 {
    my $self = $_[0];

    my %args = $self->good_new_params();
    my $expr = $self->{'expr'};

    $self->assert_died( sub { $expr->arg1( undef ) },
                        qr/missing 'arg1' parameter/ ); #')

    $self->assert_died( sub { $expr->arg2( undef ) },
                        qr/missing 'arg2' parameter/ );

    $self->assert_equals( $args{'op'}, $expr->{'op'} );
    $self->assert_equals( $args{'arg1'}, $expr->arg1 );
    $self->assert_equals( $args{'arg2'}, $expr->arg2 );

    $expr->arg1( 1 );
    $self->assert_equals( 1, $expr->arg1 );
    $expr->arg2( 2 );
    $self->assert_equals( 2, $expr->arg2 );
}

package tests::FilterExprTest::Eq;

use base qw/tests::FilterExprTest::Binary/;

sub filter_class {
    return "Lire::FilterExpr::Eq";
}

sub good_new_params {
    return ( 'arg1' => "", 'arg2' => "", 'op' => "eq" );
}

sub bad_new_params {
    return (shift->SUPER::bad_new_params())[0,1];
}

package tests::FilterExprTest::Ne;

use base qw/tests::FilterExprTest::Binary/;

sub filter_class {
    return "Lire::FilterExpr::Ne";
}

sub good_new_params {
    return ( 'arg1' => "", 'arg2' => "", 'op' => "ne" );
}

sub bad_new_params {
    return (shift->SUPER::bad_new_params())[0,1];
}

package tests::FilterExprTest::SQL;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::FilterExpr;
use Lire::ReportSpec;
use Test::Unit::TestSuite;
use Class::Inner;

sub suite {
    my $suite =
      Test::Unit::TestSuite->empty_new( 'FilterExpr::SQL Test Suite' );

    my $eq_expr =
      new Class::Inner( 'parent' => 'Lire::FilterExpr',
                        'methods' => 
                        {
                         'new' => sub { bless {}, shift; },
                         'sql_expr' => sub { 'client_host = ?' },
                         'sql_params' => sub { [ 'logreport.org' ] },
                        } );
    my $gt_expr =
      new Class::Inner( 'parent' => 'Lire::FilterExpr',
                        'methods' => 
                        {
                         'new' => sub { bless {}, shift; },
                         'sql_expr' => sub { 'file_size > ?' },
                         'sql_params' => sub { [ 102400 ] },
                        } );

    # Each test is an array with the elements:
    # 0: Test_name
    # 1: Filter class
    # 2: new() parameters
    # 3: expected result of sql_expr()
    # 4: expected result of sql_params()
    my @tests =
      (
       [ 'Ne_value',
         'Lire::FilterExpr::Ne',
         ['arg1' => '$client_host', 'arg2' => 'test.com' ],
         'client_host != ?',
         [ 'test.com' ] ],
       [ 'Ne_param',
         'Lire::FilterExpr::Ne',
         ['arg1' => '$client_host', 'arg2' => '$str_param' ],
         'client_host != ?',
         [ 'a string' ] ],
       [ 'Ne_quote',
         'Lire::FilterExpr::Ne',
         ['arg1' => '$transfer-complete', 'arg2' => 'yes' ],
         '"transfer-complete" != ?',
         [ 'yes' ] ],
       [ 'Eq_value',
         'Lire::FilterExpr::Eq',
         [ 'arg1' => '$client_host', 'arg2' => 'logreport.org' ],
         'client_host = ?',
         [ 'logreport.org' ] ],
       [ 'Eq_param',
         'Lire::FilterExpr::Eq',
         [ 'arg1' => '$client_host', 'arg2' => '$str_param' ],
         'client_host = ?',
         [ 'a string' ] ],
       [ 'Gt_value',
         'Lire::FilterExpr::Gt',
         [ 'arg1' => 1024, 'arg2' => '$file_size' ],
         '? > file_size',
         [ 1024 ] ],
       [ 'Gt_param',
         'Lire::FilterExpr::Gt',
         [ 'arg1' => '$file_size', 'arg2' => '$num_param' ],
         'file_size > ?',
         [ 5 ] ],
       [ 'Ge_value',
         'Lire::FilterExpr::Ge',
         [ 'arg1' => 1024, 'arg2' => '$file_size' ],
         '? >= file_size',
         [ 1024 ] ],
       [ 'Ge_param',
         'Lire::FilterExpr::Ge',
         [ 'arg1' => '$file_size', 'arg2' => '$num_param' ],
         'file_size >= ?',
         [ 5 ] ],
       [ 'Lt_value',
         'Lire::FilterExpr::Lt',
         [ 'arg1' => 1024, 'arg2' => '$file_size' ],
         '? < file_size',
         [ 1024 ] ],
       [ 'Lt_param',
         'Lire::FilterExpr::Lt',
         [ 'arg1' => '$file_size', 'arg2' => '$num_param' ],
         'file_size < ?',
         [ 5 ] ],
       [ 'Le_value',
         'Lire::FilterExpr::Le',
         [ 'arg1' => 1024, 'arg2' => '$file_size' ],
         '? <= file_size',
         [ 1024 ] ],
       [ 'Le_param',
         'Lire::FilterExpr::Le',
         [ 'arg1' => '$file_size', 'arg2' => '$num_param' ],
         'file_size <= ?',
         [ 5 ] ],
       [ 'Match_value_re',
         'Lire::FilterExpr::Match',
         [ 're' => '^test\\.com', 'value' => '$client_host' ],
         'lr_match(client_host,?,0)',
         [ '^test\\.com' ] ],
       [ 'Match_value',
         'Lire::FilterExpr::Match',
         [ 're' => '^test\\.com', 'value' => '$str_param' ],
         'lr_match(?,?,0)',
         [ 'a string', '^test\\.com' ] ],
       [ 'Match_param',
         'Lire::FilterExpr::Match',
         [ 're' => '$str_param', 'value' => '$client_host',
           'case-sensitive' => 1 ],
         'lr_match(client_host,?,1)',
         [ 'a string' ] ],
       [ 'Match_quote',
         'Lire::FilterExpr::Match',
         ['value' => '$transfer-complete', 're' => 'yes' ],
         'lr_match("transfer-complete",?,0)',
         [ 'yes' ] ],
       [ 'Value_value',
         'Lire::FilterExpr::Value',
         [ 'value' => 'a test' ],
         '? IS NOT NULL AND ? != 0 AND LENGTH(?) > 0',
         [ 'a test', 'a test', 'a test' ] ],
       [ 'Value_param',
         'Lire::FilterExpr::Value',
         [ 'value' => '$str_param' ],
         '? IS NOT NULL AND ? != 0 AND LENGTH(?) > 0',
         [ 'a string', 'a string', 'a string' ] ],
       [ 'Value_field',
         'Lire::FilterExpr::Value',
         [ 'value' => '$client_host' ],
         'client_host IS NOT NULL AND client_host != 0 AND LENGTH(client_host) > 0',
         [] ],
       [ 'Value_quote',
         'Lire::FilterExpr::Value',
         [ 'value' => '$transfer-complete' ],
         '"transfer-complete" IS NOT NULL AND "transfer-complete" != 0 AND LENGTH("transfer-complete") > 0',
         [] ],
       [ 'Not_expr',
         'Lire::FilterExpr::Not',
         [ 'expr' => $eq_expr ],
         'NOT (client_host = ?)',
         [ 'logreport.org' ] ],
       [ 'And_expr',
         'Lire::FilterExpr::And',
         [ 'expr' => [ $eq_expr, $gt_expr, $eq_expr ] ],
         '(client_host = ?) AND (file_size > ?) AND (client_host = ?)',
         [ 'logreport.org', 102400, 'logreport.org' ] ],
       [ 'Or_expr',
         'Lire::FilterExpr::Or',
         [ 'expr' => [ $gt_expr, $eq_expr, $gt_expr ] ],
         '(file_size > ?) OR (client_host = ?) OR (file_size > ?)',
         [ 102400, 'logreport.org', 102400 ] ],
      );

    foreach my $test ( @tests ) {
        $suite->add_test( new tests::FilterExprTest::SQL( @$test,
                                                          'test_sql_expr' ) );
        $suite->add_test( new tests::FilterExprTest::SQL( @$test,
                                                          'test_sql_params' ));
    }
    return $suite;
}

sub new {
    my $self = shift()->SUPER::new( shift() . "::$_[4]" );

    my ($filter_class, $new_params, $sql_expr, $sql_params, $test_method) = @_;

    $self->init();

    $self->{'filter_class'} = $filter_class;
    $self->{'new_params'} = $new_params;
    $self->{'sql_expr'} = $sql_expr;
    $self->{'sql_params'}=  $sql_params;
    $self->{'test_method'} = $test_method;
    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    my $spec =  new Lire::ReportSpec();
    $spec->superservice( 'test' );
    $spec->param( "num_param", new Lire::Param( 'name' => "num_param",
                                                'type' => "number",
                                                'default' => 5 ));

    $spec->param( "str_param", new Lire::Param( 'name' => "str_param",
                                                'type' => "string",
                                                'default' => 'a string' ));

    $self->{'filter_expr'} =
      $self->{'filter_class'}->new( 'container' => $spec,
                                    @{$self->{'new_params'}} );

    my %params = @{$self->{'new_params'}};
    if ( $params{'expr'} ) {
        $self->{'filter_expr'}->expr( $params{'expr'} );
    }
    return;
}

sub run_test {
    my $self = $_[0];

    my $method = $self->{'test_method'};
    $self->$method();
}

sub test_sql_expr {
    my $self = $_[0];

    $self->assert_str_equals( $self->{'sql_expr'},
                              $self->{'filter_expr'}->sql_expr() );
}

sub test_sql_params {
    my $self = $_[0];

    $self->assert_deep_equals( $self->{'sql_params'},
                               $self->{'filter_expr'}->sql_params() );
}

1;
