package tests::ReportScheduleTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture /;

use Lire::ReportSchedule;
use Time::Local;
use Class::Inner;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    $self->{'spec'} = $self->lire_default_config_spec();

    $self->{'test_cfg'} = $self->{'testdir'} . "/data/test.cfg";
    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();
    $self->set_up_test_store();
    $self->{'cfg'}{'_lr_config_spec'} = $self->{'spec'};
    $self->{'cfg'}{'lr_week_numbering'} = 'U';
    $self->{'cfg'}{'lr_reports_path'} = [ "$self->{'testdir'}/reports" ];
    $self->{'cfg'}{'lr_filters_path'} = [ "$self->{'testdir'}/filters" ];

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    $self->tear_down_test_store();

    return;
}

sub test_new {
    my $self = $_[0];

    my $sched = new Lire::ReportSchedule( "daily", "$self->{'test_cfg'}" );
    $self->assert_isa( 'Lire::ReportSchedule', $sched );

    $self->assert_str_equals( 'daily', $sched->{'_period'} );
    $self->assert_str_equals( 'daily', $sched->period() );
    $self->assert_str_equals( $self->{'test_cfg'}, $sched->{'_cfg_file'} );
    $self->assert_str_equals( $self->{'test_cfg'}, $sched->config_file() );
    $self->assert_deep_equals( [], $sched->{'_output_jobs'} );
    $self->assert_deep_equals( [], [ $sched->output_jobs() ] );
    $self->assert_dies( qr/'period' parameter should be one of 'hourly', 'daily', 'weekly', 'monthly' or 'yearly': /,
                        sub { new Lire::ReportSchedule( 'aTest', $self->{'test_cfg'} ) } );
}

sub test_add_output_job {
    my $self = $_[0];

    my $sched = new Lire::ReportSchedule( "daily", $self->{'test_cfg'} );
    my $job = bless {}, 'Lire::OutputJob';

    $self->assert_dies( qr/missing one or more 'output_job' parameters/,
                        sub { $sched->add_output_job() } );
    $sched->add_output_job( $job, $job );
    $self->assert_deep_equals( [ $job, $job ],
                               $sched->{'_output_jobs'} );
}

sub test_new_from_config {
    my $self = $_[0];

    my $job = $self->{'spec'}->get( 'report_jobs' )->get( 'report_job' )->get( 'schedules' )->get( 'schedule' )->instance();
    $job->get( 'period' )->set( 'daily' );
    $job->get( 'report_cfg' )->set( "$self->{'testdir'}/data/test.cfg" );
    my $output = $job->get( 'output_jobs' )->spec()->get( 'output_job' )->instance();
    $output->get( 'destination' )->set_plugin( 'file' );
    $output->get( 'destination' )->get( 'file' )->set( 'report.txt' );
    $output->get( 'format' )->set_plugin( 'txt' );
    $job->get( 'output_jobs' )->append( $output );

    my $jobObject = $job->as_value();
    $self->assert_isa( 'Lire::ReportSchedule', $jobObject );
    $self->assert_str_equals( 'daily', $jobObject->{'_period'} );
    $self->assert_str_equals( "$self->{'testdir'}/data/test.cfg",
                              $jobObject->{'_cfg_file'} );
    $self->assert_num_equals( 1, scalar @{$jobObject->{'_output_jobs'}} );
}

sub test_period_range {
    my $self = $_[0];

    my $dec31_2004 = timelocal( 0, 5, 23, 31, 11, 2004 );
    my $sched = new Lire::ReportSchedule( 'hourly', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 23, 31, 11, 2004 ),
                                 timelocal( 0, 0, 0, 1, 0, 2005 ) ],
                               $sched->period_range( $dec31_2004 ) );

    $sched = new Lire::ReportSchedule( 'daily', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 0, 31, 11, 2004 ),
                                 timelocal( 0, 0, 0, 1, 0, 2005 ) ],
                               $sched->period_range( $dec31_2004 ) );

    $sched = new Lire::ReportSchedule( 'monthly', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 0, 1, 11, 2004 ),
                                 timelocal( 0, 0, 0, 1, 0, 2005 ) ],
                               $sched->period_range( $dec31_2004 ) );

    my $jan1_2004 = timelocal( 0, 0, 0, 1, 0, 2004 );
    $sched = new Lire::ReportSchedule( 'monthly', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 0, 1, 0, 2004 ),
                                 timelocal( 0, 0, 0, 1, 1, 2004 ) ],
                               $sched->period_range( $jan1_2004 ) );

    $sched = new Lire::ReportSchedule( 'yearly', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 0, 1, 0, 2004 ),
                                 timelocal( 0, 0, 0, 1, 0, 2005 ) ],
                               $sched->period_range( $dec31_2004 ) );

    $sched = new Lire::ReportSchedule( 'weekly', 'report.cfg' );
    $self->assert_deep_equals( [ timelocal( 0, 0, 0, 26, 11, 2004 ),
                                 timelocal( 0, 0, 0, 2, 0, 2005 ) ],
                               $sched->period_range( $dec31_2004 ) );
}

sub test_filter_title {
    my $self = $_[0];

    my $sched = new Lire::ReportSchedule( 'hourly', 'report.cfg' );
    my $mar11_2004 = timelocal( 0, 0, 11, 11, 2, 2004 );

    $self->assert_str_equals( 'DLF records for 2004-03-11 11:00.',
                              $sched->_filter_title( $mar11_2004 ) );
    $sched->{'_period'} = 'daily';
    $self->assert_str_equals( 'DLF records for 2004-03-11.',
                              $sched->_filter_title( $mar11_2004 ) );
    $sched->{'_period'} = 'weekly';
    $self->assert_str_equals( 'DLF records for Week 10, 2004.',
                              $sched->_filter_title( $mar11_2004 ) );
    $sched->{'_period'} = 'monthly';
    $self->assert_str_equals( 'DLF records for 2004-03.',
                              $sched->_filter_title( $mar11_2004 ) );
    $sched->{'_period'} = 'yearly';
    $self->assert_str_equals( 'DLF records for 2004.',
                              $sched->_filter_title( $mar11_2004 ) );
}

sub test_init_report_config {
    my $self = $_[0];

    my $sched = new Lire::ReportSchedule( 'daily', 
                                          "$self->{'testdir'}/data/test.cfg" );

    my $mar11_2004 = timelocal( 0, 0, 0, 11, 2, 2004 );
    my $cfg = $sched->init_report_config( 'test', 0, $mar11_2004 );
    $self->assert_isa( 'Lire::ReportConfig', $cfg );
    $self->assert_isa( 'Lire::ReportSpecFactory', $cfg->{'factory'} );
    foreach my $sect ( $cfg->sections() ) {
        $self->annotate( "Checking section '" . $sect->title() . "'..." );
        my @filters = $sect->filters();
        $self->assert_isa( 'Lire::FilterSpec', $filters[-1] );

        $self->assert_str_equals( 'DLF records for 2004-03-11.',
                                  $filters[-1]->display_title() );
        my $expr = $filters[-1]->filter_spec();
        $self->assert_isa( 'Lire::FilterExpr::And', $expr );

        my $e_field = ( $sect->title() eq 'Sessions'
                        ? '$session_start' : '$time_start' );
        my ($start, $end) = @{ $expr->expr() };
        $self->assert_isa( 'Lire::FilterExpr::Ge', $start );
        $self->assert_str_equals( $e_field, $start->arg1() );
        $self->assert_num_equals( $mar11_2004, $start->arg2() );

        $self->assert_isa( 'Lire::FilterExpr::Lt', $end );
        $self->assert_str_equals( $e_field, $end->arg1() );
        $self->assert_num_equals( $mar11_2004 + 86400, $end->arg2() );

        $self->annotate( "ok\n" );
    }

    $cfg = $sched->init_report_config( 'test', 1 );
    $self->assert_isa( 'Lire::ReportConfig', $cfg );
    $self->assert_isa( 'Lire::AsciiDlf::AsciiDlfFactory', $cfg->{'factory'} );
}

1;
