package tests::ReportSpecTest;

use strict;

use base qw/ Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::ReportSpec;
use Lire::DlfSchema;
use Lire::FilterExpr;
use Lire::Param;
use Lire::Group;
use Lire::GroupField;
use Lire::Aggregate;
use Lire::Report;
use Lire::Utils qw/tempdir/;

use tests::MockAggregator;
use tests::MockAggregate;

use Class::Inner;

use File::Basename qw/dirname/;
use Cwd qw/realpath/;
use Carp;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->init();

    $self->{'dir'}    = realpath( dirname(__FILE__) );

    $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_store();

    $self->{'tmp_test_spec_dir'} = $self->{'tmpdir'} . "/test";
    mkdir $self->{'tmp_test_spec_dir'}
      or croak "failed to make $self->{'tmp_test_spec_dir'}: $!\n";

    $self->{'cfg'}{'lr_reports_path'} = [ $self->{'dir'} . "/reports",
                                      $self->{'tmpdir'} ];

    $self->set_up_simple_report_spec();
    return $self;
}

sub test_non_existent_report {
    my $self = $_[0];

    my $spec = eval { Lire::ReportSpec->load( "test", "non-existent" ) };
    $self->assert_not_null( $@, "load() should have thrown an exception" );
}

sub test_load_report {
    my $self = $_[0];


    my $spec = Lire::ReportSpec->load( "test", "top-files" );
    $self->assert_equals( "top-files", $spec->id );
}

sub create_test_report_spec {
    my $self = $_[0];

    my $spec = new Lire::ReportSpec();
    $spec->id( "test-report" );
    $spec->superservice( "test" );
    $spec->title( "Test Report Specification" );
    $spec->description( "<para>Test Report Specification</para>" );
    $spec->display_title( "Test Report" );
    $spec->param( "param_int", new Lire::Param( 'name' => "param_int",
                                                'type' => "int",
                                                'i18n_domain' => 'lire-test',
                                              ) );
    $spec->param( "param_int" )->default( 10 );
    $spec->param( "param_match", new Lire::Param( 'name' => "param_match",
                                                  'type' => "string",
                                                  'i18n_domain' => 'lire-test',
                                                ) );
    $spec->param( "param_match" )->default( ".*" );

    $spec->filter_spec( new Lire::FilterExpr::Match( 'container' => $spec,
                                                     'value' => '$file',
                                                     're' => '$param_match',
                                                   )
                      );

    my $group = new Lire::Group( 'report_spec' => $spec,
                                'limit' => '$param_int', );

    $group->group_fields( [ new Lire::GroupField( 'report_spec' => $spec,
                                                  'name' => "file",
                                                  'i18n_domain' => 'lire-test',
                                                ) ] );
    $group->ops( [ new Lire::Count( 'report_spec' => $spec,
                                    'parent' => $group,
                                    'name' => "downloads",
                                    'label' => "Downloads",
                                  ) ] );
    $group->sort_fields( [ "-downloads" ] );
    $spec->calc_spec( $group );

    $spec;
}

sub set_up_simple_report_spec {
    my $self = $_[0];

    $self->{'simple_spec'} = new Lire::ReportSpec();
    $self->{'simple_spec'}->superservice( 'test' );
    $self->{'simple_spec'}->id( 'test-spec' );
    $self->{'simple_spec'}->param( 'param1',
                                   new Lire::Param( 'name' => 'param1',
                                                    'type' => 'int',
                                                    'i18n_domain' => 'lire-test',
                                                    'default' => 5 ) );
    $self->{'simple_spec'}->charttype( 'bars' );
    $self->{'simple_spec'}->display_title( 'Top $param1 report' );
    $self->{'simple_spec'}->display_description( '<para>$param1</para>' );
    my $aggr = 
      new Class::Inner( 'parent' => 'tests::MockAggregator',
                        'args' => [ 'report_spec' => $self->{'simple_spec'} ],
                        'methods' =>
                        {
                         'create_entries' => sub {
                             # Create two empty entries
                             $_[1]->create_entry();
                             $_[1]->create_entry();
                         },
                         } );
    $aggr->ops( [ new tests::MockAggregate( 'report_spec' => $self->{'simple_spec'},
                                            'parent' => $aggr,
                                            'name' => 'aggr' ) ] );
    $self->{'simple_spec'}->calc_spec( $aggr );
    return;
}

sub test_print {
    my $self = $_[0];

    my $spec = $self->create_test_report_spec();

    my $specfile = "$self->{'tmp_test_spec_dir'}/test-report.xml";
    open my $fh, "> $specfile"
      or croak "failed to open $specfile for writing: $!\n";
    $spec->print( $fh );
    close $fh
      or croak "close failed: $!\n";

    my $spec2 = Lire::ReportSpec->load( $spec->superservice(), $spec->id() );

    $self->assert_deep_equals( $spec, $spec2 );
}

sub test_create_subreport {
    my $self = $_[0];

    my $report = new Lire::Report( 'test', 0, 1);

    my $spec = $self->{'simple_spec'};
    $self->assert_died( sub { $spec->create_subreport() },
                        qr/missing 'report' parameter/ );

    $self->assert_died( sub { $spec->create_subreport( {} ) },
                        qr/\'report\' parameter should be a \'Lire::Report\' instance, not \'HASH/ );

    my $subreport = $spec->create_subreport( $report );
    $self->assert_not_null( $subreport, 'create_subreport() returned undef' );
    $self->assert_isa( 'Lire::Report::Subreport', $subreport );
    $self->assert_equals( 'test-spec', $subreport->type() );
    $self->assert_equals( 'bars', $subreport->charttype() );
    $self->assert_equals( 'Top 5 report', $subreport->title() );
    $self->assert_equals( '<para>5</para>', $subreport->description() );

    $self->assert_equals( '<para>5</para>', $subreport->description() );
    $self->assert_equals( 2, scalar $subreport->entries() );
    $self->assert_equals( 1, ($subreport->entries())[1]->row_idx() );
    $self->assert_not_null( $subreport->table_info(),
                            "subreport's table_info() returned undef" );
    $self->assert_not_null( $subreport->table_info()->column_info_by_name( 'aggr' )->col_start(),
                            "aggr's col_start() returned undef, was finalize() called?" );
}

sub test_create_subreport_missing {
    my $self = $_[0];

    my $spec = $self->{'simple_spec'};
    $spec->mark_missing( 'Bogus reason' );

    my $report = new Lire::Report( 'test', 0, 1);
    my $subreport = $spec->create_subreport( $report );
    $self->assert( UNIVERSAL::isa( $subreport, 'Lire::Report::Subreport' ),
                   "create_subreport() didn't return a Lire::Report::Subreport: $subreport" );
    $self->assert_equals( 'test-spec', $subreport->type() );
    $self->assert( $subreport->is_missing(),
                   'is_missing() should return true' );
    $self->assert_equals( 'Bogus reason', $subreport->missing_reason() );
}

sub test_set_store {
    my $self = $_[0];

    my $spec = new Lire::ReportSpec();
    $spec->id( 'test-spec' );
    $spec->{'superservice'} = 'test-extended';
    my $aggr = new tests::MockAggregator( 'report_spec' => $spec );
    $spec->calc_spec( $aggr );

    $self->assert_dies( qr/missing \'store\' parameter/,
                        sub { $spec->set_store() } );
    $self->assert_dies( qr/\'store\' parameter should be a \'Lire::DlfStore\' instance, not \'HASH/,
                        sub { $spec->set_store( {} ) } );
    $self->assert_dies( qr/store doesn't contain a 'test-extended' DLF stream/,
                        sub { $spec->set_store( $self->{'store'} ) }  );

    $spec->{'superservice'} = 'test';
    $spec->set_store( $self->{'store'} );
    $self->assert_equals( $self->{'store'}, $aggr->{'store'} );
}

1;
