package tests::SimpleStatTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::ReportSpec;
use tests::MockAggregator;

use Class::Inner;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = shift->SUPER::set_up();

    $self->set_up_test_schema();

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'test-simplestat' );

    $self->{'mock_aggr'} =
      new tests::MockAggregator( 'report_spec' => $self->{'spec'} );

    $self->{'mock_stat'} = bless ( {}, 'Lire::SimpleStat' );
    $self->{'mock_stat'}->init( 'report_spec' => $self->{'spec'},
                                'name' => 'statTest',
                                'parent' => $self->{'mock_aggr'},
                                'field' => 'time_taken',
                                'op' => 'mock',
                                'label' => 'Time Taken' );

    $self->{'cfg'}{'lr_scale_numbers'} = 0;
    $self->{'cfg'}{'lr_scale_bytes'} = 1;
    $self->{'cfg'}{'lr_scale_seconds'} = 1;

    return  $self;
}

sub test_init {
    my $self = $_[0];

    $self->assert_equals( 'time_taken',
                          $self->{'mock_stat'}{'field'} );

    my %args = ( 'report_spec' => $self->{'spec'},
                 'name' => 'statTest',
                 'parent' => $self->{'mock_aggr'},
                 'op' => 'mock' );
    $self->assert_died( sub { $self->{'mock_stat'}->init( %args ) },
                        qr/missing 'field' parameter/ );

    $args{'field'} = 'connection_id';
    $self->assert_died( sub { $self->{'mock_stat'}->init( %args ) },
                        qr/'field' parameter should be a numerical field: 'connection_id'/ );
}

sub test_dlf_field {
    my $self = $_[0];

    my $field = $self->{'mock_stat'}->dlf_field();
    $self->assert_not_null( $field, "dlf_field() returned undef" );
    $self->assert( UNIVERSAL::isa( $field, 'Lire::Field' ),
                   "dlf_field() should have returned a Lire::Field instance, not $field" );
    $self->assert_equals( 'time_taken', $field->name() );
}

sub test_create_numerical_info {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'mock_stat'}->create_numerical_info( $info );
    my $col_info = $info->info_by_index( 0 );
    $self->assert_not_null( $col_info, 'create_numerical_info() should have create a ColumnInfo' );
    $self->assert_equals( 'statTest', $col_info->name() );
    $self->assert_equals( 'duration', $col_info->type() );
    $self->assert_equals( 'numerical', $col_info->class() );
    $self->assert_equals( 'Time Taken', $col_info->label() );
}

sub test_sql_required_fields {
    my $self = $_[0];

    $self->assert_deep_equals( [ 'time_taken' ],
                                 $self->{'mock_stat'}->sql_required_fields() );
}

sub test_create_value {
    my $self = $_[0];

    $self->{'mock_stat'}{'field'} = 'time_taken';
    my $value = $self->{'mock_stat'}->create_value( undef,
                                                    { 'statTest' => 120,
                                                      '_lr_statTest_mc' => 2 });
    $self->assert_deep_equals( { 'content' => '2m',
                                 'value' => 120,
                                 'missing_cases' => 2 }, $value );

    $self->{'mock_stat'}{'field'} = 'file_size';
    $value = $self->{'mock_stat'}->create_value( undef,
                                                 { 'statTest' => 1024,
                                                   '_lr_statTest_mc' => 0 });
    $self->assert_deep_equals( { 'content' => '1k',
                                 'value' => 1024,
                                 'missing_cases' => 0 }, $value );
}


1;
