package tests::functional::LireTest;

use strict;

use base qw/ Lire::Test::FunctionalTestCase
             Lire::Test::TestCase
             tests::functional::TestSchemaFixture /;

use POSIX qw/ setlocale LC_MESSAGES /;

use Lire::UI;
use Lire::Test::CursesUIDriver;
use Lire::Config;
use Lire::DlfConverterManager;
use Lire::Config::Parser;

sub new {
    my $self = shift->SUPER::new( @_ );

    $self->tests::functional::TestSchemaFixture::init();
    $self->{'driver'} = new Lire::Test::CursesUIDriver();

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'_old_lc_messages'} = setlocale( LC_MESSAGES );
    setlocale( LC_MESSAGES, 'C' );

    $self->set_up_test_schemas();
    $self->set_up_test_specs();
    $self->set_up_test_store();
    $self->set_up_TestDlfConverter();

    $self->{'driver'}->setup_curses_ui();

    Lire::Config->init();
    Lire::DlfConverterManager->instance()->register_default_converters();

    $self->{'lire_ui'} = new Lire::UI();
    $self->{'driver'}->set_curses_ui( $self->{'lire_ui'}->curses_ui() );

    return;
}

sub tear_down {
    my $self = $_[0];

    $self->SUPER::tear_down();
    $self->tests::functional::TestSchemaFixture::tear_down();

    $self->{'driver'}->teardown_curses_ui();

    setlocale( LC_MESSAGES, $self->{'_old_lc_messages'} );

    return;
}

sub test_lire_prefs {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->select_items( 'section_menu', 'Responder Preferences' );

        $driver->select_items( 'options_list', 'lr_mail_from' );
        $driver->enter_text( 'option_widget',
                             "\ca\ckresponder\@logreport.org" );

        $driver->select_items( 'options_list', 'lr_mail_reply_to' );
        $driver->enter_text( 'option_widget',
                             "\ca\ckresponder\@logreport.org" );

        $driver->select_items( 'options_list', 'lr_archive' );
        $driver->select_items( 'option_widget', 'yes' );

        $driver->select_items( 'section_menu', 'Program Paths' );
        $driver->select_items( 'options_list', 'gzip_path' );
        my $gzip_path = $driver->find_widget( 'option_widget' )->getobj( 'file_field' )->text();
        $self->assert_str_equals( Lire::Config->get( 'gzip_path' ),
                                  $gzip_path );

        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );

    $driver->activate_menu( 'menubar/Lire/Preferences...' );

    my $top_win = $driver->top_window();
    $self->assert_equals( $driver->find_widget( '/quickstart_win' ),
                          $top_win );

    my $prefs_file = $self->homedir() . "/.lire/config.xml";
    $self->assert( -f $prefs_file,
                   "Save didn't write the config file: '$prefs_file'" );

    my $parser =
      new Lire::Config::Parser( 'spec' => Lire::Config->config_spec() );
    my $conf = $parser->load_config_file( $prefs_file );
    $self->assert_str_equals( 'responder@logreport.org',
                              $conf->get( 'lr_mail_from' )->as_value() );
    $self->assert_str_equals( 'responder@logreport.org',
                              $conf->get( 'lr_mail_reply_to' )->as_value() );
    $self->assert_num_equals( 1,
                              $conf->get( 'lr_archive' )->as_value() );

    return;
}

sub test_new_store {
    my $self = $_[0];

    my $driver = $self->{'driver'};

    my $handler = sub {
        $driver->find_widget( 'pathvalue' )->text( $self->homedir() );
        $driver->find_widget( 'filevalue' )->text( 'new_store' );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );

    $driver->activate_menu( 'menubar/Store/New...' );

    $self->assert( -f $self->homedir() . '/new_store/dlf.db',
                   "new store 'new_store' was not created properly" );
    my $top_win = $driver->top_window();
    $self->assert_str_equals( $driver->find_widget( '/store_win' ),
                              $top_win );

    return;
}

sub test_open_store {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->find_widget( 'pathvalue' )->text( $self->{'storedir'} );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->activate_menu( 'menubar/Store/Open...' );
    $self->assert_str_equals( $driver->find_widget( '/store_win' ),
                              $driver->top_window() );

    $self->assert_dies( qr{no 'New...' submenu in 'menubar/Store' menu},
                        sub {$driver->find_menu_def('menubar/Store/New...')} );
    $self->assert_dies( qr{no 'Open...' submenu in 'menubar/Store' menu},
                        sub {$driver->find_menu_def('menubar/Store/Open...')});

    return;
}

sub test_close_store {
    my $self = $_[0];

    $self->test_open_store();
    my $driver = $self->{'driver'};
    $driver->activate_menu( 'menubar/Store/Close' );
    $self->assert_str_equals( $driver->find_widget( '/quickstart_win' ),
                              $driver->top_window() );

    return;
}

sub test_configure_jobs {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    $self->test_open_store();
    $self->do_add_import_job();
    $self->do_add_report_jobs();

    $driver->activate_menu( 'menubar/Store/Close' );

    my $parser =
      new Lire::Config::Parser( 'spec' => Lire::Config->config_spec() );

    my $conf = $parser->load_config_file( "$self->{'storedir'}/config.xml" );
    my $e_conf =
      $parser->load_config_file( $self->tests_datadir() . "/jobs-config.xml" );

    $self->assert_deep_equals( $e_conf->get( 'import_jobs' ),
                               $conf->get( 'import_jobs' ) );

    $self->assert_deep_equals( $e_conf->get( 'report_jobs' ),
                               $conf->get( 'report_jobs' ) );
}

sub do_add_import_job {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->enter_text( 'widget/name_widget', 'Daily Import' );
        $driver->select_items( 'widget/period_widget', 'daily' );
        $driver->select_items( 'widget/service_widget/list', 'test_newapi' );
        $driver->enter_text( 'widget/log_file_widget', 'test.dlf' );

        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'import_jobs/buttons', '[Add]' );
}

sub do_add_report_jobs {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->enter_text( 'widget/name_widget', 'Test_Report' );
        $driver->select_items( 'widget/superservice_widget', 'test' );
        $self->do_add_daily_schedule();
        $self->do_add_weekly_schedule();
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'report_jobs/buttons', '[Add]' );
}

sub do_add_daily_schedule {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->select_items( 'widget/period_widget', 'daily' );
        $driver->enter_text( 'widget/report_cfg_widget/file_field', 
                             "test.cfg" );
        $self->do_add_pdf_email_oj();
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/schedules_widget/buttons', '[Add]' );
}

sub do_add_weekly_schedule {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->select_items( 'widget/period_widget', 'weekly' );
        $driver->enter_text( 'widget/report_cfg_widget/file_field',
                             "test.cfg" );
        $self->do_add_html_file_oj();
        $self->do_add_txt_email_oj();
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/schedules_widget/buttons', '[Add]' );
}

sub do_add_pdf_email_oj {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->enter_text( 'widget/name_widget', 'PDF' );
        $driver->select_items( 'widget/format_widget/list', 'pdf' );
        my $pdf_properties = sub {
            $driver->select_items( 'widget/charts_widget', 'yes' );
            $driver->click_button( 'buttons', '< OK >' );
        };
        $driver->add_event_loop_handler( $pdf_properties );
        $driver->click_button( 'widget/format_widget/button', '<...>' );

        $self->set_email_destination( 'PDF Report', 'flacoste@logreport.org' );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/output_jobs_widget/buttons', '[Add]' );
}

sub do_add_txt_email_oj {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->enter_text( 'widget/name_widget', 'Text Email' );
        $driver->select_items( 'widget/format_widget/list', 'txt' );
        $self->set_email_destination( 'Text Report', 'wolfgang@logreport.org' );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/output_jobs_widget/buttons', '[Add]' );
}

sub do_add_html_file_oj {
    my $self = $_[0];

    my $driver = $self->{'driver'};
    my $handler = sub {
        $driver->enter_text( 'widget/name_widget', 'HTML' );
        $driver->select_items( 'widget/format_widget/list', 'html' );
        my $html_properties = sub {
            $driver->select_items( 'widget/charts_widget', 'yes' );
            $driver->click_button( 'buttons', '< OK >' );
        };
        $driver->add_event_loop_handler( $html_properties );
        $driver->click_button( 'widget/format_widget/button', '<...>' );

        $self->set_file_destination( "report %Y-W%V" );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/output_jobs_widget/buttons', '[Add]' );
}

sub set_email_destination {
    my ( $self, $subject, @emails ) = @_;
    my $driver = $self->{'driver'};

    $driver->select_items( 'widget/destination_widget/list', 'email' );
    my $handler = sub {
        $driver->enter_text( 'widget/subject_widget', $subject );
        foreach my $email ( @emails ) {
            $driver->enter_text( 'widget/emails_widget/value_widget', $email );
            $driver->click_button( 'widget/emails_widget/buttons', '[Add]' );
        }
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/destination_widget/button', '<...>' );
}

sub set_file_destination {
    my ( $self, $file ) = @_;
    my $driver = $self->{'driver'};

    $driver->select_items( 'widget/destination_widget/list', 'file' );
    my $handler = sub {
        $driver->enter_text( 'widget/file_widget', $file );
        $driver->click_button( 'buttons', '< OK >' );
    };
    $driver->add_event_loop_handler( $handler );
    $driver->click_button( 'widget/destination_widget/button', '<...>' );
}

1;
