package tests::functional::LrLog2MailTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase tests::functional::TestSchemaFixture/;
use Lire::Config::Build qw/ac_info/;
use Lire::ReportParser::ReportBuilder;
use Lire::Utils qw/tempdir/;
use File::Basename qw/dirname/;
use Cwd 'realpath';
use Carp;
{
    use utf8;
    use MIME::Parser;
}

sub new {
    my $self = shift->SUPER::new(@_);

    $self->tests::functional::TestSchemaFixture::init();

    $self->{'email_to'} = "flacoste\@logreport.org wsl\@logreport.org";
    $self->{'testsdir'} = realpath(dirname( __FILE__ ) . "/..");
    $self->{'report_cfg'} = "$self->{'testsdir'}/data/test.cfg";
    $self->{'parser'} = new MIME::Parser;
    my $tmpdir = tempdir( "MIME::Parser_XXXXXX", 'CLEANUP' => 1 );
    $self->{'parser'}->output_under( $tmpdir );

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schemas();
    $self->set_up_test_specs();
    $self->set_up_TestDlfConverter();

}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();
    $self->tests::functional::TestSchemaFixture::tear_down();
}

sub test_version {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail -v" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    my $VERSION = ac_info( "VERSION" );
    $self->assert_matches( qr/shipped with Lire $VERSION/, $result->stderr );
}

sub test_xml_report {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail -c $self->{'report_cfg'} -o xml test_newapi < $self->{'testsdir'}/data/test.dlf $self->{'email_to'}" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );

    my $msg = $result->sent_mail;
    $self->assert_equals( 1, scalar @$msg );
    $self->assert_deep_equals( [ split ' ', $self->{'email_to'} ],
                               $msg->[0]{'recipients'} );

    my $entity = $self->{'parser'}->parse_data( $msg->[0]{'message'} );
    my $head = $entity->head;
    $self->assert_equals( "[LogReport] test_newapi report (was: )\n",
                          $head->get( 'Subject' ) );
    $self->assert_equals( 'multipart/mixed', $head->mime_type );

    # 2 parts: report, signature and disclaimer
    $self->assert_equals( 2, scalar $entity->parts);

    # Check that the explanation is there
    my $sig_disc = $entity->parts( 0 );
    $self->assert_equals( "text/plain", $sig_disc->mime_type );
    $self->assert_matches( qr/LogReport/, $sig_disc->bodyhandle->as_string );
    $self->assert_matches( qr/raw Lire report/, 
                           $sig_disc->bodyhandle->as_string );

    my $report = $entity->parts( 1 );
    $self->assert_equals( "application/x-lire-report", $report->mime_type );
    $self->check_xml_report_from_string( "$self->{'testsdir'}/data/test-sqlite.xml",
                                         $report->bodyhandle->as_string() );

}

sub test_ascii_report {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail -c $self->{'report_cfg'} -s 'Test Subject' -o txt test_newapi < $self->{'testsdir'}/data/test.dlf $self->{'email_to'}" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );

    my $msg = $result->sent_mail;
    $self->assert_equals( 1, scalar @$msg );
    $self->assert_deep_equals( [ split ' ', $self->{'email_to'} ],
                               $msg->[0]{'recipients'} );

    my $entity = $self->{'parser'}->parse_data( $msg->[0]{'message'} );
    my $head = $entity->head;
    $self->assert_equals( "[LogReport] test_newapi report (was: Test Subject)\n",
                          $head->get( 'Subject' ) );
    $self->assert_equals( 'text/plain', $head->mime_type );

    # Check that the signature was included
    $self->assert_matches( qr/LogReport/, $entity->bodyhandle->as_string );

}

sub test_bad_service {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail no_such_service $self->{'email_to'}" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/isn't a known DLF converter/,
                           $result->stderr );
}

sub test_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_log2mail test_newapi" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/Usage lr_log2mail/,
                           $result->stderr );
}

1;
