package tests::functional::LrStoreTest;

use strict;

use base qw/Lire::Test::FunctionalTestCase tests::functional::TestSchemaFixture/;

use Lire::Utils qw/file_content/;
use Lire::Config::Build qw/ac_info/;
use File::Path qw/mkpath rmtree/;
use File::Basename;
use Cwd qw/realpath/;
use Carp;

sub new {
    my $self = shift->SUPER::new(@_);

    $self->tests::functional::TestSchemaFixture::init(@_);

    $self->{'tests_datadir'} = realpath( dirname(__FILE__) ) . "/../data";

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_store();
    $self->set_up_test_schemas();
    $self->set_up_TestDlfConverter();
}

sub set_up_test_store {
    my $self = $_[0];

    $self->{'store_path'} = $self->rundir . "/test_store";
    $self->{'store_name'} = "test_store";
    system( "cd '" . $self->rundir .  "' ; tar xf $self->{'tests_datadir'}/test_store.tar" ) == 0
      or croak "failed to untar temporary store";
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();
    $self->tests::functional::TestSchemaFixture::tear_down();

    rmtree( [ $self->{'store_path'} ] );
}

sub test_version {
    my $self = $_[0];

    my $version = ac_info( 'VERSION' );
    foreach my $arg ( "--version", "-v" ) {
        my $cmd = "lr_store $arg";
        my $result = $self->lire_run( $cmd );
        $self->annotate( $result->stderr );
        $self->assert_equals( 0, $result->status );
        $self->assert( !$result->stdout, "stdout should be empty" );
        $self->assert_does_not_match( qr/ (crit|err|warning) /,
                                      $result->stderr,
                                      "There were warnings or error messages." );
        $self->assert_matches( qr/ notice.*shipped with Lire $version/,
                               $result->stderr );
    }
}

sub test_help {
    my $self = $_[0];

    foreach my $arg ( "--help", "-h" ) {
        my $cmd = "lr_store $arg";
        my $result = $self->lire_run( $cmd );
        $self->annotate( $result->stderr );
        $self->assert_equals( 0, $result->status );
        $self->assert( !$result->stdout, "stdout should be empty" );
        $self->assert_does_not_match( qr/ (crit|err|warning) /,
                                      $result->stderr,
                                      "There were warnings or error messages." );
        $self->assert_matches( qr/Usage: lr_store/, $result->stderr );
    }
}

sub test_unknown {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store unknown comannd" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/Unknown command:.*\n.*Use lr_store --help/,
                           $result->stderr );
}

sub test_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/Missing 'command' argument.\n.*Use lr_store --help/,
                           $result->stderr );
}

sub test_create {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store create" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert_matches( qr/Usage: lr_store create/, $result->stderr );

    $result = $self->lire_run( "lr_store create $self->{'store_name'}" );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/directory.*already exists/, $result->stderr);

    $result = $self->lire_run( "lr_store create test" );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );

    my $store_dir = $self->rundir . "/test";
    $self->assert( -d $store_dir, "lr_store create didn't create $store_dir" );
    $self->assert( -f "$store_dir/dlf.db",
                   "$store_dir doesn't look like a DLF store" );
}

sub test_delete {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store delete $self->{'store_name'}" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );

    $self->assert( ! -d $self->{'store_path'}, "lr_store create didn't remove $self->{'store_path'}" );

    $result = $self->lire_run( "lr_store delete no_such_store" );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/No such store:/, $result->stderr );

    $result = $self->lire_run( "lr_store delete" );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( !$result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/missing 'store' argument/, $result->stderr );
    $self->assert_matches( qr/Usage: lr_store delete \<store\>/, $result->stderr );
}

sub test_export_dlf_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store export_dlf" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert_matches( qr/missing 'store' argument/, $result->stderr );
    $self->assert_matches( qr/Usage: lr_store export_dlf \<store> \<dlf_stream>/,
                           $result->stderr );
}

sub test_export_dlf_bad_schema {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store export_dlf $self->{'store_name'} dns" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert_matches( qr/no DLF stream 'dns' in the store/,
                           $result->stderr );
    $self->assert_matches( qr/Use lr_store list_dlf to find the available DLF streams/, $result->stderr );
}

sub test_export_dlf {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store export_dlf $self->{'store_name'} test" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    my $dlf = file_content( $self->{'tests_datadir'} . "/test.dlf" );
    $self->assert_equals( $dlf, $result->stdout );
}

sub test_list_dlf_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store list_dlf" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert_matches( qr/missing 'store' argument/, $result->stderr );
    $self->assert_matches( qr/Usage: lr_store list_dlf \<store>/,
                           $result->stderr );
}

sub test_list_dlf {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store list_dlf $self->{'store_name'}" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    my @streams = split ' ', $result->stdout;
    $self->assert_deep_equals( [ 'test' ],\@streams );
}

sub test_import_log {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store import_log $self->{'store_name'} test_newapi $self->{'tests_datadir'}/test.dlf" );
    $self->annotate( $result->stderr );
    $self->assert_equals( 0, $result->status );
    $self->assert_does_not_match( qr/ (crit|err|warning) /, $result->stderr,
                                  "There were warnings or error messages." );
    $self->assert_matches( qr/lr_import_line_count=0/, $result->stdout );
    $self->assert_matches( qr/lr_import_dlf_count=24/, $result->stdout );
    $self->assert_matches( qr/lr_import_saved_count=0/, $result->stdout );
    $self->assert_matches( qr/lr_import_error_count=0/, $result->stdout );
    $self->assert_matches( qr/lr_import_ignored_count=0/, $result->stdout );
}

sub test_import_log_usage {
    my $self = $_[0];

    my $result = $self->lire_run( "lr_store import_log" );
    $self->annotate( $result->stderr );
    $self->assert_not_equals( 0, $result->status );
    $self->assert( ! $result->stdout, "stdout should be empty" );
    $self->assert_matches( qr/missing 'store' argument/, $result->stderr );
    $self->assert_matches( qr/Usage: lr_store import_log \<store\> \<converter\> \[\<logfile\>\]/,
                           $result->stderr );

}

1;
