/* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/symbol.h"
#include "include/environment.h"

GData *LWSymbol::datalist_symbols = NULL;
GSList *LWSymbol::list_symbols = NULL;

/*! \brief Konstruktor

	Kostruktuuje symbol
*/
LWSymbol::LWSymbol (): id(0) {};

/*! \brief Destruktor

	Destruktor symbolu
*/
LWSymbol::~LWSymbol ()
{
	if (id > 0)
		g_datalist_id_remove_data (&datalist_symbols, quark);

	list_symbols = g_slist_remove (list_symbols, (gpointer) this);
}

/*! \brief Klonuje symbol

	Zwraca sklonowany symbol. 
	Jeśli symbol stanowi polecenie jest ono niezaincjalizowane.
*/
LWSymbol *LWSymbol::clone()
{
	g_return_val_if_fail (TRUE == canClone(), NULL);
	
	return factory (getName());
}

/*! \brief Pobieranie identyfikatora

	Pobiera identyfikator symbolu
	\return Pobrany identyfikator
*/
guint LWSymbol::getId ()
{
	return id;
}

/*! \brief Czy symbol jest poleceniem

	Zwraca FAŁSZ. Klasa która implementuje polecenie
	przeciąża tę metodę.
	\return PRAWDA jeśli symbol stanowi polecenie
	w przeciwnym razie FAŁSZ
*/
gboolean LWSymbol::isCommand ()
{
	return FALSE;
}

/*! \brief Czy symbol jest poleceniem

	Zwraca FAŁSZ. Klasa która implementuje wartość
	przeciąża tę metodę.
	\return PRAWDA jeśli symbol stanowi wartość
	w przeciwnym razie FAŁSZ
*/
gboolean LWSymbol::isValue ()
{
	return FALSE;
}

/*! \brief Metoda fabryczna

	Tworzy polecenie na podstawie jego nazwy 
	\param name Nazwa polecenia
*/
LWSymbol *LWSymbol::factory (const gchar *name)
{
	g_return_val_if_fail (name != NULL, NULL);
	
	typedef LWSymbol * (*SymbolConstructor) ();

	GData *symbols = LWEnvironment::getSymbols ();

	SymbolConstructor sc = (SymbolConstructor) g_datalist_get_data (&symbols, name);
	
	if (sc == NULL) g_print ("%s", name);
	
	g_return_val_if_fail (sc != NULL, NULL);

	LWSymbol *symbol = sc();
	list_symbols = g_slist_prepend (list_symbols, (gpointer) symbol);
	return sc();
}

/*! \brief Metoda fabryczna uwzględniająca identyfikator

	Tworzy polecenie na podstawie jego nazwy 
	Dodatkowo stworzene polecenie będzie miało przyporządkowany
	identyfikator.
	\param name Nazwa polecenia
	\param id Identyfikator
*/
LWSymbol *LWSymbol::factoryId (const gchar *name, guint id)
{
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (id > 0, NULL);
	
	gchar *idString = g_strdup_printf ("%s@%u", name, id);
	
	if (datalist_symbols == NULL)
		g_datalist_init (&datalist_symbols);
	
	LWSymbol *symbol = (LWSymbol *) g_datalist_get_data (&datalist_symbols, idString);
	
	if (symbol == NULL) {
		symbol = factory (name);
		symbol->id = id;
		symbol->quark = g_quark_from_string (idString);
		g_datalist_id_set_data (&datalist_symbols, symbol->quark, (gpointer) symbol);
	}
	
	g_free (idString);
	return symbol;
}

/*! \brief Niszczenie wszytskich symboli
	
	Niszczy wszystkie symbole
*/
void LWSymbol::destroyAll ()
{
	while (list_symbols != NULL) {
		LWSymbol *symbol = (LWSymbol *) list_symbols->data;
		delete symbol;
	}
}
