/*
 * fstclient.cpp - client for XFST-server
 *
 * Copyright (c) 2005 Tobias Doerffel <tobydox/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#include "qt3support.h"

#ifdef QT4

#include <QApplication>
#include <QX11EmbedWidget>
#include <QX11Info>

#else

#include <qapplication.h>
#include "qxembed.h"

#define QX11EmbedWidget QXEmbed

#endif


#include <unistd.h>
#include <cstdio>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/select.h>
#include <sys/time.h>

#include "templates.h"
#include "config_mgr.h"
#include "lmms_main_win.h"
#include "fstclient.h"



remoteVSTPlugin::remoteVSTPlugin( const QString & _plugin ) :
	m_failed( FALSE ),
	m_plugin( _plugin ),
	m_pluginWidget( NULL ),
	m_pluginWID( 0 ),
	m_serverInFD( -1 ),
	m_serverOutFD( -1 ),
	m_serverMutex(),
	m_name( "" ),
	m_vendorString( "" ),
	m_productString( "" ),
	m_vstVersion( 0 ),
	m_inputCount( 0 ),
	m_outputCount( 0 ),
	m_shmID( -1 ),
	m_shm( NULL ),
	m_shmSize( 0 )
{
	pipe( m_pipes[0] );
	pipe( m_pipes[1] );
	int child_pid = fork();
	if( child_pid < 0 )
	{
		printf( "fork() failed!\n" );
	}
	else if( child_pid == 0 )
	{
		dup2( m_pipes[0][0], 0 );
		dup2( m_pipes[1][1], 1 );
		QString xfst_server_exec = configManager::inst()->pluginDir() +
								"xfst_server";
		execlp( xfst_server_exec.
#ifdef QT4
					toAscii().constData(),
#else
					ascii(),
#endif
					xfst_server_exec.
#ifdef QT4
							toAscii().constData(),
#else
									ascii(),
#endif
								"", NULL );
		return;
	}
	m_serverInFD = m_pipes[1][0];
	m_serverOutFD = m_pipes[0][1];

	lock();
	writeValueS<Sint16>( LOAD_VST_PLUGIN );
	writeStringS( m_plugin.
#ifdef QT4
				toAscii().constData()
#else
				ascii()
#endif
			);
	
	writeValueS<Sint16>( GET_VST_VERSION );
	writeValueS<Sint16>( GET_NAME );
	writeValueS<Sint16>( GET_VENDOR_STRING );
	writeValueS<Sint16>( GET_PRODUCT_STRING );
	unlock();

	while( 1 )
	{
		Sint16 cmd = UNDEFINED_CMD;
		if( messagesLeft() )
		{
			cmd = processNextMessage();
		}
		if( cmd == SET_PRODUCT_STRING )
		{
			break;
		}
#ifdef QT4
		QApplication::processEvents( QEventLoop::AllEvents, 50 );
#else
		qApp->processEvents( 50 );
#endif
		usleep( 1 );
		if( cmd == FAILED_LOADING_PLUGIN )
		{
			m_failed = TRUE;
			break;
		}
	}
}




remoteVSTPlugin::~remoteVSTPlugin()
{
	delete m_pluginWidget;
	if( m_failed == FALSE )
	{
		setShmKeyAndSize( 0, 0 );
		// tell server to quit and wait for acknowledge
		writeValueS<Sint16>( CLOSE_VST_PLUGIN );
		sleep( 1 );
/*		while( processNextMessage() != QUIT_ACK )
		{
		}*/
/*		close( m_serverInFD );
		close( m_serverOutFD );*/
	}
/*	// close all sides of our pipes
	close( m_pipes[0][0] );
	close( m_pipes[0][1] );
	close( m_pipes[1][0] );
	close( m_pipes[1][1] );*/
}



#include <X11/Xlib.h>


void remoteVSTPlugin::showEditor( void )
{
	if( m_pluginWidget != NULL )
	{
		m_pluginWidget->show();
		return;
	}
	if( m_pluginWID == 0 )
	{
		return;
	}

	XWindowAttributes attr;
	XGetWindowAttributes(
#ifdef QT4
			QX11Info::display(),
#else
			qt_xdisplay(),
#endif
			m_pluginWID, &attr );
	m_pluginWidget = new QWidget( lmmsMainWin::inst()->workspace() );
	m_pluginWidget->setFixedSize( attr.width, attr.height );
	m_pluginWidget->setWindowTitle( name() );

	QX11EmbedWidget * xe = new QX11EmbedWidget( m_pluginWidget );
#ifdef QT4
	xe->embedInto( m_pluginWID );
#else
	xe->embed( m_pluginWID );
#endif
	xe->setFixedSize( attr.width, attr.height );
	m_pluginWidget->show();
	xe->show();
}




void remoteVSTPlugin::hideEditor( void )
{
	if( m_pluginWidget != NULL )
	{
		m_pluginWidget->hide();
	}
}




void remoteVSTPlugin::process( sampleFrame * _in_buf, sampleFrame * _out_buf )
{
	const Uint32 frames = mixer::inst()->framesPerAudioBuffer();

	if( m_shm == NULL )
	{
		// m_shm being zero means we didn't initialize everything so
		// far so process one message each time (and hope we get
		// information like SHM-key etc.) until we process
		// messages in later stage of this procedure
		if( m_shmSize == 0 )
		{
			(void) processNextMessage();
		}
		mixer::inst()->clearAudioBuffer( _out_buf, frames );
		return;
	}

	memset( m_shm, 0, m_shmSize );

	Uint8 inputs = tMax<Uint8>( m_inputCount, DEFAULT_CHANNELS );

	if( _in_buf != NULL && inputs > 0 )
	{
		for( Uint8 ch = 0; ch < inputs; ++ch )
		{
			for( Uint32 frame = 0; frame < frames; ++frame )
			{
				m_shm[ch*frames+frame] = _in_buf[frame][ch];
			}
		}
	}

	lock();
	writeValueS<Sint16>( PROCESS );
	unlock();

	if( _out_buf != NULL && m_outputCount > 0 )
	{
		// wait until server signals that process()ing is done
		while( processNextMessage() != PROCESS_DONE )
		{
		}

		Uint8 outputs = tMax<Uint8>( m_outputCount, DEFAULT_CHANNELS );
		if( outputs != DEFAULT_CHANNELS )
		{
			// clear buffer, if plugin doesn't fill up both channels
			mixer::inst()->clearAudioBuffer( _out_buf, frames );
		}
		for( Uint8 ch = 0; ch < outputs; ++ch )
		{
			for( Uint32 frame = 0; frame < frames; ++frame )
			{
				_out_buf[frame][ch] = m_shm[(m_inputCount+ch)*
								frames+frame];
			}
		}
	}
}




void remoteVSTPlugin::enqueueMidiEvent( const midiEvent & _event,
						const Uint32 _frames_ahead )
{
	lock();
	writeValueS<Sint16>( ENQUEUE_MIDI_EVENT );
	writeValueS<midiEvent>( _event );
	writeValueS<Uint32>( _frames_ahead );
	unlock();
}




void remoteVSTPlugin::setShmKeyAndSize( Uint16 _key, size_t _size )
{
	if( m_shm != NULL && m_shmSize > 0 )
	{
		shmdt( m_shm );
		m_shm = NULL;
		m_shmSize = 0;
	}
	// only called for detaching SHM?
	if( _size == 0 )
	{
		return;
	}

	int shm_id = shmget( _key, _size, 0 );
	if( shm_id == -1 )
	{
		printf( "failed getting shared memory\n" );
	}
	else
	{
		m_shm = (float *) shmat( shm_id, 0, 0 );
		// TODO: error-checking
	}
}




void remoteVSTPlugin::setPluginXID( const Sint32 _plugin_xid )
{
	m_pluginWID = _plugin_xid;
}




bool remoteVSTPlugin::messagesLeft( void ) const
{
	fd_set rfds;
	FD_ZERO( &rfds );
	FD_SET( m_serverInFD, &rfds );
	timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 1;	// can we use 0 here?
	return( select( m_serverInFD + 1, &rfds, NULL, NULL, &tv ) > 0 );
}




Sint16 remoteVSTPlugin::processNextMessage( void )
{
	fd_set rfds;
	FD_ZERO( &rfds );
	FD_SET( m_serverInFD, &rfds );
	if( select( m_serverInFD + 1, &rfds, NULL, NULL, NULL ) <= 0 )
	{
		return( UNDEFINED_CMD );
	}

	lock();
	Sint16 cmd = readValueS<Sint16>();
	switch( cmd )
	{
		case DEBUG_MSG:
			printf( "debug message from server: %s",
						readStringS().c_str() );
			break;

		case GET_SAMPLE_RATE:
			writeValueS<Sint16>( SET_SAMPLE_RATE );
			// handle is the same
			writeValueS<Sint32>(
					mixer::inst()->sampleRate() );
			break;

		case GET_BUFFER_SIZE:
			writeValueS<Sint16>( SET_BUFFER_SIZE );
			// handle is the same
			writeValueS<Uint32>(
				mixer::inst()->framesPerAudioBuffer() );
			break;

		case SET_SHM_KEY_AND_SIZE:
		{
			Uint16 shm_key = readValueS<Uint16>();
			size_t shm_size = readValueS<size_t>();
			setShmKeyAndSize( shm_key, shm_size );
			break;
		}

		case SET_INPUT_COUNT:
			m_inputCount = readValueS<Uint8>();
			break;

		case SET_OUTPUT_COUNT:
			m_outputCount = readValueS<Uint8>();
			break;

		case SET_XID:
			setPluginXID( readValueS<Window>() );
			break;

		case SET_VST_VERSION:
			m_vstVersion = readValueS<Sint32>();
			break;

		case SET_NAME:
			m_name = readStringS().c_str();
			break;

		case SET_VENDOR_STRING:
			m_vendorString = readStringS().c_str();
			break;

		case SET_PRODUCT_STRING:
			m_productString = readStringS().c_str();
			break;

		case PROCESS_DONE:
		case QUIT_ACK:
		case UNDEFINED_CMD:
		default:
			break;
	}
	unlock();

	return( cmd );
}






/*

int main( void )
{
	remoteVSTPlugin * p = new remoteVSTPlugin( "/home/toby/vst/z3ta+.dll" );
	p->showEditor();
	p->enqueueMidiEvent( midiEvent( NOTE_ON, 0, 30, 100 ), 0 );
	while( 1 )
	{
		p->process();
	}
}

*/
