/*
 * lb302.h - declaration of class lb302 which is a bass synth attempting to
 *           emulate the Roland TB303 bass synth
 *
 * Copyright (c) 2006-2007 Paul Giblock <pgib/at/users.sourceforge.net>
 * 
 * This file is part of Linux MultiMedia Studio - http://lmms.sourceforge.net
 *
 * lb302FilterIIR2 is based on the gsyn filter code by Andy Sloane.
 * 
 * lb302Filter3Pole is based on the TB303 instrument written by 
 *   Josep M Comajuncosas for the CSounds library
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA.
 *
 */


#ifndef _LB302_H_
#define _LB302_H_

#include "effect_lib.h"
#include "instrument.h"
#include "led_checkbox.h"
#include "mixer.h"


class knob;
class notePlayHandle;

class lb302FilterKnobState
{
    public:
    float cutoff;
    float reso;
    float envmod;
    float envdecay;
    float dist;
};

class lb302FilterIIR2State
{
    public:
    float vcf_c0;
    float vcf_a;
    float vcf_b;
    float vcf_c;
    float vcf_d1;
    float vcf_d2;
};

class lb302Filter3PoleState
{
    public:
    float vcf_c0;
    float kp,
          kp1h,
          kres,
          ay1,
          ay2,
          lastin,
          value;
    float aout;
};
         
typedef union 
{
    lb302FilterIIR2State iir;
    lb302Filter3PoleState pole;
} 
lb302FilterState;


class lb302Filter
{
    public:
    lb302Filter(lb302FilterKnobState* p_fs);
    virtual ~lb302Filter() {};

    virtual void recalc();
    virtual void envRecalc();
    virtual float process(const float& samp)=0;
    virtual void playNote();

    virtual void getState(lb302FilterState* fs)=0;
    virtual void setState(const lb302FilterState* fs)=0;

    protected:
    lb302FilterKnobState *fs;  
    
    // Filter Decay
	float vcf_c0;           // c0=e1 on retrigger; c0*=ed every sample; cutoff=e0+c0
	float vcf_e0,           // e0 and e1 for interpolation
          vcf_e1;           
    float vcf_rescoeff;     // Resonance coefficient [0.30,9.54]
};

class lb302FilterIIR2 : public lb302Filter
{
    public:
    lb302FilterIIR2(lb302FilterKnobState* p_fs);
	virtual ~lb302FilterIIR2();

    virtual void recalc();
    virtual void envRecalc();
    virtual float process(const float& samp);

    virtual void getState(lb302FilterState* fs);
    virtual void setState(const lb302FilterState* fs);

    protected:
    float vcf_d1,           //   d1 and d2 are added back into the sample with 
          vcf_d2;           //   vcf_a and b as coefficients. IIR2 resonance
                            //   loop.

    // IIR2 Coefficients for mixing dry and delay.
    float vcf_a,            //   Mixing coefficients for the final sound.  
          vcf_b,            //  
          vcf_c;
    
	effectLib::monoToStereoAdaptor<effectLib::distortion<> > * m_dist_fx;
	effectLib::distortion<> * m_dist;
};


class lb302Filter3Pole : public lb302Filter
{
    public:
    lb302Filter3Pole(lb302FilterKnobState* p_fs);

    //virtual void recalc();
    virtual void envRecalc();
    virtual void recalc();
    virtual float process(const float& samp);

    virtual void getState(lb302FilterState* fs);
    virtual void setState(const lb302FilterState* fs);

    protected:
    float kfcn, 
          kp, 
          kp1, 
          kp1h, 
          kres;
    float ay1, 
          ay2, 
          aout, 
          lastin, 
          value;
};

 

class lb302State
{
public:
    float vco_c;
    float vca_a;
    int vca_mode;
    int sample_cnt;

    lb302FilterState fs;
};

class lb302Note
{
public:
    float vco_inc;
    bool dead;
};


class lb302Synth : public instrument
{
    Q_OBJECT
public:
	lb302Synth( instrumentTrack * _channel_track );
	virtual ~lb302Synth();

	virtual void FASTCALL playNote( notePlayHandle * _n,
						bool _try_parallelizing );
	virtual void FASTCALL deleteNotePluginData( notePlayHandle * _n );


	virtual void FASTCALL saveSettings( QDomDocument & _doc,
							QDomElement & _parent );
	virtual void FASTCALL loadSettings( const QDomElement & _this );

	virtual QString nodeName( void ) const;

    virtual bool isMonophonic(void) const {
        return true;
    }

	virtual f_cnt_t desiredReleaseFrames( void ) const
	{
		return 4048;
	}

private:

    void initNote(lb302Note *note);


private:
	knob * vcf_cut_knob;
	knob * vcf_res_knob;
    knob * vcf_dec_knob;
	knob * vcf_mod_knob;

    knob * vco_fine_detune_knob;

    knob * dist_knob;
    knob * wave_knob;
    
    ledCheckBox * slideToggle;
    ledCheckBox * accentToggle;
    ledCheckBox * deadToggle;
    ledCheckBox * db24Toggle;

    knob * slide_dec_knob;

public slots:
    void filterChanged(float);
    void detuneChanged(float);
    void waveChanged(float);
    void db24Toggled( bool );

private:

    

private:
    // Oscillator
    float vco_inc,          // Sample increment for the frequency. Creates Sawtooth.
          vco_k,            // Raw oscillator sample [-0.5,0.5]
          vco_c;            // Raw oscillator sample [-0.5,0.5]

    float vco_slide,        //* Current value of slide exponential curve. Nonzero=sliding
          vco_slideinc,     //* Slide base to use in next node. Nonzero=slide next note
          vco_slidebase;    //* The base vco_inc while sliding.

    float vco_detune;

    enum  vco_shape_t { SAWTOOTH, INVERTED_SAWTOOTH, SQUARE, TRIANGLE, MOOG, ROUND_SQUARE };
    vco_shape_t vco_shape;

    // User settings
    lb302FilterKnobState  fs;
    lb302Filter       *vcf;
    lb302Note         hold_note;
    bool use_hold_note;
  	
    int lastFramesPlayed;
    int release_frame;

	
    // More States
    int   vcf_envpos;       // Update counter. Updates when >= ENVINC

	float vca_attack,       // Amp attack 
          vca_decay,        // Amp decay
          vca_a0,           // Initial amplifier coefficient 
          vca_a;            // Amplifier coefficient.
    
    // Envelope State
	int   vca_mode;         // 0: attack, 1: decay, 2: idle

    // My hacks
    int   sample_cnt;
    
    int   last_offset;
   

    int framesPerPeriod;
    lb302State *period_states;
    int period_states_cnt;

    int catch_frame;
    int catch_decay;

    void recalcFilter();

    int process(sampleFrame *outbuf, const Uint32 size);

} ;


#endif
