#!/usr/bin/perl

#$Header: /var/lib/cvs/LogTrend/Consolidation/ConsolidationPostgreSQLDataBase.pm,v 1.9 2002/05/02 14:47:08 lsimonneau Exp $
##*****************************************************************************
##  ConsolidationPostgreSQLDataBase isa LogTrend::DataBaseAccess::PostgreSQLDataBase
##  Description  : Extend LogTrend::DataBaseAccess::PostgreSQLDataBase
##
##  Project      : LogTrend 1.0.0.0 - Atrid Systemes
##  Author       : Laurent Simonneau l.simonneau@atrid.fr
##*****************************************************************************
#$Log: ConsolidationPostgreSQLDataBase.pm,v $
#Revision 1.9  2002/05/02 14:47:08  lsimonneau
#A lot of bugfixes.
#Critical bugfixes in consolidation.
#
#Revision 1.8  2002/02/01 17:35:16  lsimonneau
#Modify database structure. *_values tables inherite of data.
#
#Revision 1.7  2002/01/04 13:53:13  slhullier
#Attribute Kilo of tag *Graph
#
#Revision 1.6  2001/09/27 14:30:05  lsimonneau
#DIsable autocommit for DataBase and add Commit methods to Consolidation/ConsolidationPostgreSQLDataBase.pm to prevent data lost on die or sigtrem, sigint ...
#
#Revision 1.5  2001/09/24 15:29:10  lsimonneau
#Reimplementation of consolidation.
#
#Revision 1.4  2001/08/24 13:43:43  lsimonneau
#Minor bugfixes, add docs.
#
#Revision 1.3  2001/08/23 15:48:37  lsimonneau
#Parser rewriting.
#
#Revision 1.2  2001/08/22 09:51:27  lsimonneau
#First usable version of Consolidation.
#
#Revision 1.1  2001/08/17 15:35:14  lsimonneau
#Ajout de Consolidation au CVS, parser XML OK.
#

package LogTrend::Consolidation::ConsolidationPostgreSQLDataBase;

use strict;
use LogTrend::DataBaseAccess::PostgreSQLDataBase;
use LogTrend::Consolidation::ConsolidationDataBase;

use POSIX qw(strftime);

@LogTrend::Consolidation::ConsolidationPostgreSQLDataBase::ISA = ("LogTrend::DataBaseAccess::PostgreSQLDataBase", 
								  "LogTrend::Consolidation::ConsolidationDataBase");


##*****************************************************************************
## Constructor  public
##  Description  : creat a new ConsolidationPostgreSQLDataBase
##
##  Parameters   : The database name,
##                 The database server host name,
##                 The database server port,
##                 An username and a password of a database user
##*****************************************************************************
sub new
{
    my ($classname, $databasename, $host, $port, $username, $password) = @_;
  
    my $self = $classname->SUPER::new($databasename, $host, $port,
                                      $username, $password);


    bless($self, $classname);
    $self->{DBHANDLE}->{AutoCommit} = 0;
    return $self;
}


sub Commit {
    my $self = shift;

    $self->{DBHANDLE}->commit;
}

sub Rollback {
    my $self = shift;

    $self->{DBHANDLE}->rollback;
}

##*****************************************************************************
##  RemoveDataInTimeInterval public
##  Description  : Remove data 
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name,
##                 The start date,
##                 The end date (optional).
##                 Enable ancestor. String with 'enable_ancestor' or 'disable_ancestor'. 
##                                  Default value is 'enable_ancestor'.
##*****************************************************************************
sub RemoveDataInTimeInterval {
    my ($self, $source, $agent, $name, $start, $end, $enable_ancestor) = @_;


    if(!(defined $end) and !(defined $enable_ancestor)) {
	$enable_ancestor = 'enable_ancestor';
    }
    elsif(defined $end and !(defined $enable_ancestor)) {		
	if($end eq 'disable_ancestor' or $end eq 'enable_ancestor') { 
	    $enable_ancestor = $end;
	    $end = undef;
	}
	else {
	    $enable_ancestor = 'enable_ancestor';
	}
    }

    my ($id_agent, $id_data_description, $data_type, $agent_activation_date) = 
        $self->getDataAndAgentInfo($source, $agent, $name);

    return 0 unless ($data_type and $id_data_description and $id_agent);

    my $data_type_table_name = "${data_type}_values";

    my $timestring = "(date_part('epoch', $data_type_table_name.collection_date) > $start)";

    $timestring .= " AND (date_part('epoch', $data_type_table_name.collection_date) < $end)" if defined $end;

    my $st_handle = $self->{DBHANDLE}->prepare("
DELETE FROM $data_type_table_name

WHERE  (id_data_description = ?) AND 
       $timestring");
    

    $st_handle->execute($id_data_description);

    $st_handle->finish;

    my ($new_id_agent, $new_data_type, $new_agent_activation_date);
    while(defined $id_data_description and 
	  $agent_activation_date > $start and 
	  ($enable_ancestor eq 'enable_ancestor')) {
	($id_data_description, $new_id_agent, $new_data_type, $new_agent_activation_date) = 
            $self->getDataAncestorInfo($id_data_description);
	last unless defined $id_data_description;
	
	$data_type_table_name = "${new_data_type}_values";

        $st_handle = $self->{DBHANDLE}->prepare("
DELETE FROM $data_type_table_name

WHERE  (id_data_description = ?) AND 
       $timestring AND 
       (date_part('epoch', collection_date) <  $agent_activation_date)");
	
	$st_handle->execute($id_data_description);
	
	$st_handle->finish;
	
	$id_agent = $new_id_agent;
	$agent_activation_date = $new_agent_activation_date;
    }

}

##*****************************************************************************
##  AddData public
##  Description  : Add data to database
##
##  Parameters   : The source name,
##                 The agent name,
##                 The data name,
##                 A reference on a list of [value, date].
##*****************************************************************************
sub AddData {
    my ($self, $source, $agent, $name, $data_list) = @_;

    my ($id_agent, $id_data_description, $data_type, $agent_activation_date) = $self->getDataAndAgentInfo($source, $agent, $name);

    return 0 unless ($data_type and $id_data_description and $id_agent);
    
    my $data_type_table_name = "${data_type}_values";
    
    my $st_handle = $self->{DBHANDLE}->prepare("
INSERT INTO $data_type_table_name 

(id_data_description, storage_date, collection_date, value) 

VALUES (?, ?, ?, ?)");



    my $collection_date;
    my @row;
    foreach my $data (@$data_list) {
   	$collection_date = strftime("%m/%d/%Y %H:%M:%S", localtime($data->[1]));

	$st_handle->execute($id_data_description, 'now', $collection_date, $data->[0]);
    }

    $st_handle->finish;

}


##*****************************************************************************
##  GetListOfAgentTypeOnSource public
##  Description  : Return the list of agent with the specified type on a source
##
##  Parameters   : The source's number,
##                 The agent's type
##
##  Return a list of [source, agent]
##*****************************************************************************
sub GetListOfAgentTypeOnSource {
    my ($self, $sourcenumber, $agenttype) = @_;

    my $st_handle = $self->{DBHANDLE}->prepare("

SELECT id_source, number 

FROM agents 

WHERE id_source = ? AND information ~* '^$agenttype' 

ORDER BY number ASC")
	or die "Can't get agent of type $agenttype on source $sourcenumber (prepare)";

    $st_handle->execute($sourcenumber);

    my $result = $st_handle->fetchall_arrayref;

    $st_handle->finish();

    if($#$result == -1) {
	return 0;
    }
    
    return $result;    
}
