#!/usr/bin/perl

package LogTrend::Agent::SNMPAgent;

use strict;

use vars qw( @ISA );
use sigtrap qw(die INT TERM QUIT HUP);

use XML::DOM;
use LogTrend::Agent;
use LogTrend::Common::LogDie;
use Getopt::Long;
use SNMP::MIB::Compiler;
use Net::SNMP;
use IPC::SysV qw( IPC_CREAT IPC_EXCL S_IRWXU IPC_NOWAIT);
use IPC::Msg;

@ISA = ("LogTrend::Agent");

my $name = "SNMPAgent";
my $version = "1.0.0.0";
my $defaultIPCqid = 1024;

##******************************************************************************
## Constructor  public > Agent
##  Description  : create a new SNMPAgent
##  Parameters   : none
##******************************************************************************
sub new {
    my($class) = shift;
    my($self) = {};

    $class = ref($class) || $class;
    $self = $class->SUPER::new( $name, $version );

    bless($self, $class);
}

##******************************************************************************
## Method ParseAgentOptions  public  (>Agent)
##  Description  : parses agent specific switches (overload to enable)
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub ParseAgentOptions {
    my($self) = shift;
    my($qid);

    GetOptions('queueid|q=i', \$qid)
    or  do {
        print "Usage: $name [options] -- [--queue|-q number]\n",
              "\n",
              "     --queue number: to specify the IPC Message queue ID (defaut 1024)\n",
              "\n";
        exit(1);
    };

        $qid
    and $self->{SNMPAgent}{Traps}{Qid} = $qid;
}

##******************************************************************************
## Method ParseXMLConfigFile
## Description  : Parse the XML SNMP configuration file
## Parameters   : none
## Return value : none
##******************************************************************************
sub ParseXMLConfigFile {
    my ($self,$file,$agentstate) = @_;

    my($parser, $node);

    my($SNMPAgent) = sub {
        my($node) = @_;
        my($src, $out, $sfx);

        my($MIB) = sub {
            my($node) = @_;
            my($mibobj);

            my($Load) = sub {
                my($node) = @_;
                my($count);

                for my $node ($node->getElementsByTagName('Load', 0)) {
                    my($name);
                    
                        $name = $node->getAttribute('name')
                    or  Die("$file: tag 'Load': attribute 'name': missing value");

                    eval { $mibobj->compile($name); };

                        $@
                    and Die "Error loading '$name'";
                    ++$count;
                }
                    $count
                or  Die("$file: tag 'MIB': no tag 'Load'");

            };

                $node = $node->getElementsByTagName('MIB', 0)->item(0)
            or  Die("$file: Missing tag 'MIB' tag");

                $src = $node->getAttribute('src_path')
            or  Die("$file: tag 'MIB': missing attribute 'src_path'");

                $out = $node->getAttribute('out_path')
            or  Die("$file: tag 'MIB': missing attribute 'out_path'");

                $sfx = $node->getAttribute('sfx')
            or  Die("$file: tag 'MIB': missing attribute 'sfx'");

            $sfx = [ eval "( $sfx )" ];

                $@
            and Die("$file: tag 'MIB': attribute 'sfx' invalid content");

                $mibobj = new SNMP::MIB::Compiler
            or  Die ("Unable to create object 'SNMP::MIB::Compiler'");

            $mibobj->add_path($src);
            $mibobj->repository($out);
            $mibobj->add_extension(@$sfx);

            &$Load($node);

            $self->{SNMPAgent}{MIBObject} = $mibobj;
        };

        my($ValidateOID) = sub {
            my($oid) = @_;
            my(@soid) = split(/\s+|\s*\.\s*/, $oid);
            my($mib) = $self->{SNMPAgent}{MIBObject};
            my(@noid);
            my($type);

            my($GetObjectType) = sub {
                my($object) = @_;
                my($node);
                my($type, $basetype);
                my($GetBaseType);# Needs predeclare for recursion

                $GetBaseType = sub {
                    my($type) = @_;

                        exists($mib->{types}{$type})
                    or  return $type;

                    &$GetBaseType($mib->{types}{$type}{type});
                };

                    exists($mib->{nodes}{$object})
                or  Die "Root node object '$object' unsupported";

                $node = $mib->{nodes}{$object};

                $type =   $node->{type} eq 'OBJECT-TYPE'
                        ? $node->{syntax}{type}
                        : $node->{type};

                $basetype = &$GetBaseType($type);

                if($basetype eq 'OCTET STRING') {
                    return 'Text';
                }
                elsif($basetype eq 'INTEGER') {
                    return 'Real'
                }

                Die "Object '$object': type '$type' unsupported ($basetype)";
            };


            # root node
            if($soid[0] =~ /^[0-9]+$/) { # Numeric node
                for my $oid (keys(%{$mib->{root}})) {
                        $mib->{root}{$oid}{oid}[0] == $soid[0]
                    and do {
                        $noid[0] = $soid[0];
                        $soid[0] = $oid;
                        last;
                    };
                }
                    @noid
                or  Die "root node number '$noid[0]' unknown";
            }
            else { # Symbolic node
                    exists($mib->{root}{$soid[0]})
                or  Die "root node '$soid[0]' unknown";
                $noid[0] =  $mib->{root}{$soid[0]}{oid}[0];
            }

            # Tree nodes
            for(my $i = 1; $i < @soid; ++$i) { # Numeric node
                if($soid[$i] =~ /^[0-9]+$/) {
                        exists($mib->{tree}{$soid[$i-1]}{$soid[$i]})
                    and do {
                        $noid[$i] = $soid[$i];
                        $soid[$i] = $mib->{tree}{$soid[$i-1]}{$soid[$i]};
                        next;
                    };
                    Die "Node number '$soid[$i]' unknown under node '" .
                        join('.', @soid[0..$i-1]) .
                        "'";
                }
                else { # Symbolic node
                        exists($mib->{nodes}{$soid[$i]})
                    and $soid[$i-1] eq $mib->{nodes}{$soid[$i]}{oid}[-2]
                    and do {
                        $noid[$i] = $mib->{nodes}{$soid[$i]}{oid}[-1];
                        next;
                    };
                    Die "Node '$soid[$i]' unknown under node '" .
                        join('.', @soid[0..$i-1]) .
                        "'";
                }
            }

            $type = &$GetObjectType($soid[-1]);

            ( join('.',@soid), join('.',@noid), $type )

        };

        my($SNMP) = sub {
            my($node) = @_;
            my($attr) = {};

            my($Objects) = sub {
                my($node) = shift;
                my($prefix);
                my($objects) = [];

                my($Object) = sub {
                    my($node) = @_;

                    for my $node ($node->getElementsByTagName('Object', 0)) {
                        my($object) = {};

                            $object->{description} = $node->getAttribute('description')
                        or  Die("$file: tag 'Object': missing attribute 'description'");

                            $object->{OID} = $node->getAttribute('oid')
                        or  Die("$file: tag 'Object': missing attribute 'oid'");

                            $prefix
                        and $object->{OID} = $prefix . '.' . $object->{OID};

                        ($object->{SOID},
                         $object->{NOID},
                         $object->{type}) = &$ValidateOID($object->{OID});

                            $object->{index} = $node->getAttribute('index')
                        or  $object->{index} = '0';

                        push(@$objects, $object);
                    }

                        @$objects
                    or  Die("$file: tag 'Objects' contains no tag 'Object'");
                };

                for my $node($node->getElementsByTagName('Objects', 0)) {

                    $prefix = $node->getAttribute('prefix');

                    &$Object($node);
                }

                $objects
            };

            my($Traps) = sub {
                my($node) = shift;
                my($prefix);
                my($traps) = [];

                my($Trap) = sub {

                    my($Match) = sub {
                        my($node) = shift;
                        my($matches) = [];
                        my($oid, $index, $value);

                        for my $node ($node->getElementsByTagName('Match', 0)) {

                                $oid = $node->getAttribute('oid')
                            or  Die("$file: tag 'Match': missing attribute 'oid'");

                                $prefix
                            and $oid = $prefix . '.' . $oid;

                            (undef, $oid, undef) = &$ValidateOID($oid);

                                $index = $node->getAttribute('index')
                            or  $index = '0';

                            $oid .= '.' . $index;

                                $value = $node->getAttribute('value')
                            or  $value = undef;

                            push(@$matches, $oid, $value);
                        }
                            @$matches
                        or  Die("$file: tag 'Trap': contains no tag 'Match'");

                        $matches;
                    };

                    for my $node ($node->getElementsByTagName('Trap', 0)) {
                        my($trap) = {};
                        my($type);

                            $trap->{description} = $node->getAttribute('description')
                        or  Die("$file: tag 'Trap': missing attribute 'description'");

                            $type = $node->getAttribute('type')
                        or  Die("$file: tag 'Trap': missing attribute 'type'");

                        $type = ucfirst(lc($type));

                            $type =~ /(?:Info|Warning|Error)$/
                        or  Die("$file: tag '$name': unknown alarm type '$type'");

                        $trap->{type} = $type;

                        $trap->{match} = &$Match($node);

                        push(@$traps, $trap);
                    }

                        @$traps
                    or  Die("$file: tag 'Traps' contains no tag 'Trap'");
                };

                    $node = $node->getElementsByTagName('Traps', 0)->item(0)
                or  return;

                $prefix = $node->getAttribute('prefix');

                &$Trap($node);

                $traps
            };

                $node = $node->getElementsByTagName('SNMP', 0)->item(0)
            or  Die("$file: Missing tag 'SNMP' tag");

                $attr->{host} = $node->getAttribute('host')
            or  Die("$file: tag 'MIB': missing attribute 'host'");
                $attr->{community} = $node->getAttribute('community')
            or  Die("$file: tag 'MIB': missing attribute 'community'");
                $attr->{port} = $node->getAttribute('port')
            or  Die("$file: tag 'MIB': missing attribute 'port'");
                $attr->{timeout} = $node->getAttribute('timeout')
            or  Die("$file: tag 'MIB': missing attribute 'timeout'");

            $self->{SNMPAgent}{Session} = $attr;
            $self->{SNMPAgent}{Objects} = &$Objects($node);
            $self->{SNMPAgent}{Traps}{Desc} = &$Traps($node);

        };


            $node = $node->getElementsByTagName('SNMPAgent', 0)->item(0)
        or  Die("$file: Missing tag 'SNMPAgent' tag");

        &$MIB($node);
        &$SNMP($node);

    };

    my($LoadSpecificNode) = sub {
        my($parser, $document, $rootnode);

        # Create a XML::DOM::parser object
        #
            $parser = new XML::DOM::Parser()
        or  Die("XML::DOM::Parser: $!");

        # Parse and get the root document node
        #
            $document = $parser->parsefile( $file )
        or  Die("$file :$!");

        # Get Configuration tag node
        #
            $rootnode = $document->getElementsByTagName('Configuration', 0)->item(0)
        or  Die("$file: tag 'Configuration' not found");

        # And then the Specific tag node
        #
            $node = $rootnode->getElementsByTagName('Specific', 0)->item(0)
        or  Die("$file: tag 'Specific' not found");
    };

    $self->SUPER::ParseXMLConfigFile( $file, $agentstate );

    &$LoadSpecificNode();
    &$SNMPAgent($node);

        $self->IsRunning()
    and $self->{SNMPAgent}{Traps}{Desc}
    and $self->_RegisterTrapDaemon();

}

##******************************************************************************
## Method CreateAgentDescription  public  (>Agent)
##  Description  : creates an agent's description
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CreateAgentDescription {
   my($self) = shift;
   my($dnum) = 0;
   my($anum) = 0;
   my($objects) = $self->{SNMPAgent}{Objects};
   my($traps) = $self->{SNMPAgent}{Traps}{Desc};

   for my $object (@$objects) {
       $self->AddADataDescription(++$dnum, $object->{type},
                                  "none", $object->{description}, "");
   }

   for my $trap (@$traps) {
       $self->AddAnAlarmDescription(++$anum,
                                    $trap->{type},
                                    $trap->{description},
                                    "" );
   }
}

##******************************************************************************
## Method CollectData  public  (>Agent)
##  Description  : collects data and alarms
##  Parameters   : none
##  Return value : none
##******************************************************************************
sub CollectData {
    my $self = shift;
    my($sessprm) = $self->{SNMPAgent}{Session};
    my($objects) = $self->{SNMPAgent}{Objects};
    my($traps) = $self->{SNMPAgent}{Traps}{Desc};
    my($dnum) = 0;
    my(@oids);
    my(@types);
    my($session, $error, $answers);

    my($CollectObjects) = sub {

        for my $object (@$objects) {
            push(@oids, $object->{NOID} . '.' . $object->{index});
            push(@types, $object->{type});

        }

        ($session, $error) = Net::SNMP->session(Hostname => $sessprm->{host},
                                                Community => $sessprm->{community},
                                                Port => $sessprm->{port},
                                                Version => 1,
                                                Timeout => $sessprm->{timeout});
            defined($session)
        or  do {
            SysLog "Cannot create SNMP session: $error";
            $session->close();
            return
        };

        $answers = $session->get_request(@oids);

            $error = $session->error()
        and do {
            SysLog "Get-Request error: $error";
            $session->close();
            return
        };


        $session->close();

        for(my $i = 0; $i < @oids; ++$i) {
            for($types[$i]) {
                /Text/    and do {
                        $self->AddDataText(++$dnum, $answers->{$oids[$i]});
                        last;
                    };
                /Real/   and do {
                        $self->AddDataReal(++$dnum, $answers->{$oids[$i]});
                        last;
                    };
                Die "CollectData(): unkown type '$_'";
            }
        }
    };

    my($CollectTraps) = sub {
        my($msg) = "<Message type= \"G\" pid=\"$$\" />";
        my($alarms) = $self->_TrapDaemonDialog($msg);

            $alarms
        or  return;

        $alarms =~ s/\s*//gs;

            $alarms =~ /^(?:[0-9]+(?:,[0-9]+)*)?$/
        or  die "Invalid alarm list from the trap daemon: $alarms";

        for my $anum (split/,/, $alarms) {
            $self->AddAlarm($anum);
        }
    };

        defined($objects)
    and &$CollectObjects();

        defined($traps)
    and &$CollectTraps();
}

##########################################################
# Method _RegisterTrapDaemon:  private
# Description : contact the trap daemon and register traps
#               to catch and report
# Parameters: none
# Return value: none
##########################################################
sub _RegisterTrapDaemon {
    my($self) = shift;
    my($msg,$queue);
    my($host) = $self->{SNMPAgent}{Session}{host};
    my($traplist) = $self->{SNMPAgent}{Traps}{Desc};

    $self->{SNMPAgent}{Traps}{Qid} ||= $defaultIPCqid;

        $queue =  new IPC::Msg($self->{SNMPAgent}{Traps}{Qid}, S_IRWXU )
    or  Die "Cannot open message queue [$self->{SNMPAgent}{Traps}{Qid}]: $!";

    $self->{SNMPAgent}{Traps}{Queue} = $queue;

    $msg = qq[<Message type="R" pid="$$">\n];

    for(my $i = 0; $i < @$traplist; ++$i) {
        my($num) = $i + 1;
        my($match) = $traplist->[$i]{match};

        $msg .= qq[  <Trap num="$num" ip="$host">\n];

        for(my $j=0; $j < @$match;) {
            my($oid) = $match->[$j++];
            my($value) = $match->[$j++];

            $msg .= qq[    <Object oid="$oid" ];
                $value
            and $msg .= qq[value="$value" ];
            $msg .= qq[/>\n];
        }

        $msg .= qq[  </Trap>\n];
    }

    $msg .= qq[</Message>];

    $self->_TrapDaemonDialog($msg);

    # YES. This is a callback :-)
    # It will unregister the agent from the trap daemon
    # upon termination
    eval 'END { $self->_TrapDaemonDialog(qq[<Message type="U" pid="$$" />]); }';

}

##########################################################
# Method _TrapDaemonDialog:  private
# Description : Send an XML message to the trap daemon,
#               get the answer, parse it. Die if status ne "ok"
#               and return the CDATA content if any
# Parameters: A valid XML Message
# Return value: CDATA in the Answer message body, if any
##########################################################

sub _TrapDaemonDialog {
    my($self) = shift;
    my($msg) = @_;
    my($queue) = $self->{SNMPAgent}{Traps}{Queue};
    my($status, $content);
    my($parser, $node);

        $queue->snd(1, $msg)
    or  die "Failed to deliver message to trap server: $!";

        $queue->rcv($msg, 1024, $$)
    or  die "Failed receiving message from trap server: $!";

        $parser = new XML::DOM::Parser()
    or  Die("XML::DOM::Parser: $!");

        $node = $parser->parse( $msg )
    or  Die("Unable to parse trap server answer :$!");

        $node = $node->getElementsByTagName('Answer', 0)->item(0)
    or  Die("Trap server answer: tag 'Answer' not found");

        $status = $node->getAttribute('status')
    or  Die("Trap server answer: tag 'Answer': missing attribute 'status'");

    $node = $node->getFirstChild();

        $node
    and $node->getNodeType() != TEXT_NODE
    and Die("Trap server answer: invalid answer:\n$msg");

        $node
    and $content = $node->getData;

        $content
    and $content =~ s/^\s+|\s+$//g;

        $status eq "ok"
    or  Die("Trap server answer: Error: $content");

    $content;
}

1;

__END__

=head1 NAME

SNMPAgent.pm - Perl Extension for LogTrend : SNMP Agent

=head1 SYNOPSIS

  use LogTrend::Agent::SNMPAgent;

  LogTrend::Agent::SNMPAgent->new();
  
=head1 DESCRIPTION

LogTrend::Agent::SNMPAgent is a Perl extention implementing an
SNMP Agent for LogTrend.

This module is not intended for direct use, but to be called
through its intertface utility called SNMPAgent.

As it inherits from LogTrend::Agent, the various Agent command
line switches apply to it.

Beside those switches, it accepts an optional specific switch
C<--queue> or C<-q> followed by a queue number to specify the
IPC Message queue to be used to collect various traps from the
SNMPTrapd daemon (see manual pages). This switch is separated
from the Agent ones by a double dash, " la POSIX".

The SNMP Agent is aimed at collecting SNMP objects and/or SNMP
traps from a given source and then send them to the LogTrend
storage server. Objects are collected as data, while traps
generate alarms.

Note that only numeric or string objects can be collected.

=head1 PRE-REQUISITES

The following Perl modules are definitly needed for this agent to
work:

    SNMP::MIB::Compiler
    Net::SNMP

The former one is needed to compile the various MIBs and SMIs
needed, while the later one is used to encode/decode SNMP PDUs.

Note that to be able to generate alarms from traps, the equipment
must be configured to send its traps to the local system and the
SNMPTrapd daemon should be started prior to the agent.

=head1 CONFIGURATION

The SNMP Agent configuration is done using an XML file.

As of every LogTrend agents, it is divided in two parts: the
generic and the specific:

    <?xml version="1.0" standalone="no"?>
    <!DOCTYPE Configuration SYSTEM "Configuration.dtd">
    <Configuration>
       <Generic>
       ....
       </Generic>
       <Specific>
           <SNMPAgent>
           ...
           </SNMPAgent>
       </Specific>
    </Configuration>
    
The generic part is described in the LogTrend::Agent module
documentation.

The specific part includes an C<SNMPAgent> tag which groups
all the various configuration tags for this agent.

    <MIB> [Mandatory] Contains the definitions of the various MIB/SMI
                      used by the agent.

        Attributes:

            src_path [Mandatory] The path where the source MIB/SMI
                                 files reside (must have read access).

            dst_path [Mandatory] The path where the compiled MIB/SMI
                                 files reside (must have write access).

            sfx [Mandatory] Comma separated list of all possible file
                            extention for the MIB and SMI files, single
                            quoted (see SNMP::MIB::Compiler documentation
                            for more information).

        Tags:

            <Load> : to specify a MIB/SMI filname to load.

                Attributes:

                    name [Mandatory] Base name of the MIB/SMI file (its
                                     extention is to be specified by the
                                     'sfx' attribute above).

                Example:
                
                    <Load name="RFC-1212" />

            

        Example:

            <MIB src_path="/etc/SNMPAgent.d/mibs"
                 out_path="/etc/SNMPAgent.d/out"
                 sfx="'', '.txt', '.mib', '.my', '.smi'" >

                <Load name="RFC1155-SMI" />
                <Load name="RFC1158-MIB" />
                <Load name="RFC-1212" />
                <Load name="CISCO-SMI" />
                <Load name="OLD-CISCO-INTERFACES-MIB" />
            </MIB>

    <SNMP> [Mandatory] Contains the definitions of objects and traps to handle,
                       and the SNMP connection parameters


        Attributes:

            host [Mandatory] Host name or IP address of the equipment
            
            community [Mandatory] SNMP community name

            port [Mandatory] Port number to use for sending/reading the SNMP
                             GET-REQUEST PDU.

            timeout [Mandatory] maximum time to wait for the GET-REQUEST answers

        Tags:

            <Objects> [Optional] Liste of requested objects. Maybe given multiple
                                 times to use different OBJECT-IDENTIFIER prefixes.

                Attributes:

                    prefix [Mandatory] OBJECT-IDENTIFIER prefix to add in front of
                                       listed objects.

                Tags:
                
                    <Object> [At least one] SNMP object definition

                        Attributes:

                            description [Mandatory] object description in a human
                                                    readable form.

                            oid [Mandatory] OBJECT-IDENTIFIER to be added to the
                                            'prefix' attibute of <Objects> tag.
                                            Maybe numeric, symbolic or mixed.

                            index [Optional] Defaults to 0. Index to be added to
                                             the OBJECT-IDENTIFIER for a GET-REQUEST
                                             PDU (remember that not-indexed objects
                                             are postfixed with '.0' while indexed ones
                                             start from '.1'). 

                        Example:

                            <Object description="Interface[2]-Number of input bytes"
                                    oid="interfaces.ifTable.ifEntry.ifInOctets"
                                    index="2" />

                    Example:

                        <Objects prefix="iso.org.dod.internet.mgmt.mib-2" >
                            <Object description="Interface[2]-Number of input bytes"
                                    oid="interfaces.ifTable.ifEntry.ifInOctets"
                                    index="2" />
                            <Object description="Interface[2]-Number of output bytes"
                                    oid="interfaces.ifTable.ifEntry.ifOutOctets"
                                    index="2" />
                            <Object description="Second interface type"
                                    oid="interfaces.ifTable.ifEntry.ifDescr"
                                    index="2"/>
                        </Objects>


            <Traps> [Optional] Liste of traps that should generate alarms. Maybe
                               given multiple times to use different OBJECT-IDENTIFIER
                               prefixes.


                Attributes:

                    prefix [Mandatory] OBJECT-IDENTIFIER prefix to add in front of
                                       traps OID.

                Tags:

                    <Trap> [At least one] Trap bag description

                        Attributes:

                            description [Mandatory] alarm description in a human
                                                    readable form.

                            type [Mandatory] alarm type to generate. Maybe one of
                                             'info', 'warning', 'error'.

                        Tags:

                            <Match> [At least one] Specify a OID and optionaly its
                                                   value which must be found in the
                                                   trap bag to generate the alarm.
                                                   
                                Attributes:
                                
                                    oid [Mandatory] OID to be found in the trap bag.

                                    index [Optional] '.0' if ommited. SNMP object
                                                     index.

                                    value [Optional] If specified, if the oid is
                                                     found in the trap bag, it
                                                     must have this value to match.

            Example:
                                     

                <Traps prefix="iso.org.dod.internet" >
                    <Trap description="Interface[1]-Down" type="warning" >
                          <Match oid="mgmt.mib-2.interfaces.ifTable.ifEntry.ifIndex"
                                 index="1" />
                          <Match oid="private.enterprises.cisco.local.linterfaces.lifTable.lifEntry.locIfReason"
                                 index="1"
                                 value="down" />
                    </Trap>
                    <Trap description="Interface[2]-Down" type="warning" >
                          <Match oid="mgmt.mib-2.interfaces.ifTable.ifEntry.ifIndex"
                                 index="2" />
                          <Match oid="private.enterprises.cisco.local.linterfaces.lifTable.lifEntry.locIfReason"
                                 index="2"
                                 value="down" />
                </Trap>

            
=head1 AUTHOR

Franois Dsarmnien -- Atrid Systmes (f.desarmenien@atrid.fr)

=head1 COPYRIGHT

Copyright 2001, Atrid Systme http://www.atrid.fr/

Project home page: http://www.logtrend.org/

Licensed under the same terms as LogTrend project is.

=head1 WARRANTY

THIS SOFTWARE COMES WITH ABSOLUTLY NO WARRANTY OF ANY KIND.
IT IS PROVIDED "AS IS" FOR THE SOLE PURPOSE OF EVENTUALLY
BEEING USEFUL FOR SOME PEOPLE, BUT ONLY AT THEIR OWN RISK.


=cut


