#!/usr/bin/perl -w
use strict;
##########################################################################
# $Id: logwatch.pl,v 1.142 2005/11/12 18:59:21 kirk Exp $
##########################################################################
# Most current version can always be found at:
# ftp://ftp.logwatch.org/pub/redhat/RPMS

########################################################
# Specify version and build-date:
my $Version = '7.1';
my $VDate = '11/12/05';

#######################################################
# This was written and is maintained by:
#    Kirk Bauer <kirk@kaybee.org>
#
# Please send all comments, suggestions, bug reports,
#    etc, to logwatch@logwatch.org.
#
########################################################

my $BaseDir = "/usr/share/logwatch";
my $ConfigDir = "/etc/logwatch";
my $PerlVersion = "$^X";

#############################################################################

use Getopt::Long;
use POSIX qw(uname);
use File::Temp qw/ tempdir /;

eval "use lib \"$BaseDir/lib\";";
eval "use Logwatch \':dates\'";

my (%Config, @ServiceList, @LogFileList, %ServiceData, %LogFileData);
my (@AllShared, @AllLogFiles, @FileList);
# These need to not be global variables one day
my (@ReadConfigNames, @ReadConfigValues);

# Default config here...
$Config{'detail'} = 0;
$Config{'mailto'} = "root";
$Config{'save'} = "";
$Config{'print'} = 0;
$Config{'range'} = "yesterday";
$Config{'debug'} = 0;
$Config{'archives'} = 0;
$Config{'tmpdir'} = "/var/cache/logwatch";
$Config{'splithosts'} = 0;
$Config{'multiemail'} = 0;
$Config{'numeric'} = 0;
$Config{'pathtocat'} = "cat";
$Config{'pathtozcat'} = "zcat";
$Config{'output'} = "unformatted";
$Config{'sendmail'} = "sendmail";


# Logwatch now does some basic searching for logs
# So if the log file is not in the log path it will check /var/adm
# and then /var/log -mgt
$Config{'logdir'} = "/var/log";

#Added to create switches for different os options -mgt
#Changed to POSIX to remove calls to uname and hostname
my ($OSname, $hostname, $release, $version, $machine) = POSIX::uname();
$Config{'hostname'} = "$hostname";

#############################################################################

sub Usage () {
   # Show usage for this program
   print "\nUsage: $0 [--detail <level>] [--logfile <name>]\n" . 
      "   [--print] [--mailto <addr>] [--archives] [--range <range>] [--debug <level>]\n" .
      "   [--save <filename>] [--help] [--version] [--service <name>]\n" .
      "   [--numeric] [--output <output_type>]\n" .
      "   [--splithosts] [--multiemail]\n\n";
   print "--detail <level>: Report Detail Level - High, Med, Low or any #.\n";
   print "--logfile <name>: *Name of a logfile definition to report on.\n";
   print "--logdir <name>: Name of default directory where logs are stored.\n";
   print "--service <name>: *Name of a service definition to report on.\n";
   print "--print: Display report to stdout.\n";
   print "--mailto <addr>: Mail report to <addr>.\n";
   print "--archives: Use archived log files too.\n";
   print "--save <filename>: Save to <filename>.\n";
   print "--range <range>: Date range: Yesterday, Today, All, Help\n";
   print "                             where help will describe additional options\n";
   print "--numeric: Display ddresses numerically rather than symbolically and numerically\n";
   print "           (saves  a  nameserver address-to-name lookup).\n";
   print "--debug <level>: Debug Level - High, Med, Low or any #.\n";
# hostname needs to be cleaned up and explained
#   print "--hostname <host>: 
   print "--splithosts: Create a report for each host in syslog.\n";
   print "--multiemail: Send each host report in a separate email.  Ignored if \n";
   print "              not using --splithosts.\n";
   print "--output <output type>: Report Format - mail, html-embed, html or unformatted#.\n";
   print "--version: Displays current version.\n";
   print "--help: This message.\n";
   print "* = Switch can be specified multiple times...\n\n";
   exit (99);
}

my %wordsToInts = (yes  => 1,  no     => 0,
                   true => 1,  false  => 0,
                   on   => 1,  off    => 0,
                   high => 10,
                   med  => 5,  medium => 5,
                   low  => 0);

sub getInt {
   my $word = shift;
   my $tmpWord = lc $word;
   $tmpWord =~ s/\W//g;
   return $wordsToInts{$tmpWord} if (defined $wordsToInts{$tmpWord});
   unless ($word =~ s/^"(.*)"$/$1/) {
      return lc $word;
   }
   return $word;
}
              
sub CleanVars {
   foreach (keys %Config) {
      $Config{$_} = getInt($Config{$_});
   }
}

sub PrintStdArray (@) {
   my @ThisArray = @_;
   my $i;
   for ($i=0;$i<=$#ThisArray;$i++) {
      print "[" . $i . "] = " . $ThisArray[$i] . "\n";
   }
}

sub PrintConfig () {
   # for debugging, print out config...
   foreach (keys %Config) {
      print $_ . ' -> ' . $Config{$_} . "\n";
   }
   print "Service List:\n";
   PrintStdArray @ServiceList;
   print "\n";
   print "LogFile List:\n";
   PrintStdArray @LogFileList;
   print "\n\n";
}

# for debugging...
sub PrintServiceData () {
   my ($ThisKey1,$ThisKey2,$i);
   foreach $ThisKey1 (keys %ServiceData) {
      print "\nService Name: " . $ThisKey1 . "\n";
      foreach $ThisKey2 (keys %{$ServiceData{$ThisKey1}}) {
         next unless ($ThisKey2 =~ /^\d+-/);
         print "   $ThisKey2 = $ServiceData{$ThisKey1}{$ThisKey2}\n";
      }
      for ($i=0;$i<=$#{$ServiceData{$ThisKey1}{'logfiles'}};$i++) {
         print "   Logfile = " . $ServiceData{$ThisKey1}{'logfiles'}[$i] . "\n";
      }
   }
}

# for debugging...
sub PrintLogFileData () {
   my ($ThisKey1,$ThisKey2,$i);
   foreach $ThisKey1 (keys %LogFileData) {
      print "\nLogfile Name: " . $ThisKey1 . "\n";
      foreach $ThisKey2 (keys %{$LogFileData{$ThisKey1}}) {
         next unless ($ThisKey2 =~ /^\d+-/);
         print "   $ThisKey2 = $LogFileData{$ThisKey1}{$ThisKey2}\n";
      }
      for ($i=0;$i<=$#{$LogFileData{$ThisKey1}{'logfiles'}};$i++) {
         print "   Logfile = " . $LogFileData{$ThisKey1}{'logfiles'}[$i] . "\n";
      }
      for ($i=0;$i<=$#{$LogFileData{$ThisKey1}{'archives'}};$i++) {
         print "   Archive = " . $LogFileData{$ThisKey1}{'archives'}[$i] . "\n";
      }
   }
}

sub ReadConfigFile {
   my $FileName = $_[0];
   my $Prefix = $_[1];

   if ( ! -f $FileName ) {
      return(0);
   }

   if ($Config{'debug'} > 5) {
      print "ReadConfigFile: Opening " . $FileName . "\n";
   }
   open (READCONFFILE, $FileName) or die "Cannot open file $FileName: $!\n";
   my $line;
   while ($line = <READCONFFILE>) {
      if ($Config{'debug'} > 9) {
         print "ReadConfigFile: Read Line: " . $line;
      }
      $line =~ s/\#.*\\\s*$/\\/;
      $line =~ s/\#.*$//;
      next if ($line =~ /^\s*$/);

      if ($Prefix) {
         next if ($line !~ m/\Q$Prefix:\E/);
         $line =~ s/\Q$Prefix:\E//;
      }

      if ($line =~ s/\\\s*$//) {
	  $line .= <READCONFFILE>;
          redo unless eof(READCONFFILE);
      }

      my ($name, $value) = split /=/, $line, 2;
      $name =~ s/^\s+//; $name =~ s/\s+$//;
      if ($value) { $value =~ s/^\s+//; $value =~ s/\s+$//; }
      else { $value = ''; }

      push @ReadConfigNames, lc $name;
      push @ReadConfigValues, getInt $value;
      if ($Config{'debug'} > 7) {
         print "ReadConfigFile: Name=" . $name . ", Value=" . $value . "\n";
      }
   }
   close READCONFFILE;
}

#############################################################################

# Add / to BaseDir
unless ($BaseDir =~ m=/$=) {
   $BaseDir = $BaseDir . "/";
}

# Load main config file...
if ($Config{'debug'} > 8) {
   print "\nDefault Config:\n";
   PrintConfig();
}

CleanVars();

my $OldMailTo = $Config{'mailto'};
my $OldPrint  = $Config{'print'};

# For each of the configuration sets (logwatch.conf here, and
# logfiles,and services later), we do the following:
#  1. read the different configuration files
#  2. for each parameter, if it is cummulative, check if
#     it the special case empty string
#  3. check to see if duplicate

@ReadConfigNames = ();
@ReadConfigValues = ();

ReadConfigFile ("$BaseDir/default.conf/logwatch.conf", "");
ReadConfigFile ("$BaseDir/dist.conf/logwatch.conf", "");
ReadConfigFile ("$ConfigDir/conf/logwatch.conf", "");
ReadConfigFile ("$ConfigDir/conf/override.conf", "logwatch");


for (my $i = 0; $i <= $#ReadConfigNames; $i++) {
   if ($ReadConfigNames[$i] eq "logfile") {
      if ($ReadConfigValues[$i] eq "") {
	  @LogFileList = ();
      } elsif (! grep(/^$ReadConfigValues[$i]$/, @LogFileList)) {
         push @LogFileList, $ReadConfigValues[$i];
      }
   } elsif ($ReadConfigNames[$i] eq "service") {
      if ($ReadConfigValues[$i] eq "") {
	  @ServiceList = ();
      } elsif (! grep(/^$ReadConfigValues[$i]$/, @ServiceList)) {
         push @ServiceList, $ReadConfigValues[$i];
      }
   } else {
      $Config{$ReadConfigNames[$i]} = $ReadConfigValues[$i];
   }
}

CleanVars();

if ($OldMailTo ne $Config{'mailto'}) {
   $Config{'print'} = 0;
} elsif ($OldPrint ne $Config{'print'}) {
   $Config{'mailto'} = "";
}

if ($Config{'debug'} > 8) {
   print "\nConfig After Config File:\n";
   PrintConfig();
}

# Options time...

my @TempLogFileList = ();
my @TempServiceList = ();
my $Help = 0;
my $ShowVersion = 0;

$OldMailTo = $Config{'mailto'};
$OldPrint  = $Config{'print'};

GetOptions ( "d|detail=s"   => \$Config{'detail'},
             "l|logfile=s@" => \@TempLogFileList,
             "logdir=s"     => \$Config{'logdir'},
             "s|service=s@" => \@TempServiceList,
             "p|print"      => \$Config{'print'},
             "m|mailto=s"   => \$Config{'mailto'},
             "save=s"       => \$Config{'save'},
             "a|archives"   => \$Config{'archives'},
             "debug=s"      => \$Config{'debug'},
             "r|range=s"    => \$Config{'range'},
	     "n|numeric"    => \$Config{'numeric'},
             "h|help"       => \$Help,
             "v|version"    => \$ShowVersion,
             "hostname=s"   => \$Config{'hostname'},
             "splithosts"   => \$Config{'splithosts'},
             "multiemail"   => \$Config{'multiemail'},
	     "o|output=s"   => \$Config{'output'},
           ) or Usage();

$Help and Usage();

if ($Config{'range'} =~ /help/i) {
    RangeHelpDM();
    exit(0);
}

if ($ShowVersion) {
   print "Logwatch $Version (released $VDate)\n";
   exit 0;
}

CleanVars();


my $outtype_mail=0;
my $outtype_html=0;
my $outtype_htmlembed=0;
my $outtype_unformatted=0;
my $index_par=0;
my @format = (250);


my (%out_body, $p);

if ( $Config{'output'} eq 'mail' ) {
	    $outtype_mail = 1;
	} elsif ( $Config{'output'} eq 'html' ) {
	    $outtype_html = 1;
	} elsif ( $Config{'output'} eq 'html-embed' ) {
	    $outtype_htmlembed = 1;
	} elsif ( $Config{'output'} eq 'unformatted' ) {
	    $outtype_unformatted = 1;
	} else {
	    print STDERR "$0: unknown output-format: $Config{'output'}\n\n";
	}


my @reports = ();

my $out_head ='';
my $out_mime ='';
my $out_reference ='';


if ($OldMailTo ne $Config{'mailto'}) {
   $Config{'print'} = 0;
} elsif ($OldPrint ne $Config{'print'}) {
   $Config{'mailto'} = "";
}

if ($Config{'debug'} > 8) {
   print "\nCommand Line Parameters:\n   Log File List:\n";
   PrintStdArray @TempLogFileList;
   print "\n   Service List:\n";
   PrintStdArray @TempServiceList;
   print "\nConfig After Command Line Parsing:\n";
   PrintConfig();
}

if ($#TempLogFileList > -1) {
   @LogFileList = @TempLogFileList;
   for (my $i = 0; $i <= $#LogFileList; $i++) {
      $LogFileList[$i] = lc($LogFileList[$i]);
   }
   @ServiceList = ();
}

if ($#TempServiceList > -1) {
   @ServiceList = @TempServiceList;
   for (my $i = 0; $i <= $#ServiceList; $i++) {
      $ServiceList[$i] = lc($ServiceList[$i]);
   }
}

if ( ($#ServiceList == -1) and ($#LogFileList == -1) ) {
   push @ServiceList, 'all';
}

if ($Config{'debug'} > 5) {
   print "\nConfig After Everything:\n";
   PrintConfig();
}

#############################################################################

# Find out what services are defined...
my @TempAllServices = ();
my @services = ();
my (@CmdList, @CmdArgList, @Separators, $ServicesDir, $ThisFile, $count);


foreach my $ServicesDir ("$BaseDir/default.conf", "$BaseDir/dist.conf", "$ConfigDir/conf") {
   if (-d "$ServicesDir/services") {
      opendir(SERVICESDIR, "$ServicesDir/services") or
         die "$ServicesDir $!";
      while (defined($ThisFile = readdir(SERVICESDIR))) {
	 if ((-f "$ServicesDir/services/$ThisFile") && (!grep (/^$ThisFile$/, @services))) {
	     push @services, $ThisFile;
         }
      }
      closedir SERVICESDIR;
   }
}

foreach my $f (@services) {
   my $ThisService = lc $f;
   $ThisService =~ s/\.conf$//;
   push @TempAllServices, $ThisService;

   # @Separators tells us where each of the config files start, and
   # is used only for the commands (entries that start with '*')
   @ReadConfigNames = ();
   @ReadConfigValues = ();
   @Separators = ();
   push (@Separators, scalar(@ReadConfigNames));
   ReadConfigFile("$BaseDir/default.conf/services/$f", "");
   push (@Separators, scalar(@ReadConfigNames));
   ReadConfigFile("$BaseDir/dist.conf/services/$f", "");
   push (@Separators, scalar(@ReadConfigNames));
   ReadConfigFile("$ConfigDir/conf/services/$f","");
   push (@Separators, scalar(@ReadConfigNames));
   ReadConfigFile("$ConfigDir/conf/override.conf", "services/$ThisService");

   @CmdList = ();
   @CmdArgList = ();

   for (my $i = 0; $i <= $#ReadConfigNames; $i++) {
      if (grep(/^$i$/, @Separators)) {
         $count = 0;
      }
      if ($ReadConfigNames[$i] eq 'logfile') {
	 if ($ReadConfigValues[$i] eq "") {
	    @{$ServiceData{$ThisService}{'logfiles'}} = ();
         } elsif (! grep(/^$ReadConfigValues[$i]$/, @{$ServiceData{$ThisService}{'logfiles'}})) {
            push @{$ServiceData{$ThisService}{'logfiles'}}, $ReadConfigValues[$i];
	 }
      } elsif ($ReadConfigNames[$i] =~ /^\*/) {
	 if ($count == 0) {
	    @CmdList = ();
	    @CmdArgList = ();
	 }
         $count++;
         push (@CmdList, $ReadConfigNames[$i]);
         push (@CmdArgList, $ReadConfigValues[$i]);
      } else {
         $ServiceData{$ThisService}{$ReadConfigNames[$i]} = $ReadConfigValues[$i];
      }
   }
   for my $i (0..$#CmdList) {
       $ServiceData{$ThisService}{+sprintf("%03d-%s", $i, $CmdList[$i])} = $CmdArgList[$i];
   }
}
my @AllServices = sort @TempAllServices;

# Find out what logfiles are defined...
opendir(LOGFILEDIR, $BaseDir . "/default.conf/logfiles") or die $BaseDir . "/default.conf/logfiles/, no such directory.\n";
while (defined($ThisFile = readdir(LOGFILEDIR))) {
   unless (-d $BaseDir . "/default.conf/logfiles/" . $ThisFile) {
      my $ThisLogFile = $ThisFile;
      if ($ThisLogFile =~ s/\.conf$//i) {
         push @AllLogFiles, $ThisLogFile;
         @ReadConfigNames = ();
         @ReadConfigValues = ();
         @Separators = ();
         push (@Separators, scalar(@ReadConfigNames));
         ReadConfigFile("$BaseDir/default.conf/logfiles/" . $ThisFile, "");
         push (@Separators, scalar(@ReadConfigNames));
         ReadConfigFile("$BaseDir/dist.conf/logfiles/" . $ThisFile, "");
         push (@Separators, scalar(@ReadConfigNames));
         ReadConfigFile("$ConfigDir/conf/logfiles/" . $ThisFile, "");
         push (@Separators, scalar(@ReadConfigNames));
         ReadConfigFile("$ConfigDir/conf/override.conf", "logfiles/$ThisLogFile");

         @CmdList = ();
         @CmdArgList = ();

         for (my $i = 0; $i <= $#ReadConfigNames; $i++) {
            if (grep(/^$i$/, @Separators)) {
               $count = 0;
            }
            if ($ReadConfigNames[$i] eq "logfile") {
               #Lets try and find the logs -mgt
               if ($ReadConfigValues[$i] eq "") {
		  @{$LogFileData{$ThisLogFile}{'logfiles'}} = ();
	       } else {
		  my $TempLogFileName;
                  if (-e "$Config{'logdir'}/$ReadConfigValues[$i]") {
                     $TempLogFileName = $ReadConfigValues[$i];
                  } elsif (-e "/var/adm/$ReadConfigValues[$i]") {
                     $TempLogFileName = "adm/$ReadConfigValues[$i]";
                  } elsif (-e "/var/log/$ReadConfigValues[$i]") {
                     $TempLogFileName = "log/$ReadConfigValues[$i]";
                  } else {
                     #Fallback to default even if it doesn't exist -mgt
                     $TempLogFileName = $ReadConfigValues[$i];
		  }
                  # We attempt to remove duplicates.  This is not foolproof
                  # because you can create duplicates through convoluted
                  # globbing.  We don't do that with the shipped logfiles.
                  # Same applies to archives, in the next block.
                  if (! grep(/^\Q$TempLogFileName\E$/, @{$LogFileData{$ThisLogFile}{'logfiles'}})) {
                     push @{$LogFileData{$ThisLogFile}{'logfiles'}}, $TempLogFileName;
		  }
	       }
            } elsif ($ReadConfigNames[$i] eq "archive") {
               if ($ReadConfigValues[$i] eq "") {
		  @{$LogFileData{$ThisLogFile}{'archives'}} = ();
	       } elsif (! grep(/^\Q$ReadConfigValues[$i]\E$/, @{$LogFileData{$ThisLogFile}{'archives'}})) {
                  push @{$LogFileData{$ThisLogFile}{'archives'}}, $ReadConfigValues[$i];
	       }
            } elsif ($ReadConfigNames[$i] =~ /^\*/) {
               if ($count == 0) {
	       @CmdList = ();
	       @CmdArgList = ();
	       }
               $count++;
               push (@CmdList, $ReadConfigNames[$i]);
               push (@CmdArgList, $ReadConfigValues[$i]);
            } else {
               $LogFileData{$ThisLogFile}{$ReadConfigNames[$i]} = $ReadConfigValues[$i];
            }
            for my $i (0..$#CmdList) {
                $LogFileData{$ThisLogFile}{+sprintf("%03d-%s", $i, $CmdList[$i])} = $CmdArgList[$i];
            }
         }
      }
   }
}
closedir(LOGFILEDIR);

# Find out what shared functions are defined...
opendir(SHAREDDIR,$BaseDir . "scripts/shared") or die $BaseDir . "scripts/shared/, no such directory.\n";
while (defined($ThisFile = readdir(SHAREDDIR))) {
   unless (-d $BaseDir . "scripts/shared/" . $ThisFile) {
      push @AllShared, lc($ThisFile);
   }
}
closedir(SHAREDDIR);

if ($Config{'debug'} > 5) {
   print "\nAll Services:\n";
   PrintStdArray @AllServices;
   print "\nAll Log Files:\n";
   PrintStdArray @AllLogFiles;
   print "\nAll Shared:\n";
   PrintStdArray @AllShared;
}

#############################################################################

# Time to expand @ServiceList, using @LogFileList if defined...

if ((scalar @ServiceList > 0) && (grep /^all$/i, @ServiceList)) {
    # This means we are doing *all* services ... but excluding some
    my %tmphash;
    foreach my $item (@AllServices) {
      $tmphash{lc $item} = "";
    }
    foreach my $service (@ServiceList) {
      next if $service =~ /^all$/i;
      if ($service =~ /^\-(.+)$/) {
          my $offservice = lc $1;
          if (! grep (/^$offservice$/, @AllServices)) {
             die "Nonexistent service to disable: $offservice\n";
          }
          if (exists $tmphash{$offservice}) {
             delete $tmphash{$offservice};
          }
          
      } else {
          die "Wrong configuration entry for \"Service\", if \"All\" selected, only \"-\" items are allowed\n";
      }
    }
    @ServiceList = ();
    foreach my $keys (keys %tmphash) {
      push @ServiceList, $keys;
    }
    @LogFileList = ();
} else {
   my $ThisOne;
   while (defined($ThisOne = pop @LogFileList)) {
      unless ($LogFileData{$ThisOne}) {
         die "Logwatch is not configured to use logfile: $ThisOne\n";
      }
      foreach my $ThisService (keys %ServiceData) {
         for (my $i = 0; $i <= $#{$ServiceData{$ThisService}{'logfiles'}}; $i++) {
            if ( $ServiceData{$ThisService}{'logfiles'}[$i] eq $ThisOne ) {
               push @ServiceList,$ThisService;
            }
         }
      }
   }
   @TempServiceList = sort @ServiceList;
   @ServiceList = ();
   my $LastOne = "";
   while (defined($ThisOne = pop @TempServiceList)) {
      unless ( ($ThisOne eq $LastOne) or ($ThisOne eq 'all') or ($ThisOne =~ /^-/)) {
         unless ($ServiceData{$ThisOne}) {
            die "Logwatch does not know how to process service: $ThisOne\n";
         }
         push @ServiceList, $ThisOne;
      }
      $LastOne = $ThisOne;
   }
}

# Now lets fill up @LogFileList again...
foreach my $ServiceName (@ServiceList) {
   foreach my $LogName ( @{$ServiceData{$ServiceName}{'logfiles'} } ) {
      unless ( grep m/^$LogName$/, @LogFileList ) { 
         push @LogFileList, $LogName;
      }
   }
}

if ($Config{'debug'} > 7) {
   print "\n\nAll Service Data:\n";
   PrintServiceData;
   print "\nServices that will be processed:\n";
   PrintStdArray @ServiceList;
   print "\n\n";
   print "\n\nAll LogFile Data:\n";
   PrintLogFileData;
   print "\nLogFiles that will be processed:\n";
   PrintStdArray @LogFileList;
   print "\n\n";
}

#############################################################################

#Set very strict permissions because we deal with security logs
umask 0177;
#Making temp dir with File::Temp  -mgt
my $TempDir = tempdir( 'logwatch.XXXXXXXX', DIR => $Config{tmpdir},
      CLEANUP => 0 );

if ($Config{'debug'}>7) {
      print "\nMade Temp Dir: " . $TempDir . " with tempdir\n";
}

unless ($TempDir =~ m=/$=) {
    $TempDir .= "/";
}
   
#############################################################################

# Set up the environment...

$ENV{'LOGWATCH_DATE_RANGE'} = $Config{'range'};
$ENV{'LOGWATCH_GLOBAL_DETAIL'} = $Config{'detail'};
$ENV{'LOGWATCH_OUTPUT_TYPE'} = $Config{'output'};
$ENV{'LOGWATCH_DEBUG'} = $Config{'debug'};
$ENV{'LOGWATCH_TEMP_DIR'} = $TempDir;
$ENV{'LOGWATCH_NUMERIC'} = $Config{'numeric'};
$ENV{'HOSTNAME'} = $Config{'hostname'};
$ENV{'OSname'} = $OSname;

if ($Config{'hostlimit'}) {
   $ENV{'LOGWATCH_ONLY_HOSTNAME'} = $Config{'hostname'};
   $ENV{'LOGWATCH_ONLY_HOSTNAME'} =~ s/\..*//;
}
if ($Config{'debug'}>4) {
   foreach ('LOGWATCH_DATE_RANGE', 'LOGWATCH_GLOBAL_DETAIL', 'LOGWATCH_OUTPUT_TYPE', 
            'LOGWATCH_TEMP_DIR', 'LOGWATCH_DEBUG', 'LOGWATCH_ONLY_HOSTNAME') {
      if ($ENV{$_}) {
         print "export $_='$ENV{$_}'\n";
      }
   }
}

my $LibDir = "$BaseDir/lib";
if ($ENV{PERL5LIB}) {
    # User dirs should be able to override this setting
    $ENV{PERL5LIB} = "$ENV{PERL5LIB}:$LibDir";
} else {
    $ENV{PERL5LIB} = $LibDir;
}

#############################################################################

unless ($Config{'logdir'} =~ m=/$=) {
   $Config{'logdir'} .= "/";
}

# Okay, now it is time to do pre-processing on all the logfiles...

my @EnvList = ();
my $LogFile;
foreach $LogFile (@LogFileList) {
   next if ($LogFile eq 'none');
	if (!defined($LogFileData{$LogFile}{'logfiles'})) {
		print "*** Error: There is no logfile defined. Do you have a " . $ConfigDir . "/conf/logfiles/" . $LogFile . ".conf file ?\n";
		next;
	}

   @FileList = $TempDir . $LogFile . "-archive";
   push @FileList, @{$LogFileData{$LogFile}{'logfiles'}};
   my $Archive;
   foreach my $ArchiveWild (@{$LogFileData{$LogFile}{'archives'}}) {
      unless ($ArchiveWild =~ m=^/=) {
         $ArchiveWild = ($Config{'logdir'} . $ArchiveWild);
      } #End unless
   my $DestFile =  $TempDir . $LogFile . "-archive";
   foreach $Archive (reverse glob $ArchiveWild) {
      unless ($Archive =~ m=^/=) {
         $Archive = ($Config{'logdir'} . $Archive);
      } #End unless

      my $CheckTime;
      # We need to find out what's the earliest log we need
      my @time_t = TimeBuild();
      if ($Config{'range'} eq 'all') {
         if ($Config{'archives'} == 0) {
            # range is 'all', but we don't get archive files
	    $CheckTime = time;
         } else {
            # range is 'all', and we get all archive files
	    $CheckTime = 0;
	 }
      } elsif ($time_t[0]) {
	 # range is something else, and we need to get one
	 # day ahead. A day has 86400 seconds.  (We double
	 # that to deal with different timezones.)
	 $CheckTime = $time_t[0] - 86400*2;
     } else {
         # range is wrong
         $CheckTime = time;
     }

      my @FileStat = stat($Archive);
      if ($CheckTime <= ($FileStat[9])) {
         if (($Archive =~ m/gz$/) && (-e "$Archive")) {
         #These system calls are not secure but we are getting closer
         #What needs to go is all the pipes and instead we need a command loop
         #For each filter to apply -mgt
            my $arguments = "$Archive 2>/dev/null >> $DestFile";
            system("$Config{'pathtozcat'} $arguments") == 0
               or die "system $Config{'pathtozcat'} failed: $?" 
         } elsif (-e "$Archive") {
            my $arguments = "$Archive 2>/dev/null >> $DestFile";
            system("$Config{'pathtocat'} $arguments") == 0
               or die "system $Config{'pathtocat'} failed: $?" 
         } #End if/elsif existence
      } #End if $CheckTime

    } #End Archive
   } #End ArchiveWild
#   } #End if Archives
   my $FileText = "";
   foreach my $ThisFileWild (@FileList) {
    unless ( ($ThisFileWild =~ m=^/=) || ($ThisFileWild =~ m=^$TempDir=) )  {
      $ThisFileWild = ($Config{'logdir'} . $ThisFileWild);
    } #End unless

    foreach my $ThisFile (glob $ThisFileWild) {
      #Existence check for files -mgt
      next unless ( (-e $ThisFile) || (-e $Config{'logdir'} . $ThisFile));
      if ( ($ThisFile =~ m=^/=) || ($ThisFile =~ m=^$TempDir=) ) {
         $FileText .= ($ThisFile . " ");
      } else {
         $FileText .= ( $Config{'logdir'} . $ThisFile . " ");
      } #End if
    } #End glob
   } #End foreach Wild

   # remove the ENV entries set by previous service
   foreach my $Parm (@EnvList) {
      delete $ENV{$Parm};
   }
   @EnvList = ();

   my $FilterText = " 2>/dev/null ";
   foreach (sort keys %{$LogFileData{$LogFile}}) {
      my $cmd = $_;
      if ($cmd =~ s/^\d+-\*//) {
         if (-f "$ConfigDir/scripts/shared/$cmd") {
            $FilterText .= ("| $PerlVersion $ConfigDir/scripts/shared/$cmd '$LogFileData{$LogFile}{$_}'" );
         } elsif (-f "$BaseDir/scripts/shared/$cmd") {
            $FilterText .= ("| $PerlVersion $BaseDir/scripts/shared/$cmd '$LogFileData{$LogFile}{$_}'" );
         } else {
	     die "Cannot find shared script $cmd\n";
         }
      } elsif ($cmd =~ s/^\$//) {
         push @EnvList, $cmd;
         $ENV{$cmd} = $LogFileData{$LogFile}{$_};
         if ($Config{'debug'}>4) {
            print "export $cmd='$LogFileData{$LogFile}{$_}'\n";
         }
      }
   }
   if (opendir (LOGDIR,$ConfigDir . "scripts/logfiles/" . $LogFile)) {
      foreach (sort readdir(LOGDIR)) {
         unless ( -d $ConfigDir . "scripts/logfiles/$LogFile/$_") {
            $FilterText .= ("| $PerlVersion $ConfigDir" . "scripts/logfiles/$LogFile/$_");
         }
      }
      closedir (LOGDIR);
   }
   if (opendir (LOGDIR,$BaseDir . "scripts/logfiles/" . $LogFile)) {
      foreach (sort readdir(LOGDIR)) {
         unless (( -d $BaseDir . "scripts/logfiles/$LogFile/$_") or
                 # if in ConfigDir, then the ConfigDir version is used
                 ( -f "$ConfigDir/scripts/logfiles/$LogFile/$_")) {
            $FilterText .= ("| $PerlVersion $BaseDir" . "scripts/logfiles/$LogFile/$_");
         }
      }
      closedir (LOGDIR);
   }

   #Instead of trying to cat non-existant logs we test for it above -mgt
   if ($FileText) {
      my $Command = $FileText . $FilterText . ">" . $TempDir . $LogFile;
      if ($Config{'debug'}>4) {
         print "\nPreprocessing LogFile: " . $LogFile . "\n" . $Command . "\n";
      }
      if ($LogFile !~ /^[-_\w\d]+$/) {
         print STDERR "Unexpected filename: [[$LogFile]]. Not used\n"
      } else {
      #These system calls are not secure but we are getting closer
      #What needs to go is all the pipes and instead we need a command loop
      #For each filter to apply -mgt
         system("$Config{'pathtocat'} $Command") == 0
            or die "system $Config{'pathtocat'} $Command failed: $?" 
      }
   }
}

#populate the host lists if we're splitting hosts
my @hosts;
if ($Config{'splithosts'} eq 1) {
   my $newlogfile;
   my @logarray;
   opendir (LOGDIR,$TempDir) || die "Cannot open dir";
   @logarray = readdir(LOGDIR);
   closedir (LOGDIR);
   my $ecpcmd = ("| $PerlVersion $BaseDir" . "scripts/shared/hostlist");
   foreach $newlogfile (@logarray) {
     my $eeefile = ("$TempDir" . "$newlogfile");
     if ((!(-d $eeefile)) && (!($eeefile =~ m/-archive/))) {
         system("$Config{'pathtocat'} $eeefile $ecpcmd") == 0
            or die "system $Config{'pathtocat'} $eeefile $ecpcmd failed: $?" 
     }
   }
   #read in the final host list
   open (HOSTFILE,"$TempDir/hostfile") || die $!;
   @hosts = <HOSTFILE>;
   close (HOSTFILE);
   chomp @hosts;
   @hosts = sort(@hosts);
}

#############################################################################

my $report_finish = "\n ###################### LogWatch End ######################### \n\n";
my $printing = '';
my $emailopen = '';

sub initprint {
   return if $printing;
   if ($Config{'print'} eq 1) {
      *OUTFILE = *STDOUT;
   } elsif ($Config{'save'} ne "") {
      open(OUTFILE,">" . $Config{'save'}) or die "Can't open output file: $Config{'save'}\n";
   } elsif ($OSname eq "SunOS") {
      #Solaris mail doesn't know -s -mgt
      if (($Config{'multiemail'} eq 1) || ($emailopen eq "")) {
         open(OUTFILE,"|$Config{'mailer'} $Config{'mailto'}") or die "Can't execute /bin/mail\n";
         print OUTFILE "From: LogWatcher\n";
         print OUTFILE "To: $Config{'mailto'}\n";
	 if ( $outtype_htmlembed ) {
	     print OUTFILE "MIME-Version: 1.0\n"; 
        print OUTFILE "Content-Transfer-Encoding: 7bit\n";
        print OUTFILE "Content-Type: text/html; charset=us-ascii\n";

	 }
	 print OUTFILE "Subject: LogWatch for $Config{'hostname'}\n\n";
         if (($Config{'splithosts'} eq 1) && ($Config{'multiemail'} eq 0)) {
            print OUTFILE "Reporting on hosts: @hosts\n";
         }
         $emailopen = 'y';
      }
   } else {
       if (($Config{'multiemail'} eq 1) || ($emailopen eq "")) {
	   if ( $outtype_html) {
	       open(OUTFILE,"|uuencode $Config{'hostname'}.html |$Config{'mailer'} -s \"LogWatch for $Config{'hostname'}\" " . $Config{'mailto'}) or die "Can't execute /bin/mail\n";
	   } elsif ( $outtype_htmlembed ) {
	       #open(OUTFILE,"|$Config{'sendmail'} -s \"LogWatch for $Config{'hostname'}\" " . $Config{'mailto'}) or die "Can't execute sendmail\n";
	       open(OUTFILE,"|$Config{'sendmail'} -t") or die "Can't execute sendmail\n";
	       $out_mime.=  "To: " . $Config{'mailto'} . "\n";
	       $out_mime.=  "From: LogWatcher\n";
	       $out_mime.=  "Subject: LogWatch for " . $Config{'hostname'} . "\n";
	       $out_mime.= "MIME-Version: 1.0\n"; 
          $out_mime.= "Content-Transfer-Encoding: 7bit\n";
          $out_mime.= "Content-Type: text/html; charset=\"iso-8859-1\"\n\n"


	   } else {
	       open(OUTFILE,"|$Config{'mailer'} -s \"LogWatch for $Config{'hostname'}\" " . $Config{'mailto'}) or die "Can't execute /bin/mail\n";
	   }
	   if (($Config{'splithosts'} eq 1) && ($Config{'multiemail'} eq 0)) {
	       print OUTFILE "Reporting on hosts: @hosts\n";
	   }
	   $emailopen = 'y';
       }
   }
   $printing = 'y';

   # simple parse of the dates
   my $simple_timematch = TimeFilter(" %Y-%b-%d %Hh %Mm %Ss ");
   my @simple_range = split(/\|/, $simple_timematch);
   if ($#simple_range > 1) {
       # delete all array entries, except first and last
       splice(@simple_range, 1, $#simple_range-1);
   }
   for (my $range_index=0; $range_index<$#simple_range+1; $range_index++) {
       $simple_range[$range_index] =~ s/\.\.[hms]//g;
       $simple_range[$range_index] =~ s/\.//g;
       $simple_range[$range_index] =~ tr/--//s;
       $simple_range[$range_index] =~ s/ -|- //;
       $simple_range[$range_index] =~ tr/ //s;
   }

   my $print_range = join("/",@simple_range);

   $index_par++;
   outstart($index_par,"LOGWATCH Summary");
   if ( $outtype_html or $outtype_htmlembed ) {
       outheader( $index_par, "\n <h2><font color=\"blue\"> LogWatch $Version ($VDate)</font></h2>\n");
   }       else {
       outline( $index_par, "\n ################### LogWatch $Version ($VDate) #################### \n");
   }
   outline( $index_par, "       Processing Initiated: " . localtime(time) . "\n");
   outline( $index_par, "       Date Range Processed: $Config{'range'}\n");
   outline( $index_par, "                             $print_range\n") if ($Config{'range'} ne 'all');
   outline( $index_par, "                             Period is " . GetPeriod() . ".\n")
      if ($Config{'range'} ne 'all');
   outline( $index_par, "     Detail Level of Output: $Config{'detail'}\n");
   outline( $index_par, "             Type of Output: $Config{'output'}\n");
   outline( $index_par, "          Logfiles for Host: $Config{'hostname'}\n");

   if ( $outtype_html or $outtype_htmlembed ) {
   outline( $index_par, "\n");
   } else {
   outline( $index_par, " ################################################################## \n");
   }
   outstop( $index_par );


}

sub parselogs {
   my $Service;

   #Load our ignore file
   my @IGNORE;
   if ( -e "$ConfigDir/conf/ignore.conf") {
   open( IGNORE, "$ConfigDir/conf/ignore.conf" )  or return undef;
   @IGNORE = grep {!/(^#|^\s+$)/} <IGNORE>;
   close IGNORE;
   }

   my @EnvList = ();
   foreach $Service (sort @ServiceList) {
      my $Ignored = 0;
      $ENV{'PRINTING'} = $printing;
      if (defined $ServiceData{$Service}{'detail'}) {
	 $ENV{'LOGWATCH_DETAIL_LEVEL'} = $ServiceData{$Service}{'detail'};
      } else {
         $ENV{'LOGWATCH_DETAIL_LEVEL'} = $ENV{'LOGWATCH_GLOBAL_DETAIL'};
      }
      @FileList = @{$ServiceData{$Service}{'logfiles'}};
      my $FileText = "";
      foreach $ThisFile (@FileList) {
         if (-s $TempDir . $ThisFile) {
            $FileText .= ( $TempDir . $ThisFile . " ");
         }
      }

      # remove the ENV entries set by previous service
      foreach my $Parm (@EnvList) {
         delete $ENV{$Parm};
      }
      @EnvList = ();

      my $FilterText = " ";
      foreach (sort keys %{$ServiceData{$Service}}) {
         my $cmd = $_;
         if ($cmd =~ s/^\d+-\*//) {
            if (-f "$ConfigDir/scripts/shared/$cmd") {
               $FilterText .= ("$PerlVersion $ConfigDir/scripts/shared/$cmd '$ServiceData{$Service}{$_}' |" );
            } elsif (-f "$BaseDir/scripts/shared/$cmd") {
               $FilterText .= ("$PerlVersion $BaseDir/scripts/shared/$cmd '$ServiceData{$Service}{$_}' |" );
            } else {
               die "Cannot find shared script $cmd\n";
            }
         } elsif ($cmd =~ s/^\$//) {
            $ENV{$cmd} = $ServiceData{$Service}{$_};
            push @EnvList, $cmd;
            if ($Config{'debug'}>4) {
               print "export $cmd='$ServiceData{$Service}{$_}'\n";
            }
         }
      }
# ECP - insert the host stripping now
      my $HostStrip = " ";
      if ($Config{'splithosts'} eq 1) {
         $HostStrip .= ("$PerlVersion $BaseDir" . "scripts/shared/onlyhost");
      }
      my $ServiceExec = "$BaseDir/scripts/services/$Service";
      if (-f "$ConfigDir/scripts/services/$Service") {
	 $ServiceExec = "$ConfigDir/scripts/services/$Service";
      } else {
         $ServiceExec = "$BaseDir/scripts/services/$Service";
      }

      if (-f $ServiceExec ) {
         #If shell= was set in service.conf we will use it
         if ($ServiceData{$Service}{shell}) {
            my $shelltest = $ServiceData{$Service}{shell};
            $shelltest =~ s/([\w\/]+).*/$1/;
            if (-e "$shelltest") {
               $FilterText .= "$ServiceData{$Service}{shell} $ServiceExec";
            } else {
               die "Can't use $ServiceData{$Service}{shell} for $ServiceExec";
            }
         } else {
            $FilterText .= "$PerlVersion $ServiceExec";
         } #End if shell
      }
      else {
         die "Can't open: " . $ServiceExec;
      }

      my $Command = '';
      if ($FileList[0] eq 'none') {
         $Command = " $FilterText 2>&1 "; 
      } elsif ($FileText) {
         if ($HostStrip ne " ") {
            $Command = " ( $Config{'pathtocat'} $FileText | $HostStrip | $FilterText) 2>&1 "; 
         } else {
            $Command = " ( $Config{'pathtocat'} $FileText | $FilterText) 2>&1 "; 
         }
      }
   
      if ($Command) {
         if ($Config{'debug'}>4) {
            print "\nProcessing Service: " . $Service . "\n" . $Command . "\n";
         }
         open (TESTFILE,$Command . " |");
         my $ThisLine;
         my $has_output = 0;
         LINE: while (defined ($ThisLine = <TESTFILE>)) {
            next LINE if ((not $printing) and $ThisLine =~ /^\s*$/);
            IGNORE: for my $ignore_filter (@IGNORE) {
               chomp $ignore_filter;
               if ($ThisLine =~ m/$ignore_filter/) {
                  $Ignored++;
                  next LINE;
                  }
            }
            initprint();
            if (($has_output == 0) and ($ServiceData{$Service}{'title'})) {
               $index_par++;
               outstart($index_par, $ServiceData{$Service}{'title'} );
               my $BeginVar;
               if ($ENV{'LOGWATCH_GLOBAL_DETAIL'} == $ENV{'LOGWATCH_DETAIL_LEVEL'}) {
		  $BeginVar = "Begin";
               } else {
		  $BeginVar = "Begin (detail=" . $ENV{'LOGWATCH_DETAIL_LEVEL'} . ")";
               }
               if ( $outtype_html or $outtype_htmlembed ) {
                   outheader( $index_par, "\n <h3><font color=\"blue\">$ServiceData{$Service}{'title'} $BeginVar </font></h3>\n");
               } else {
                   outline( $index_par, "\n --------------------- $ServiceData{$Service}{'title'} $BeginVar ------------------------ \n\n");
               }
               $has_output = 1;
            }
            outline( $index_par, $ThisLine);
         }
         close (TESTFILE);
         if ($has_output and $ServiceData{$Service}{'title'}) {
            if ( $outtype_html or $outtype_htmlembed ) {
                if ($Ignored > 0) {  outheader( $index_par, "\n $Ignored Ignored Lines\n"); };
                outheader( $index_par,  "\n <h3><font color=\"blue\">$ServiceData{$Service}{'title'} End </font></h3>\n");
            } else {
                if ($Ignored > 0) { outline( $index_par, "\n $Ignored Ignored Lines\n"); };
                outline( $index_par,  "\n ---------------------- $ServiceData{$Service}{'title'} End ------------------------- \n\n");
            }
            outstop($index_par);
         }
      }
   }

   if ( $outtype_htmlembed ) {
       $out_head .= $out_mime;
   }
   
   if ( $outtype_html or $outtype_htmlembed ) {
   $out_head .=
   "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\"\"http://www.w3.org/TR/REC-html40/loose.dtd\">
   <html>
    <head>
   ";
   $out_head .="<title>LogWatch ". $Version . " (" .$VDate.")</title>";
   $out_head .="<meta name=\"generator\" content=\"LogWatch " . $Version . "(".$VDate.")" . "\">";
   $out_head .="
   <style type=\"text/css\">
         h1 {color: gray; border-bottom: 3px double silver}
         h2,h3 {color: gray; border-bottom: 1px solid silver}
         .sample {border-left: 3px double gray; padding-left: 1%}
         .explan {color: green; border-top: 1px solid green;
         border-bottom: 1px solid green;
         padding: 1%; margin-top: 1%; margin-bottom: 1%;}
         .copyright {color: black; border-top: 1px solid grey;
         border-bottom: 1px solid grey;
         padding: 1%; margin-top: 1%; margin-bottom: 1%;}
       </style>
    </head>
    <body style=\"width:70%; margin-left: 15%\" bgcolor=\"#FFFFFF\">
    <hr>";
   }
   
   
      if ( $outtype_html or $outtype_htmlembed ) {
          $index_par++;
          outstart( $index_par,  "LogWatch End" );
          outheader( $index_par,  "\n <h3><font color=\"blue\">LogWatch ended at ".  localtime(time) ."</font></h3>\n") if ($printing);
          outstop($index_par);
      }   else {
          outline( $index_par, $report_finish) if ($printing);
      }
   
      if ( $outtype_html or $outtype_htmlembed ) {
   
      print OUTFILE $out_head;
      print OUTFILE $out_body{'E'} if defined $out_body{'E'};
   
  
     print OUTFILE "<hr><ul>\n";
     #foreach ( 'E', 0 .. $index_par ) {
     foreach ( 0 .. $index_par ) {
         outref($_,$reports[$_]) if defined( $reports[$_] );
     }
     print OUTFILE $out_reference;
     print OUTFILE "</ul></hr>\n";
  
  
     }
  
  ######## In Any Case print the report ########
  
     foreach ( 'E', 0 .. $index_par ) {
         if ( $outtype_html or $outtype_htmlembed ) {
        print OUTFILE $out_body{$_} if defined( $out_body{$_} );
         }
         else {
        print OUTFILE $out_body{$_} if defined( $out_body{$_} );
         }
  
     }
  
  
     if ( $outtype_html or $outtype_htmlembed ) {
  
    #Copyright notice -- FIXEME KIRK -mgt
    print OUTFILE '<div class=\"copyright\">
     <hr>
     <p>LogWatch HTML report is also based on the work done by the people of Calamaris, see address for details</p>
     <address><a href="http://Calamaris.Cord.de/">Calamaris</a>
     $Revision: 1.142 $, Copyright &copy; 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004
     <a href="http://Cord.de/">Cord Beermann</a>.
    </address>
    <p>Calamaris comes with ABSOLUTELY NO WARRANTY. It is free software, and you
     are welcome to redistribute it under certain conditions.  See source for
     details.
    </p></div>
  ';
      print OUTFILE ' </body></html>', "\n";
  
    #if ($outtype_htmlembed) {
    #  print OUTFILE '--=_alternative 003B1B25C1256EE7_=--', "\n";
    #}
  }
  
   if ($Config{'multiemail'} eq 1) {
      close(OUTFILE) unless ($Config{'print'} eq 1);
   }
}

if ($Config{'splithosts'} eq 1) {
   my $Host;
   foreach $Host (@hosts) {
      $printing = '';
      $ENV{'LOGWATCH_ONLY_HOSTNAME'} = $Host;
      $ENV{'LOGWATCH_ONLY_HOSTNAME'} =~ s/\..*//;
      $Config{'hostname'} = $Host;
      parselogs();
   } # ECP
} else {
   parselogs();
}
close(OUTFILE) unless ($Config{'print'} eq 1);
#############################################################################

# Get rid of temp directory...
if ($Config{'debug'}<100) {
   opendir(TEMPDIR, $TempDir);
   my @files = grep(!/^\./, readdir(TEMPDIR));
   closedir(TEMPDIR);

   for my $tempfile (@files) { unlink "$TempDir/$tempfile"; };
   rmdir $TempDir;
}

exit(0);

sub outref {
    my $name = shift (@_);
    $out_reference .= "   <li><a href=\"#$name\">$reports[$name]</a>\n";
}

sub outstart {
    my $index = shift (@_);
    my $name = shift (@_);
    $reports[$index]=$name;
    if ( $outtype_html or $outtype_htmlembed ) {
	if ( $index eq 'E' ) {
	    $out_body{$index} .= "  <hr><table border=\"0\">\n";
	} else {
	    $out_body{$index} .= "  <hr>
  <h2><a name=\"$index\">$reports[$index]</a></h2>
  <div class=\"sample\">
  <table border=\"1\">\n";
	}
   # } else {
	#$out_body{$index} .= "\n# $reports[$index]\n" unless ( $index eq 'E' );
    }
}

sub outheader {
    my $index = shift (@_);
    my $print;
    my $no = 0;
    $out_body{$index} .= "   <tr>\n" if ( $outtype_html or $outtype_htmlembed );
    foreach (@_) {
	$p = $_;
	if ($outtype_unformatted) {
	    $out_body{$index} .= "$p ";
	} elsif ( $outtype_html or $outtype_htmlembed ) {
      $p =~ s# +# #go;
      $p =~ s#(^ | $)##go;
      $p = '&nbsp;' if ( $p eq '' );
      $out_body{$index} .= "    <th>$p</th>\n";
  } elsif ( $format[$no] eq '%' ) {
      $out_body{$index} .=
	  sprintf( ' ' x ( 6 - length($p) ) ) . substr( $p, 0, 6 ) . ' ';
  } elsif ( $format[$no] eq 'kbs' ) {
      $out_body{$index} .=
	  sprintf( substr( $p, 0, 7 ) . ' ' x ( 7 - length($p) ) . ' ' );
  } else {
      $out_body{$index} .=
	  sprintf(
		  substr( $p, 0, $format[$no] )
		  . ' ' x( $format[$no] - length($p) ) . ' ' );
  }
	$no++;
    }
    $out_body{$index} .= '   </tr>' if ( $outtype_html or $outtype_htmlembed );
    $out_body{$index} .= "\n";
}

sub outline {
    my $index = shift (@_);
    my $print;
    my $no = 0;
    $out_body{$index} .= "   <tr>\n" if ( $outtype_html or $outtype_htmlembed );
    foreach (@_) {
	$print = $_;
	if ($outtype_unformatted or $outtype_mail) {
	    $out_body{$index} .= "$print ";
	} elsif ( $outtype_html or $outtype_htmlembed ) {
      $print =~ s# +# #go;
      $print =~ s# $##go;
      $print =~ s#<#\&lt\;#go;
      $print =~ s#>#\&gt\;#go;
      if ( $no == 0 ) {
	  unless ( $print =~ s#^ ##go ) {
		   $out_body{$index} .= "    <th align=\"left\">$print</th>\n";
	       } else {
		   $out_body{$index} .= "    <td bgcolor=\"#dddddd\">&nbsp;&nbsp;$print</td>\n";
	       }
      } elsif ( $format[$no] eq '%' or $format[$no] eq 'kbs' ) {
	  if ( $print eq '' or $print eq '-' ) {
	      $out_body{$index} .= "    <td>&nbsp;</td>\n";
	  } else {
	      $out_body{$index} .=
		  sprintf( "    <td align=\"right\">%.2f</td>\n", $print );
	  }
      } elsif ( $no == 1 or $print =~ m#^[\d\.e\-\+]+$#o ) {
		$out_body{$index} .=
		sprintf( "    <td align=\"right\">%d</td>\n", $print );
	    } else {
		if ($print) {
		    $out_body{$index} .= "    <td align=\"right\">$print</td>\n";
		} else {
		    $out_body{$index} .= "    <td align=\"right\">&nbsp;</td>\n";
		}
	    }
  } else {
      if ( $no == 0 ) {
	  if ( length($print) > $format[$no] ) {
	      $out_body{$index} .=
		  sprintf( $print . "\n" . ' ' x $format[$no] . ' ' );
	  } else {
	      $out_body{$index} .=
		  sprintf( $print . ' ' x ( $format[$no] - length($print) ) . ' ' );
	  }
      } elsif ( $format[$no] eq '%' ) {
	  if ( $print eq ' ' ) {
	      $out_body{$index} .= ' ' x 7;
	  } else {
	      $out_body{$index} .= sprintf( "%6.2f ", $print );
	  }
      } elsif ( $format[$no] eq 'kbs' ) {
	  if ( $print eq '-' ) {
	      $out_body{$index} .= '    -   ';
	  } else {
	      if ( $print >= 10000 ) {
		  $out_body{$index} .= sprintf( "%7.0f ", $print );
	      } else {
		  $out_body{$index} .= sprintf( "%7.2f ", $print );
	      }
	  }
      } else {
	  $print = sprintf( "%d", $print + .5 ) if $print =~ m#^[\d\.e\-\+]+$#o;
	  $out_body{$index} .=
	        sprintf( ' ' x ( $format[$no] - length($print) )
			 . substr( $print, 0, $format[$no] ) . ' ' );
      }
  }
	$no++;
    }
    $out_body{$index} .= '   </tr>' if ( $outtype_html or $outtype_htmlembed );
    $out_body{$index} .= "\n" if ( $outtype_html or $outtype_htmlembed );
}

sub outseperator {
    my $index = shift (@_);
    my $print;
    $out_body{$index} .= "   <tr>\n" if ( $outtype_html or $outtype_htmlembed );
    foreach $print (@format) {
	if ($outtype_unformatted) {
	    $out_body{$index} .= "--- ";
	} elsif ( $outtype_html or $outtype_htmlembed ) {
	    $out_body{$index} .= "    <td></td>\n";
	} elsif ( $print eq '%' ) {
	    $out_body{$index} .= sprintf( '-' x 6 . ' ' );
	} elsif ( $print eq 'kbs' ) {
	    $out_body{$index} .= sprintf( '-' x 7 . ' ' );
	} else {
	    $out_body{$index} .= sprintf( '-' x $print . ' ' );
	}
    }
    $out_body{$index} .= '   </tr>' if ( $outtype_html or $outtype_htmlembed );
    $out_body{$index} .= "\n";
}

sub outstop {
    my $index = shift (@_);
    if ( $outtype_html or $outtype_htmlembed ) {
	$out_body{$index} .= "  </table></div>\n";
	$out_body{$index} .= "  <div class=\"explan\"><p><a href=\"#top\">Back to Top</a></p></div>\n";
    }
}


# vi: shiftwidth=3 tabstop=3 et

