# Copyright (C) 2009-2010 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""
Make it possible for Apport to catch crashes of LottaNZB and make it easy for
users to report them using Launchpad.

The log and both the configuration of LottaNZB and HellaNZB are included in the
bug report. Sensitive information like passwords, usernames and Usenet servers
are stripped from the report for privacy reasons.

See https://wiki.ubuntu.com/Apport/DeveloperHowTo for more information.
"""

import os
import re
import getpass
import apport

from os.path import join, expanduser
from apport.hookutils import *

try:
    from lottanzb.resources import get_config
except:
    # Gracefully fallback, if there's a problem importing the base modules.
    # This could for example be caused by a syntax error.
    def get_config(*subpaths):
        return join(expanduser("~/.config/lottanzb"), *subpaths)

def add_info(report):
    # Don't file bugs against the Ubuntu package in Launchpad, but directly
    # to the application project page if the package has been downloaded from
    # the LottaNZB homepage.
    # Note: This won't work with Apport 1.10 or newer unless bug #517272 has
    # been fixed.
    if not apport.packaging.is_distro_package(report["Package"].split()[0]):
        report["ThirdParty"] = "True"
        report["CrashDB"] = "lottanzb"
    
    attach_related_packages(report, ["lottanzb", "hellanzb"])
    
    attach_file_if_exists(report, get_config("log"), "log")
    attach_file_if_exists(report, get_config("lottanzb.conf"), "lottanzb.conf")
    attach_file_if_exists(report, get_config("hellanzb.conf"), "hellanzb.conf")
    
    luser = getpass.getuser()
    repl = "@@APPORTREPLACED@@"
    
    # Strip passwords, usernames as well as Usenet servers URLs from the report.
    if "log" in report:
        report["log"] = report["log"].replace(luser, repl)
    
    if "hellanzb.conf" in report:
        config = report["hellanzb.conf"]
        config = re.sub(r"hosts=\['.+'\]", "hosts=['%s']" % repl, config)
        
        for key in ["id", "username", "password", "bindTo"]:
            config = re.sub("%s='.+?'" % key, "%s='%s'" % (key, repl), config)
        
        for key in ["NEWZBIN_USERNAME", "NEWZBIN_PASSWORD", "GROWL_PASSWORD", 
                    "GROWL_SERVER", "XMLRPC_PASSWORD", "XMLRPC_SERVER_BIND", 
                    "XMLRPC_SERVER"]:
            clean_line = "%s = \"%s\"" % (key, repl)
            config = re.sub("%s = \"?.+\"?" % key, clean_line, config)
            
        config = config.replace(luser, repl)
        
        report["hellanzb.conf"] = config
    
    if "lottanzb.conf" in report:
        config = report["lottanzb.conf"]
        
        for key in ["address", "username", "password"]:
            config = re.sub("%s = .+" % key, "%s = %s" % (key, repl), config)
            
        config = config.replace(luser, repl)
        
        report["lottanzb.conf"] = config
