/*
 * dlock.c - Linux /proc-based lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1998 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1998 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.4 99/11/10 15:02:35 abe Exp $";
#endif


#include "lslk.h"


/*
 * Local static variables
 */

static FILE *Pls = (FILE *)NULL;	/* /proc/locks stream */
static int PPath = 0;			/* /proc/<PID>/fd has path names */


/*
 * Local definitions
 */


/*
 * Local function prototypes
 */

_PROTOTYPE(static void get_path_and_sz,(void));
_PROTOTYPE(static int savelock,(char **fp));


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	char buf[MAXPATHLEN], **fp;
/*
 * Read and process the lines of /proc/locks.
 */
	while (fgets(buf, sizeof(buf), Pls)) {
	    if (get_fields(buf, ":", &fp) < 10)
		continue;
	    if (!fp[1] || strcmp(fp[1], "->") == 0)
		continue;
	    if (savelock(fp)) {
		if (is_lock_sel()) {
		    (void) get_path_and_sz();
		    NLockU++;
		    Lp = (struct llock_info *)NULL;
		}
	    }
	}
	(void) fclose(Pls);
}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[MAXPATHLEN];
	char *cp, **fp;
	FILE *ss;

	(void) sprintf(buf, "%s/%lu/stat", PROCFS, lp->pid);
	if ((ss = fopen(buf, "r"))) {
	    cp = fgets(buf, sizeof(buf), ss);
	    (void) fclose(ss);
	    if (cp
	    &&  get_fields(buf, (char *)NULL, &fp) >= 2
	    &&  fp[0] && lp->pid == (unsigned long)atoi(fp[0])
	    &&  fp[1] && *fp[1] == '(' && strlen(fp[1] + 1) >= 2)
	    {
		if ((cp = strrchr(fp[1] + 1, ')')))
		   *cp = '\0';
		return(fp[1] + 1);
	    }
	}
	return("(unknown)");
}


/*
 * get_path_and_sz() - get path name and size for lock file
 */

static void
get_path_and_sz()
{
	char buf[MAXPATHLEN], *cp, path[MAXPATHLEN];
	struct dirent *dp;
	DIR *ds;
	int f, fx, i, n, sv;
	MALLOC_S len;
	struct stat sb;
/*
 * Prepare to search the /proc/<PID>/fd/ directory for the device and inode.
 */
	(void) sprintf(buf, "%s/%lu/fd", PROCFS, Lp->pid);
	if (!(ds = opendir(buf)))
	    return;
	if ((fx = strlen(buf)) >= (sizeof(buf) - 2)) {
	    (void) closedir(ds);
	    return;
	}
	buf[fx++] = '/';
/*
 * Read the fd entries for the process.
 */
	while ((dp = readdir(ds))) {

	/*
	 * Check for a numeric file descriptor number.
	 */
	     for (cp = dp->d_name, f = n = 0; *cp; *cp++) {

#if	defined(__STDC__)	/* { */
		if (!isdigit((unsigned char)*cp))
#else	/* !defined(__STDC__)	   } { */
		if (!isascii(*cp) || !isdigit((unsigned char)*cp))
#endif	/* defined(__STDC__)	   } */

		{
		    f = 1;
		    break;
		}
		n++;
	    }
	/*
	 * Skip non-file-descriptor names and names that are too long.
	 */
	    if (f)
		continue;
	    if ((fx + n + 1) >= (sizeof(buf) - 1))
		continue;
	/*
	 * Stat(2) the file descriptor file to get the device, inode, and
	 * file length.  Stat(2) a file on an NFS file system via the
	 * local statsafely() function.  Ignore a result whose device and
	 * inode don't match those of the lock.  If a match is located, save
	 * the file size.
	 */
	    (void) strcpy(&buf[fx], dp->d_name);
	    if (HasNFS) {
		for (i = 0; i < NMnt; i++) {
		    if (!Mnt[i].priv || strcasecmp((char *)Mnt[i].priv, "nfs"))
			continue;
		    if (Lp->dev == Mnt[i].dev)
			break;
		}
	    } else
		i = NMnt;
	    if (i < NMnt)
		sv = statsafely(buf, &sb);
	    else
		sv = stat(buf, &sb);
	    if (sv || Lp->dev != sb.st_dev || Lp->inum != sb.st_ino)
		continue;
	    Lp->sz = sb.st_size;
	    Lp->szs = 1;
	/*
	 * If path names are available and none has yet been recorded
	 * for this lock, try to obtain it.
	 */
	    if (!PPath || Lp->path)
		break;
	    if ((len = readlink(buf, path, sizeof(path))) < 1)
		break;
	    if (!(Lp->path = (char *)malloc(len + 1)))
		break;
	    (void) strncpy(Lp->path, path, len);
	    Lp->path[len] = '\0';
	    break;
	}
	(void) closedir(ds);
}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	char p1[MAXPATHLEN], p2[MAXPATHLEN], p3[MAXPATHLEN];
	int fd;
	size_t len;
/*
 * Open a stream to /proc/locks.
 */
	(void) sprintf(p1, "%s/locks", PROCFS);
	if (!(Pls = fopen(p1, "r"))) {
	    (void) fprintf(stderr, "%s: can't fopen(%s)\n", Pn, p1);
	    Exit(1);
	}
/*
 * See if the /proc/<PID>/fd/<file_descriptor> for the /proc/locks
 * stream has a path name.  Set PPath if it does to enable later
 * path name storage for locks.
 */
	fd = fileno(Pls);
	(void) sprintf(p2, "%s/%d/fd/%d", PROCFS, getpid(), fd);
	if ((len = readlink(p2, p3, sizeof(p3) - 1)) > 0) {
	    p3[len] = '\0';
	    if (strcmp(p1, p3) == 0)
		PPath = 1;
	}
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}


/*
 * savelock() -- save lock information
 */

static int
savelock(fp)
	char **fp;			/* /proc/locks field pointers */
{
	unsigned long end, start;
	char *ep;
	long maj, min;
/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save the lock's mandatory status.
 */
	if (!fp[2])
	    return(0);
	if (*fp[2] == 'M')
	    Lp->mand = 1;
/*
 * Save the lock's type.
 */
	if (!fp[3])
	    return(0);
	if (*fp[3] == 'R')
	    Lp->type = 1;
	else if (*fp[3] == 'W')
	    Lp->type = 2;
	else
	    return(0);
/*
 * Save the lock's PID.
 */
	if (!fp[4] || !*fp[4])
	   return(0);
	Lp->pid = (unsigned long)atoi(fp[4]);
/*
 * Save the lock's device number.
 */
	ep = (char *)NULL;
	if (!fp[5] || !*fp[5]
	||  (maj = strtol(fp[5], &ep, 16)) == LONG_MIN || maj == LONG_MAX
	||  !ep || *ep)
	    return(0);
	ep = (char *)NULL;
	if (!fp[6] || !*fp[6]
	||  (min = strtol(fp[6], &ep, 16)) == LONG_MIN || min == LONG_MAX
	||  !ep || *ep)
	    return(0);
	Lp->dev = (dev_t)makedev((int)maj, (int)min);
/*
 * Save the lock's inode number.
 */
	if (!fp[7] || !*fp[7])
	    return(0);
	Lp->inum = (ino_t)atoi(fp[7]);
/*
 * Save the lock's start and end values.
 */
	if (!fp[8] || !*fp[8] || !fp[9] || !*fp[9])
	    return(0);
	Lp->start = (unsigned long)atoi(fp[8]);
	Lp->ss = 1;
	Lp->end = (unsigned long)atoi(fp[9]);
	Lp->es = 1;
/*
 * Defer the getting of file size and path name until it is known that
 * the lock has been selected.
 *
 * Set the lock source to local; Linux doesn't support remote locking.
 */
	Lp->src = 0;
}
