/** vim:sw=4:sts=4
 *
 * Hash search when the cmph library is used.  Only the FCH algorithm is
 * supported.
 * by Wolfgang Oertl 2007
 *
 * This code is derived from the cmph library (http://cmph.sourceforge.net/)
 * version 0.6 by Davi de Castro Reis and Fabiano Cupertino Botelho.
 *
 * I could have linked lua-gtk with the cmph library, but that would have
 * resulted in an almost 50 kB larger lua-gtk.  This is so because the cmph
 * library supports more algorithms and includes the generation code too,
 * whereas I only need the very much simpler lookup code at runtime.
 */

#include "hash-cmph.h"
#include <stdio.h>	/* NULL */


/**
 * From fch.c.
 *
 * Note: the type of p1 and p2 was changed from float to unsigned int, because
 * they can only contain integers anyway.
 */
static int mixh10h1h12(unsigned int b, unsigned int p1, unsigned int p2,
    unsigned int i)
{
    if (i < p1) {
	i %= p2;
    } else {
	i %= b;
	if (i < p2)
	    i += p2;
    }

    return i;
}


/**
 * Hash lookup function.  Returns a bucket number; any input string results
 * in a valid bucket.  Whether the key is in the hash table has to be
 * determined later from the contents of the bucket.
 */
unsigned int fch_search(const struct hash_info *hi, const unsigned char *key,
    int keylen, unsigned int *hash_value)
{
    const struct fch_data *fch = (const struct fch_data*) hi->algo_data;
    unsigned int h1, h2, g;

    // Calculate a first hash value; it is also used for comparison with the
    // hash value stored in the bucket to identify hits and misses.
    *hash_value = h1 = compute_hash(&fch->h1, key, keylen, NULL);

    // The first hash value is used to achieve the "minimal" and "perfect"
    // properties of the hash algorithm.  Using it an entry in the "g"
    // table is looked up, which is added to the second hash value and
    // mapping it to the interval [0, n-1] without holes or duplicates.
    h1 = mixh10h1h12(fch->b, fch->p1, fch->p2, h1 % fch->m);

    // The "g" table may contain 16 or 32 bit entries depending on the
    // number of buckets; up to 2 ^ 16 it is enough to store 16 bit.
    switch (fch->g_size) {
	case 16:
	g = fch->g[h1];
	break;

	case 32:
	g = fch->g[h1 * 2] + (fch->g[h1*2 + 1] << 16);
	break;

	default:
	return -1;
    }

    // Calculate the second hash value and the final bucket number.
    h2 = compute_hash(&fch->h2, key, keylen, NULL) % fch->m;
    return (h2 + g) % fch->m;
}


