/*
Copyright (c) 1996-2006 Taco Hoekwater <taco@luatex.org>

This file is part of LuaTeX.

LuaTeX is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

LuaTeX is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with LuaTeX; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

$Id$
*/

#include "ptexlib.h"

#include "luatex-api.h"

/* 
   The truetype and opentype specifications are a bit of a mess.  I
   had no desire to spend months studying the sometimes conflicting
   standards by Adobe and Microsoft, so I have deduced the internal
   file stucture of ttf&otf from George Williams' font editor,
   FontForge. 
*/

#include "ttf.h"

#undef strstart
#undef name

#include "ustring.h"
#include "chardata.h"

extern char *AdobeExpertEncoding[], *AdobeStandardEncoding[];

extern const char *ttfstandardnames[];

/* there are likely preference setting */
int ask_user_for_cmap = 0;
char *SaveTablesPref = NULL;


/* When given a postscript string it SHOULD be in ASCII. But it will often*/
/* contain a copyright symbol (sometimes in latin1, sometimes in macroman)*/
/* unfortunately both encodings use 0xa9 for copyright so we can't distinguish */
/* guess that it's latin1 (or that copyright is the only odd char which */
/* means a latin1 conversion will work for macs too). */

int  u_strlen(register const unichar_t *str) {
    register int len = 0;

    while ( *str++!='\0' )
        ++len;
    return( len );
}


int32 utf8_ildb(const char **_text) {
    int32 val= -1;
    int ch;
    const uint8 *text = (const uint8 *) *_text;
    /* Increment and load character */

    if ( (ch = *text++)<0x80 ) {
        val = ch;
    } else if ( ch<=0xbf ) {
        /* error */
    } else if ( ch<=0xdf ) {
        if ( *text>=0x80 && *text<0xc0 )
            val = ((ch&0x1f)<<6) | (*text++&0x3f);
    } else if ( ch<=0xef ) {
        if ( *text>=0x80 && *text<0xc0 && text[1]>=0x80 && text[1]<0xc0 ) {
            val = ((ch&0xf)<<12) | ((text[0]&0x3f)<<6) | (text[1]&0x3f);
            text += 2;
        }
    } else {
        int w = ( ((ch&0x7)<<2) | ((text[0]&0x30)>>4) )-1, w2;
        w = (w<<6) | ((text[0]&0xf)<<2) | ((text[1]&0x30)>>4);
        w2 = ((text[1]&0xf)<<6) | (text[2]&0x3f);
        val = w*0x400 + w2 + 0x10000;
        if ( *text<0x80 || text[1]<0x80 || text[2]<0x80 ||
                *text>=0xc0 || text[1]>=0xc0 || text[2]>=0xc0 )
            val = -1;
        else
            text += 3;
    }
    *_text = (const char *) text;
return( val );
}


char *u2utf8_strcpy(char *utf8buf,const unichar_t *ubuf) {
    char *pt = utf8buf;

    while ( *ubuf ) {
        if ( *ubuf<0x80 )
            *pt++ = *ubuf;
        else if ( *ubuf<0x800 ) {
            *pt++ = 0xc0 | (*ubuf>>6);
            *pt++ = 0x80 | (*ubuf&0x3f);
        } else if ( *ubuf>=0xd800 && *ubuf<0xdc00 && ubuf[1]>=0xdc00 && ubuf[1]<0xe000 ) {
            int u = ((*ubuf>>6)&0xf)+1, y = ((*ubuf&3)<<4) | ((ubuf[1]>>6)&0xf);
            *pt++ = 0xf0 | (u>>2);
            *pt++ = 0x80 | ((u&3)<<4) | ((*ubuf>>2)&0xf);
            *pt++ = 0x80 | y;
            *pt++ = 0x80 | (ubuf[1]&0x3f);
        } else {
            *pt++ = 0xe0 | (*ubuf>>12);
            *pt++ = 0x80 | ((*ubuf>>6)&0x3f);
            *pt++ = 0x80 | (*ubuf&0x3f);
        }
        ++ubuf;
    }
    *pt = '\0';
    return( utf8buf );
}

char *u2utf8_copy(const unichar_t *ubuf) {
    int len;
    char *utf8buf;

    if ( ubuf==NULL )
      return( NULL );

    len = u_strlen(ubuf);
    utf8buf = galloc((len+1)*3);
    return( u2utf8_strcpy(utf8buf,ubuf));
}

unichar_t *uc_copy(const char *pt) {
    unichar_t *res, *rpt;
    int n;

    if(!pt)
return((unichar_t *)0);

    n = strlen(pt);
    res = galloc((n+1)*sizeof(unichar_t));
    for ( rpt=res; --n>=0 ; *rpt++ = *(unsigned char *) pt++ );
    *rpt = '\0';
return(res);
}

int utf8_valid(const char *str) {
    /* Is this a valid utf8 string? */
    int ch;

    while ( (ch=utf8_ildb(&str))!='\0' )
        if ( ch==-1 )
	  return( false );
    return( true );
}

char *latin1_2_utf8_strcpy(char *utf8buf,const char *lbuf) {
    char *pt = utf8buf;
    const unsigned char *lpt = (const unsigned char *) lbuf;

    while ( *lpt ) {
        if ( *lpt<0x80 )
            *pt++ = *lpt;
        else {
            *pt++ = 0xc0 | (*lpt>>6);
            *pt++ = 0x80 | (*lpt&0x3f);
        }
        ++lpt;
    }
    *pt = '\0';
return( utf8buf );
}

char *latin1_2_utf8_copy(const char *lbuf) {
    int len;
    char *utf8buf;

    if ( lbuf==NULL )
return( NULL );

    len = strlen(lbuf);
    utf8buf = galloc(2*len+1);
return( latin1_2_utf8_strcpy(utf8buf,lbuf));
}


char *utf8_verify_copy(const char *str) {
  if ( str==NULL )
    return( NULL );

    if ( utf8_valid(str))
      return( copy(str));             /* Either in ASCII (good) or appears to be utf8*/
    return( latin1_2_utf8_copy(str));
}

/* ucharmap.c */
int local_encoding = e_iso8859_1;

static int bad_enc_warn = false;


unichar_t *encoding2u_strncpy(unichar_t *uto, const char *_from, int n, enum encoding cs) {
    unichar_t *upt=uto;
    const unichar_t *table;
    int offset;
    const unsigned char *from = (const unsigned char *) _from;

    if ( cs<e_first2byte ) {
	table = unicode_from_alphabets[cs];
	if ( table==NULL ) {
	    while ( *from && n>0 ) {
		*upt++ = *(unsigned char *) (from++);
		--n;
	    }
	} else {
	    while ( *from && n>0 ) {
		*upt ++ = table[*(unsigned char *) (from++)];
		--n;
	    }
	}
    } else if ( cs<e_unicode ) {
	*uto = '\0';
	switch ( cs ) {
	  default:
	    if ( !bad_enc_warn ) {
		bad_enc_warn = true;
		fprintf( stderr, "Unexpected encoding %d, I'll pretend it's latin1\n", cs );
	    }
return( encoding2u_strncpy(uto,_from,n,e_iso8859_1));
	  case e_johab: case e_big5: case e_big5hkscs:
	    if ( cs==e_big5 ) {
		offset = 0xa100;
		table = unicode_from_big5;
	    } else if ( cs==e_big5hkscs ) {
		offset = 0x8100;
		table = unicode_from_big5hkscs;
	    } else {
		offset = 0x8400;
		table = unicode_from_johab;
	    }
	    while ( *from && n>0 ) {
		if ( *from>=(offset>>8) && from[1]!='\0' ) {
		    *upt++ = table[ ((*from<<8) | from[1]) - offset ];
		    from += 2;
		} else
		    *upt++ = *from++;
		--n;
	    }
	  break;
	  case e_wansung:
	    while ( *from && n>0 ) {
		if ( *from>=0xa1 && from[1]>=0xa1 ) {
		    *upt++ = unicode_from_ksc5601[ (*from-0xa1)*94+(from[1]-0xa1) ];
		    from += 2;
		} else
		    *upt++ = *from++;
		--n;
	    }
	  break;
	  case e_jisgb:
	    while ( *from && n>0 ) {
		if ( *from>=0xa1 && from[1]>=0xa1 ) {
		    *upt++ = unicode_from_gb2312[ (*from-0xa1)*94+(from[1]-0xa1) ];
		    from += 2;
		} else
		    *upt++ = *from++;
		--n;
	    }
	  break;
	  case e_sjis:
	    while ( *from && n>0 ) {
		if ( *from<127 || ( *from>=161 && *from<=223 )) {
		    *upt++ = unicode_from_jis201[*from++];
		} else {
		    int ch1 = *from++;
		    int ch2 = *from++;
		    if ( ch1 >= 129 && ch1<= 159 )
			ch1 -= 112;
		    else
			ch1 -= 176;
		    ch1 <<= 1;
		    if ( ch2>=159 )
			ch2-= 126;
		    else if ( ch2>127 ) {
			--ch1;
			ch2 -= 32;
		    } else {
			--ch1;
			ch2 -= 31;
		    }
		    *upt++ = unicode_from_jis208[(ch1-0x21)*94+(ch2-0x21)];
		}
		--n;
	    }
	  break;
	}
    } else if ( cs==e_unicode ) {
	unichar_t *ufrom = (unichar_t *) from;
	while ( *ufrom && n>0 ) {
	    *upt++ = *ufrom++;
	    --n;
	}
    } else if ( cs==e_unicode_backwards ) {
	unichar_t *ufrom = (unichar_t *) from;
	while ( *ufrom && n>0 ) {
	    unichar_t ch = (*ufrom>>8)||((*ufrom&0xff)<<8);
	    *upt++ = ch;
	    ++ufrom;
	    --n;
	}
    } else if ( cs==e_utf8 ) {
	while ( *from && n>0 ) {
	    if ( *from<=127 )
		*upt = *from++;
	    else if ( *from<=0xdf ) {
		if ( from[1]>=0x80 ) {
		    *upt = ((*from&0x1f)<<6) | (from[1]&0x3f);
		    from += 2;
		} else {
		    ++from;	/* Badly formed utf */
		    *upt = 0xfffd;
		}
	    } else if ( *from<=0xef ) {
		if ( from[1]>=0x80 && from[2]>=0x80 ) {
		    *upt = ((*from&0xf)<<12) | ((from[1]&0x3f)<<6) | (from[2]&0x3f);
		    from += 3;
		} else {
		    ++from;	/* Badly formed utf */
		    *upt = 0xfffd;
		}
	    } else if ( n>2 ) {
		if ( from[1]>=0x80 && from[2]>=0x80 && from[3]>=0x80 ) {
		    int w = ( ((*from&0x7)<<2) | ((from[1]&0x30)>>4) )-1;
		    *upt++ = 0xd800 | (w<<6) | ((from[1]&0xf)<<2) | ((from[2]&0x30)>>4);
		    *upt   = 0xdc00 | ((from[2]&0xf)<<6) | (from[3]&0x3f);
		    from += 4;
		} else {
		    ++from;	/* Badly formed utf */
		    *upt = 0xfffd;
		}
	    } else {
		/* no space for surrogate */
		from += 4;
	    }
	    ++upt;
	}
    } else {
	if ( !bad_enc_warn ) {
	    bad_enc_warn = true;
	    fprintf( stderr, "Unexpected encoding %d, I'll pretend it's latin1\n", cs );
	}
return( encoding2u_strncpy(uto,_from,n,e_iso8859_1));
    }

    if ( n>0 )
	*upt = '\0';

return( uto );
}

char *u2encoding_strncpy(char *to, const unichar_t *ufrom, int n, enum encoding cs) {
    char *pt = to;

    /* we just ignore anything that doesn't fit in the encoding we look at */
    if ( cs<e_first2byte ) {
	struct charmap *table = NULL;
	unsigned char *plane;
	table = alphabets_from_unicode[cs];
	if ( table==NULL ) {	/* ASCII */
	    while ( *ufrom && n>0 ) {
		int ch = *ufrom;
		if ( ch<127 ) {
		    *pt++ = ch;
		    --n;
		}
		++ufrom;
	    }
	} else {
	    while ( *ufrom && n>0 ) {
		int highch = *ufrom>>8, ch;
		if ( highch>=table->first && highch<=table->last &&
			    (plane = table->table[highch])!=NULL &&
			    (ch=plane[*ufrom&0xff])!=0 ) {
		    *pt++ = ch;
		    --n;
		}
		++ufrom;
	    }
	}
	if ( n>0 )
	    *pt = '\0';
    } else if ( cs<e_unicode ) {
	struct charmap2 *table;
	unsigned short *plane;
	unsigned char *plane1;

	*to = '\0';
	switch ( cs ) {
	  default:
	    if ( !bad_enc_warn ) {
		bad_enc_warn = true;
		fprintf( stderr, "Unexpected encoding %d, I'll pretend it's latin1\n", cs );
	    }
return( u2encoding_strncpy(to,ufrom,n,e_iso8859_1));
	  case e_johab: case e_big5: case e_big5hkscs:
	    table = cs==e_big5 ? &big5_from_unicode :
		    cs==e_big5hkscs ? &big5hkscs_from_unicode :
			    &johab_from_unicode;
	    while ( *ufrom && n>0 ) {
		int highch = *ufrom>>8, ch;
		if ( *ufrom<0x80 ) {
		    *pt++ = *ufrom;
		    --n;
		} else if ( highch>=table->first && highch<=table->last &&
			(plane = table->table[highch-table->first])!=NULL &&
			(ch=plane[*ufrom&0xff])!=0 ) {
		    *pt++ = ch>>8;
		    *pt++ = ch&0xff;
		    n -= 2;
		}
		ufrom ++;
	    }
	  break;
	  case e_wansung:
	    while ( *ufrom && n>0 ) {
		int highch = *ufrom>>8, ch;
		if ( *ufrom<0x80 ) {
		    *pt++ = *ufrom;
		    --n;
		} else if ( highch>=ksc5601_from_unicode.first && highch<=ksc5601_from_unicode.last &&
			(plane = ksc5601_from_unicode.table[highch-ksc5601_from_unicode.first])!=NULL &&
			(ch=plane[*ufrom&0xff])!=0 ) {
		    *pt++ = (ch>>8) + 0x80;
		    *pt++ = (ch&0xff) + 0x80;
		    n -= 2;
		}
		ufrom ++;
	    }
	  break;
	  case e_jisgb:
	    while ( *ufrom && n>0 ) {
		int highch = *ufrom>>8, ch;
		if ( *ufrom<0x80 ) {
		    *pt++ = *ufrom;
		    --n;
		} else if ( highch>=gb2312_from_unicode.first && highch<=gb2312_from_unicode.last &&
			(plane = gb2312_from_unicode.table[highch-gb2312_from_unicode.first])!=NULL &&
			(ch=plane[*ufrom&0xff])!=0 ) {
		    *pt++ = (ch>>8) + 0x80;
		    *pt++ = (ch&0xff) + 0x80;
		    n -= 2;
		}
		ufrom ++;
	    }
	  break;
	  case e_sjis:
	    while ( *ufrom && n>0 ) {
		int highch = *ufrom>>8, ch;
		if ( highch>=jis201_from_unicode.first && highch<=jis201_from_unicode.last &&
			(plane1 = jis201_from_unicode.table[highch-jis201_from_unicode.first])!=NULL &&
			(ch=plane1[*ufrom&0xff])!=0 ) {
		    *pt++ = ch;
		    --n;
		} else if ( *ufrom<' ' ) {	/* control chars */
		    *pt++ = *ufrom;
		    --n;
		} else if ( highch>=jis_from_unicode.first && highch<=jis_from_unicode.last &&
			(plane = jis_from_unicode.table[highch-jis_from_unicode.first])!=NULL &&
			(ch=plane[*ufrom&0xff])!=0 && ch<0x8000 ) {	/* no jis212 */
		    int j1 = ch>>8, j2 = ch&0xff;
		    int ro = j1<95 ? 112 : 176;
		    int co = (j1&1) ? (j2>95?32:31) : 126;
		    *pt++ = ((j1+1)>>1)+ro;
		    *pt++ = j2+co;
		    n -= 2;
		}
		++ufrom;
	    }
	  break;
	}
	if ( n>0 )
	    *pt = '\0';
    } else if ( cs==e_unicode ) {
	unichar_t *uto = (unichar_t *) to;
	while ( *ufrom && n>1 ) {
	    *uto++ = *ufrom++;
	    n-=2;
	}
	if ( n>1 )
	    *uto = '\0';
    } else if ( cs==e_unicode_backwards ) {
	unichar_t *uto = (unichar_t *) to;
	while ( *ufrom && n>1 ) {
	    unichar_t ch = (*ufrom>>8)||((*ufrom&0xff)<<8);
	    *uto++ = ch;
	    ++ufrom;
	    n-=2;
	}
	if ( n>1 )
	    *uto = '\0';
    } else if ( cs==e_utf8 ) {
	while ( *ufrom ) {
	    if ( *ufrom<0x80 ) {
		if ( n<=1 )
	break;
		*pt++ = *ufrom;
		--n;
	    } else if ( *ufrom<0x800 ) {
		if ( n<=2 )
	break;
		*pt++ = 0xc0 | (*ufrom>>6);
		*pt++ = 0x80 | (*ufrom&0x3f);
		n -= 2;
	    } else if ( *ufrom>=0xd800 && *ufrom<0xdc00 && ufrom[1]>=0xdc00 && ufrom[1]<0xe000 ) {
		int u = ((*ufrom>>6)&0xf)+1, y = ((*ufrom&3)<<4) | ((ufrom[1]>>6)&0xf);
		if ( n<=4 )
	    break;
		*pt++ = 0xf0 | (u>>2);
		*pt++ = 0x80 | ((u&3)<<4) | ((*ufrom>>2)&0xf);
		*pt++ = 0x80 | y;
		*pt++ = 0x80 | (ufrom[1]&0x3f);
		n -= 4;
	    } else {
		if ( n<=3 )
	    break;
		*pt++ = 0xe0 | (*ufrom>>12);
		*pt++ = 0x80 | ((*ufrom>>6)&0x3f);
		*pt++ = 0x80 | (*ufrom&0x3f);
	    }
	    ++ufrom;
	}
	if ( n>1 )
	    *pt = '\0';
    } else {
	if ( !bad_enc_warn ) {
	    bad_enc_warn = true;
	    fprintf( stderr, "Unexpected encoding %d, I'll pretend it's latin1\n", cs );
	}
return( u2encoding_strncpy(to,ufrom,n,e_iso8859_1));
    }

return( to );
}


char *def2utf8_copy(const char *from) {
    int len;
    char *ret;
    unichar_t *temp, *uto;

    if ( from==NULL )
return( NULL );
    len = strlen(from);
#if HAVE_ICONV_H
    if ( my_iconv_setup() ) {
	size_t in_left = len, out_left = 2*len;
	char *cto = (char *) galloc(3*(len+1)), *cret = cto;
	iconv(to_utf8, (iconv_arg2_t) &from, &in_left, &cto, &out_left);
	*cto++ = '\0';
	*cto++ = '\0';
return( cret );
    }
#endif
    uto = galloc(2*len+2);
    temp = encoding2u_strncpy(uto,from,len,local_encoding);
    if ( temp==NULL ) {
	free( uto );
return( NULL );
    }
    uto[len] = '\0';
    ret = u2utf8_copy(uto);
    free(uto);
return( ret );
}


char *xuid = NULL;

/* these four are user prefs */
int default_fv_font_size = 24;
int default_fv_antialias = false;
int default_fv_row_count = 4;
int default_fv_col_count = 16;
int default_fv_bbsized = true;
int new_fonts_are_order2 = false;
int loaded_fonts_same_as_new = false;
int prefer_cjk_encodings=false;
char *TTFFoundry=NULL;

extern  Encoding custom ;

int getushort(FILE *ttf) {
    int ch1 = getc(ttf);
    int ch2 = getc(ttf);
    if ( ch2==EOF )
return( EOF );
return( (ch1<<8)|ch2 );
}

int get3byte(FILE *ttf) {
    int ch1 = getc(ttf);
    int ch2 = getc(ttf);
    int ch3 = getc(ttf);
    if ( ch3==EOF )
return( EOF );
return( (ch1<<16)|(ch2<<8)|ch3 );
}

int32 getlong(FILE *ttf) {
    int ch1 = getc(ttf);
    int ch2 = getc(ttf);
    int ch3 = getc(ttf);
    int ch4 = getc(ttf);
    if ( ch4==EOF )
return( EOF );
return( (ch1<<24)|(ch2<<16)|(ch3<<8)|ch4 );
}

static int32 getoffset(FILE *ttf, int offsize) {
    if ( offsize==1 )
return( getc(ttf));
    else if ( offsize==2 )
return( getushort(ttf));
    else if ( offsize==3 )
return( get3byte(ttf));
    else
return( getlong(ttf));
}

real getfixed(FILE *ttf) {
    int32 val = getlong(ttf);
    int mant = val&0xffff;
    /* This oddity may be needed to deal with the first 16 bits being signed */
    /*  and the low-order bits unsigned */
return( (real) (val>>16) + (mant/65536.0) );
}

real get2dot14(FILE *ttf) {
    int32 val = getushort(ttf);
    int mant = val&0x3fff;
    /* This oddity may be needed to deal with the first 2 bits being signed */
    /*  and the low-order bits unsigned */
return( (real) ((val<<16)>>(16+14)) + (mant/16384.0) );
}

static Encoding *enc_from_platspec(int platform,int specific) {
  char *enc;
  Encoding *e;
  
  enc = "Custom";
  if ( platform==0 ) {
	enc = "Unicode";
	if ( specific==4 )
	  enc = "UnicodeFull";
  } else if ( platform==1 ) {
	if ( specific==0 )
	  enc = "Mac";
	else if ( specific==1 )
	  enc = "Sjis";
	else if ( specific==2 )
	  enc = "Big5hkscs";		/* Or should we just guess big5? Both are wrong sometimes */
	else if ( specific==3 )
	  enc = "EUC-KR";
	else if ( specific==25 )
	  enc = "EUC-CN";
  } else if ( platform==2 ) {		/* obselete */
	if ( specific==0 )
	  enc = "ASCII";
	else if ( specific==1 )
	  enc = "Unicode";
	else if ( specific==2 )
	  enc = "ISO8859-1";
  } else if ( platform==3 ) {
	if ( specific==1 || specific==0 )	/* symbol (sp=0) is just unicode (PUA) */
	  enc = "Unicode";
	else if ( specific==2 )
	  enc = "Sjis";
	else if ( specific==3 )
	  enc = "EUC-CN";
	else if ( specific==4 )
	  enc = "Big5hkscs";
	else if ( specific==5 )
	  enc = "EUC-KR";
	else if ( specific==6 )
	  enc = "Johab";
	else if ( specific==10 )
	  enc = "UnicodeFull";
  } else if ( platform==7 ) {		/* Used internally in freetype, but */
	if ( specific==0 )		/*  there's no harm in looking for it */
	  enc = "AdobeStandard";	/*  even if it never happens */
	else if ( specific==1 )
	  /* adobe_expert */;
	else if ( specific==2 )
	  /* adobe_custom */;
  }
  e = FindOrMakeEncoding(enc);
  if ( e==NULL ) {
	static int p = -1,s = -1;
	if (( p!=platform || s!=specific ) 
		&& 
		!(platform==1 && specific==0)  /* todo */
		) {
	  LogError( _("The truetype encoding specified by platform=%d specific=%d (which we map to %s) is not supported by your version of iconv(3).\n"),
				platform, specific, enc );
	  p = platform; s = specific;
	}
  }
  return( e );
}

static char *_readencstring(FILE *ttf,int offset,int len,
	int platform,int specific,int language) {
  long pos = ftell(ttf);
  unichar_t *str, *pt;
  char *ret;
  int i, ch;
  Encoding *enc;
	
  fseek(ttf,offset,SEEK_SET);
  
  if ( platform==1 ) {
    /* Mac is screwy, there are several different varients of MacRoman */
    /*  depending on the language, they didn't get it right when they  */
    /*  invented their script system */
    char *cstr, *cpt;
    cstr = cpt = galloc(len+1);
    for ( i=0; i<len; ++i )
      *cpt++ = getc(ttf);
    *cpt = '\0';
    ret = MacStrToUtf8(cstr,specific,language);
    free(cstr);
  } else {
    enc = enc_from_platspec(platform,specific);
    if ( enc==NULL )
      return( NULL );
    if ( enc->is_unicodebmp ) {
      str = pt = galloc(len+2);
      for ( i=0; i<len/2; ++i ) {
	ch = getc(ttf)<<8;
	*pt++ = ch | getc(ttf);
      }
      *pt = 0;
    } else if ( enc->unicode!=NULL ) {
      str = pt = galloc(2*len+2);
      for ( i=0; i<len; ++i )
	*pt++ = enc->unicode[getc(ttf)];
      *pt = 0;
    } else if ( enc->tounicode!=NULL ) {
      size_t inlen = len+1, outlen = 2*len+2;
      char *cstr = galloc(inlen);
      ICONV_CONST char *in = cstr;
      char *out;
      str = galloc(outlen+2);
      out = (char *) str;
      iconv(enc->tounicode,&in,&inlen,&out,&outlen);
      out[0] = '\0'; out[1] = '\0';
      free(cstr);
    } else {
      str = uc_copy("");
    }
    ret = u2utf8_copy(str);
    free(str);
  }
  fseek(ttf,pos,SEEK_SET);
  return( ret );
}

char *TTFGetFontName(FILE *ttf,int32 offset,int32 off2) {
    int i,num;
    int32 tag, nameoffset, length, stringoffset;
    int plat, spec, lang, name, len, off, val;
    int fullval, fullstr, fulllen, famval, famstr, famlen;
    Encoding *enc;
    int fullplat, fullspec, fulllang, famplat, famspec, famlang;

    fseek(ttf,offset,SEEK_SET);
    /* version = */ getlong(ttf);
    num = getushort(ttf);
    /* srange = */ getushort(ttf);
    /* esel = */ getushort(ttf);
    /* rshift = */ getushort(ttf);
    for ( i=0; i<num; ++i ) {
	tag = getlong(ttf);
	/* checksum = */ getlong(ttf);
	nameoffset = off2+getlong(ttf);
	length = getlong(ttf);
	if ( tag==CHR('n','a','m','e'))
    break;
    }
    if ( i==num )
return( NULL );

    fseek(ttf,nameoffset,SEEK_SET);
    /* format = */ getushort(ttf);
    num = getushort(ttf);
    stringoffset = nameoffset+getushort(ttf);
    fullval = famval = 0;
    for ( i=0; i<num; ++i ) {
	plat = getushort(ttf);
	spec = getushort(ttf);
	lang = getushort(ttf);
	name = getushort(ttf);
	len = getushort(ttf);
	off = getushort(ttf);
	enc = enc_from_platspec(plat,spec);
	if ( enc==NULL )
    continue;
	val = 0;
    /* I really want an english name */
	if ( (plat==0 || plat==1) && !enc->is_custom && lang==0 )
	    val = 11;
	else if ( plat==3 && !enc->is_custom && (lang&0xff)==0x09 )
	    val = 12;
    /* failing that I'll take what I can get */
	else if ( !enc->is_custom )
	    val = 1;
	if ( name==4 && val>fullval ) {
	    fullval = val;
	    fullstr = off;
	    fulllen = len;
	    fullplat = plat;
	    fullspec = spec;
	    fulllang = lang;
	    if ( val==12 )
    break;
	} else if ( name==1 && val>famval ) {
	    famval = val;
	    famstr = off;
	    famlen = len;
	    famplat = plat;
	    famspec = spec;
	    famlang = lang;
	}
    }
    if ( fullval==0 ) {
	if ( famval==0 )
return( NULL );
	fullstr = famstr;
	fulllen = famlen;
	fullplat = famplat;
	fullspec = famspec;
	fulllang = famlang;
    }
return( _readencstring(ttf,stringoffset+fullstr,fulllen,fullplat,fullspec,fulllang));
}

static int PickTTFFont(FILE *ttf,char *filename,char **chosenname) {
    int32 *offsets, cnt, i, choice, j;
    char **names;
    char *pt, *lparen;

    /* TTCF version = */ getlong(ttf);
    cnt = getlong(ttf);
    if ( cnt==1 ) {
	/* This is easy, don't bother to ask the user, there's no choice */
	int32 offset = getlong(ttf);
	fseek(ttf,offset,SEEK_SET);
return( true );
    }
    offsets = galloc(cnt*sizeof(int32));
    for ( i=0; i<cnt; ++i )
	offsets[i] = getlong(ttf);
    names = galloc(cnt*sizeof(char *));
    for ( i=j=0; i<cnt; ++i ) {
	names[j] = TTFGetFontName(ttf,offsets[i],0);
	if ( names[j]!=NULL ) ++j;
    }
    pt = strrchr(filename,'/');
    if ( pt==NULL ) pt = filename;
    if ( (lparen = strchr(pt,'('))!=NULL && strchr(lparen,')')!=NULL ) {
	char *find = copy(lparen+1);
	pt = strchr(find,')');
	if ( pt!=NULL ) *pt='\0';
	for ( choice=cnt-1; choice>=0; --choice )
	    if ( strcmp(names[choice],find)==0 )
	break;
	if ( choice==-1 ) {
	    char *fn = copy(filename);
	    fn[lparen-filename] = '\0';
/* GT: The user is trying to open a font file which contains multiple fonts and */
/* GT: has asked for a font which is not in that file. */
/* GT: The string will look like: <fontname> is not in <filename> */
	    /* gwwv_post_error(_("Not in Collection"),
	         _("%1$s is not in %2$.100s"),find,fn);*/
	    free(fn);
	}
	free(find);
    } else
	choice = 0;
    if ( choice!=-1 ) {
	fseek(ttf,offsets[choice],SEEK_SET);
	*chosenname = copy(names[choice]);
    }
    for ( i=0; i<j; ++i )
	free(names[i]);
    free(names);
    free(offsets);
return( choice!=-1);
}

static int PickCFFFont(char **fontnames) {
    unichar_t **names;
    int cnt, i, choice;

    for ( cnt=0; fontnames[cnt]!=NULL; ++cnt);
    names = gcalloc(cnt+1,sizeof(unichar_t *));
    for ( i=0; i<cnt; ++i )
	names[i] = uc_copy(fontnames[i]);
    choice = 0;
    for ( i=0; i<cnt; ++i )
	free(names[i]);
    free(names);
return( choice );
}

static void ParseSaveTablesPref(struct ttfinfo *info) {
#if 0    
  extern char *SaveTablesPref;
#endif
    char *pt, *spt;
    int cnt;

    info->savecnt = 0;
    info->savetab = NULL;
    if ( SaveTablesPref==NULL || *SaveTablesPref=='\0' )
return;
    for ( pt=SaveTablesPref, cnt=0; *pt; ++pt )
	if ( *pt==',' )
	    ++cnt;
    info->savecnt = cnt+1;
    info->savetab = gcalloc(cnt+1,sizeof(struct savetab));
    for ( pt=spt=SaveTablesPref, cnt=0; ; ++pt ) {
	if ( *pt==',' || *pt=='\0' ) {
	    uint32 tag;
	    tag  = ( ( spt  <pt )? spt[0] : ' ' )<<24;
	    tag |= ( ( spt+1<pt )? spt[1] : ' ' )<<16;
	    tag |= ( ( spt+2<pt )? spt[2] : ' ' )<<8 ;
	    tag |= ( ( spt+3<pt )? spt[3] : ' ' )    ;
	    info->savetab[cnt++].tag = tag;
	    if ( *pt )
		spt = pt+1;
	    else
    break;
	}
    }
}

static int readttfheader(FILE *ttf, struct ttfinfo *info,char *filename,
						 char **choosenname) {
  int i, j;
  int tag, checksum, offset, length, version;
  
  version=getlong(ttf);
  if ( version==CHR('t','t','c','f')) {
	/* TrueType font collection */
	info->is_ttc = true;
	if ( !PickTTFFont(ttf,filename,choosenname))
	  return( 0 );
	/* If they picked a font, then we should be left pointing at the */
	/*  start of the Table Directory for that font */
	info->one_of_many = true;
	version = getlong(ttf);
  }

  /* Apple says that 'typ1' is a valid code for a type1 font wrapped up in */
  /*  a truetype table structure, but gives no docs on what tables get used */
  /*  or how */
  if ( version==CHR('t','y','p','1')) {
	if ( filename==NULL ) 
	  filename = "it";
	LogError( _("Interesting, I've never seen an example of this type of font, could you\nsend me a copy of %s?\nThanks\n  gww@silcom.com\n"), filename );
  }
  if ( version!=0x00010000 && version!=CHR('t','r','u','e') &&
	   version!=0x00020000 &&	/* Windows 3.1 Chinese version used this version for some arphic fonts */
	   /* See discussion on freetype list, july 2004 */
	    version!=CHR('O','T','T','O'))
	return( 0 );			/* Not version 1 of true type, nor Open Type */
  info->numtables = getushort(ttf);
    /* searchRange = */ getushort(ttf);
  /* entrySelector = */ getushort(ttf);
    /* rangeshift = */ getushort(ttf);
  
  ParseSaveTablesPref(info);

  for ( i=0; i<info->numtables; ++i ) {
	tag = getlong(ttf);
	checksum = getlong(ttf);
	offset = getlong(ttf);
	length = getlong(ttf);
#ifdef DEBUG
 printf( "%c%c%c%c\n", tag>>24, (tag>>16)&0xff, (tag>>8)&0xff, tag&0xff );
#endif
	switch ( tag ) {
	  case CHR('C','F','F',' '):
	    info->cff_start = offset;
	    info->cff_length = length;
	  break;
	  case CHR('c','m','a','p'):
	    info->encoding_start = offset;
	  break;
	  case CHR('g','a','s','p'):
	    info->gasp_start = offset;
	  break;
	  case CHR('g','l','y','f'):
	    info->glyph_start = offset;
	    info->glyph_length = length;
	  break;
	  case CHR('G','D','E','F'):
	    info->gdef_start = offset;
	    info->gdef_length = length;
	  break;
	  case CHR('G','P','O','S'):
	    info->gpos_start = offset;
	    info->gpos_length = length;
	  break;
	  case CHR('G','S','U','B'):
	    info->gsub_start = offset;
	    info->gsub_length = length;
	  break;
	  case CHR('b','d','a','t'):		/* Apple/MS use a different tag, but the same format. Great. */
	  case CHR('E','B','D','T'):
	    info->bitmapdata_start = offset;
	    info->bitmapdata_length = length;
	  break;
	  case CHR('b','l','o','c'):		/* Apple/MS use a different tag. Great. */
	  case CHR('E','B','L','C'):
	    info->bitmaploc_start = offset;
	    info->bitmaploc_length = length;
	  break;
	  case CHR('b','h','e','d'):		/* Apple uses bhed for fonts with only bitmaps */
	  case CHR('h','e','a','d'):
	    info->head_start = offset;
	  break;
	  case CHR('h','h','e','a'):
	    info->hhea_start = offset;
	  break;
	  case CHR('h','m','t','x'):
	    info->hmetrics_start = offset;
	  break;
	  case CHR('k','e','r','n'):
	    info->kern_start = offset;
	  break;
	  case CHR('l','o','c','a'):
	    info->glyphlocations_start = offset;
	    info->loca_length = length;
	    info->glyph_cnt = length/2-1;	/* the minus one is because there is one extra entry to give the length of the last glyph */
	    if ( info->glyph_cnt<0 ) info->glyph_cnt = 0;
	  break;
	  case CHR('m','a','x','p'):
	    info->maxp_start = offset;
	    info->maxp_len = length;
	  break;
	  case CHR('n','a','m','e'):
	    info->copyright_start = offset;
	  break;
	  case CHR('p','o','s','t'):
	    info->postscript_start = offset;
	  break;
	  case CHR('O','S','/','2'):
	    info->os2_start = offset;
	  break;
	  case CHR('v','h','e','a'):
	    info->vhea_start = offset;
	  break;
	  case CHR('v','m','t','x'):
	    info->vmetrics_start = offset;
	  break;
	  case CHR('V','O','R','G'):
	    info->vorg_start = offset;
	  break;
	      /* Apple stuff */
	  case CHR('a','c','n','t'):
	    info->acnt_start = offset;
	  break;
	  case CHR('f','e','a','t'):
	    info->feat_start = offset;
	  break;
	  case CHR('l','c','a','r'):
	    info->lcar_start = offset;
	  break;
	  case CHR('m','o','r','t'):
	    info->mort_start = offset;
	  break;
	  case CHR('m','o','r','x'):
	    info->morx_start = offset;
	  break;
	  case CHR('o','p','b','d'):
	    info->opbd_start = offset;
	  break;
	  case CHR('p','r','o','p'):
	    info->prop_start = offset;
	  break;
	      /* to make sense of instrs */
	  case CHR('c','v','t',' '):
	    info->cvt_start = offset;
	    info->cvt_len = length;
	  break;
	  case CHR('p','r','e','p'):
	    info->prep_start = offset;
	    info->prep_len = length;
	  break;
	  case CHR('f','p','g','m'):
	    info->fpgm_start = offset;
	    info->fpgm_len = length;
	  break;

	    /* non-standard tables I've added */
	  case CHR('P','f','E','d'):
	    info->pfed_start = offset;
	  break;
	  case CHR('F','F','T','M'):
	    info->fftm_start = offset;
	  break;
	  case CHR('T','e','X',' '):
	    info->tex_start = offset;
	  break;
	  case CHR('B','D','F',' '):
	    info->bdf_start = offset;
	  break;

	    /* Apple's mm fonts */
	  case CHR('g','v','a','r'):
	    info->gvar_start = offset;
	    info->gvar_len = length;
	  break;
	  case CHR('f','v','a','r'):
	    info->fvar_start = offset;
	    info->fvar_len = length;
	  break;
	  case CHR('a','v','a','r'):
	    info->avar_start = offset;
	    info->avar_len = length;
	  break;
	  case CHR('c','v','a','r'):
	    info->cvar_start = offset;
	    info->cvar_len = length;
	  break;

	  default:
	    for ( j=0; j<info->savecnt; ++j ) if ( info->savetab[j].tag == tag ) {
		info->savetab[j].offset = offset;
		info->savetab[j].len = length;
	    break;
	    }
	}
    }
return( true );
}

static void readdate(FILE *ttf,struct ttfinfo *info,int ismod) {
    int date[4], date1970[4], year[2];
    int i;
    /* Dates in sfnt files are seconds since 1904. I adjust to unix time */
    /*  seconds since 1970 by figuring out how many seconds were in between */

    date[3] = getushort(ttf);
    date[2] = getushort(ttf);
    date[1] = getushort(ttf);
    date[0] = getushort(ttf);
    memset(date1970,0,sizeof(date1970));
    year[0] = (60*60*24*365L)&0xffff;
    year[1] = (60*60*24*365L)>>16;
    for ( i=1904; i<1970; ++i ) {
	date1970[0] += year[0];
	date1970[1] += year[1];
	if ( (i&3)==0 && (i%100!=0 || i%400==0))
	    date1970[0] += 24*60*60L;		/* Leap year */
	date1970[1] += (date1970[0]>>16);
	date1970[0] &= 0xffff;
	date1970[2] += date1970[1]>>16;
	date1970[1] &= 0xffff;
	date1970[3] += date1970[2]>>16;
	date1970[2] &= 0xffff;
    }

    for ( i=0; i<3; ++i ) {
	date[i] -= date1970[i];
	date[i+1] += date[i]>>16;
	date[i] &= 0xffff;
    }
    date[3] -= date1970[3];

    *(ismod ? &info->modificationtime : &info->creationtime) =
#ifdef _HAS_LONGLONG
	    (((long long) date[3])<<48) |
	    (((long long) date[2])<<32) |
#endif
	    (             date[1] <<16) |
			  date[0];
}

static void readttfhead(FILE *ttf,struct ttfinfo *info) {
    /* Here I want units per em, and size of loca entries */
    /* oh... also creation/modification times */
    int i, flags;

    fseek(ttf,info->head_start+4*4,SEEK_SET);		/* skip over the version number and a bunch of junk */
    flags = getushort(ttf);
    info->optimized_for_cleartype = (flags&(1<<13))?1:0;
    info->emsize = getushort(ttf);
    for ( i=0; i<12; ++i )
	getushort(ttf);
    info->macstyle = getushort(ttf);
    for ( i=0; i<2; ++i )
	getushort(ttf);
    info->index_to_loc_is_long = getushort(ttf);
    if ( info->index_to_loc_is_long )
	info->glyph_cnt = info->loca_length/4-1;
    if ( info->glyph_cnt<0 ) info->glyph_cnt = 0;

    if ( info->fftm_start!=0 ) {
	fseek(ttf,info->fftm_start+3*4,SEEK_SET);
    } else {
	fseek(ttf,info->head_start+4*4+2+2,SEEK_SET);
    }
    readdate(ttf,info,0);
    readdate(ttf,info,1);
}

static void readttfhhea(FILE *ttf,struct ttfinfo *info) {
    /* Here I want ascent, descent and the number of horizontal metrics */
    int i;

    fseek(ttf,info->hhea_start+4,SEEK_SET);		/* skip over the version number */
    info->pfminfo.hhead_ascent = getushort(ttf);
    info->pfminfo.hhead_descent = (short) getushort(ttf);
    info->pfminfo.hheadascent_add = info->pfminfo.hheaddescent_add = false;
    info->pfminfo.linegap = getushort(ttf);
    info->pfminfo.hheadset = true;
    /*info->ascent = info->pfminfo.hhead_ascent;*/

    info->ascent = .8*info->emsize;
    info->descent = info->emsize-info->ascent;

    for ( i=0; i<12; ++i )
	getushort(ttf);
    info->width_cnt = getushort(ttf);
}

static void readttfmaxp(FILE *ttf,struct ttfinfo *info) {
    /* All I want here is the number of glyphs */
    int cnt;
    fseek(ttf,info->maxp_start+4,SEEK_SET);		/* skip over the version number */
    cnt = getushort(ttf);
    if ( info->glyph_cnt==0 && info->glyph_length==0 && info->loca_length<=4 &&
	    info->cff_length==0 ) {
	/* X11 OpenType bitmap format */;
	info->onlystrikes = true;
    } else if ( cnt!=info->glyph_cnt && info->loca_length!=0 ) {
	if ( cnt>info->glyph_cnt )
	    cnt = info->glyph_cnt;		/* Use the smaller of the two values */
    }
    /* Open Type fonts have no loca table, so we can't calculate the glyph */
    /*  count from it */
    info->glyph_cnt = cnt;
    if ( cnt<0 ) info->glyph_cnt = 0;
}

static char *stripspaces(char *str) {
    char *str2 = str, *base = str;

    if ( str==NULL )
return( NULL );

    while ( *str ) {
	if ( *str==' ' )
	    ++str;
	else
	    *str2++ = *str++;
    }
    *str2 = '\0';
return( base );
}

static struct macname *AddMacName(FILE *ttf,
	int strlen, int stroff,int spec,int language, struct macname *last) {
    struct macname *new = chunkalloc(sizeof(struct macname));
    long pos = ftell(ttf);
    char *pt;
    int i;

    new->next = last;
    new->enc = spec;
    new->lang = language;
    new->name = pt = galloc(strlen+1);

    fseek(ttf,stroff,SEEK_SET);

    for ( i=0; i<strlen; ++i )
	*pt++ = getc(ttf);
    *pt = '\0';

    fseek(ttf,pos,SEEK_SET);
return( new );
}

static void MacFeatureAdd(FILE *ttf, struct ttfinfo *info, int id,
	int strlen, int stroff,int spec,int language) {
    MacFeat *f;
    struct macsetting *s;

    for ( f=info->features; f!=NULL; f=f->next ) {
	if ( f->strid==id ) {
	    f->featname = AddMacName(ttf,strlen,stroff,spec,language,f->featname);
return;
	} else {
	    for ( s=f->settings; s!=NULL; s=s->next ) {
		if ( s->strid==id ) {
		    s->setname = AddMacName(ttf,strlen,stroff,spec,language,s->setname);
return;
		}
	    }
	}
    }
    /* Well, there are some things in the name table other than feature/setting*/
    /*  names. Let's keep track of everything just in case.... */
    if ( info->fvar_start!=0 ) {
	struct macidname *mi, *p;
	for ( p=NULL, mi=info->macstrids; mi!=NULL && mi->id!=id; p = mi, mi=mi->next );
	if ( mi==NULL ) {
	    mi = chunkalloc(sizeof(struct macidname));
	    mi->id = id;
	    mi->last = mi->head = AddMacName(ttf,strlen,stroff,spec,language,NULL);
	    if ( p==NULL )
		info->macstrids = mi;
	    else
		p->next = mi;
	} else {
	    mi->last->next = AddMacName(ttf,strlen,stroff,spec,language,NULL);
	    mi->last = mi->last->next;
	}
    }
}

static void ValidatePostScriptFontName(char *str) {
    char *end, *pt, *npt;
    int complained = false;

    /* someone gave me a font where the fontname started with the utf8 byte */
    /*  order mark.  PLRM says only ASCII encoding is supported. CFF says */
    /*  only printable ASCII should be used */
    if ( ((uint8 *) str)[0] == 0xef && ((uint8 *) str)[1]==0xbb && ((uint8 *) str)[2] == 0xbf ) {
	LogError(_("The fontname begins with the utf8 byte order sequence. This is illegal. %s"), str+3 );
	for ( pt=str+3; *pt; ++pt )
	    pt[-3] = *pt;		/* ANSI says we can't strcpy overlapping strings */
    }
    strtod(str,&end);
    if ( (*end=='\0' || (isdigit(str[0]) && strchr(str,'#')!=NULL)) &&
	    *str!='\0' ) {
	*str = 'a';
	complained = true;
    }
    for ( pt=str; *pt; ++pt ) {
	if ( *pt<=' ' || *pt>=0x7f ||
		*pt=='(' || *pt=='[' || *pt=='{' || *pt=='<' ||
		*pt==')' || *pt==']' || *pt=='}' || *pt=='>' ||
		*pt=='%' || *pt=='/' ) {
	    if ( !complained ) {
	    }
	    complained = true;
	    for ( npt=pt; npt[1]; ++npt )
		*npt = npt[1];
	    *npt = '\0';
	    --pt;
	}
    }
    if ( strlen(str)>63 ) {
	str[63] = '\0';
    }
}

char *EnforcePostScriptName(char *old) {
    char *end, *pt, *npt, *str = copy(old);

    if ( old==NULL )
return( old );

    strtod(str,&end);
    if ( (*end=='\0' || (isdigit(str[0]) && strchr(str,'#')!=NULL)) &&
	    *str!='\0' ) {
	free(str);
	str=galloc(strlen(old)+2);
	*str = 'a';
	strcpy(str+1,old);
    }
    for ( pt=str; *pt; ++pt ) {
	if ( *pt<=' ' || *pt>=0x7f ||
		*pt=='(' || *pt=='[' || *pt=='{' || *pt=='<' ||
		*pt==')' || *pt==']' || *pt=='}' || *pt=='>' ||
		*pt=='%' || *pt=='/' ) {
	    for ( npt=pt; npt[1]; ++npt )
		*npt = npt[1];
	    *npt = '\0';
	}
    }
    if ( strlen(str)>63 )
	str[63] = '\0';
return( str );
}

static int IsSubSetOf(const char *substr,const char *fullstr ) {
    /* The mac string is often a subset of the unicode string. Certain */
    /*  characters can't be expressed in the mac encoding and are omitted */
    /*  or turned to question marks or some such */
    const char *pt1, *pt2;
    uint32 ch1, ch2;

    for ( pt1=substr, pt2=fullstr, ch1=utf8_ildb(&pt1); ch1!=0 ; ) {
	if ( *pt2=='\0' )
    break;
	ch2 = utf8_ildb(&pt2);
	if ( ch1==ch2 )
	    ch1 = utf8_ildb(&pt1);
    }
    if ( ch1=='\0' )
return( true );

    for ( pt1=substr, pt2=fullstr, ch1=utf8_ildb(&pt1); ch1!=0 ; ) {
	if ( *pt2=='\0' )
    break;
	ch2 = utf8_ildb(&pt2);
	if ( ch1==ch2 || ch1=='?' )
	    ch1 = utf8_ildb(&pt1);
    }
return( ch1=='\0' );
}

static uint16 _WinLangFromMac[] = {
	0x409,		/* English */
	0x40c,		/* French */
	0x407,		/* German */
	0x410,		/* Italian */
	0x413,		/* Dutch */
	0x41d,		/* Swedish */
	0x40a,		/* Spanish */
	0x406,		/* Danish */
	0x416,		/* Portuguese */
	0x414,		/* Norwegian */
/*10*/	0x40d,		/* Hebrew */
	0x411,		/* Japanese */
	0x401,		/* Arabic */
	0x40b,		/* Finnish */
	0x408,		/* Greek */
	0x40f,		/* Icelandic */
	0x43a,		/* Maltese */
	0x41f,		/* Turkish */
	0x41a,		/* Croatian */
	0x404,		/* Traditional Chinese */
/*20*/	0x420,		/* Urdu */
	0x439,		/* Hindi */
	0x41e,		/* Thai */
	0x412,		/* Korean */
	0x427,		/* Lithuanian */
	0x415,		/* Polish */
	0x40e,		/* Hungarian */
	0x425,		/* Estonian */
	0x426,		/* Latvian */
	0x43b,		/* Sami (Lappish) */
/*30*/	0x438,		/* Faroese (Icelandic) */
	0x429,		/* Farsi/Persian */
	0x419,		/* Russian */
	0x804,		/* Simplified Chinese */
	0x813,		/* Flemish */
	0x43c,		/* Irish Gaelic */
	0x41c,		/* albanian */
	0x418,		/* Romanian */
	0x405,		/* Czech */
	0x41b,		/* Slovak */
/*40*/	0x424,		/* Slovenian */
	0x43d,		/* Yiddish */
	0xc1a,		/* Serbian */
	0x42f,		/* Macedonian */
	0x402,		/* Bulgarian */
	0x422,		/* Ukrainian */
	0x423,		/* Byelorussian */
	0x843,		/* Uzbek */
	0x43f,		/* Kazakh */
	0x42c,		/* Azerbaijani (Cyrillic) */
/*50*/	0x82c,		/* Azerbaijani (Arabic) */
	0x42b,		/* Armenian */
	0x437,		/* Georgian */
	0x818,		/* Moldavian */
	0x440,		/* Kirghiz */
	0x428,		/* Tajiki */
	0x442,		/* Turkmen */
	0x450,		/* Mongolian (Mongolian) */
	0x850,		/* Mongolian (cyrillic) */
	0x463,		/* Pashto */
/*60*/	0xffff,		/* Kurdish */
	0x860,		/* Kashmiri */
	0x459,		/* Sindhi */
	0xffff,		/* Tibetan */
	0x461,		/* Nepali */
	0x43b,		/* Sanskrit */
	0x44e,		/* Marathi */
	0x445,		/* Bengali */
	0x44d,		/* Assamese */
	0x447,		/* Gujarati */
/*70*/	0x446,		/* Punjabi */
	0x448,		/* Oriya */
	0x44c,		/* Malayalam */
	0x44b,		/* Kannada */
	0x449,		/* Tamil */
	0x44a,		/* Telugu */
	0x45b,		/* Sinhalese */
	0x455,		/* Burmese */
	0x453,		/* Khmer */
	0x454,		/* Lao */
/*80*/	0x42a,		/* Vietnamese */
	0x421,		/* Indonesian */
	0x464,		/* Tagalog */
	0x43e,		/* Malay (latin) */
	0x83e,		/* Malay (arabic) */
	0x45e,		/* Amharic */
	0x473,		/* Tigrinya */
	0x472,		/* Galla, oromo, afan */
	0x477,		/* Somali */
	0x441,		/* Swahili */
/*90*/	0xffff,		/* Kinyarwanda/Ruanda */
	0xffff,		/* Rundi/Kirundi */
	0xffff,		/* Nyanja/Chewa */
	0xffff,		/* Malagasy */
/*94*/	0xffff,		/* Esperanto */
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
/*100*/	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
/*110*/	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
/*120*/	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
	0xffff,
/*128*/	0x452,		/* Welsh */
	0x42d,		/* Basque */
/*130*/	0x403,		/* Catalan */
	0x476,		/* Latin */
	0xffff,		/* Quechua */
	0x474,		/* Guarani */
	0xffff,		/* Aymara */
	0x444,		/* Tatar */
	0xffff,		/* Uighur */
	0xffff,		/* Dzongkha/Bhutani */
	0xffff,		/* Javanese (roman) */
	0xffff,		/* Sundanese (roman) */
/*140*/	0x456,		/* Galician */
	0x436,		/* Afrikaans */
	0xffff,		/* Breton */
	0x45d,		/* Inuktitut */
	0x43c,		/* Scottish Gaelic */
	0xc3c,		/* Manx Gaelic */
	0x83c,		/* Irish Gaelic (with dot) */
	0xffff,		/* Tongan */
	0xffff,		/* Greek (polytonic) */
	0xffff,		/* Greenlandic */	/* Presumably icelandic? */
/*150*/	0x42c,		/* Azebaijani (roman) */
	0xffff
};


uint16 WinLangFromMac(int maclang) {

    if ( maclang<0 || maclang>=sizeof(_WinLangFromMac)/sizeof(_WinLangFromMac[0]))
return( 0xffff );

return( _WinLangFromMac[maclang] );
}

#define N_(a) a


static struct ttf_nameids { char *text; int data; }   ttfnameids[] = {
    { N_("Fullname"), 4},
    { N_("Styles (SubFamily)"), 2},
    { N_("Copyright"), 0},
    { N_("Family"), 1},
    { N_("UniqueID"), 3},
    { N_("Version"), 5},
    { N_("Trademark"), 7},
    { N_("Manufacturer"), 8},
    { N_("Designer"), 9},
    { N_("Descriptor"), 10},
    { N_("Vendor URL"), 11},
    { N_("Designer URL"), 12},
    { N_("License"), 13},
    { N_("License URL"), 14},
    { N_("Preferred Family"), 16},
    { N_("Preferred Styles"), 17},
    { N_("Compatible Full"), 18},
    { N_("Sample Text"), 19},
    { N_("CID findfont Name"), 20},
    { NULL }};


const char *TTFNameIds(int id) {
    int i;
    for ( i=0; ttfnameids[i].text!=NULL; ++i )
	if ( ttfnameids[i].data == id )
return( (char *) ttfnameids[i].text );

    if ( id==6 )
return( "Postscript" );

return( "Unknown" );
}


static struct ms_languages { char *text; int data; }  mslanguages[] = {
    { N_("Afrikaans"), 0x436},
    { N_("Albanian"), 0x41c},
    { N_("Lang|Amharic"), 0x45e},
    { N_("Arabic (Saudi Arabia)"), 0x401},
    { N_("Arabic (Iraq)"), 0x801},
    { N_("Arabic (Egypt)"), 0xc01},
    { N_("Arabic (Libya)"), 0x1001},
    { N_("Arabic (Algeria)"), 0x1401},
    { N_("Arabic (Morocco)"), 0x1801},
    { N_("Arabic (Tunisia)"), 0x1C01},
    { N_("Arabic (Oman)"), 0x2001},
    { N_("Arabic (Yemen)"), 0x2401},
    { N_("Arabic (Syria)"), 0x2801},
    { N_("Arabic (Jordan)"), 0x2c01},
    { N_("Arabic (Lebanon)"), 0x3001},
    { N_("Arabic (Kuwait)"), 0x3401},
    { N_("Arabic (U.A.E.)"), 0x3801},
    { N_("Arabic (Bahrain)"), 0x3c01},
    { N_("Arabic (Qatar)"), 0x4001},
    { N_("Lang|Armenian"), 0x42b},
    { N_("Assamese"), 0x44d},
    { N_("Azeri (Latin)"), 0x42c},
    { N_("Azeri (Cyrillic)"), 0x82c},
    { N_("Basque"), 0x42d},
    { N_("Byelorussian"), 0x423},
    { N_("Lang|Bengali"), 0x445},
    { N_("Bengali Bangladesh"), 0x845},
    { N_("Bulgarian"), 0x402},
    { N_("Burmese"), 0x455},
    { N_("Catalan"), 0x403},
    { N_("Cambodian"), 0x453},
    { N_("Lang|Cherokee"), 0x45c},
    { N_("Chinese (Taiwan)"), 0x404},
    { N_("Chinese (PRC)"), 0x804},
    { N_("Chinese (Hong Kong)"), 0xc04},
    { N_("Chinese (Singapore)"), 0x1004},
    { N_("Chinese (Macau)"), 0x1404},
    { N_("Croatian"), 0x41a},
    { N_("Croatian Bosnia/Herzegovina"), 0x101a},
    { N_("Czech"), 0x405},
    { N_("Danish"), 0x406},
    { N_("Divehi"), 0x465},
    { N_("Dutch"), 0x413},
    { N_("Flemish (Belgian Dutch)"), 0x813},
    { N_("Edo"), 0x466},
    { N_("English (British)"), 0x809},
    { N_("English (US)"), 0x409},
    { N_("English (Canada)"), 0x1009},
    { N_("English (Australian)"), 0xc09},
    { N_("English (New Zealand)"), 0x1409},
    { N_("English (Irish)"), 0x1809},
    { N_("English (South Africa)"), 0x1c09},
    { N_("English (Jamaica)"), 0x2009},
    { N_("English (Caribbean)"), 0x2409},
    { N_("English (Belize)"), 0x2809},
    { N_("English (Trinidad)"), 0x2c09},
    { N_("English (Zimbabwe)"), 0x3009},
    { N_("English (Philippines)"), 0x3409},
    { N_("English (Indonesia)"), 0x3809},
    { N_("English (Hong Kong)"), 0x3c09},
    { N_("English (India)"), 0x4009},
    { N_("English (Malaysia)"), 0x4409},
    { N_("Estonian"), 0x425},
    { N_("Faeroese"), 0x438},
    { N_("Lang|Farsi"), 0x429},
    { N_("Filipino"), 0x464},
    { N_("Finnish"), 0x40b},
    { N_("French French"), 0x40c},
    { N_("French Belgium"), 0x80c},
    { N_("French Canadian"), 0xc0c},
    { N_("French Swiss"), 0x100c},
    { N_("French Luxembourg"), 0x140c},
    { N_("French Monaco"), 0x180c},
    { N_("French West Indies"), 0x1c0c},
    { N_("French Réunion"), 0x200c},
    { N_("French D.R. Congo"), 0x240c},
    { N_("French Senegal"), 0x280c},
    { N_("French Camaroon"), 0x2c0c},
    { N_("French Côte d'Ivoire"), 0x300c},
    { N_("French Mali"), 0x340c},
    { N_("French Morocco"), 0x380c},
    { N_("French Haiti"), 0x3c0c},
    { N_("French North Africa"), 0xe40c},
    { N_("Frisian"), 0x462},
    { N_("Fulfulde"), 0x467},
    { N_("Gaelic (Scottish)"), 0x43c},
    { N_("Gaelic (Irish)"), 0x83c},
    { N_("Galician"), 0x467},
    { N_("Lang|Georgian"), 0x437},
    { N_("German German"), 0x407},
    { N_("German Swiss"), 0x807},
    { N_("German Austrian"), 0xc07},
    { N_("German Luxembourg"), 0x1007},
    { N_("German Liechtenstein"), 0x1407},
    { N_("Lang|Greek"), 0x408},
    { N_("Guarani"), 0x474},
    { N_("Lang|Gujarati"), 0x447},
    { N_("Hausa"), 0x468},
    { N_("Hawaiian"), 0x475},
    { N_("Lang|Hebrew"), 0x40d},
    { N_("Hindi"), 0x439},
    { N_("Hungarian"), 0x40e},
    { N_("Ibibio"), 0x469},
    { N_("Icelandic"), 0x40f},
    { N_("Igbo"), 0x470},
    { N_("Indonesian"), 0x421},
    { N_("Inuktitut"), 0x45d},
    { N_("Italian"), 0x410},
    { N_("Italian Swiss"), 0x810},
    { N_("Japanese"), 0x411},
    { N_("Lang|Kannada"), 0x44b},
    { N_("Kanuri"), 0x471},
    { N_("Kashmiri (India)"), 0x860},
    { N_("Kazakh"), 0x43f},
    { N_("Lang|Khmer"), 0x453},
    { N_("Kirghiz"), 0x440},
    { N_("Konkani"), 0x457},
    { N_("Korean"), 0x412},
    { N_("Korean (Johab)"), 0x812},
    { N_("Lao"), 0x454},
    { N_("Latvian"), 0x426},
    { N_("Lang|Latin"), 0x476},
    { N_("Lithuanian"), 0x427},
    { N_("Lithuanian (Classic)"), 0x827},
    { N_("Macedonian"), 0x42f},
    { N_("Malay"), 0x43e},
    { N_("Malay (Brunei)"), 0x83e},
    { N_("Lang|Malayalam"), 0x44c},
    { N_("Maltese"), 0x43a},
    { N_("Manipuri"), 0x458},
    { N_("Maori"), 0x481},
    { N_("Marathi"), 0x44e},
    { N_("Mongolian (Cyrillic)"), 0x450},
    { N_("Mongolian (Mongolian)"), 0x850},
    { N_("Nepali"), 0x461},
    { N_("Nepali (India)"), 0x861},
    { N_("Norwegian (Bokmal)"), 0x414},
    { N_("Norwegian (Nynorsk)"), 0x814},
    { N_("Lang|Oriya"), 0x448},
    { N_("Oromo"), 0x472},
    { N_("Papiamentu"), 0x479},
    { N_("Pashto"), 0x463},
    { N_("Polish"), 0x415},
    { N_("Portugese (Portugal)"), 0x416},
    { N_("Portuguese (Brasil)"), 0x816},
    { N_("Punjabi (India)"), 0x446},
    { N_("Punjabi (Pakistan)"), 0x846},
    { N_("Quecha (Bolivia)"), 0x46b},
    { N_("Quecha (Ecuador)"), 0x86b},
    { N_("Quecha (Peru)"), 0xc6b},
    { N_("Rhaeto-Romanic"), 0x417},
    { N_("Romanian"), 0x418},
    { N_("Romanian (Moldova)"), 0x818},
    { N_("Russian"), 0x419},
    { N_("Russian (Moldova)"), 0x819},
    { N_("Sami (Lappish)"), 0x43b},
    { N_("Sanskrit"), 0x43b},
    { N_("Sepedi"), 0x46c},
    { N_("Serbian (Cyrillic)"), 0xc1a},
    { N_("Serbian (Latin)"), 0x81a},
    { N_("Sindhi India"), 0x459},
    { N_("Sindhi Pakistan"), 0x859},
    { N_("Lang|Sinhalese"), 0x45b},
    { N_("Slovak"), 0x41b},
    { N_("Slovenian"), 0x424},
    { N_("Sorbian"), 0x42e},
    { N_("Spanish (Traditional)"), 0x40a},
    { N_("Spanish Mexico"), 0x80a},
    { N_("Spanish (Modern)"), 0xc0a},
    { N_("Spanish (Guatemala)"), 0x100a},
    { N_("Spanish (Costa Rica)"), 0x140a},
    { N_("Spanish (Panama)"), 0x180a},
    { N_("Spanish (Dominican Republic)"), 0x1c0a},
    { N_("Spanish (Venezuela)"), 0x200a},
    { N_("Spanish (Colombia)"), 0x240a},
    { N_("Spanish (Peru)"), 0x280a},
    { N_("Spanish (Argentina)"), 0x2c0a},
    { N_("Spanish (Ecuador)"), 0x300a},
    { N_("Spanish (Chile)"), 0x340a},
    { N_("Spanish (Uruguay)"), 0x380a},
    { N_("Spanish (Paraguay)"), 0x3c0a},
    { N_("Spanish (Bolivia)"), 0x400a},
    { N_("Spanish (El Salvador)"), 0x440a},
    { N_("Spanish (Honduras)"), 0x480a},
    { N_("Spanish (Nicaragua)"), 0x4c0a},
    { N_("Spanish (Puerto Rico)"), 0x500a},
    { N_("Spanish (United States)"), 0x540a},
    { N_("Spanish (Latin America)"), 0xe40a},
    { N_("Sutu"), 0x430},
    { N_("Swahili (Kenyan)"), 0x441},
    { N_("Swedish (Sweden)"), 0x41d},
    { N_("Swedish (Finland)"), 0x81d},
    { N_("Lang|Syriac"), 0x45a},
    { N_("Lang|Tagalog"), 0x464},
    { N_("Tajik"), 0x428},
    { N_("Tamazight (Arabic)"), 0x45f},
    { N_("Tamazight (Latin)"), 0x85f},
    { N_("Lang|Tamil"), 0x449},
    { N_("Tatar (Tatarstan)"), 0x444},
    { N_("Lang|Telugu"), 0x44a},
    { N_("Lang|Thai"), 0x41e},
    { N_("Tibetan (PRC)"), 0x451},
    { N_("Tibetan Bhutan"), 0x851},
    { N_("Tigrinya Ethiopia"), 0x473},
    { N_("Tigrinyan Eritrea"), 0x873},
    { N_("Tsonga"), 0x431},
    { N_("Tswana"), 0x432},
    { N_("Turkish"), 0x41f},
    { N_("Turkmen"), 0x442},
    { N_("Lang|Uighur"), 0x480},
    { N_("Ukrainian"), 0x422},
    { N_("Urdu (Pakistan)"), 0x420},
    { N_("Urdu (India)"), 0x820},
    { N_("Uzbek (Latin)"), 0x443},
    { N_("Uzbek (Cyrillic)"), 0x843},
    { N_("Venda"), 0x433},
    { N_("Vietnamese"), 0x42a},
    { N_("Welsh"), 0x452},
    { N_("Xhosa"), 0x434},
    { N_("Lang|Yi"), 0x478},
    { N_("Yiddish"), 0x43d},
    { N_("Yoruba"), 0x46a},
    { N_("Zulu"), 0x435},
    { NULL }};


const char *MSLangString(int language) {
    int i;
    for ( i=0; mslanguages[i].text!=NULL; ++i )
        if ( mslanguages[i].data == language )
return( (char *) mslanguages[i].text );

    language &= 0xff;
    for ( i=0; mslanguages[i].text!=NULL; ++i )
        if ( (mslanguages[i].data & 0xff) == language )
return( (char *) mslanguages[i].text );

return "Unknown" ;
}


static void TTFAddLangStr(FILE *ttf, struct ttfinfo *info, int id,
	int strlen, int stroff,int plat,int spec,int language) {
    struct ttflangname *cur, *prev;
    char *str;

    if ( plat==1 && id>=256 && (info->features!=NULL || info->fvar_start!=0)) {
	MacFeatureAdd(ttf,info,id,strlen,stroff,spec,language);
	return;
    } else if ( id<0 || id>=ttf_namemax )
      return;

    str = _readencstring(ttf,stroff,strlen,plat,spec,language);
    if ( str==NULL )		/* we didn't understand the encoding */
      return;
    if ( id==ttf_postscriptname )
	  ValidatePostScriptFontName(str);
    if ( *str=='\0' ) {
	  /*  free(str);*/
	  return;
    }

    if ( plat==1 || plat==0 )
	language = WinLangFromMac(language);
    if ( (language&0xff00)==0 ) language |= 0x400;

    for ( prev=NULL, cur=info->names; cur!=NULL && cur->lang!=language; prev = cur, cur=cur->next );
    if ( cur==NULL ) {
	cur = chunkalloc(sizeof(struct ttflangname));
	cur->lang = language;
	if ( prev==NULL )
	    info->names = cur;
	else
	    prev->next = cur;
    }
    if ( cur->names[id]==NULL ) {
	cur->names[id] = str;
	if ( plat==1 || plat==0 )
	    cur->frommac[id/32] |= (1<<(id&0x1f));
/* There's some wacky bug in gcc. If the optimizer is on then strcmp gets turned */
/*  into some inline call. Which would be fine, except the following line goes */
/*  bluey. "Called object is not a function", etc. Compiles fine if we turn off */
/*  the inlining magic */
#ifdef strcmp
# undef strcmp
#endif
    } else if ( strcmp(str,cur->names[id])==0 ) {
	free(str);
	if ( plat==3 )
	    cur->frommac[id/32] &= ~(1<<(id&0x1f));
    } else if ( plat==1 ) {
	/* Mac string doesn't match mac unicode string */
	if ( !IsSubSetOf(str,cur->names[id]) )
	    LogError( _("Warning: Mac and Unicode entries in the 'name' table differ for the\n %s string in the language %s\n Mac String: %s\nMac Unicode String: %s\n"),
		    TTFNameIds(id),MSLangString(language),
		    str,cur->names[id]);
	else
	    LogError( _("Warning: Mac string is a subset of the Unicode string in the 'name' table\n for the %s string in the %s language.\n"),
		    TTFNameIds(id),MSLangString(language));
	free(str);
    } else if ( plat==3 && (cur->frommac[id/32] & (1<<(id&0x1f))) ) {
	if ( !IsSubSetOf(cur->names[id],str) )
	    LogError( _("Warning: Mac and Windows entries in the 'name' table differ for the\n %s string in the language %s\n Mac String: %s\nWindows String: %s\n"),
		    TTFNameIds(id),MSLangString(language),
		    cur->names[id],str);
	else
	    LogError( _("Warning: Mac string is a subset of the Windows string in the 'name' table\n for the %s string in the %s language.\n"),
		    TTFNameIds(id),MSLangString(language));
	free(cur->names[id]);
	cur->names[id] = str;
	cur->frommac[id/32] &= ~(1<<(id&0x1f));
    } else {
	int ret;
	if ( info->dupnamestate!=0 )
	    ret = info->dupnamestate;
	else
	    ret = 3;
	if ( ret==1 || ret==2 )
	  info->dupnamestate = ret;
	
	if ( ret==0 || ret==1 )
	    free(str);
	else {
	    free(cur->names[id]);
	    cur->names[id] = str;
	}
    }
}

static int is_ascii(char *str) {	/* isascii is in ctype */
    if ( str==NULL )
return( false );
    while ( *str && *str<127 && *str>=' ' )
	++str;
return( *str=='\0' );
}

static char *FindLangEntry(struct ttfinfo *info, int id ) {
    /* Look for an entry with string id */
    /* we prefer english, if we can't find english look for something in ascii */
    struct ttflangname *cur;
    char *ret;

    for ( cur=info->names; cur!=NULL && cur->lang!=0x409; cur=cur->next );
    if ( cur!=NULL && cur->names[id]==NULL ) cur = NULL;
    if ( cur==NULL )
	for ( cur=info->names; cur!=NULL && (cur->lang&0xf)!=0x09; cur=cur->next );
    if ( cur!=NULL && cur->names[id]==NULL ) cur = NULL;
    if ( cur==NULL )
	for ( cur=info->names; cur!=NULL && !is_ascii(cur->names[id]); cur=cur->next );
    if ( cur==NULL )
	for ( cur=info->names; cur!=NULL && cur->names[id]==NULL; cur=cur->next );
    if ( cur==NULL )
return( NULL );
    ret = copy(cur->names[id]);	
return( ret );
}

struct otfname *FindAllLangEntries(FILE *ttf, struct ttfinfo *info, int id ) {
    /* Look for all entries with string id under windows platform */
    int32 here = ftell(ttf);
    int i, cnt, tableoff;
    int platform, specific, language, name, str_len, stroff;
    struct otfname *head=NULL, *cur;

    if ( info->copyright_start!=0 && id!=0 ) {
	fseek(ttf,info->copyright_start,SEEK_SET);
	/* format selector = */ getushort(ttf);
	cnt = getushort(ttf);
	tableoff = info->copyright_start+getushort(ttf);
	for ( i=0; i<cnt; ++i ) {
	    platform = getushort(ttf);
	    specific = getushort(ttf);
	    language = getushort(ttf);
	    name = getushort(ttf);
	    str_len = getushort(ttf);
	    stroff = getushort(ttf);

	    if ( platform==3 && name==id ) {
		char *temp = _readencstring(ttf,tableoff+stroff,str_len,platform,specific,language);
		if ( temp!=NULL ) {
		    cur = chunkalloc(sizeof(struct otfname));
		    cur->next = head;
		    head = cur;
		    cur->lang = language;
		    cur->name = temp;
		}
	    }
	}
	fseek(ttf,here,SEEK_SET);
    }
return( head );
}

static struct macname *reversemacnames(struct macname *mn) {
    struct macname *next, *prev=NULL;

    if ( mn==NULL )
return( NULL );

    next = mn->next;
    while ( next!=NULL ) {
	mn->next = prev;
	prev = mn;
	mn = next;
	next = mn->next;
    }
    mn->next = prev;
return( mn );
}

static void readttfcopyrights(FILE *ttf,struct ttfinfo *info) {
    int i, cnt, tableoff;
    int platform, specific, language, name, str_len, stroff;

    if ( info->feat_start!=0 )
	readmacfeaturemap(ttf,info);
    if ( info->copyright_start!=0 ) {
	fseek(ttf,info->copyright_start,SEEK_SET);
	/* format selector = */ getushort(ttf);
	cnt = getushort(ttf);
	tableoff = info->copyright_start+getushort(ttf);
	for ( i=0; i<cnt; ++i ) {
	    platform = getushort(ttf);
	    specific = getushort(ttf);
	    language = getushort(ttf);
	    name = getushort(ttf);
	    str_len = getushort(ttf);
	    stroff = getushort(ttf);
    
	    TTFAddLangStr(ttf,info,name,str_len,tableoff+stroff,
		    platform,specific,language);
	}
    }

    if ( info->copyright==NULL )
	info->copyright = FindLangEntry(info,ttf_copyright);
    if ( info->familyname==NULL )
	info->familyname = FindLangEntry(info,ttf_family);
    if ( info->fullname==NULL )
	info->fullname = FindLangEntry(info,ttf_fullname);
    if ( info->version==NULL )
	info->version = FindLangEntry(info,ttf_version);
    if ( info->fontname==NULL )
	info->fontname = FindLangEntry(info,ttf_postscriptname);

    if ( info->fontname != NULL && *info->fontname=='\0' ) {
	free(info->fontname);
	info->fontname = NULL;
    }
    if ( info->familyname != NULL && *info->familyname=='\0' ) {
	free(info->familyname);
	info->familyname = NULL;
    }
    if ( info->fullname != NULL && *info->fullname=='\0' ) {
	free(info->fullname);
	info->fullname = NULL;
    }

    /* OpenType spec says the version string should begin with "Version " and */
    /*  end with a space and have a number in between */
    if ( info->version==NULL ) info->version = copy("1.0");
    else if ( strnmatch(info->version,"Version ",8)==0 ) {
	char *temp = copy(info->version+8);
	if ( temp[strlen(temp)-1]==' ' )
	    temp[strlen(temp)-1] = '\0';
	free(info->version);
	info->version = temp;
    }
    if ( info->fontname==NULL ) {
	if ( info->fullname!=NULL )
	    info->fontname = stripspaces(copy(info->fullname));
	if ( info->fontname==NULL && info->familyname!=NULL )
	    info->fontname = stripspaces(copy(info->familyname));
	if ( info->fontname!=NULL )
	    ValidatePostScriptFontName(info->fontname);
    }

    if ( info->features ) {
	MacFeat *mf;
	struct macsetting *ms;
	for ( mf=info->features; mf!=NULL; mf = mf->next ) {
	    mf->featname = reversemacnames(mf->featname);
	    for ( ms=mf->settings; ms!=NULL; ms=ms->next )
		ms->setname = reversemacnames(ms->setname);
	}
    }
}

static void readttfpreglyph(FILE *ttf,struct ttfinfo *info) {
    readttfhead(ttf,info);
    readttfhhea(ttf,info);
    readttfmaxp(ttf,info);
    readttfcopyrights(ttf,info);
}

#define _On_Curve	1
#define _X_Short	2
#define _Y_Short	4
#define _Repeat		8
#define _X_Same		0x10
#define _Y_Same		0x20

static void FigureControls(SplinePoint *from, SplinePoint *to, BasePoint *cp,
	int is_order2) {
    /* What are the control points for 2 cp bezier which will provide the same*/
    /*  curve as that for the 1 cp bezier specified above */
    real b, c, d;

    if ( is_order2 ) {
	from->nextcp = to->prevcp = *cp;
	if ( cp->x==to->me.x && cp->y==to->me.y ) {
	    /* I would lose track of the proper location of this cp if I left */
	    /* it here (would end up with from->nonextcp, which would mean I'd*/
	    /* use from->me rather than to->me in tottf.c:SSAddPoints. So we  */
	    /* distort it a little */
	    BasePoint off;
	    double len;
	    off.x = from->me.x-to->me.x; off.y = from->me.y-to->me.y;
	    len = sqrt(off.x*off.x+off.y*off.y);
	    if ( len>3 ) {
		/* move the cp slightly toward from, but on the line between the two */
		from->nextcp.x = (to->prevcp.x += rint(off.x/len));
		from->nextcp.y = (to->prevcp.y += rint(off.y/len));
	    }
	}
    } else {
	d = from->me.x;
	c = 2*cp->x - 2*from->me.x;
	b = to->me.x+from->me.x-2*cp->x;
	from->nextcp.x = d+c/3;
	to->prevcp.x = from->nextcp.x + (c+b)/3;

	d = from->me.y;
	c = 2*cp->y - 2*from->me.y;
	b = to->me.y+from->me.y-2*cp->y;
	from->nextcp.y = d+c/3;
	to->prevcp.y = from->nextcp.y + (c+b)/3;
    }

    if ( from->me.x!=from->nextcp.x || from->me.y!=from->nextcp.y )
	from->nonextcp = false;
    if ( to->me.x!=to->prevcp.x || to->me.y!=to->prevcp.y )
	to->noprevcp = false;
    if ( is_order2 && (to->noprevcp || from->nonextcp)) {
	to->noprevcp = from->nonextcp = true;
	from->nextcp = from->me;
	to->prevcp = to->me;
    }
}

static SplineSet *ttfbuildcontours(int path_cnt,uint16 *endpt, char *flags,
	BasePoint *pts, int is_order2) {
    SplineSet *head=NULL, *last=NULL, *cur;
    int i, path, start, last_off;
    SplinePoint *sp;

    for ( path=i=0; path<path_cnt; ++path ) {
	if ( endpt[path]<i )	/* Sigh. Yes there are fonts with bad endpt info */
	  continue;
	cur = chunkalloc(sizeof(SplineSet));
	if ( head==NULL )
	    head = cur;
	else
	    last->next = cur;
	last = cur;
	last_off = false;
	start = i;
	sp = NULL;
	while ( i<=endpt[path] ) {
	    if ( flags[i]&_On_Curve ) {
		sp = chunkalloc(sizeof(SplinePoint));
		sp->me = sp->nextcp = sp->prevcp = pts[i];
		sp->nonextcp = sp->noprevcp = true;
		sp->ttfindex = i;
		sp->nextcpindex = 0xffff;
		if ( last_off && cur->last!=NULL )
		    FigureControls(cur->last,sp,&pts[i-1],is_order2);
		last_off = false;
	    } else if ( last_off ) {
		/* two off curve points get a third on curve point created */
		/* half-way between them. Now isn't that special */
		sp = chunkalloc(sizeof(SplinePoint));
		sp->me.x = (pts[i].x+pts[i-1].x)/2;
		sp->me.y = (pts[i].y+pts[i-1].y)/2;
		sp->nextcp = sp->prevcp = sp->me;
		sp->nonextcp = true;
		sp->ttfindex = 0xffff;
		sp->nextcpindex = i;
		if ( last_off && cur->last!=NULL )
		    FigureControls(cur->last,sp,&pts[i-1],is_order2);
		/* last_off continues to be true */
	    } else {
		if ( cur->first!=NULL )
		    cur->last->nextcpindex = i;
		last_off = true;
		sp = NULL;
	    }
	    if ( sp!=NULL ) {
		if ( cur->first==NULL )
		    cur->first = sp;
		else
		    SplineMake(cur->last,sp,is_order2);
		cur->last = sp;
	    }
	    ++i;
	}
	if ( start==i-1 ) {
	    /* MS chinese fonts have contours consisting of a single off curve*/
	    /*  point. What on earth do they think that means? */
	    /* Oh. I see. It's used to possition marks and such */
	    if ( cur->first==NULL ) {
		sp = chunkalloc(sizeof(SplinePoint));
		sp->me.x = pts[start].x;
		sp->me.y = pts[start].y;
		sp->nextcp = sp->prevcp = sp->me;
		sp->nonextcp = sp->noprevcp = true;
		sp->ttfindex = i-1;
		sp->nextcpindex = 0xffff;
		cur->first = cur->last = sp;
	    }
	} else if ( !(flags[start]&_On_Curve) && !(flags[i-1]&_On_Curve) ) {
	    sp = chunkalloc(sizeof(SplinePoint));
	    sp->me.x = (pts[start].x+pts[i-1].x)/2;
	    sp->me.y = (pts[start].y+pts[i-1].y)/2;
	    sp->nextcp = sp->prevcp = sp->me;
	    sp->nonextcp = true;
	    sp->ttfindex = 0xffff;
	    sp->nextcpindex = start;
	    FigureControls(cur->last,sp,&pts[i-1],is_order2);
	    SplineMake(cur->last,sp,is_order2);
	    cur->last = sp;
	    FigureControls(sp,cur->first,&pts[start],is_order2);
	} else if ( !(flags[i-1]&_On_Curve)) {
	    FigureControls(cur->last,cur->first,&pts[i-1],is_order2);
	    cur->last->nextcpindex = i-1;
	} else if ( !(flags[start]&_On_Curve) ) {
	    FigureControls(cur->last,cur->first,&pts[start],is_order2);
	    sp->nextcpindex = start;
	}
	if ( cur->last!=cur->first ) {
	    SplineMake(cur->last,cur->first,is_order2);
	    cur->last = cur->first;
	}
	for ( sp=cur->first; ; ) {
	    if ( sp->ttfindex!=0xffff && SPInterpolate(sp) )
		sp->dontinterpolate = true;
	    if ( sp->next==NULL )
	break;
	    sp=sp->next->to;
	    if ( sp==cur->first )
	break;
	}
    }
return( head );
}

static void readttfsimpleglyph(FILE *ttf,struct ttfinfo *info,SplineChar *sc, int path_cnt) {
    uint16 *endpt = galloc((path_cnt+1)*sizeof(uint16));
    uint8 *instructions;
    char *flags;
    BasePoint *pts;
    int i, j, tot, len;
    int last_pos;

    for ( i=0; i<path_cnt; ++i ) {
	endpt[i] = getushort(ttf);
	if ( i!=0 && endpt[i]<endpt[i-1] ) {
	    LogError( _("Bad tt font: contour ends make no sense in glyph %d.\n"),
		    sc->orig_pos );
return;
	}
    }
    if ( path_cnt==0 ) {
	tot = 0;
	pts = galloc(sizeof(BasePoint));
    } else {
	tot = endpt[path_cnt-1]+1;
	pts = galloc(tot*sizeof(BasePoint));
    }

    len = getushort(ttf);
    instructions = galloc(len);
    for ( i=0; i<len; ++i )
	instructions[i] = getc(ttf);

    flags = galloc(tot);
    for ( i=0; i<tot; ++i ) {
	flags[i] = getc(ttf);
	if ( flags[i]&_Repeat ) {
	    int cnt = getc(ttf);
	    if ( i+cnt>=tot ) {
		IError("Flag count is wrong (or total is): %d %d", i+cnt, tot );
		cnt = tot-i-1;
	    }
	    for ( j=0; j<cnt; ++j )
		flags[i+j+1] = flags[i];
	    i += cnt;
	}
	if ( feof(ttf))
    break;
    }
    if ( i!=tot )
	IError("Flag count is wrong (or total is): %d %d in glyph %d", i, tot, sc->orig_pos );

    last_pos = 0;
    for ( i=0; i<tot; ++i ) {
	if ( flags[i]&_X_Short ) {
	    int off = getc(ttf);
	    if ( !(flags[i]&_X_Same ) )
		off = -off;
	    pts[i].x = last_pos + off;
	} else if ( flags[i]&_X_Same )
	    pts[i].x = last_pos;
	else
	    pts[i].x = last_pos + (short) getushort(ttf);
	last_pos = pts[i].x;
    }

    last_pos = 0;
    for ( i=0; i<tot; ++i ) {
	if ( flags[i]&_Y_Short ) {
	    int off = getc(ttf);
	    if ( !(flags[i]&_Y_Same ) )
		off = -off;
	    pts[i].y = last_pos + off;
	} else if ( flags[i]&_Y_Same )
	    pts[i].y = last_pos;
	else
	    pts[i].y = last_pos + (short) getushort(ttf);
	last_pos = pts[i].y;
    }
    /* TH there is no need to actually build the truetype contours */
    sc->layers[ly_fore].splines = NULL; /* ttfbuildcontours(path_cnt,endpt,flags,pts,info->to_order2);*/
    if ( info->to_order2 && len!=0 ) {
	sc->ttf_instrs_len = len;
	sc->ttf_instrs = instructions;
    } else
	free(instructions);
    SCCatagorizePoints(sc);
    free(endpt);
    free(flags);
    free(pts);
    if ( feof(ttf))
	LogError( _("Reached end of file when reading simple glyph\n") );
}

static void readttfcompositglyph(FILE *ttf,struct ttfinfo *info,SplineChar *sc, int32 end) {
    RefChar *head=NULL, *last=NULL, *cur;
    int flags=0, arg1, arg2;

    if ( ftell(ttf)>=end ) {
	LogError( _("Empty composite %d\n"), sc->orig_pos );
return;
    }

    do {
	if ( ftell(ttf)>=end ) {
	    LogError( _("Bad flags value, implied MORE components at end of glyph %d\n"), sc->orig_pos );
    break;
	}
	cur = RefCharCreate();
	flags = getushort(ttf);
	cur->orig_pos = getushort(ttf);
	if ( feof(ttf) || cur->orig_pos>=info->glyph_cnt ) {
	    LogError(_("Reference to glyph %d out of bounds when parsing 'glyf' table.\n"), cur->orig_pos );
	    cur->orig_pos = 0;
	}
	if ( info->inuse!=NULL )
	    info->inuse[cur->orig_pos] = true;
	if ( flags&_ARGS_ARE_WORDS ) {
	    arg1 = (short) getushort(ttf);
	    arg2 = (short) getushort(ttf);
	} else {
	    arg1 = (signed char) getc(ttf);
	    arg2 = (signed char) getc(ttf);
	}
	cur->use_my_metrics =		 (flags & _USE_MY_METRICS) ? 1 : 0;
	cur->round_translation_to_grid = (flags & _ROUND) ? 1 : 0;
	if ( flags & _ARGS_ARE_XY ) {
	    /* There is some very strange stuff (half-)documented on the apple*/
	    /*  site about how these should be interpretted when there are */
	    /*  scale factors, or rotations */
	    /* It isn't well enough described to be comprehensible */
	    /*  http://fonts.apple.com/TTRefMan/RM06/Chap6glyf.html */
	    /* Microsoft says nothing about this */
	    /* Adobe implies this is a difference between MS and Apple */
	    /*  MS doesn't do this, Apple does (GRRRGH!!!!) */
	    /* Adobe says that setting bit 12 means that this will not happen */
	    /* Adobe says that setting bit 11 means that this will happen */
	    /*  So if either bit is set we know when this happens, if neither */
	    /*  we guess... But I still don't know how to interpret the */
	    /*  apple mode under rotation... */
	    /* I notice that FreeType does nothing about rotation nor does it */
	    /*  interpret bits 11&12 */
	    /* Ah. It turns out that even Apple does not do what Apple's docs */
	    /*  claim it does. I think I've worked it out (see below), but... */
	    /*  Bleah! */
	    cur->transform[4] = arg1;
	    cur->transform[5] = arg2;
	} else {
	    /* Somehow we can get offsets by looking at the points in the */
	    /*  points so far generated and comparing them to the points in */
	    /*  the current componant */
	    /* How exactly is not described on any of the Apple, MS, Adobe */
	    /* freetype looks up arg1 in the set of points we've got so far */
	    /*  looks up arg2 in the new component (before renumbering) */
	    /*  offset.x = arg1.x - arg2.x; offset.y = arg1.y - arg2.y; */
	    /* This fixup needs to be done later though (after all glyphs */
	    /*  have been loaded) */
	    cur->match_pt_base = arg1;
	    cur->match_pt_ref = arg2;
	    cur->point_match = true;
	}
	cur->transform[0] = cur->transform[3] = 1.0;
	if ( flags & _SCALE )
	    cur->transform[0] = cur->transform[3] = get2dot14(ttf);
	else if ( flags & _XY_SCALE ) {
	    cur->transform[0] = get2dot14(ttf);
	    cur->transform[3] = get2dot14(ttf);
	} else if ( flags & _MATRIX ) {
	    cur->transform[0] = get2dot14(ttf);
	    cur->transform[1] = get2dot14(ttf);
	    cur->transform[2] = get2dot14(ttf);
	    cur->transform[3] = get2dot14(ttf);
	}
	if ( flags & _ARGS_ARE_XY ) {	/* Only muck with these guys if they are real offsets and not point matching */
#ifdef __Mac
	/* On mac assume scaled offsets unless told unscaled explicitly */
	if ( !(flags&_UNSCALED_OFFSETS) &&
#else
	/* everywhere else assume unscaled offsets unless told scaled explicitly */
	if ( (flags & _SCALED_OFFSETS) &&
#endif
		(flags & _ARGS_ARE_XY) && (flags&(_SCALE|_XY_SCALE|_MATRIX))) {
	    /*static int asked = 0;*/
	    /* This is not what Apple documents on their website. But it is */
	    /*  what appears to match the behavior of their rasterizer */
	    /* Apple has changed their documentation (without updating their */
	    /*  changelog), but I believe they are still incorrect */
	    cur->transform[4] *= sqrt(cur->transform[0]*cur->transform[0]+
		    cur->transform[1]*cur->transform[1]);
	    cur->transform[5] *= sqrt(cur->transform[2]*cur->transform[2]+
		    cur->transform[3]*cur->transform[3]);
	}
	}
	if ( cur->orig_pos>=info->glyph_cnt ) {
	    LogError(_("Glyph %d attempts to reference glyph %d which is outside the font\n"), sc->orig_pos, cur->orig_pos );
	    chunkfree(cur,sizeof(*cur));
	} else {
	    if ( head==NULL )
		head = cur;
	    else
		last->next = cur;
	    last = cur;
	}
	if ( feof(ttf)) {
	    LogError(_("Reached end of file when reading composit glyph\n") );
    break;
	}
    } while ( flags&_MORE );
    if ( (flags & _INSTR ) && info->to_order2 && ftell(ttf)<end ) {
	sc->ttf_instrs_len = getushort(ttf);
	if ( sc->ttf_instrs_len > 0 && ftell(ttf)+sc->ttf_instrs_len<=end ) {
	    uint8 *instructions = galloc(sc->ttf_instrs_len);
	    int i;
	    for ( i=0; i<sc->ttf_instrs_len; ++i )
		instructions[i] = getc(ttf);
	    sc->ttf_instrs = instructions;
	} else
	    sc->ttf_instrs_len = 0;
    }
    /* I'm ignoring many things that I don't understand here */
    /* what happens if ARGS AREN'T XY */
    /* ROUND means that offsets should rounded to the grid before being added */
    /*  (it's irrelevant for my purposes) */
    sc->layers[ly_fore].refs = head;
}

static SplineChar *readttfglyph(FILE *ttf,struct ttfinfo *info,int start, int end,int gid) {
    int path_cnt;
    SplineChar *sc = SplineCharCreate();

    sc->unicodeenc = -1;
    sc->vwidth = info->emsize;
    /* sc->manualhints = 1; */ /* But only when I know how to read them in!!!! */

    if ( end>info->glyph_length ) {
	if ( !info->complainedbeyondglyfend )
	    LogError(_("Bad glyph (%d), its definition extends beyond the end of the glyf table\n"), gid );
	info->complainedbeyondglyfend = true;
	SplineCharFree(sc);
	return( NULL );
    }
    if ( start==end ) {
	/* This isn't mentioned, but we seem to get some glyphs with no size,*/
	/*  not even a path cnt. They appear to be empty glyphs */
      return( sc );
    }
    fseek(ttf,info->glyph_start+start,SEEK_SET);
    path_cnt = (short) getushort(ttf);
    sc->xmin = getushort(ttf);
    sc->ymin = getushort(ttf);
    sc->xmax = getushort(ttf);
    sc->ymax = getushort(ttf);
    /* TH: this is just plain ugly, and no longer needed
    sc->lsidebearing = sc->ymax;
    */
    sc->orig_pos = gid;
    if ( path_cnt>=0 )
	readttfsimpleglyph(ttf,info,sc,path_cnt);
    else
	readttfcompositglyph(ttf,info,sc,info->glyph_start+end);
    if ( start>end )
	LogError(_("Bad glyph (%d), disordered 'loca' table (start comes after end)\n"), gid );
    else if ( ftell(ttf)>info->glyph_start+end )
	LogError(_("Bad glyph (%d), its definition extends beyond the space allowed for it\n"), gid );
    return( sc );
}

static void readttfencodings(FILE *ttf,struct ttfinfo *info, int justinuse);

static void readttfglyphs(FILE *ttf,struct ttfinfo *info) {
    int i, anyread;
    uint32 *goffsets = galloc((info->glyph_cnt+1)*sizeof(uint32));

    /* First we read all the locations. This might not be needed, they may */
    /*  just follow one another, but nothing I've noticed says that so let's */
    /*  be careful */
    fseek(ttf,info->glyphlocations_start,SEEK_SET);
    if ( info->index_to_loc_is_long ) {
	for ( i=0; i<=info->glyph_cnt ; ++i )
	    goffsets[i] = getlong(ttf);
    } else {
	for ( i=0; i<=info->glyph_cnt ; ++i )
	    goffsets[i] = 2*getushort(ttf);
    }

    info->chars = gcalloc(info->glyph_cnt,sizeof(SplineChar *));
    if ( !info->is_ttc ) {
	/* read all the glyphs */
	for ( i=0; i<info->glyph_cnt ; ++i ) {
	    info->chars[i] = readttfglyph(ttf,info,goffsets[i],goffsets[i+1],i);
	}
    } else {
	/* only read the glyphs we actually use in this font */
	/* this is complicated by references (and substitutions), */
	/* we can't just rely on the encoding to tell us what is used */
	info->inuse = gcalloc(info->glyph_cnt,sizeof(char));
	readttfencodings(ttf,info,true);
	if ( info->gsub_start!=0 )		/* Some glyphs may appear in substitutions and not in the encoding... */
	    readttfgsubUsed(ttf,info);
	    /* I don't bother to read the morx table because mac doesn't */
	    /*  support ttc files */
	anyread = true;
	while ( anyread ) {
	    anyread = false;
	    for ( i=0; i<info->glyph_cnt ; ++i ) {
		if ( info->inuse[i] && info->chars[i]==NULL ) {
		    info->chars[i] = readttfglyph(ttf,info,goffsets[i],goffsets[i+1],i);
		    anyread = info->chars[i]!=NULL;
		}
	    }
	}
	free(info->inuse); info->inuse = NULL;
    }
    free(goffsets);
    for ( i=0; i<info->glyph_cnt ; ++i )
	if ( info->chars[i]!=NULL )
	    info->chars[i]->orig_pos = i;
}

/* Standard names for cff */
const char *cffnames[] = {
 ".notdef",
 "space",
 "exclam",
 "quotedbl",
 "numbersign",
 "dollar",
 "percent",
 "ampersand",
 "quoteright",
 "parenleft",
 "parenright",
 "asterisk",
 "plus",
 "comma",
 "hyphen",
 "period",
 "slash",
 "zero",
 "one",
 "two",
 "three",
 "four",
 "five",
 "six",
 "seven",
 "eight",
 "nine",
 "colon",
 "semicolon",
 "less",
 "equal",
 "greater",
 "question",
 "at",
 "A",
 "B",
 "C",
 "D",
 "E",
 "F",
 "G",
 "H",
 "I",
 "J",
 "K",
 "L",
 "M",
 "N",
 "O",
 "P",
 "Q",
 "R",
 "S",
 "T",
 "U",
 "V",
 "W",
 "X",
 "Y",
 "Z",
 "bracketleft",
 "backslash",
 "bracketright",
 "asciicircum",
 "underscore",
 "quoteleft",
 "a",
 "b",
 "c",
 "d",
 "e",
 "f",
 "g",
 "h",
 "i",
 "j",
 "k",
 "l",
 "m",
 "n",
 "o",
 "p",
 "q",
 "r",
 "s",
 "t",
 "u",
 "v",
 "w",
 "x",
 "y",
 "z",
 "braceleft",
 "bar",
 "braceright",
 "asciitilde",
 "exclamdown",
 "cent",
 "sterling",
 "fraction",
 "yen",
 "florin",
 "section",
 "currency",
 "quotesingle",
 "quotedblleft",
 "guillemotleft",
 "guilsinglleft",
 "guilsinglright",
 "fi",
 "fl",
 "endash",
 "dagger",
 "daggerdbl",
 "periodcentered",
 "paragraph",
 "bullet",
 "quotesinglbase",
 "quotedblbase",
 "quotedblright",
 "guillemotright",
 "ellipsis",
 "perthousand",
 "questiondown",
 "grave",
 "acute",
 "circumflex",
 "tilde",
 "macron",
 "breve",
 "dotaccent",
 "dieresis",
 "ring",
 "cedilla",
 "hungarumlaut",
 "ogonek",
 "caron",
 "emdash",
 "AE",
 "ordfeminine",
 "Lslash",
 "Oslash",
 "OE",
 "ordmasculine",
 "ae",
 "dotlessi",
 "lslash",
 "oslash",
 "oe",
 "germandbls",
 "onesuperior",
 "logicalnot",
 "mu",
 "trademark",
 "Eth",
 "onehalf",
 "plusminus",
 "Thorn",
 "onequarter",
 "divide",
 "brokenbar",
 "degree",
 "thorn",
 "threequarters",
 "twosuperior",
 "registered",
 "minus",
 "eth",
 "multiply",
 "threesuperior",
 "copyright",
 "Aacute",
 "Acircumflex",
 "Adieresis",
 "Agrave",
 "Aring",
 "Atilde",
 "Ccedilla",
 "Eacute",
 "Ecircumflex",
 "Edieresis",
 "Egrave",
 "Iacute",
 "Icircumflex",
 "Idieresis",
 "Igrave",
 "Ntilde",
 "Oacute",
 "Ocircumflex",
 "Odieresis",
 "Ograve",
 "Otilde",
 "Scaron",
 "Uacute",
 "Ucircumflex",
 "Udieresis",
 "Ugrave",
 "Yacute",
 "Ydieresis",
 "Zcaron",
 "aacute",
 "acircumflex",
 "adieresis",
 "agrave",
 "aring",
 "atilde",
 "ccedilla",
 "eacute",
 "ecircumflex",
 "edieresis",
 "egrave",
 "iacute",
 "icircumflex",
 "idieresis",
 "igrave",
 "ntilde",
 "oacute",
 "ocircumflex",
 "odieresis",
 "ograve",
 "otilde",
 "scaron",
 "uacute",
 "ucircumflex",
 "udieresis",
 "ugrave",
 "yacute",
 "ydieresis",
 "zcaron",
 "exclamsmall",
 "Hungarumlautsmall",
 "dollaroldstyle",
 "dollarsuperior",
 "ampersandsmall",
 "Acutesmall",
 "parenleftsuperior",
 "parenrightsuperior",
 "twodotenleader",
 "onedotenleader",
 "zerooldstyle",
 "oneoldstyle",
 "twooldstyle",
 "threeoldstyle",
 "fouroldstyle",
 "fiveoldstyle",
 "sixoldstyle",
 "sevenoldstyle",
 "eightoldstyle",
 "nineoldstyle",
 "commasuperior",
 "threequartersemdash",
 "periodsuperior",
 "questionsmall",
 "asuperior",
 "bsuperior",
 "centsuperior",
 "dsuperior",
 "esuperior",
 "isuperior",
 "lsuperior",
 "msuperior",
 "nsuperior",
 "osuperior",
 "rsuperior",
 "ssuperior",
 "tsuperior",
 "ff",
 "ffi",
 "ffl",
 "parenleftinferior",
 "parenrightinferior",
 "Circumflexsmall",
 "hyphensuperior",
 "Gravesmall",
 "Asmall",
 "Bsmall",
 "Csmall",
 "Dsmall",
 "Esmall",
 "Fsmall",
 "Gsmall",
 "Hsmall",
 "Ismall",
 "Jsmall",
 "Ksmall",
 "Lsmall",
 "Msmall",
 "Nsmall",
 "Osmall",
 "Psmall",
 "Qsmall",
 "Rsmall",
 "Ssmall",
 "Tsmall",
 "Usmall",
 "Vsmall",
 "Wsmall",
 "Xsmall",
 "Ysmall",
 "Zsmall",
 "colonmonetary",
 "onefitted",
 "rupiah",
 "Tildesmall",
 "exclamdownsmall",
 "centoldstyle",
 "Lslashsmall",
 "Scaronsmall",
 "Zcaronsmall",
 "Dieresissmall",
 "Brevesmall",
 "Caronsmall",
 "Dotaccentsmall",
 "Macronsmall",
 "figuredash",
 "hypheninferior",
 "Ogoneksmall",
 "Ringsmall",
 "Cedillasmall",
 "questiondownsmall",
 "oneeighth",
 "threeeighths",
 "fiveeighths",
 "seveneighths",
 "onethird",
 "twothirds",
 "zerosuperior",
 "foursuperior",
 "fivesuperior",
 "sixsuperior",
 "sevensuperior",
 "eightsuperior",
 "ninesuperior",
 "zeroinferior",
 "oneinferior",
 "twoinferior",
 "threeinferior",
 "fourinferior",
 "fiveinferior",
 "sixinferior",
 "seveninferior",
 "eightinferior",
 "nineinferior",
 "centinferior",
 "dollarinferior",
 "periodinferior",
 "commainferior",
 "Agravesmall",
 "Aacutesmall",
 "Acircumflexsmall",
 "Atildesmall",
 "Adieresissmall",
 "Aringsmall",
 "AEsmall",
 "Ccedillasmall",
 "Egravesmall",
 "Eacutesmall",
 "Ecircumflexsmall",
 "Edieresissmall",
 "Igravesmall",
 "Iacutesmall",
 "Icircumflexsmall",
 "Idieresissmall",
 "Ethsmall",
 "Ntildesmall",
 "Ogravesmall",
 "Oacutesmall",
 "Ocircumflexsmall",
 "Otildesmall",
 "Odieresissmall",
 "OEsmall",
 "Oslashsmall",
 "Ugravesmall",
 "Uacutesmall",
 "Ucircumflexsmall",
 "Udieresissmall",
 "Yacutesmall",
 "Thornsmall",
 "Ydieresissmall",
 "001.000",
 "001.001",
 "001.002",
 "001.003",
 "Black",
 "Bold",
 "Book",
 "Light",
 "Medium",
 "Regular",
 "Roman",
 "Semibold",
 NULL
};
const int nStdStrings = sizeof(cffnames)/sizeof(cffnames[0])-1;

static char **readcfffontnames(FILE *ttf,int *cnt) {
    uint16 count = getushort(ttf);
    int offsize;
    uint32 *offsets;
    char **names;
    int i,j;

    if ( cnt!=NULL ) *cnt = count;

    if ( count==0 )
return( NULL );
    offsets = galloc((count+1)*sizeof(uint32));
    offsize = getc(ttf);
    for ( i=0; i<=count; ++i )
	offsets[i] = getoffset(ttf,offsize);
    names = galloc((count+1)*sizeof(char *));
    for ( i=0; i<count; ++i ) {
	if ( offsets[i+1]<offsets[i] ) {
/* GT: The CFF font type contains a thing called a name INDEX, and that INDEX */
/* GT: is bad. It is an index of many of the names used in the CFF font. */
/* GT: We hope the user will never see this. */
	    LogError( _("Bad CFF name INDEX\n") );
	    while ( i<count ) {
		names[i] = copy("");
		++i;
	    }
	    --i;
	} else {
	    names[i] = galloc(offsets[i+1]-offsets[i]+1);
	    for ( j=0; j<offsets[i+1]-offsets[i]; ++j )
		names[i][j] = getc(ttf);
	    names[i][j] = '\0';
	}
    }
    names[i] = NULL;
    free(offsets);
return( names );
}

static char *addnibble(char *pt, int nib) {
    if ( nib<=9 )
	*pt++ = nib+'0';
    else if ( nib==10 )
	*pt++ = '.';
    else if ( nib==11 )
	*pt++ = 'E';
    else if ( nib==12 ) {
	*pt++ = 'E';
	*pt++ = '-';
    } else if ( nib==14 )
	*pt++ = '-';
    else if ( nib==15 )
	*pt++ = '\0';
return( pt );
}

static int readcffthing(FILE *ttf,int *_ival,real *dval,int *operand) {
    char buffer[50], *pt;
    int ch, ival;

    ch = getc(ttf);
    if ( ch==12 ) {
	*operand = (12<<8) | getc(ttf);
return( 3 );
    } else if ( ch<=21 ) {
	*operand = ch;
return( 3 );
    } else if ( ch==30 ) {
	/* fixed format doesn't exist in dict data but does in type2 strings */
	pt = buffer;
	do {
	    ch = getc(ttf);
	    if ( pt<buffer+44 || (ch&0xf)==0xf || (ch&0xf0)==0xf0 ) {
		pt = addnibble(pt,ch>>4);
		pt = addnibble(pt,ch&0xf);
	    }
	} while ( pt[-1]!='\0' );
	*dval = strtod(buffer,NULL);
return( 2 );
    } else if ( ch>=32 && ch<=246 ) {
	*_ival = ch-139;
return( 1 );
    } else if ( ch>=247 && ch<=250 ) {
	*_ival = ((ch-247)<<8) + getc(ttf)+108;
return( 1 );
    } else if ( ch>=251 && ch<=254 ) {
	*_ival = -((ch-251)<<8) - getc(ttf)-108;
return( 1 );
    } else if ( ch==28 ) {
	ival = getc(ttf)<<8;
	*_ival = (short) (ival | getc(ttf));
return( 1 );
    } else if ( ch==29 ) {
	/* 4 byte integers exist in dict data but not in type2 strings */
	ival = getc(ttf)<<24;
	ival = ival | getc(ttf)<<16;
	ival = ival | getc(ttf)<<8;
	*_ival = (int) (ival | getc(ttf));
return( 1 );
    }
    LogError(_("Unexpected value in dictionary %d\n"), ch );
    *_ival = 0;
return( 0 );
}

static void skipcfft2thing(FILE *ttf) {
    /* The old CFF spec allows little type2 programs to live in the CFF dict */
    /*  indices. These are designed to allow interpolation of values for mm */
    /*  fonts. */
    /* The Type2 program is terminated by an "endchar" operator */
    /* I don't support this, but I shall try to skip over them properly */
    /* There's no discussion about how values move from the t2 stack to the */
    /*  cff stack, as there are no examples of this, it's hard to guess */
    int ch;

/* GT: DICT is a magic term inside CFF fonts, as is INDEX, and I guess CFF and type2 */
    LogError( _("FontForge does not support type2 programs embedded in CFF DICT INDICES.\n") );
    forever {
	ch = getc(ttf);
	if ( ch>=247 && ch<=254 )
	    getc(ttf);		/* Two byte number */
	else if ( ch==255 ) {
	    getc(ttf); getc(ttf); getc(ttf); getc(ttf);
	    /* 16.16 number */
	} else if ( ch==28 ) {
	    getc(ttf);
	    getc(ttf);
	} else if ( ch==12 ) {
	    getc(ttf);		/* Two byte operator */
	} else if ( ch==14 ) {
return;
	}
    }
}

struct topdicts {
    int32 cff_start;

    char *fontname;	/* From Name Index */

    int version;	/* SID */
    int notice;		/* SID */
    int copyright;	/* SID */
    int fullname;	/* SID */
    int familyname;	/* SID */
    int weight;		/* SID */
    int isfixedpitch;
    real italicangle;
    real underlinepos;
    real underlinewidth;
    int painttype;
    int charstringtype;
    real fontmatrix[6];
    int uniqueid;
    real fontbb[4];
    real strokewidth;
    int xuid[20];
    int charsetoff;	/* from start of file */
    int encodingoff;	/* from start of file */
    int charstringsoff;	/* from start of file */
    int private_size;
    int private_offset;	/* from start of file */
    int synthetic_base;	/* font index */
    int postscript_code;	/* SID */
 /* synthetic fonts only (whatever they are) */
    int basefontname;		/* SID */
 /* Multiple master/synthetic fonts */
    real basefontblend[16];	/* delta */	/* No description of why this is relevant for mm fonts */
 /* Multiple master fonts only */
    int blendaxistypes[17];	/* SID */
    int nMasters;
    int nAxes;
    real weightvector[17];
    int lenBuildCharArray;	/* No description of what this means */
    int NormalizeDesignVector;	/* SID */	/* No description of what this does */
    int ConvertDesignVector;	/* SID */	/* No description of what this does */
 /* CID fonts only */
    int ros_registry;		/* SID */
    int ros_ordering;		/* SID */
    int ros_supplement;
    real cidfontversion;
    int cidfontrevision;
    int cidfonttype;
    int cidcount;
    int uidbase;
    int fdarrayoff;	/* from start of file */
    int fdselectoff;	/* from start of file */
    int sid_fontname;	/* SID */
/* Private stuff */
    real bluevalues[14];
    real otherblues[10];
    real familyblues[14];
    real familyotherblues[10];
    real bluescale;
    real blueshift;
    real bluefuzz;
    int stdhw;
    int stdvw;
    real stemsnaph[10];
    real stemsnapv[10];
    int forcebold;
    real forceboldthreshold;
    int languagegroup;
    real expansionfactor;
    int initialRandomSeed;
    int subrsoff;	/* from start of this private table */
    int defaultwidthx;
    int nominalwidthx;

    struct pschars glyphs;
    struct pschars local_subrs;
    uint16 *charset;
};

static void TopDictFree(struct topdicts *dict) {
    int i;

    free(dict->charset);
    for ( i=0; i<dict->glyphs.cnt; ++i )
	free(dict->glyphs.values[i]);
    free(dict->glyphs.values);
    free(dict->glyphs.lens);
    for ( i=0; i<dict->local_subrs.cnt; ++i )
	free(dict->local_subrs.values[i]);
    free(dict->local_subrs.values);
    free(dict->local_subrs.lens);
    free(dict);
}

static void readcffsubrs(FILE *ttf, struct pschars *subs) {
    uint16 count = getushort(ttf);
    int offsize;
    uint32 *offsets;
    int i,j, base;
    int err = false;

    memset(subs,'\0',sizeof(struct pschars));
    if ( count==0 )
return;
    subs->cnt = count;
    subs->lens = galloc(count*sizeof(int));
    subs->values = galloc(count*sizeof(uint8 *));
    offsets = galloc((count+1)*sizeof(uint32));
    offsize = getc(ttf);
    for ( i=0; i<=count; ++i )
	offsets[i] = getoffset(ttf,offsize);
    base = ftell(ttf)-1;
    for ( i=0; i<count; ++i ) {
	if ( offsets[i+1]>offsets[i] && offsets[i+1]-offsets[i]<0x10000 ) {
	    subs->lens[i] = offsets[i+1]-offsets[i];
	    subs->values[i] = galloc(offsets[i+1]-offsets[i]+1);
	    for ( j=0; j<offsets[i+1]-offsets[i]; ++j )
		subs->values[i][j] = getc(ttf);
	    subs->values[i][j] = '\0';
	} else {
	    if ( !err )
		LogError( _("Bad subroutine INDEX in cff font.\n" ));
	    err = true;
	    subs->lens[i] = 1;
	    subs->values[i] = galloc(2);
	    subs->values[i][0] = 11;		/* return */
	    subs->values[i][1] = '\0';
	    fseek(ttf,base+offsets[i+1],SEEK_SET);
	}
    }
    free(offsets);
}

static struct topdicts *readcfftopdict(FILE *ttf, char *fontname, int len) {
    struct topdicts *td = gcalloc(1,sizeof(struct topdicts));
    long base = ftell(ttf);
    int ival, oval, sp, ret, i;
    real stack[50];

    if ( fontname!=NULL )
	ValidatePostScriptFontName(fontname);

    td->fontname = fontname;
    td->underlinepos = -100;
    td->underlinewidth = 50;
    td->charstringtype = 2;
    td->fontmatrix[0] = td->fontmatrix[3] = .001;

    td->notice = td->copyright = td->fullname = td->familyname = td->weight = td->version = -1;
    td->postscript_code = td->basefontname = -1;
    td->synthetic_base = td->ros_registry = -1;
    td->fdarrayoff = td->fdselectoff = td->sid_fontname = -1;
    td->blendaxistypes[0] = -1;

    /* Multiple master fonts can have Type2 operators here, particularly */
    /*  blend operators. We're ignoring that */
    while ( ftell(ttf)<base+len ) {
	sp = 0;
	while ( (ret=readcffthing(ttf,&ival,&stack[sp],&oval))!=3 && ftell(ttf)<base+len ) {
	    if ( ret==1 )
		stack[sp]=ival;
	    if ( ret!=0 && sp<45 )
		++sp;
	}
	if ( ret==3 && oval==31 /* "T2" operator, can have 0 arguments */ ) {
	    skipcfft2thing(ttf);
	} else if ( sp==0 )
	    LogError( _("No argument to operator\n") );
	else if ( ret==3 ) switch( oval ) {
	  case 0:
	    td->version = stack[sp-1];
	  break;
	  case 1:
	    td->notice = stack[sp-1];
	  break;
	  case (12<<8)+0:
	    td->copyright = stack[sp-1];
	  break;
	  case 2:
	    td->fullname = stack[sp-1];
	  break;
	  case 3:
	    td->familyname = stack[sp-1];
	  break;
	  case 4:
	    td->weight = stack[sp-1];
	  break;
	  case (12<<8)+1:
	    td->isfixedpitch = stack[sp-1];
	  break;
	  case (12<<8)+2:
	    td->italicangle = stack[sp-1];
	  break;
	  case (12<<8)+3:
	    td->underlinepos = stack[sp-1];
	  break;
	  case (12<<8)+4:
	    td->underlinewidth = stack[sp-1];
	  break;
	  case (12<<8)+5:
	    td->painttype = stack[sp-1];
	  break;
	  case (12<<8)+6:
	    td->charstringtype = stack[sp-1];
	  break;
	  case (12<<8)+7:
	    memcpy(td->fontmatrix,stack,(sp>=6?6:sp)*sizeof(real));
	  break;
	  case 13:
	    td->uniqueid = stack[sp-1];
	  break;
	  case 5:
	    memcpy(td->fontbb,stack,(sp>=4?4:sp)*sizeof(real));
	  break;
	  case (12<<8)+8:
	    td->strokewidth = stack[sp-1];
	  break;
	  case 14:
	    for ( i=0; i<sp && i<20; ++i )
		td->xuid[i] = stack[i];
	  break;
	  case 15:
	    td->charsetoff = stack[sp-1];
	  break;
	  case 16:
	    td->encodingoff = stack[sp-1];
	  break;
	  case 17:
	    td->charstringsoff = stack[sp-1];
	  break;
	  case 18:
	    td->private_size = stack[0];
	    td->private_offset = stack[1];
	  break;
	  case (12<<8)+20:
	    LogError( _("FontForge does not support synthetic fonts\n") );
	    td->synthetic_base = stack[sp-1];
	  break;
	  case (12<<8)+21:
	    td->postscript_code = stack[sp-1];
	  break;
	  case (12<<8)+22:
	    td->basefontname = stack[sp-1];
	  break;
	  case (12<<8)+23:
	    for ( i=0; i<sp && i<16; ++i )
		td->basefontblend[i] = stack[i];
	  break;
	  case (12<<8)+24:
	    LogError( _("FontForge does not support type2 multiple master fonts\n") );
	    td->nMasters = stack[0];
	    td->nAxes = sp-4;
	    memcpy(td->weightvector,stack+1,(sp-4)*sizeof(real));
	    td->lenBuildCharArray = stack[sp-3];
	    td->NormalizeDesignVector = stack[sp-2];	/* These are type2 charstrings, even in type1 fonts */
	    td->ConvertDesignVector = stack[sp-1];
	  break;
	  case (12<<8)+26:
	    for ( i=0; i<sp && i<16; ++i )
		td->blendaxistypes[i] = stack[i];
	    td->blendaxistypes[i] = -1;
	  break;
	  case (12<<8)+30:
	    td->ros_registry = stack[0];
	    td->ros_ordering = stack[1];
	    td->ros_supplement = stack[2];
	  break;
	  case (12<<8)+31:
	    td->cidfontversion = stack[sp-1];
	  break;
	  case (12<<8)+32:
	    td->cidfontrevision = stack[sp-1];
	  break;
	  case (12<<8)+33:
	    td->cidfonttype = stack[sp-1];
	  break;
	  case (12<<8)+34:
	    td->cidcount = stack[sp-1];
	  break;
	  case (12<<8)+35:
	    td->uidbase = stack[sp-1];
	  break;
	  case (12<<8)+36:
	    td->fdarrayoff = stack[sp-1];
	  break;
	  case (12<<8)+37:
	    td->fdselectoff = stack[sp-1];
	  break;
	  case (12<<8)+38:
	    td->sid_fontname = stack[sp-1];
	  break;
	  case (12<<8)+39:
	    LogError(_("FontForge does not support Chameleon fonts\n"));;
	  break;
	  default:
	    LogError(_("Unknown operator in %s: %x\n"), fontname, oval );
	  break;
	}
    }
return( td );
}

static void readcffprivate(FILE *ttf, struct topdicts *td) {
    int ival, oval, sp, ret, i;
    real stack[50];
    int32 end = td->cff_start+td->private_offset+td->private_size;

    fseek(ttf,td->cff_start+td->private_offset,SEEK_SET);

    td->subrsoff = -1;
    td->expansionfactor = .06;
    td->bluefuzz = 1;
    td->blueshift = 7;
    td->bluescale = .039625;

    while ( ftell(ttf)<end ) {
	sp = 0;
	while ( (ret=readcffthing(ttf,&ival,&stack[sp],&oval))!=3 && ftell(ttf)<end ) {
	    if ( ret==1 )
		stack[sp]=ival;
	    if ( ret!=0 && sp<45 )
		++sp;
	}
	if ( ret==3 && oval==31 /* "T2" operator, can have 0 arguments */ ) {
	    skipcfft2thing(ttf);
	} else if ( sp==0 && oval!=6 && oval!=7 && oval!=8 && oval!=9 && oval !=(12<<8)+12 && oval !=(12<<8)+13)
	    LogError( _("No argument to operator %d in private dict\n"), oval );
	else if ( ret==3 ) switch( oval ) {
	  case 6:
	    for ( i=0; i<sp && i<14; ++i ) {
		td->bluevalues[i] = stack[i];
		if ( i!=0 )
		    td->bluevalues[i] += td->bluevalues[i-1];
	    }
	  break;
	  case 7:
	    for ( i=0; i<sp && i<10; ++i ) {
		td->otherblues[i] = stack[i];
		if ( i!=0 )
		    td->otherblues[i] += td->otherblues[i-1];
	    }
	  break;
	  case 8:
	    for ( i=0; i<sp && i<14; ++i ) {
		td->familyblues[i] = stack[i];
		if ( i!=0 )
		    td->familyblues[i] += td->familyblues[i-1];
	    }
	  break;
	  case 9:
	    for ( i=0; i<sp && i<10; ++i ) {
		td->familyotherblues[i] = stack[i];
		if ( i!=0 )
		    td->familyotherblues[i] += td->familyotherblues[i-1];
	    }
	  break;
	  case (12<<8)+9:
	    td->bluescale = stack[sp-1];
	  break;
	  case (12<<8)+10:
	    td->blueshift = stack[sp-1];
	  break;
	  case (12<<8)+11:
	    td->bluefuzz = stack[sp-1];
	  break;
	  case 10:
	    td->stdhw = stack[sp-1];
	  break;
	  case 11:
	    td->stdvw = stack[sp-1];
	  break;
	  case (12<<8)+12:
	    for ( i=0; i<sp && i<10; ++i ) {
		td->stemsnaph[i] = stack[i];
		if ( i!=0 )
		    td->stemsnaph[i] += td->stemsnaph[i-1];
	    }
	  break;
	  case (12<<8)+13:
	    for ( i=0; i<sp && i<10; ++i ) {
		td->stemsnapv[i] = stack[i];
		if ( i!=0 )
		    td->stemsnapv[i] += td->stemsnapv[i-1];
	    }
	  break;
	  case (12<<8)+14:
	    td->forcebold = stack[sp-1];
	  break;
	  case (12<<8)+15:
	    td->forceboldthreshold = stack[sp-1];
	  break;
	  case (12<<8)+16:
	    /* lenIV. -1 => unencrypted charstrings */
	    /* obsolete */
	  break;
	  case (12<<8)+17:
	    td->languagegroup = stack[sp-1];
	  break;
	  case (12<<8)+18:
	    td->expansionfactor = stack[sp-1];
	  break;
	  case (12<<8)+19:
	    td->initialRandomSeed = stack[sp-1];
	  break;
	  case 19:
	    td->subrsoff = stack[sp-1];
	  break;
	  case 20:
	    td->defaultwidthx = stack[sp-1];
	  break;
	  case 21:
	    td->nominalwidthx = stack[sp-1];
	  break;
	  default:
	    LogError(_("Unknown operator in %s: %x\n"), td->fontname, oval );
	  break;
	}
    }

    if ( td->subrsoff!=-1 ) {
	fseek(ttf,td->cff_start+td->private_offset+td->subrsoff,SEEK_SET);
	readcffsubrs(ttf,&td->local_subrs);
    }
}

static struct topdicts **readcfftopdicts(FILE *ttf, char **fontnames, int32 cff_start) {
    uint16 count = getushort(ttf);
    int offsize;
    uint32 *offsets;
    struct topdicts **dicts;
    int i;

    if ( count==0 )
return( NULL );
    offsets = galloc((count+1)*sizeof(uint32));
    offsize = getc(ttf);
    for ( i=0; i<=count; ++i )
	offsets[i] = getoffset(ttf,offsize);
    dicts = galloc((count+1)*sizeof(struct topdicts *));
    for ( i=0; i<count; ++i ) {
	dicts[i] = readcfftopdict(ttf,fontnames!=NULL?fontnames[i]:NULL,
		offsets[i+1]-offsets[i]);
	dicts[i]->cff_start = cff_start;
    }
    dicts[i] = NULL;
    free(offsets);
return( dicts );
}

static const char *getsid(int sid,char **strings,int scnt) {
    if ( sid==-1 )
return( NULL );
    else if ( sid<nStdStrings )
return( cffnames[sid] );
    else if ( sid-nStdStrings>scnt ) {
	LogError( _("Bad sid %d (must be less than %d)\n"), sid, scnt+nStdStrings );
return( NULL );
    } else
return( strings[sid-nStdStrings]);
}

/* I really expect to deal with encodings in ttf cmap, but ocasionally we */
/*  get a bare cff */
static void readcffenc(FILE *ttf,struct topdicts *dict,struct ttfinfo *info,
	char **strings, int scnt) {
    int format, cnt, i, j, pos, first, last, dupenc, sid;
#if 0
    extern char *AdobeStandardEncoding[], *AdobeExpertEncoding[];
#endif
    const char *name;
    EncMap *map;

    if ( info->encoding_start!=0 )		/* Use the cmap instead */
return;
    if ( info->subfontcnt!=0 )
return;						/* Use cids instead */

    for ( i=0; i<info->glyph_cnt; ++i ) {
	if ( info->chars[i]->unicodeenc==-1 )
	    info->chars[i]->unicodeenc = UniFromName(info->chars[i]->name,ui_none,&custom);
    }

    map = EncMapNew(256,256,&custom);
    if ( dict->encodingoff==0 || dict->encodingoff==1 ) {
	/* Standard Encodings */
	char **enc = dict->encodingoff==0 ? AdobeStandardEncoding : AdobeExpertEncoding;
	map->enc = FindOrMakeEncoding( dict->encodingoff==0 ?
		"AdobeStandard" : "Custom" );
	if ( map->enc==NULL )
	    map->enc = &custom;
	for ( i=0; i<info->glyph_cnt; ++i ) {
	    for ( pos=0; pos<256; ++pos )
		if ( strcmp(info->chars[i]->name,enc[pos])==0 )
	    break;
	    if ( pos<256 )
		map->map[pos] = i;
	}
    } else {
	fseek(ttf,dict->cff_start+dict->encodingoff,SEEK_SET);
	format = getc(ttf);
	if ( (format&0x7f)==0 ) {
	    cnt = getc(ttf);
	    for ( i=1; i<=cnt && i<info->glyph_cnt; ++i )
		map->map[getc(ttf)] = i;
	} else if ( (format&0x7f)==1 ) {
	    cnt = getc(ttf);
	    pos = 0;
	    for ( i=0; i<cnt ; ++i ) {
		first = getc(ttf);
		last = first + getc(ttf)-1;
		while ( first<=last && first<256 ) {
		    if ( pos<info->glyph_cnt )
			map->map[first] = pos;
		    ++pos;
		    ++first;
		}
	    }
	} else
	    LogError( _("Unexpected encoding format in cff: %d\n"), format );
	if ( format&0x80 ) {
	    cnt = getc(ttf);
	    for ( i=0; i<cnt; ++i ) {
		dupenc = getc(ttf);
		sid = getushort(ttf);
		name = getsid(sid,strings,scnt);
		if ( name==NULL )	/* Table is erroneous */
	    break;
		for ( j=0; j<info->glyph_cnt; ++j )
		    if ( strcmp(name,info->chars[j]->name)==0 )
		break;
		if ( j!=info->glyph_cnt )
		    map->map[dupenc] = j;
	    }
	}
    }
    info->map = map;
}

static void readcffset(FILE *ttf,struct topdicts *dict) {
    int len = dict->glyphs.cnt;
    int i;
    int format, cnt, j, first;

    i = 0;
    if ( dict->charsetoff==0 ) {
	/* ISO Adobe charset */
	dict->charset = galloc(len*sizeof(uint16));
	for ( i=0; i<len && i<=228; ++i )
	    dict->charset[i] = i;
    } else if ( dict->charsetoff==1 ) {
	/* Expert charset */
	dict->charset = galloc((len<162?162:len)*sizeof(uint16));
	dict->charset[0] = 0;		/* .notdef */
	dict->charset[1] = 1;
	for ( i=2; i<len && i<=238-227; ++i )
	    dict->charset[i] = i+227;
	dict->charset[12] = 13;
	dict->charset[13] = 14;
	dict->charset[14] = 15;
	dict->charset[15] = 99;
	for ( i=16; i<len && i<=248-223; ++i )
	    dict->charset[i] = i+223;
	dict->charset[25] = 27;
	dict->charset[26] = 28;
	for ( i=27; i<len && i<=266-222; ++i )
	    dict->charset[i] = i+222;
	dict->charset[44] = 109;
	dict->charset[45] = 110;
	for ( i=46; i<len && i<=318-221; ++i )
	    dict->charset[i] = i+221;
	dict->charset[96] = 158;
	dict->charset[97] = 155;
	dict->charset[98] = 163;
	for ( i=99; i<len && i<=326-220; ++i )
	    dict->charset[i] = i+220;
	dict->charset[107] = 150;
	dict->charset[108] = 164;
	dict->charset[109] = 169;
	for ( i=110; i<len && i<=378-217; ++i )
	    dict->charset[i] = i+217;
    } else if ( dict->charsetoff==2 ) {
	/* Expert subset charset */
	dict->charset = galloc((len<130?130:len)*sizeof(uint16));
	dict->charset[0] = 0;		/* .notdef */
	dict->charset[1] = 1;
	for ( i=2; i<len && i<=238-227; ++i )
	    dict->charset[i] = i+227;
	dict->charset[12] = 13;
	dict->charset[13] = 14;
	dict->charset[14] = 15;
	dict->charset[15] = 99;
	for ( i=16; i<len && i<=248-223; ++i )
	    dict->charset[i] = i+223;
	dict->charset[25] = 27;
	dict->charset[26] = 28;
	for ( i=27; i<len && i<=266-222; ++i )
	    dict->charset[i] = i+222;
	dict->charset[44] = 109;
	dict->charset[45] = 110;
	for ( i=46; i<len && i<=272-221; ++i )
	    dict->charset[i] = i+221;
	dict->charset[51] = 300;
	dict->charset[52] = 301;
	dict->charset[53] = 302;
	dict->charset[54] = 305;
	dict->charset[55] = 314;
	dict->charset[56] = 315;
	dict->charset[57] = 158;
	dict->charset[58] = 155;
	dict->charset[59] = 163;
	for ( i=60; i<len && i<=326-260; ++i )
	    dict->charset[i] = i+260;
	dict->charset[67] = 150;
	dict->charset[68] = 164;
	dict->charset[69] = 169;
	for ( i=110; i<len && i<=346-217; ++i )
	    dict->charset[i] = i+217;
    } else {
	dict->charset = galloc(len*sizeof(uint16));
	dict->charset[0] = 0;		/* .notdef */
	fseek(ttf,dict->cff_start+dict->charsetoff,SEEK_SET);
	format = getc(ttf);
	if ( format==0 ) {
	    for ( i=1; i<len; ++i )
		dict->charset[i] = getushort(ttf);
	} else if ( format==1 ) {
	    for ( i = 1; i<len; ) {
		first = dict->charset[i++] = getushort(ttf);
		cnt = getc(ttf);
		for ( j=0; j<cnt; ++j )
		    dict->charset[i++] = ++first;
	    }
	} else if ( format==2 ) {
	    for ( i = 1; i<len; ) {
		first = dict->charset[i++] = getushort(ttf);
		cnt = getushort(ttf);
		for ( j=0; j<cnt; ++j )
		    dict->charset[i++] = ++first;
	    }
	} else
	    LogError( _("Unexpected charset format in cff: %d\n"), format );
    }
    while ( i<len ) dict->charset[i++] = 0;
}

static uint8 *readfdselect(FILE *ttf,int numglyphs) {
    uint8 *fdselect = gcalloc(numglyphs,sizeof(uint8));
    int i, j, format, nr, first, end, fd;

    format = getc(ttf);
    if ( format==0 ) {
	for ( i=0; i<numglyphs; ++i )
	    fdselect[i] = getc(ttf);
    } else if ( format==3 ) {
	nr = getushort(ttf);
	first = getushort(ttf);
	for ( i=0; i<nr; ++i ) {
	    fd = getc(ttf);
	    end = getushort(ttf);
	    for ( j=first; j<end; ++j ) {
		if ( j>=numglyphs )
		    LogError( _("Bad fdselect\n") );
		else
		    fdselect[j] = fd;
	    }
	    first = end;
	}
    } else {
	LogError( _("Didn't understand format for fdselect %d\n"), format );
    }
return( fdselect );
}


static char *intarray2str(int *array, int size) {
    int i,j;
    char *pt, *ret;

    for ( i=size-1; i>=0 && array[i]==0; --i );
    if ( i==-1 )
return( NULL );
    ret = pt = galloc((i+1)*12+12);
    *pt++ = '[';
    for ( j=0; j<=i; ++j ) {
	sprintf( pt, "%d ", array[j]);
	pt += strlen(pt);
    }
    pt[-1]=']';
return( ret );
}

static char *realarray2str(real *array, int size) {
    int i,j;
    char *pt, *ret;

    for ( i=size-1; i>=0 && array[i]==0; --i );
    if ( i==-1 )
return( NULL );
    ret = pt = galloc((i+1)*20+12);
    *pt++ = '[';
    for ( j=0; j<=i; ++j ) {
	sprintf( pt, "%g ", (double) array[j]);
	pt += strlen(pt);
    }
    pt[-1]=']';
return( ret );
}

static void privateadd(struct psdict *private,char *key,char *value) {
    if ( value==NULL )
return;
    private->keys[private->next] = copy(key);
    private->values[private->next++] = value;
}

static void privateaddint(struct psdict *private,char *key,int val) {
    char buf[10];
    if ( val==0 )
return;
    sprintf( buf,"%d", val );
    privateadd(private,key,copy(buf));
}

static void privateaddintarray(struct psdict *private,char *key,int val) {
    char buf[10];
    if ( val==0 )
return;
    sprintf( buf,"[%d]", val );
    privateadd(private,key,copy(buf));
}

static void privateaddreal(struct psdict *private,char *key,double val) {
    char buf[10];
    if ( val==0 )
      return;
    sprintf( buf,"%g", val );
    privateadd(private,key,copy(buf));
}

static void cffprivatefillup(struct psdict *private, struct topdicts *dict) {
    private->cnt = 14;
    private->keys = galloc(14*sizeof(char *));
    private->values = galloc(14*sizeof(char *));
    privateadd(private,"BlueValues",
	    realarray2str(dict->bluevalues,sizeof(dict->bluevalues)/sizeof(dict->bluevalues[0])));
    privateadd(private,"OtherBlues",
	    realarray2str(dict->otherblues,sizeof(dict->otherblues)/sizeof(dict->otherblues[0])));
    privateadd(private,"FamilyBlues",
	    realarray2str(dict->familyblues,sizeof(dict->familyblues)/sizeof(dict->familyblues[0])));
    privateadd(private,"FamilyOtherBlues",
	    realarray2str(dict->familyotherblues,sizeof(dict->familyotherblues)/sizeof(dict->familyotherblues[0])));
    privateaddreal(private,"BlueScale",dict->bluescale);
    privateaddreal(private,"BlueShift",dict->blueshift);
    privateaddreal(private,"BlueFuzz",dict->bluefuzz);
    privateaddintarray(private,"StdHW",dict->stdhw);
    privateaddintarray(private,"StdVW",dict->stdvw);
    privateadd(private,"SnapStemH",
	    realarray2str(dict->stemsnaph,sizeof(dict->stemsnaph)/sizeof(dict->stemsnaph[0])));
    privateadd(private,"SnapStemV",
	    realarray2str(dict->stemsnapv,sizeof(dict->stemsnapv)/sizeof(dict->stemsnapv[0])));
    if ( dict->forcebold )
	privateadd(private,"ForceBold",copy("true"));
    if ( dict->forceboldthreshold!=0 )
	privateaddreal(private,"ForceBoldThreshold",dict->forceboldthreshold);
    privateaddint(private,"LanguageGroup",dict->languagegroup);
    privateaddreal(private,"ExpansionFactor",dict->expansionfactor);
}

static SplineFont *cffsffillup(struct topdicts *subdict, char **strings,
	int scnt) {
    SplineFont *sf = SplineFontEmpty();
    int emsize;
    static int nameless;

    sf->fontname = utf8_verify_copy(getsid(subdict->sid_fontname,strings,scnt));
    if ( sf->fontname==NULL ) {
	char buffer[40];
	sprintf(buffer,"UntitledSubFont_%d", ++nameless );
	sf->fontname = copy(buffer);
    }

    if ( subdict->fontmatrix[0]==0 )
	emsize = 1000;
    else
	emsize = rint( 1/subdict->fontmatrix[0] );
    sf->ascent = .8*emsize;
    sf->descent = emsize - sf->ascent;
    if ( subdict->copyright!=-1 )
	sf->copyright = utf8_verify_copy(getsid(subdict->copyright,strings,scnt));
    else
	sf->copyright = utf8_verify_copy(getsid(subdict->notice,strings,scnt));
    sf->familyname = utf8_verify_copy(getsid(subdict->familyname,strings,scnt));
    sf->fullname = utf8_verify_copy(getsid(subdict->fullname,strings,scnt));
    sf->weight = utf8_verify_copy(getsid(subdict->weight,strings,scnt));
    sf->version = utf8_verify_copy(getsid(subdict->version,strings,scnt));
    sf->italicangle = subdict->italicangle;
    sf->upos = subdict->underlinepos;
    sf->uwidth = subdict->underlinewidth;
    sf->xuid = intarray2str(subdict->xuid,sizeof(subdict->xuid)/sizeof(subdict->xuid[0]));
    sf->uniqueid = subdict->uniqueid;
    sf->strokewidth = subdict->strokewidth;
    sf->strokedfont = subdict->painttype==2;

    if ( subdict->private_size>0 ) {
	sf->private = gcalloc(1,sizeof(struct psdict));
	cffprivatefillup(sf->private,subdict);
    }
return( sf );
}

static void cffinfofillup(struct ttfinfo *info, struct topdicts *dict,
	char **strings, int scnt ) {

    info->glyph_cnt = dict->glyphs.cnt;
    if ( info->glyph_cnt<0 ) info->glyph_cnt = 0;

    if ( dict->fontmatrix[0]==0 )
	info->emsize = 1000;
    else
	info->emsize = rint( 1/dict->fontmatrix[0] );
#if 1
    info->ascent = .8*info->emsize;
#else
    info->ascent = dict->fontbb[3]*info->emsize/(dict->fontbb[3]-dict->fontbb[1]);
#endif
    info->descent = info->emsize - info->ascent;
    if ( dict->copyright!=-1 || dict->notice!=-1 )
	free( info->copyright );
    if ( dict->copyright!=-1 )
	info->copyright = utf8_verify_copy(getsid(dict->copyright,strings,scnt));
    else if ( dict->notice!=-1 )
	info->copyright = utf8_verify_copy(getsid(dict->notice,strings,scnt));
    if ( dict->familyname!=-1 ) {
	free(info->familyname);
	info->familyname = utf8_verify_copy(getsid(dict->familyname,strings,scnt));
    }
    if ( dict->fullname!=-1 ) {
	free(info->fullname);
	info->fullname = utf8_verify_copy(getsid(dict->fullname,strings,scnt));
    }
    if ( dict->weight!=-1 ) {
	free(info->weight);
	info->weight = utf8_verify_copy(getsid(dict->weight,strings,scnt));
    }
    if ( dict->version!=-1 ) {
	free(info->version);
	info->version = utf8_verify_copy(getsid(dict->version,strings,scnt));
    }
    if ( dict->fontname!=NULL ) {
	free(info->fontname);
	info->fontname = utf8_verify_copy(dict->fontname);
    }
    info->italicAngle = dict->italicangle;
    info->upos = dict->underlinepos;
    info->uwidth = dict->underlinewidth;
    info->xuid = intarray2str(dict->xuid,sizeof(dict->xuid)/sizeof(dict->xuid[0]));
    info->uniqueid = dict->uniqueid;
    info->strokewidth = dict->strokewidth;
    info->strokedfont = dict->painttype==2;

    if ( dict->private_size>0 ) {
	info->private = gcalloc(1,sizeof(struct psdict));
	cffprivatefillup(info->private,dict);
    }
    if ( dict->ros_registry!=-1 ) {
	info->cidregistry = copy(getsid(dict->ros_registry,strings,scnt));
	info->ordering = copy(getsid(dict->ros_ordering,strings,scnt));
	info->supplement = dict->ros_supplement;
	info->cidfontversion = dict->cidfontversion;
    }
}

SplineChar *SplineCharCreate(void) {
  SplineChar *sc = chunkalloc(sizeof(SplineChar));
  sc->color = COLOR_DEFAULT;
  sc->orig_pos = 0xffff;
  sc->unicodeenc = -1;
  sc->layer_cnt = 2;
  sc->tex_height = sc->tex_depth = sc->tex_sub_pos = sc->tex_super_pos =
            TEX_UNDEF;
  return( sc );
}


static void cfffigure(struct ttfinfo *info, struct topdicts *dict,
	char **strings, int scnt, struct pschars *gsubrs) {
    int i, cstype;
    DBounds bb;
    struct pschars *subrs;
    struct pscontext pscontext;

    memset(&pscontext,0,sizeof(pscontext));

    cffinfofillup(info, dict, strings, scnt );

/* The format allows for some dicts that are type1 strings and others that */
/*  are type2s. Which means that the global subrs will have a different bias */
/*  as we flip from font to font. So we can't set the bias when we read in */
/*  the subrs but must wait until we know which font we're working on. */
    cstype = dict->charstringtype;
    pscontext.is_type2 = cstype-1;
    pscontext.painttype = dict->painttype;
    gsubrs->bias = cstype==1 ? 0 :
	    gsubrs->cnt < 1240 ? 107 :
	    gsubrs->cnt <33900 ? 1131 : 32768;
    subrs = &dict->local_subrs;
    subrs->bias = cstype==1 ? 0 :
	    subrs->cnt < 1240 ? 107 :
	    subrs->cnt <33900 ? 1131 : 32768;

    info->chars = gcalloc(info->glyph_cnt,sizeof(SplineChar *));
    for ( i=0; i<info->glyph_cnt; ++i ) {
	  info->chars[i] = PSCharStringToSplines(
		dict->glyphs.values[i], dict->glyphs.lens[i],&pscontext,
		subrs,gsubrs,getsid(dict->charset[i],strings,scnt));
	  
	  SplineCharFindBounds(info->chars[i],&bb);
	  info->chars[i]->xmin = bb.minx;
	  info->chars[i]->ymin = bb.miny;
	  info->chars[i]->xmax = bb.maxx;
	  info->chars[i]->ymax = bb.maxy;
	  info->chars[i]->vwidth = info->emsize;
	  if ( cstype==2 ) {
	    if ( info->chars[i]->width == (int16) 0x8000 )
		info->chars[i]->width = dict->defaultwidthx;
	    else
		info->chars[i]->width += dict->nominalwidthx;
	  }
    }
    /* Need to do a reference fixup here !!!!! just in case some idiot */
    /*  used type1 char strings -- or used the depreciated meaning of */
    /*  endchar (==seac) */
}


static void cidfigure(struct ttfinfo *info, struct topdicts *dict,
	char **strings, int scnt, struct pschars *gsubrs, struct topdicts **subdicts,
	uint8 *fdselect) {
    int i, j, cstype, uni, cid;
    struct pschars *subrs;
    SplineFont *sf;
    DBounds bb;
    struct cidmap *map;
    char buffer[100];
    struct pscontext pscontext;
    EncMap *encmap = NULL;

    memset(&pscontext,0,sizeof(pscontext));

    cffinfofillup(info, dict, strings, scnt );

    /* We'll set the encmap later */
    /*info->map = encmap = EncMapNew(info->glyph_cnt,info->glyph_cnt,&custom);*/

    for ( j=0; subdicts[j]!=NULL; ++j );
    info->subfontcnt = j;
    info->subfonts = gcalloc(j+1,sizeof(SplineFont *));
    for ( j=0; subdicts[j]!=NULL; ++j )  {
	info->subfonts[j] = cffsffillup(subdicts[j],strings,scnt);
	info->subfonts[j]->map = encmap;
    }
    for ( i=0; i<info->glyph_cnt; ++i ) {
	sf = info->subfonts[ fdselect[i] ];
	cid = dict->charset[i];
	if ( cid>=sf->glyphcnt ) sf->glyphcnt = sf->glyphmax = cid+1;
	/*if ( cid>=encmap->enccount ) encmap->enccount = cid+1;*/
    }
    for ( j=0; subdicts[j]!=NULL; ++j )
	info->subfonts[j]->glyphs = gcalloc(info->subfonts[j]->glyphcnt,sizeof(SplineChar *));
    /*encmap->encmax = encmap->enccount;*/
    /*encmap->map = galloc(encmap->enccount*sizeof(int));*/
    /*memset(encmap->map,-1,encmap->enccount*sizeof(int));*/

    info->chars = gcalloc(info->glyph_cnt,sizeof(SplineChar *));

    /* info->chars provides access to the chars ordered by glyph, which the */
    /*  ttf routines care about */
    /* sf->glyphs provides access to the chars ordered by CID. Not sure what */
    /*  would happen to a kern from one font to another... */

    map = FindCidMap(info->cidregistry,info->ordering,info->supplement,NULL);

    for ( i=0; i<info->glyph_cnt; ++i ) {
	j = fdselect[i];
	sf = info->subfonts[ j ];
/* The format allows for some dicts that are type1 strings and others that */
/*  are type2s. Which means that the global subrs will have a different bias */
/*  as we flip from font to font. So we can't set the bias when we read in */
/*  the subrs but must wait until we know which font we're working on. */
	cstype = subdicts[j]->charstringtype;
	pscontext.is_type2 = cstype-1;
	pscontext.painttype = subdicts[j]->painttype;
	gsubrs->bias = cstype==1 ? 0 :
		gsubrs->cnt < 1240 ? 107 :
		gsubrs->cnt <33900 ? 1131 : 32768;
	subrs = &subdicts[j]->local_subrs;
	subrs->bias = cstype==1 ? 0 :
		subrs->cnt < 1240 ? 107 :
		subrs->cnt <33900 ? 1131 : 32768;

	cid = dict->charset[i];
	/*encmap->map[cid] = cid;*/
	uni = CID2NameUni(map,cid,buffer,sizeof(buffer));
	info->chars[i] = PSCharStringToSplines(
		dict->glyphs.values[i], dict->glyphs.lens[i],&pscontext,
		subrs,gsubrs,buffer);
	info->chars[i]->vwidth = sf->ascent+sf->descent;
	info->chars[i]->unicodeenc = uni;

	SplineCharFindBounds(info->chars[i],&bb);
	info->chars[i]->xmin = bb.minx;
	info->chars[i]->ymin = bb.miny;
	info->chars[i]->xmax = bb.maxx;
	info->chars[i]->ymax = bb.maxy;

	sf->glyphs[cid] = info->chars[i];
	sf->glyphs[cid]->parent = sf;
	sf->glyphs[cid]->orig_pos = cid;		/* Bug! should be i, but I assume sf->chars[orig_pos]->orig_pos==orig_pos */
	if ( sf->glyphs[cid]->layers[ly_fore].refs!=NULL )
	    IError( "Reference found in CID font. Can't fix it up");
	if ( cstype==2 ) {
	    if ( sf->glyphs[cid]->width == (int16) 0x8000 )
		sf->glyphs[cid]->width = subdicts[j]->defaultwidthx;
	    else
		sf->glyphs[cid]->width += subdicts[j]->nominalwidthx;
	}
    }
    /* No need to do a reference fixup here-- the chars aren't associated */
    /*  with any encoding as is required for seac */
}

static int readcffglyphs(FILE *ttf,struct ttfinfo *info) {
    int offsize;
    int hdrsize;
    char **fontnames, **strings;
    struct topdicts **dicts, **subdicts;
    int i, j, which;
    struct pschars gsubs;
    uint8 *fdselect;
    int scnt;

    fseek(ttf,info->cff_start,SEEK_SET);
    if ( getc(ttf)!='\1' ) {		/* Major version */
	LogError( _("CFF version mismatch\n" ));
return( 0 );
    }
    getc(ttf);				/* Minor version */
    hdrsize = getc(ttf);
    offsize = getc(ttf);
    if ( hdrsize!=4 )
	fseek(ttf,info->cff_start+hdrsize,SEEK_SET);
    fontnames = readcfffontnames(ttf,NULL);
    which = 0;
    if ( fontnames[1]!=NULL ) {		/* More than one? Can that even happen in OpenType? */
	which = PickCFFFont(fontnames);
	if ( which==-1 ) {
	    for ( i=0; fontnames[i]!=NULL; ++i )
		free(fontnames[i]);
	    free(fontnames);
return( 0 );
	}
    }
    dicts = readcfftopdicts(ttf,fontnames,info->cff_start);
	/* String index is just the same as fontname index */
    strings = readcfffontnames(ttf,&scnt);
    readcffsubrs(ttf,&gsubs );
    /* Can be many fonts here. Only decompose the one */
    if ( dicts[which]->charstringsoff!=-1 ) {
	fseek(ttf,info->cff_start+dicts[which]->charstringsoff,SEEK_SET);
	readcffsubrs(ttf,&dicts[which]->glyphs);
    }
    if ( dicts[which]->private_offset!=-1 )
	readcffprivate(ttf,dicts[which]);
    if ( dicts[which]->charsetoff!=-1 )
	readcffset(ttf,dicts[which]);
    if ( dicts[which]->fdarrayoff==-1 )
	cfffigure(info,dicts[which],strings,scnt,&gsubs);
    else {
	fseek(ttf,info->cff_start+dicts[which]->fdarrayoff,SEEK_SET);
	subdicts = readcfftopdicts(ttf,NULL,info->cff_start);
	fseek(ttf,info->cff_start+dicts[which]->fdselectoff,SEEK_SET);
	fdselect = readfdselect(ttf,dicts[which]->glyphs.cnt);
	for ( j=0; subdicts[j]!=NULL; ++j ) {
	    if ( subdicts[j]->private_offset!=-1 )
		readcffprivate(ttf,subdicts[j]);
	    if ( subdicts[j]->charsetoff!=-1 )
		readcffset(ttf,subdicts[j]);
	}
	cidfigure(info,dicts[which],strings,scnt,&gsubs,subdicts,fdselect);
	for ( j=0; subdicts[j]!=NULL; ++j )
	    TopDictFree(subdicts[j]);
	free(subdicts); free(fdselect);
    }
    if ( dicts[which]->encodingoff!=-1 )
	readcffenc(ttf,dicts[which],info,strings,scnt);

    if ( dicts[which]->fdarrayoff==-1 ) {
	for ( i=0; i<info->glyph_cnt ; ++i )
	    if ( info->chars[i]!=NULL )
		info->chars[i]->orig_pos = i;
    }

    if ( info->to_order2 ) {
	for ( i=0; i<info->glyph_cnt; ++i )
	    SCConvertToOrder2(info->chars[i]);
    }

    for ( i=0; fontnames[i]!=NULL && i<1; ++i ) {
	free(fontnames[i]);
	TopDictFree(dicts[i]);
    }
    free(fontnames); free(dicts);
    if ( strings!=NULL ) {
	for ( i=0; strings[i]!=NULL; ++i )
	    free(strings[i]);
	free(strings);
    }
    for ( i=0; i<gsubs.cnt; ++i )
	free(gsubs.values[i]);
    free(gsubs.values); free(gsubs.lens);

return( 1 );
}

static void readttfwidths(FILE *ttf,struct ttfinfo *info) {
    int i,j;
    int lastwidth = info->emsize;
    /* I'm not interested in the lsb, I'm not sure what it means if it differs*/
    /*  from that is specified in the outline. Do we move the outline? */
    int check_width_consistency = info->cff_start!=0 && info->glyph_start==0;
    SplineChar *sc;

    fseek(ttf,info->hmetrics_start,SEEK_SET);
    for ( i=0; i<info->width_cnt && i<info->glyph_cnt; ++i ) {
      lastwidth = getushort(ttf);
      if ( (sc = info->chars[i])!=NULL ) {	/* can happen in ttc files */
		if ( check_width_consistency && sc->width!=lastwidth && abs(sc->width-lastwidth)>1) {
	  if ( info->fontname!=NULL && sc->name!=NULL )
	    LogError("In %s, in glyph %s, 'CFF ' advance width (%d) and\n  'hmtx' width (%d) do not match. (Subsequent mismatches will not be reported)\n",
		     info->fontname, sc->name, sc->width, lastwidth );
	  else
	    LogError("In GID %d, 'CFF ' advance width (%d) and 'hmtx' width (%d) do not match.\n  (Subsequent mismatches will not be reported)\n",
		     i, sc->width, lastwidth );
	  check_width_consistency = false;
	}
	sc->width = lastwidth;
	sc->widthset = true;
      }
      /* lsb = */ getushort(ttf);
    }
    if ( i==0 )
	LogError( _("Invalid ttf hmtx table (or hhea), numOfLongMetrics is 0\n") );
	
    for ( j=i; j<info->glyph_cnt; ++j ) {
	if ( info->chars[j]!=NULL ) {		/* In a ttc file we may skip some */
	    info->chars[j]->width = lastwidth;
	    info->chars[j]->widthset = true;
	}
    }
}


static void readttfvwidths(FILE *ttf,struct ttfinfo *info) {
    int i,j;
    int lastvwidth = info->emsize, vwidth_cnt, tsb, cnt=0;
    int32 voff=0;

    fseek(ttf,info->vhea_start+4+4,SEEK_SET);		/* skip over the version number & typo right/left */
    info->pfminfo.vlinegap = getushort(ttf);
    info->pfminfo.vheadset = true;

    for ( i=0; i<12; ++i )
	getushort(ttf);
    vwidth_cnt = getushort(ttf);

    fseek(ttf,info->vmetrics_start,SEEK_SET);
    for ( i=0; i<vwidth_cnt && i<info->glyph_cnt; ++i ) {
	lastvwidth = getushort(ttf);
	tsb = getushort(ttf);
	if ( info->chars[i]!=NULL ) {		/* can happen in ttc files */
	    info->chars[i]->vwidth = lastvwidth;
	    if ( info->cff_start==0 ) {
	      voff += tsb + info->chars[i]->ymax ;
		++cnt;
	    }
	}
    }
    if ( i==0 )
	LogError( _("Invalid ttf vmtx table (or vhea), numOfLongVerMetrics is 0\n") );
	
    for ( j=i; j<info->glyph_cnt; ++j ) {
	if ( info->chars[j]!=NULL )		/* In a ttc file we may skip some */
	    info->chars[j]->vwidth = lastvwidth;
    }

    /* for truetype fonts the vertical offset is found by adding the ymax of a */
    /*  character to the top side bearing. I set the font wide value to the */
    /*  average of them all */
    /* But opentype doesn't give us the ymax easily, and rather than compute */
    /*  the bounding box I'll just punt and pick a reasonable value */
    /* Of course I hope it will be over riden by the VORG table */
    if ( cnt!=0 )
	info->vertical_origin = (voff+cnt/2)/cnt;
    if ( info->vertical_origin==0 )
	info->vertical_origin = info->ascent;
    if ( info->vorg_start!=0 ) {
	fseek(ttf,info->vorg_start+4,SEEK_SET);
	info->vertical_origin = (short) getushort(ttf);
    }
}

static int modenc(int enc,int modtype) {
return( enc );
}

static int umodenc(int enc,int modtype) {
    if ( modtype==-1 )
return( -1 );
    if ( modtype<=1 /* Unicode */ ) {
	/* No conversion */;
    } else if ( modtype==2 /* SJIS */ ) {
	if ( enc<=127 ) {
	    /* Latin */
	    if ( enc=='\\' ) enc = 0xa5;	/* Yen */
	} else if ( enc>=161 && enc<=223 ) {
	    /* Katakana */
	    enc = unicode_from_jis201[enc];
	} else {
	    int ch1 = enc>>8, ch2 = enc&0xff;
	    if ( ch1 >= 129 && ch1<= 159 )
		ch1 -= 112;
	    else
		ch1 -= 176;
	    ch1 <<= 1;
	    if ( ch2>=159 )
		ch2-= 126;
	    else if ( ch2>127 ) {
		--ch1;
		ch2 -= 32;
	    } else {
		--ch1;
		ch2 -= 31;
	    }
	    enc = unicode_from_jis208[(ch1-0x21)*94+(ch2-0x21)];
	}
    } else if ( modtype==3 /* GB2312 offset by 0x8080, parse just like wansung */ ) {
	if ( enc>0xa1a1 ) {
	    enc -= 0xa1a1;
	    enc = (enc>>8)*94 + (enc&0xff);
	    enc = unicode_from_gb2312[enc];
	    if ( enc==0 ) enc = -1;
	} else if ( enc>0x100 )
	    enc = -1;
    } else if ( modtype==4 /* BIG5 */ ) {	/* old ms docs say big5 is modtype==3, but new ones say 4 */
	if ( enc>0x8100 )
	    enc = unicode_from_big5hkscs[enc-0x8100];
	else if ( enc>0x100 )
	    enc = -1;
    } else if ( modtype==5 /* Wansung == KSC 5601-1987, I hope */ ) {
	if ( enc>0xa1a1 ) {
	    enc -= 0xa1a1;
	    enc = (enc>>8)*94 + (enc&0xff);
	    enc = unicode_from_ksc5601[enc];
	    if ( enc==0 ) enc = -1;
	} else if ( enc>0x100 )
	    enc = -1;
    } else if ( modtype==6 /* Johab */ ) {
	if ( enc>0x8400 )
	    enc = unicode_from_johab[enc-0x8400];
	else if ( enc>0x100 )
	    enc = -1;
    }
    if ( enc==0 )
	enc = -1;
return( enc );
}

struct cmap_encs {
    int platform;
    int specific;
    int offset;
    int lang;
    int format;
    Encoding *enc;
};

static int SubtableIsntSupported(FILE *ttf,uint32 offset,struct cmap_encs *cmap_enc) {
    uint32 here = ftell(ttf);
    int format, len, ret=false;

    fseek(ttf,offset,SEEK_SET);

    cmap_enc->format = format = getushort(ttf);
    if ( format<0 || (format&1) || format>12 ) {
	LogError( _("Encoding subtable for platform=%d, specific=%d has an unsupported format %d.\n"),
		cmap_enc->platform, cmap_enc->specific, format );
	ret = true;
    }

    if ( format!=12 && format!=10 && format!=8 ) {
	len = getushort(ttf);
	cmap_enc->lang = getushort(ttf);
    } else {
	/* padding */ getushort(ttf);
	len = getlong(ttf);
	cmap_enc->lang = getlong(ttf);
    }
    if ( len==0 ) {
	LogError( _("Encoding subtable for platform=%d, specific=%d has a 0 length subtable.\n"),
		cmap_enc->platform, cmap_enc->specific );
	ret = true;
    }
    fseek(ttf,here,SEEK_SET);
return( ret );
}

static enum uni_interp amscheck(struct ttfinfo *info, EncMap *map) {
    int cnt = 0;
    /* Try to guess if the font uses the AMS math PUA assignments */

    if ( map==NULL )
return( ui_none );

    if ( 0xe668<map->enccount && map->map[0xe668]!=-1 &&
	    info->chars[map->map[0xe668]]->unicodeenc=='b' )
	++cnt;
    if ( 0xe3c8<map->enccount && map->map[0xe626]!=-1 &&
	    info->chars[map->map[0xe626]]->unicodeenc==0xe626 )
	++cnt;
    if ( 0xe3c8<map->enccount && map->map[0xe3c8]!=-1 &&
	    info->chars[map->map[0xe3c8]]->unicodeenc==0x29e1 )
	++cnt;
    if ( 0x2A7C<map->enccount && map->map[0x2A7C]!=-1 &&
	    info->chars[map->map[0x2A7C]]->unicodeenc==0xE32A )
	++cnt;
    if ( 0x2920<map->enccount && map->map[0x2920]!=-1 &&
	    info->chars[map->map[0x2920]]->unicodeenc==0xE221 )
	++cnt;
return( cnt>=2 ? ui_ams : ui_none );
}

static int PickCMap(struct cmap_encs *cmap_encs,int enccnt,int def) {
    char buffer[500];
    char **choices, *encname;
    int i, ret;
    static char *macscripts[]= { N_("Script|Roman"), N_("Script|Japanese"), N_("Script|Traditional Chinese"), N_("Script|Korean"),
	N_("Script|Arabic"), N_("Script|Hebrew"),  N_("Script|Greek"),
/* GT: Don't ask me what RSymbol means, I don't know either. It's in apple's */
/* GT:  docs though */
	N_("Script|Cyrillic"), N_("Script|RSymbol"), N_("Script|Devanagari"),
/* 10*/ N_("Script|Gurmukhi"), N_("Script|Gujarati"), NULL, NULL, NULL,
	NULL, NULL, NULL, NULL, NULL,
/* 20*/	NULL, N_("Script|Thai"), NULL, NULL, NULL, N_("Script|Simplified Chinese"),
	NULL, NULL, NULL, N_("Script|Central European"),
/* 30*/ NULL, NULL, NULL };

    choices = galloc(enccnt*sizeof(char *));
    for ( i=0; i<enccnt; ++i ) {
	encname = NULL;
	if ( cmap_encs[i].platform==1 && cmap_encs[i].specific<32 ) {
	    encname = macscripts[cmap_encs[i].specific];
	    if ( encname!=NULL )
		encname = S_(encname);
	} else if ( cmap_encs[i].platform==0 ) {
	    switch ( cmap_encs[i].specific ) {
	      case 0:
		encname = N_("Unicode 1.0");
	      break;
	      case 1:
		encname = N_("Unicode 1.1");
	      break;
	      case 2:
		encname = N_("ISO 10646:1993");
	      break;
	      case 3:
		encname = N_("Unicode 2.0+, BMP only");
	      break;
	      case 4:
		encname = N_("Unicode 2.0+, all planes");
	      break;
	    }
	} else if ( cmap_encs[i].platform==3 && cmap_encs[i].specific==0 )
	    encname = N_("\"Symbol\"");
	if ( encname==NULL )
	    encname = cmap_encs[i].enc->enc_name;

	sprintf(buffer,"%d (%s) %d %s %s  %s",
		cmap_encs[i].platform,
		    cmap_encs[i].platform==0 ? _("Apple Unicode") :
		    cmap_encs[i].platform==1 ? _("Apple") :
		    cmap_encs[i].platform==2 ? _("ISO (Deprecated)") :
		    cmap_encs[i].platform==3 ? _("MicroSoft") :
		    cmap_encs[i].platform==4 ? _("Custom") :
		    cmap_encs[i].platform==7 ? _("FreeType internals") :
					       _("Unknown"),
		cmap_encs[i].specific,
		encname,
		cmap_encs[i].platform==1 && cmap_encs[i].lang!=0? MacLanguageFromCode(cmap_encs[i].lang-1) : "",
		cmap_encs[i].format == 0 ? "Byte encoding table" :
		cmap_encs[i].format == 2 ? "High-byte mapping through table" :
		cmap_encs[i].format == 4 ? "Segment mapping to delta values" :
		cmap_encs[i].format == 6 ? "Trimmed table mapping" :
		cmap_encs[i].format == 8 ? "mixed 16-bit and 32-bit coverage" :
		cmap_encs[i].format == 10 ? "Trimmed array" :
		cmap_encs[i].format == 12 ? "Segmented coverage" :
		    "Unknown format" );
	choices[i] = copy(buffer);
    }
    ret = gwwv_choose(_("Pick a CMap subtable"),(const char **) choices,enccnt,def,
	    _("Pick a CMap subtable"));
    for ( i=0; i<enccnt; ++i )
	free(choices[i]);
    free(choices);
return( ret );
}

/* 'cmap' table: readttfcmap */
static void readttfencodings(FILE *ttf,struct ttfinfo *info, int justinuse) {
  int i,j, def, unicode_cmap, unicode4_cmap, dcnt, dcmap_cnt, dc;
  int nencs, version, usable_encs;
  Encoding *enc = &custom;
  const int32 *trans=NULL;
  enum uni_interp interp = ui_none;
  int platform, specific;
  int offset, encoff=0;
  int format, len;
  uint16 table[256];
  int segCount;
  uint16 *endchars, *startchars, *delta, *rangeOffset, *glyphs;
  int index, last;
  int mod = 0;
  SplineChar *sc;
  uint8 *used;
  int badencwarned=false;
  int glyph_tot;
  Encoding *temp;
  EncMap *map;
  struct cmap_encs *cmap_encs, desired_cmaps[2], *dcmap;

  fseek(ttf,info->encoding_start,SEEK_SET);
  version = getushort(ttf);
  nencs = getushort(ttf);
  if ( version!=0 && nencs==0 )
	nencs = version;		/* Sometimes they are backwards */
  cmap_encs = galloc(nencs*sizeof(struct cmap_encs));
  for ( i=usable_encs=0; i<nencs; ++i ) {
	cmap_encs[usable_encs].platform =  getushort(ttf);
	cmap_encs[usable_encs].specific = getushort(ttf);
	cmap_encs[usable_encs].offset = getlong(ttf);
	temp = enc_from_platspec(cmap_encs[usable_encs].platform,cmap_encs[usable_encs].specific);
	if ( temp==NULL )	/* iconv doesn't support this. Some sun iconvs seem limited */
	  temp = FindOrMakeEncoding("Custom");
	cmap_encs[usable_encs].enc = temp;
	if ( SubtableIsntSupported(ttf,info->encoding_start+cmap_encs[usable_encs].offset,
							   &cmap_encs[usable_encs]))
	  continue;
	++usable_encs;
  }
  if ( usable_encs==0 ) {
	LogError( _("Could not find any valid encoding tables" ));
	free(cmap_encs);
	return;
  }
  def = -1;
  enc = &custom;
  unicode_cmap = unicode4_cmap = -1;
  for ( i=0; i<usable_encs; ++i ) {
	temp = cmap_encs[i].enc;
	platform = cmap_encs[i].platform;
	specific = cmap_encs[i].specific;
	offset = cmap_encs[i].offset;
	
	if ( (platform==3 && specific==10) || (platform==0 && specific==4) ) { /* MS Unicode 4 byte */
	  enc = temp;
	  def = i;
	  unicode4_cmap = i;
	} else if ( !enc->is_unicodefull && 
				(!prefer_cjk_encodings ||
				 (!enc->is_japanese && !enc->is_korean && !enc->is_tradchinese &&
				  !enc->is_simplechinese)) &&
				(( platform==3 && specific==1 ) || /* MS Unicode */
				 /* Well I should only deal with apple unicode specific==0 (default) and 3 (U2.0 semantics) */
				 /*  but apple ships dfonts with specific==1 (Unicode 1.1 semantics) */
				 /*  which is stupid of them */
				 ( platform==0 /*&& (specific==0 || specific==3)*/ ))) {	/* Apple Unicode */
	  enc = temp;
	  def = i;
	} else if ( platform==3 && specific==0 && enc->is_custom ) {
	  /* Only select symbol if we don't have something better */
	  enc = temp;
	  def = i;
	  /* Now I had assumed this would be a 1 byte encoding, but it turns*/
	  /*  out to map into the unicode private use area at U+f000-U+F0FF */
	  /*  so it's a 2 byte enc */
	  /* Mac platform specific encodings are script numbers. 0=>roman, 1=>jap, 2=>big5, 3=>korean, 4=>arab, 5=>hebrew, 6=>greek, 7=>cyrillic, ... 25=>simplified chinese */
	} else if ( platform==1 && specific==0 && enc->is_custom ) {
	  enc = temp;
	  def = i;
	} else if ( platform==1 && (specific==2 ||specific==1||specific==3||specific==25) &&
				!enc->is_unicodefull &&
				(prefer_cjk_encodings || !enc->is_unicodebmp) ) {
	  enc = temp;
	  def = i;
	} else if ( platform==3 && (specific>=2 && specific<=6 ) &&
				!enc->is_unicodefull &&
				(prefer_cjk_encodings || !enc->is_unicodebmp) ) {
	  /* Old ms docs say that specific==3 => big 5, new docs say specific==4 => big5 */
	  /*  Ain't that jus' great? */
	  enc = temp;
	  def = i;
	}
	if ( (platform==3 && specific==1) ||
		 (platform==0 && specific==3))
	  unicode_cmap = i;
  }
  
  if ( justinuse || !ask_user_for_cmap || (i = PickCMap(cmap_encs,usable_encs,def))==-1 )
	i = def;
  info->platform = cmap_encs[i].platform;
  info->specific = cmap_encs[i].specific;
  
  desired_cmaps[0] = cmap_encs[i]; dcnt = 1;
  if ( unicode4_cmap!=-1 ) {
	if ( i!=unicode4_cmap ) {
	  desired_cmaps[1] = cmap_encs[unicode4_cmap];
	  ++dcnt;
	}
  } else if ( unicode_cmap!=-1 ) {
	if ( i!=unicode_cmap ) {
	  desired_cmaps[1] = cmap_encs[unicode_cmap];
	  ++dcnt;
	}
  } else {
	if ( i!=def ) {
	    desired_cmaps[1] = cmap_encs[def];
	    ++dcnt;
	}
  }
  
  map = NULL;
  if ( justinuse ) {
	dcmap_cnt = usable_encs;
	dcmap = cmap_encs;
  } else {
	dcmap_cnt = dcnt;
	dcmap = desired_cmaps;
  }
  for ( dc=dcmap_cnt-1; dc>=0; --dc ) {
	/* if justinuse then look at all cmaps and tick the glyphs they use */
	/* otherwise dcmap_cnt will be either 1 or 2. If 1 then this subtable */
	/* contains both the encoding and the source for unicode encodings */
	/* if dcmap_cnt==2 then when dc==0 we are setting up the encoding */
	/*  and when dc==1 we are setting up the unicode code points */
	int dounicode = (dc==dcmap_cnt-1);
	enc = dcmap[dc].enc;
	encoff = dcmap[dc].offset;
	
	if ( dc==0 && !justinuse ) {
	  interp = interp_from_encoding(enc,ui_none);
	  mod = 0;
	  if ( dcmap[dc].platform==3 && (dcmap[dc].specific>=2 && dcmap[dc].specific<=6 ))
		mod = dcmap[dc].specific;
	  else if ( dcmap[dc].platform==1 && (dcmap[dc].specific==2 ||dcmap[dc].specific==1||dcmap[dc].specific==3||dcmap[dc].specific==25))
		mod = dcmap[dc].specific==1?2:dcmap[dc].specific==2?4:dcmap[dc].specific==3?5:3;		/* convert to ms specific */
	  info->map = map = EncMapNew(enc->char_cnt,info->glyph_cnt,enc);
	  info->uni_interp = interp;
	}
	
	fseek(ttf,info->encoding_start+encoff,SEEK_SET);
	format = getushort(ttf);
	if ( format!=12 && format!=10 && format!=8 ) {
	  len = getushort(ttf);
	  /* version/language = */ getushort(ttf);
	} else {
	  /* padding */ getushort(ttf);
	  len = getlong(ttf);
	  /* language = */ getlong(ttf);
	}
	if ( enc->is_unicodebmp && (format==8 || format==10 || format==12))
	  enc = FindOrMakeEncoding("UnicodeFull");
	
	if ( format==0 ) {
	  if ( !justinuse && map!=NULL && map->enccount<256 ) {
		map->map = grealloc(map->map,256*sizeof(int));
		memset(map->map,-1,(256-map->enccount)*sizeof(int));
		map->enccount = map->encmax = 256;
	  }
	  for ( i=0; i<len-6; ++i )
		table[i] = getc(ttf);
	  trans = enc->unicode;
	  if ( trans==NULL && dcmap[dc].platform==1 )
		trans = MacEncToUnicode(dcmap[dc].specific,dcmap[dc].lang-1);
	  for ( i=0; i<256 && i<len-6; ++i )
		if ( !justinuse ) {
		  if ( table[i]<info->glyph_cnt && info->chars[table[i]]!=NULL ) {
			if ( map!=NULL )
			  map->map[i] = table[i];
			if ( dounicode && trans!=NULL )
			  info->chars[table[i]]->unicodeenc = trans[i];
		  }
		} else if ( table[i]<info->glyph_cnt && info->chars[table[i]]!=NULL )
		  info->inuse[table[i]] = 1;
	} else if ( format==4 ) {
	  segCount = getushort(ttf)/2;
	  /* searchRange = */ getushort(ttf);
	  /* entrySelector = */ getushort(ttf);
	  /* rangeShift = */ getushort(ttf);
	  endchars = galloc(segCount*sizeof(uint16));
	  used = gcalloc(65536,sizeof(uint8));
	  for ( i=0; i<segCount; ++i )
		endchars[i] = getushort(ttf);
	  if ( getushort(ttf)!=0 )
		IError("Expected 0 in 'cmap' format 4 subtable");
	  startchars = galloc(segCount*sizeof(uint16));
	  for ( i=0; i<segCount; ++i )
		startchars[i] = getushort(ttf);
	  delta = galloc(segCount*sizeof(uint16));
	  for ( i=0; i<segCount; ++i )
		delta[i] = getushort(ttf);
	  rangeOffset = galloc(segCount*sizeof(uint16));
	  for ( i=0; i<segCount; ++i )
		rangeOffset[i] = getushort(ttf);
	  len -= 8*sizeof(uint16) +
		4*segCount*sizeof(uint16);
	  /* that's the amount of space left in the subtable and it must */
	  /*  be filled with glyphIDs */
	  if ( len<0 ) {
		IError("This font has an illegal format 4 subtable with too little space for all the segments.\nThis error is not recoverable.\nBye" );
		exit(1);
	  }
	  glyphs = galloc(len);
	  glyph_tot = len/2;
	  for ( i=0; i<glyph_tot; ++i )
		glyphs[i] = getushort(ttf);
	  for ( i=0; i<segCount; ++i ) {
		if ( rangeOffset[i]==0 && startchars[i]==0xffff )
		  /* Done */;
		else if ( rangeOffset[i]==0 ) {
		    for ( j=startchars[i]; j<=endchars[i]; ++j ) {
			if ( justinuse && (uint16) (j+delta[i])<info->glyph_cnt )
			    info->inuse[(uint16) (j+delta[i])] = true;
			else if ( (uint16) (j+delta[i])>=info->glyph_cnt || info->chars[(uint16) (j+delta[i])]==NULL )
			    LogError( _("Attempt to encode missing glyph %d to %d (0x%x)\n"),
				    (uint16) (j+delta[i]), modenc(j,mod), modenc(j,mod));
			else {
			    int uenc = umodenc(j,mod);
			    int lenc = modenc(j,mod);
			    if ( uenc!=-1 && used[uenc] ) {
				if ( !badencwarned ) {
				    LogError( _("Multiple glyphs map to the same unicode encoding U+%04X, only one will be used\n"), uenc );
			            badencwarned = true;
				}
			    } else {
				if ( uenc!=-1 && dounicode ) used[uenc] = true;
				if ( dounicode && info->chars[(uint16) (j+delta[i])]->unicodeenc==-1 )
				    info->chars[(uint16) (j+delta[i])]->unicodeenc = uenc;
			        if ( map!=NULL && lenc<map->enccount )
				    map->map[lenc] = (uint16) (j+delta[i]);
			    }
			}
		    }
		} else if ( rangeOffset[i]!=0xffff ) {
		    /* Apple says a rangeOffset of 0xffff means no glyph */
		    /*  OpenType doesn't mention this */
		    for ( j=startchars[i]; j<=endchars[i]; ++j ) {
			int temp = (i-segCount+rangeOffset[i]/2) + j-startchars[i];
			if ( temp<glyph_tot )
			    index = glyphs[ temp ];
			else {
			    /* This happened in mingliu.ttc(PMingLiU) */
			    if ( !justinuse )
				LogError( _("Glyph index out of bounds. Was %d, must be less than %d.\n In attempt to associate a glyph with encoding %x in segment %d\n with platform=%d, specific=%d (in 'cmap')\n"),
					temp, glyph_tot, j, i, dcmap[dc].platform, dcmap[dc].specific );
			    index = 0;
			}
			if ( index!=0 ) {
			    index = (unsigned short) (index+delta[i]);
			    if ( index>=info->glyph_cnt )
				/* This isn't mentioned either, but in some */
			        /*  MS Chinese fonts (kaiu.ttf) the index */
			        /*  goes out of bounds. and MS's ttf dump */
			        /*  program says it is treated as 0 */
				LogError( _("Attempt to encode missing glyph %d to %d (0x%x)\n"),
					index, modenc(j,mod), modenc(j,mod));
			    else if ( justinuse )
				info->inuse[index] = 1;
			    else if ( info->chars[index]==NULL )
				LogError( _("Attempt to encode missing glyph %d to %d (0x%x)\n"),
					index, modenc(j,mod), modenc(j,mod));
			    else {
				int uenc = umodenc(j,mod);
				int lenc = modenc(j,mod);
				if ( uenc!=-1 && used[uenc] ) {
				    if ( !badencwarned ) {
					LogError( _("Multiple glyphs map to the same unicode encoding U+%04X, only one will be used\n"), uenc );
					badencwarned = true;
				    }
				} else {
				    if ( uenc!=-1 && dounicode ) used[uenc] = true;
				    if ( dounicode && info->chars[index]->unicodeenc==-1 )
					info->chars[index]->unicodeenc = uenc;
				    if ( map!=NULL && lenc<map->enccount )
					map->map[lenc] = index;
				}
			    }
			}
		    }
		} else
		    LogError( _("Use of a range offset of 0xffff to mean a missing glyph in cmap table\n") );
	    }
	    free(glyphs);
	    free(rangeOffset);
	    free(delta);
	    free(startchars);
	    free(endchars);
	    free(used);
	} else if ( format==6 ) {
	    /* trimmed array format */
	    /* Well, the docs say it's for 2byte encodings, but Apple actually*/
	    /*  uses it for 1 byte encodings which don't fit into the require-*/
	    /*  ments for a format 0 sub-table. See Zapfino.dfont */
	    int first, count;
	    first = getushort(ttf);
	    count = getushort(ttf);
	    trans = enc->unicode;
	    if ( trans==NULL && dcmap[dc].platform==1 && first+count<=256 )
		trans = MacEncToUnicode(dcmap[dc].specific,dcmap[dc].lang-1);
	    if ( justinuse )
		for ( i=0; i<count; ++i )
		    info->inuse[getushort(ttf)]= 1;
	    else {
		for ( i=0; i<count; ++i ) {
		    int gid = getushort(ttf);
		    if ( dounicode )
			info->chars[gid]->unicodeenc = trans==NULL ? trans[first+1] : first+i;
		    if ( map!=NULL && first+i < map->enccount )
			map->map[first+i] = gid;
		}
	    }
	} else if ( format==2 ) {
	    int max_sub_head_key = 0, cnt, max_pos= -1;
	    struct subhead *subheads;
	    
	    for ( i=0; i<256; ++i ) {
		table[i] = getushort(ttf)/8;	/* Sub-header keys */
		if ( table[i]>max_sub_head_key ) {
		    max_sub_head_key = table[i];	/* The entry is a byte pointer, I want a pointer in units of struct subheader */
		    max_pos = i;
		}
	    }
	    subheads = galloc((max_sub_head_key+1)*sizeof(struct subhead));
	    for ( i=0; i<=max_sub_head_key; ++i ) {
		subheads[i].first = getushort(ttf);
		subheads[i].cnt = getushort(ttf);
		subheads[i].delta = getushort(ttf);
		subheads[i].rangeoff = (getushort(ttf)-
				(max_sub_head_key-i)*sizeof(struct subhead)-
				sizeof(short))/sizeof(short);
	    }
	    cnt = (len-(ftell(ttf)-(info->encoding_start+encoff)))/sizeof(short);
	    /* The count is the number of glyph indexes to read. it is the */
	    /*  length of the entire subtable minus that bit we've read so far */
	    glyphs = galloc(cnt*sizeof(short));
	    for ( i=0; i<cnt; ++i )
		glyphs[i] = getushort(ttf);
	    last = -1;
	    for ( i=0; i<256; ++i ) {
		if ( table[i]==0 ) {
		    /* Special case, single byte encoding entry, look it up in */
		    /*  subhead */
		    /* In the one example I've got of this encoding (wcl-02.ttf) the chars */
		    /* 0xfd, 0xfe, 0xff are said to exist but there is no mapping */
		    /* for them. */
		    if ( i>=max_pos )
			index = 0;	/* the subhead says there are 256 entries, but in fact there are only 193, so attempting to find these guys should give an error */
		    else if ( i<subheads[0].first || i>=subheads[0].first+subheads[0].cnt ||
			    subheads[0].rangeoff+(i-subheads[0].first)>=cnt )
			index = 0;
		    else if ( (index = glyphs[subheads[0].rangeoff+(i-subheads[0].first)])!= 0 )
			index = (uint32) (index+subheads[0].delta);
		    /* I assume the single byte codes are just ascii or latin1*/
		    if ( index!=0 && index<info->glyph_cnt ) {
			if ( justinuse )
			    info->inuse[index] = 1;
			else if ( info->chars[index]==NULL )
			    /* Do Nothing */;
			else {
			    int lenc = modenc(i,mod);
			    if ( dounicode && info->chars[index]->unicodeenc==-1 )
				info->chars[index]->unicodeenc = i;
			    if ( map!=NULL && lenc<map->enccount )
				map->map[lenc] = index;
			}
		    }
		} else {
		    int k = table[i];
		    for ( j=0; j<subheads[k].cnt; ++j ) {
			int enc, lenc;
			if ( subheads[k].rangeoff+j>=cnt )
			    index = 0;
			else if ( (index = glyphs[subheads[k].rangeoff+j])!= 0 )
			    index = (uint16) (index+subheads[k].delta);
			if ( index!=0 && index<info->glyph_cnt ) {
			    enc = (i<<8)|(j+subheads[k].first);
			    lenc = modenc(enc,mod);
			    if ( justinuse )
				info->inuse[index] = 1;
			    else if ( info->chars[index]==NULL )
				/* Do Nothing */;
			    else {
				if ( dounicode && info->chars[index]->unicodeenc==-1 )
				    info->chars[index]->unicodeenc = umodenc(enc,mod);
				if ( map!=NULL && lenc<map->enccount )
				    map->map[lenc] = index;
			    }
			}
		    }
		    /*if ( last==-1 ) last = i;*/
		}
	    }
	    free(subheads);
	    free(glyphs);
	} else if ( format==8 ) {
	    uint32 ngroups, start, end, startglyph;
	    if ( !enc->is_unicodefull ) {
		IError("I don't support 32 bit characters except for the UCS-4 (MS platform, specific=10)" );
		enc = FindOrMakeEncoding("UnicodeFull");
	    }
	    /* I'm now assuming unicode surrogate encoding, so I just ignore */
	    /*  the is32 table (it will be set for the surrogates and not for */
	    /*  anything else */
	    fseek(ttf,8192,SEEK_CUR);
	    ngroups = getlong(ttf);
	    for ( j=0; j<ngroups; ++j ) {
		start = getlong(ttf);
		end = getlong(ttf);
		startglyph = getlong(ttf);
		if ( justinuse )
		    for ( i=start; i<=end; ++i )
			info->inuse[startglyph+i-start]= 1;
		else
		    for ( i=start; i<=end; ++i ) {
			int uenc = ((i>>16)-0xd800)*0x400 + (i&0xffff)-0xdc00 + 0x10000;
			sc = info->chars[startglyph+i-start];
			if ( dounicode && sc->unicodeenc==-1 )
			    sc->unicodeenc = uenc;
			if ( map!=NULL && sc->unicodeenc < map->enccount )
			    map->map[uenc] = startglyph+i-start;
		    }
	    }
	} else if ( format==10 ) {
	    /* same as format 6, except for 4byte chars */
	    int first, count;
	    if ( !enc->is_unicodefull ) {
		IError("I don't support 32 bit characters except for the UCS-4 (MS platform, specific=10)" );
		enc = FindOrMakeEncoding("UnicodeFull");
	    }
	    first = getlong(ttf);
	    count = getlong(ttf);
	    if ( justinuse )
		for ( i=0; i<count; ++i )
		    info->inuse[getushort(ttf)]= 1;
	    else
		for ( i=0; i<count; ++i ) {
		    int gid = getushort(ttf);
		    if ( dounicode )
			info->chars[gid]->unicodeenc = first+i;
		    if ( map!=NULL && first+i < map->enccount )
			map->map[first+i] = gid;
		}
	} else if ( format==12 ) {
	    uint32 ngroups, start, end, startglyph;
	    if ( !enc->is_unicodefull ) {
		IError("I don't support 32 bit characters except for the UCS-4 (MS platform, specific=10)" );
		enc = FindOrMakeEncoding("UnicodeFull");
	    }
	    ngroups = getlong(ttf);
	    for ( j=0; j<ngroups; ++j ) {
		start = getlong(ttf);
		end = getlong(ttf);
		startglyph = getlong(ttf);
		if ( justinuse ) {
		    for ( i=start; i<=end; ++i )
			if ( startglyph+i-start < info->glyph_cnt )
			    info->inuse[startglyph+i-start]= 1;
			else
		    break;
		} else
		    for ( i=start; i<=end; ++i ) {
			if ( startglyph+i-start >= info->glyph_cnt ||
				info->chars[startglyph+i-start]==NULL ) {
			    LogError( _("Bad font: Encoding data out of range.\n") );
		    break;
			} else {
			    if ( dounicode )
				info->chars[startglyph+i-start]->unicodeenc = i;
			    if ( map!=NULL && i < map->enccount )
				map->map[i] = startglyph+i-start;
			}
		    }
	    }
	}
    }
    free(cmap_encs);
    if ( info->chars!=NULL )
	for ( i=0; i<info->glyph_cnt; ++i )
	    if ( info->chars[i]!=NULL && info->chars[i]->unicodeenc==0xffff )
		info->chars[i]->unicodeenc = -1;
    if ( !justinuse ) {
	  if ( interp==ui_none )
	    info->uni_interp = amscheck(info,map);
	  map->enc = enc;		/* This can be changed from the initial value */
    }
    info->map = map;
}

static void readttfos2metrics(FILE *ttf,struct ttfinfo *info) {
    int i, sel;

    fseek(ttf,info->os2_start,SEEK_SET);
    info->os2_version = getushort(ttf);
    info->pfminfo.avgwidth = getushort(ttf);
    info->pfminfo.weight = getushort(ttf);
    info->pfminfo.width = getushort(ttf);
    info->pfminfo.fstype = getushort(ttf);
    info->pfminfo.os2_subxsize = getushort(ttf);
    info->pfminfo.os2_subysize = getushort(ttf);
    info->pfminfo.os2_subxoff = getushort(ttf);
    info->pfminfo.os2_subyoff = getushort(ttf);
    info->pfminfo.os2_supxsize = getushort(ttf);
    info->pfminfo.os2_supysize = getushort(ttf);
    info->pfminfo.os2_supxoff = getushort(ttf);
    info->pfminfo.os2_supyoff = getushort(ttf);
    info->pfminfo.os2_strikeysize = getushort(ttf);
    info->pfminfo.os2_strikeypos = getushort(ttf);
    info->pfminfo.os2_family_class = getushort(ttf);
    for ( i=0; i<10; ++i )
	info->pfminfo.panose[i] = getc(ttf);
    info->pfminfo.pfmfamily = info->pfminfo.panose[0]==2 ? 0x11 :	/* might be 0x21 */ /* Text & Display maps to either serif 0x11 or sans 0x21 or monospace 0x31 */
		      info->pfminfo.panose[0]==3 ? 0x41 :	/* Script */
		      info->pfminfo.panose[0]==4 ? 0x51 :	/* Decorative */
		      0x51;					/* And pictorial doesn't fit into pfm */
    /* unicoderange[] */ getlong(ttf);
    /* unicoderange[] */ getlong(ttf);
    /* unicoderange[] */ getlong(ttf);
    /* unicoderange[] */ getlong(ttf);
    info->pfminfo.os2_vendor[0] = getc(ttf);
    info->pfminfo.os2_vendor[1] = getc(ttf);
    info->pfminfo.os2_vendor[2] = getc(ttf);
    info->pfminfo.os2_vendor[3] = getc(ttf);
    sel = getushort(ttf);
    if ( info->os2_version>=4 ) {
	info->use_typo_metrics = (sel&128)?1:0;
	info->weight_width_slope_only = (sel&256)?1:0;
    }
    info->pfminfo.firstchar = getushort(ttf);
    info->pfminfo.lastchar = getushort(ttf);
    info->pfminfo.os2_typoascent = getushort(ttf);
    info->pfminfo.os2_typodescent = (short) getushort(ttf);
    if ( info->pfminfo.os2_typoascent-info->pfminfo.os2_typodescent == info->emsize ) {
	info->ascent = info->pfminfo.os2_typoascent;
	info->descent = -info->pfminfo.os2_typodescent;
    }
    info->pfminfo.os2_typolinegap = getushort(ttf);
    info->pfminfo.os2_winascent = getushort(ttf);
    info->pfminfo.os2_windescent = getushort(ttf);
    if ( info->os2_version>=3 ) { /* TH just in case */
      /* unicoderange[] */ getlong(ttf);
      /* unicoderange[] */ getlong(ttf);
      info->pfminfo.os2_xheight     = getushort(ttf); /* four new fields */
      info->pfminfo.os2_capheight   = getushort(ttf);
      info->pfminfo.os2_defaultchar = getushort(ttf);
      info->pfminfo.os2_breakchar   = getushort(ttf);
    }
    info->pfminfo.winascent_add = info->pfminfo.windescent_add = false;
    info->pfminfo.typoascent_add = info->pfminfo.typodescent_add = false;
    info->pfminfo.pfmset = true;
    info->pfminfo.panose_set = true;
    info->pfminfo.subsuper_set = true;
}

static int cmapEncFromName(struct ttfinfo *info,const char *nm, int glyphid) {
    int uni;
    int i;

    if ( info->map!=NULL )
	uni = EncFromName(nm,info->uni_interp,info->map->enc);
    else
	uni = EncFromName(nm,ui_none,&custom);
    if ( uni==-1 )
return( -1 );

    for ( i=0; i<info->glyph_cnt; ++i ) if ( info->chars[i]!=NULL ) {
	if ( info->chars[i]->unicodeenc==uni ) {
	    if ( info->complainedmultname )
		/* Don't do it again */;
	    else if ( info->chars[i]->name!=NULL && strcmp(info->chars[i]->name,nm)==0 )
		LogError( _("Warning: Glyph %d has the same name (%s) as Glyph %d\n"),
			i, nm, glyphid );
	    else
		LogError( _("Warning: Glyph %d is named %s which should mean it is mapped to\n Unicode U+%04X, but Glyph %d already has that encoding.\n"),
			glyphid, nm, uni, i);
	    info->complainedmultname = true;
return( -1 );
	}
    }
return( uni );
}

static void readttfpostnames(FILE *ttf,struct ttfinfo *info) {
    int i,j;
    int format, len, gc, gcbig, val;
    const char *name;
    char buffer[30];
    uint16 *indexes;
#if 0
    extern const char *ttfstandardnames[];
#endif
    int notdefwarned = false;


    /* Give ourselves an xuid, just in case they want to convert to PostScript*/
    /*  (even type42)							      */
    if ( xuid!=NULL && info->fd==NULL && info->xuid==NULL ) {
	info->xuid = galloc(strlen(xuid)+20);
	sprintf(info->xuid,"[%s %d]", xuid, (rand()&0xffffff));
    }

    if ( info->postscript_start!=0 ) {
	fseek(ttf,info->postscript_start,SEEK_SET);
	format = getlong(ttf);
	info->italicAngle = getfixed(ttf);
	info->upos = (short) getushort(ttf);
	info->uwidth = (short) getushort(ttf);
	/* fixedpitch = */ getlong(ttf);
	/* mem1 = */ getlong(ttf);
	/* mem2 = */ getlong(ttf);
	/* mem3 = */ getlong(ttf);
	/* mem4 = */ getlong(ttf);
	if ( format==0x00020000 ) {
	    gc = getushort(ttf);
	    indexes = gcalloc(65536,sizeof(uint16));
	    /* the index table is backwards from the way I want to use it */
	    gcbig = 0;
	    for ( i=0; i<gc; ++i ) {
		indexes[val = getushort(ttf)] = i;
		if ( val>=258 ) ++gcbig;
	    }

	    /* if we are only loading bitmaps, we can get holes in our data */
	    for ( i=0; i<258; ++i ) if ( indexes[i]!=0 || i==0 ) if ( indexes[i]<info->glyph_cnt && info->chars[indexes[i]]!=NULL ) {
		info->chars[indexes[i]]->name = copy(ttfstandardnames[i]);
		if ( info->chars[indexes[i]]->unicodeenc==-1 )
		    info->chars[indexes[i]]->unicodeenc = cmapEncFromName(info,ttfstandardnames[i],indexes[i]);
	    }
	    gcbig += 258;
	    for ( i=258; i<gcbig; ++i ) {
		char *nm;
		len = getc(ttf);
		if ( len<0 )		/* Don't crash on EOF */
	    break;
		nm = galloc(len+1);
		for ( j=0; j<len; ++j )
		    nm[j] = getc(ttf);
		nm[j] = '\0';
		if ( indexes[i]<info->glyph_cnt && info->chars[indexes[i]]!=NULL ) {
		    info->chars[indexes[i]]->name = nm;
		    if ( info->chars[indexes[i]]->unicodeenc==-1 )
			info->chars[indexes[i]]->unicodeenc = cmapEncFromName(info,nm,indexes[i]);
		}
	    }
	    free(indexes);
	}
    }

    if ( info->fd!=NULL && info->fd->chars!=NULL) {
	EncMap *map = NULL;
	struct pschars *chars = info->fd->chars;
	if ( info->map==NULL )
	    info->map = map = EncMapNew(65536,65536,FindOrMakeEncoding("UnicodeBmp"));
	/* In type42 fonts the names are stored in a postscript /CharStrings dictionary */
	for ( i=0; i<chars->next; ++i ) {
	    int gid = (intpt) (chars->values[i]);
	    if ( gid>=0 && gid<info->glyph_cnt && chars->keys[i]!=NULL ) {
		free(info->chars[gid]->name);
		info->chars[gid]->name = chars->keys[i];
		info->chars[gid]->unicodeenc = UniFromName(chars->keys[i],info->uni_interp,info->map->enc);
		if ( map!=NULL && info->chars[gid]->unicodeenc!=-1 &&
			info->chars[gid]->unicodeenc<map->enccount)
		    map->map[ info->chars[gid]->unicodeenc ] = gid;
		chars->keys[i] = NULL;
		chars->values[i] = NULL;
	    } else
		chars->values[i] = NULL;
	}
    }

    /* where no names are given, but we've got a unicode encoding use */
    /*  that to guess at them */
    for ( i=0; i<info->glyph_cnt; ++i ) if ( info->chars[i]!=NULL ) {
	/* info->chars[i] can be null in some TTC files */
	if ( i!=0 && info->chars[i]->name!=NULL &&
		strcmp(info->chars[i]->name,".notdef")==0 ) {
	    /* for some reason MS puts out fonts where several characters */
	    /* are called .notdef (and only one is a real notdef). So if we */
	    /* find a glyph other than 0 called ".notdef" then pretend it had */
	    /* no name */
	    if ( !notdefwarned ) {
		notdefwarned = true;
		LogError( _("Glyph %d is called \".notdef\", a singularly inept choice of name (only glyph 0\n may be called .notdef)\nFontForge will rename it.\n"), i );
	    }
	    free(info->chars[i]->name);
	    info->chars[i]->name = NULL;
	}
	/* And some volt files actually assign nul strings to the name */
	if ( (info->chars[i]->name!=NULL && *info->chars[i]->name!='\0' ))
	  continue;
	free(info->chars[i]->name);	/* If it's a null string get rid of it */
	if ( i==0 )
	  name = ".notdef";
	else if ( info->chars[i]->unicodeenc==-1 ) {
	  /* Do this later */;
	  name = NULL;
	} else {
	  name = StdGlyphName(buffer,info->chars[i]->unicodeenc,info->uni_interp,NULL);
	}
	info->chars[i]->name = copy(name);
    }

    /* If we have a GSUB table we can give some unencoded glyphs names */
    /*  for example if we have a vrt2 substitution of A to <unencoded> */
    /*  we could name the unencoded "A.vrt2" (though in this case we might */
    /*  try A.vert instead */ /* Werner suggested this */
    /* We could try this from morx too, except that apple features don't */
    /*  use meaningful ids. That is A.15,3 isn't very readable */
    for ( i=info->glyph_cnt-1; i>=0 ; --i )
	if ( info->chars[i]!=NULL && info->chars[i]->name==NULL )
    break;
    if ( i>=0 && info->gsub_start!=0 )
	GuessNamesFromGSUB(ttf,info);

    for ( i=0; i<info->glyph_cnt; ++i ) {
	/* info->chars[i] can be null in some TTC files */
	if ( info->chars[i]==NULL )
    continue;
	if ( info->chars[i]->name!=NULL )
    continue;
	if ( info->ordering!=NULL )
	    sprintf(buffer, "%.20s-%d", info->ordering, i );
	else if ( info->map!=NULL && info->map->map[i]!=-1 )
	    sprintf(buffer, "nounicode-%d-%d-%x", info->platform, info->specific,
		    (int) info->map->map[i] );
	else
	    sprintf( buffer, "glyph%d", i );
	info->chars[i]->name = copy(buffer);
    }
}

static void readttfgasp(FILE *ttf,struct ttfinfo *info) {
    int i, cnt;

    if ( info->gasp_start==0 )
return;

    fseek(ttf,info->gasp_start,SEEK_SET);
    info->gasp_version = getushort(ttf);
    if ( info->gasp_version!=0 && info->gasp_version!=1 )
return;			/* We only support 'gasp' versions 0&1 (no other versions currently) */
    info->gasp_cnt = cnt = getushort(ttf);
    if ( cnt==0 )
return;
    info->gasp = galloc(cnt*sizeof(struct gasp));
    for ( i=0; i<cnt; ++i ) {
	info->gasp[i].ppem = getushort(ttf);
	info->gasp[i].flags = getushort(ttf);
    }
}

static void UnfigureControls(Spline *spline,BasePoint *pos) {
    pos->x = rint( (spline->splines[0].c+2*spline->splines[0].d)/2 );
    pos->y = rint( (spline->splines[1].c+2*spline->splines[1].d)/2 );
}

int ttfFindPointInSC(SplineChar *sc,int pnum,BasePoint *pos,
	RefChar *bound) {
    SplineSet *ss;
    SplinePoint *sp;
    int last=0, ret;
    RefChar *refs;

    for ( ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
	for ( sp=ss->first; ; ) {
	    if ( sp->ttfindex==pnum ) {
		*pos = sp->me;
return(-1);
	    } else if ( sp->nextcpindex==pnum ) {
		if ( sp->next!=NULL && sp->next->order2 )
		    *pos = sp->nextcp;
		else {
		    /* fix this up to be 2 degree bezier control point */
		    UnfigureControls(sp->next,pos);
		}
return( -1 );
	    }
	    if ( !sp->nonextcp && last<sp->nextcpindex )
		last = sp->nextcpindex;
	    else if ( sp->ttfindex!=0xffff )
		last = sp->ttfindex;
	    if ( sp->next==NULL )
	break;
	    sp = sp->next->to;
	    if ( sp==ss->first )
	break;
	}
    }
    for ( refs=sc->layers[ly_fore].refs; refs!=NULL; refs=refs->next ) {
	if ( refs==bound ) {
	    LogError( _("Invalid point match. Point would be after this reference.\n") );
return( 0x800000 );
	}
	ret = ttfFindPointInSC(refs->sc,pnum-last,pos,NULL);
	if ( ret==-1 ) {
	    BasePoint p;
	    p.x = refs->transform[0]*pos->x + refs->transform[2]*pos->y + refs->transform[4];
	    p.y = refs->transform[1]*pos->x + refs->transform[3]*pos->y + refs->transform[5];
	    *pos = p;
return( -1 );
	}
	last += ret;
    }
return( last );		/* Count of number of points in the character */
}

static void ttfPointMatch(SplineChar *sc,RefChar *rf) {
    BasePoint sofar, inref;

    if ( ttfFindPointInSC(sc,rf->match_pt_base,&sofar,rf)!=-1 ||
	    ttfFindPointInSC(rf->sc,rf->match_pt_ref,&inref,NULL)!=-1 ) {
	LogError( _("Could not match points in composite glyph (%d to %d) when adding %s to %s\n"),
		rf->match_pt_base, rf->match_pt_ref, rf->sc->name, sc->name);
return;
    }
    rf->transform[4] = sofar.x-inref.x;
    rf->transform[5] = sofar.y-inref.y;
}

int ttfFixupRef(SplineChar **chars,int i) {
    RefChar *ref, *prev, *next;

    if ( chars[i]==NULL )		/* Can happen in ttc files */
return( false );
    if ( chars[i]->ticked )
return( false );
    chars[i]->ticked = true;
    prev = NULL;
    for ( ref=chars[i]->layers[ly_fore].refs; ref!=NULL; ref=next ) {
	if ( ref->sc!=NULL )
    break;				/* Already done */
	next = ref->next;
	if ( !ttfFixupRef(chars,ref->orig_pos)) {
	    if ( prev==NULL )
		chars[i]->layers[ly_fore].refs = next;
	    else
		prev->next = next;
	    chunkfree(ref,sizeof(RefChar));
	} else {
	    ref->sc = chars[ref->orig_pos];
	    ref->adobe_enc = getAdobeEnc(ref->sc->name);
	    if ( ref->point_match )
		ttfPointMatch(chars[i],ref);
	    SCReinstanciateRefChar(chars[i],ref);
	    SCMakeDependent(chars[i],ref->sc);
	    prev = ref;
	}
    }
    chars[i]->ticked = false;
return( true );
}

static void ttfFixupReferences(struct ttfinfo *info) {
    int i;

    for ( i=0; i<info->glyph_cnt; ++i ) {
	ttfFixupRef(info->chars,i);
    }
}

static void TtfCopyTableBlindly(struct ttfinfo *info,FILE *ttf,
	uint32 start,uint32 len,uint32 tag) {
    struct ttf_table *tab;

    if ( start==0 || len==0 )
return;
    if ( len>0x1000000 ) {
	LogError( _("Unlikely length for table, so I'm ignoring it. %u\n"), len );
return;
    }

    tab = chunkalloc(sizeof(struct ttf_table));
    tab->tag = tag;
    tab->len = len;
    tab->data = galloc(len);
    fseek(ttf,start,SEEK_SET);
    fread(tab->data,1,len,ttf);
    tab->next = info->tabs;
    info->tabs = tab;
}

 static int readttf(FILE *ttf, struct ttfinfo *info, char *filename, char *choice) {
    int i;

    if ( !readttfheader(ttf,info,filename,&info->chosenname)) {
      return( 0 );
    }
    readttfpreglyph(ttf,info);

    /* never read bitmap fonts */
    if (strcmp(choice,"ttf") == 0) {
      if (info->glyphlocations_start==0 || info->glyph_length==0)
	return (0);
      info->cff_start=0;      
    } else if (strcmp(choice,"otf") == 0)  {
      if (info->cff_start == 0)
	return (0);
      info->glyph_start = info->glyphlocations_start = 0;
    } else {
      return(0);
    }
    
    if ( info->glyphlocations_start!=0 && info->glyph_start!=0 ) {
      info->to_order2 = (!loaded_fonts_same_as_new ||
			 (loaded_fonts_same_as_new && new_fonts_are_order2));
      /* If it's an apple mm font, then we don't want to change the order */
      /*  This messes up the point count */
      if ( info->gvar_start!=0 && info->fvar_start!=0 )
	info->to_order2 = true;
      readttfglyphs(ttf,info);
    } else if ( info->cff_start!=0 ) {
      info->to_order2 = (loaded_fonts_same_as_new && new_fonts_are_order2);
      if ( !readcffglyphs(ttf,info) ) {
	return( 0 );
      }
    } else {
      return( 0 );
    }

    if ( info->hmetrics_start!=0 )
	readttfwidths(ttf,info);

    if ( info->vmetrics_start!=0 && info->vhea_start!=0 )
	readttfvwidths(ttf,info);
    /* 'cmap' is not meaningful for cid keyed fonts, and not supplied for */
    /*  type42 fonts */
    if ( info->cidregistry==NULL && info->encoding_start!=0 )
	readttfencodings(ttf,info,false);
    if ( info->os2_start!=0 )
	readttfos2metrics(ttf,info);
    readttfpostnames(ttf,info);		/* If no postscript table we'll guess at names */
    if ( info->gdef_start!=0 )		/* ligature caret positioning info */
	readttfgdef(ttf,info);
    else {
      if ( info->prop_start!=0 )
	readttfprop(ttf,info);
      if ( info->lcar_start!=0 )
	readttflcar(ttf,info);
    }
    if ( info->gasp_start!=0 )
	readttfgasp(ttf,info);
    /* read the cvt table before reading variation data */
    if ( info->to_order2 ) {
      /* Yes, even though we've looked at maxp already, let's make a blind */
      /*  copy too for those fields we can't compute on our own */
      /* Like size of twilight zone, etc. */
	TtfCopyTableBlindly(info,ttf,info->maxp_start,info->maxp_len,CHR('m','a','x','p'));
	
	/* TH: we do not need the actual instructions copied in luatex */
	/* TtfCopyTableBlindly(info,ttf,info->cvt_start,info->cvt_len,CHR('c','v','t',' ')); */
	/* TtfCopyTableBlindly(info,ttf,info->fpgm_start,info->fpgm_len,CHR('f','p','g','m'));*/
	/* TtfCopyTableBlindly(info,ttf,info->prep_start,info->prep_len,CHR('p','r','e','p'));*/
    }
    for ( i=0; i<info->savecnt; ++i ) if ( info->savetab[i].offset!=0 )
	TtfCopyTableBlindly(info,ttf,info->savetab[i].offset,info->savetab[i].len,info->savetab[i].tag);
    /* Do this before reading kerning info */
    if ( info->to_order2 && info->gvar_start!=0 && info->fvar_start!=0 )
	readttfvariations(info,ttf);
    if ( info->gpos_start!=0 )		/* kerning info may live in the gpos table too */
	readttfgpossub(ttf,info,true);
    else {
      if ( info->kern_start!=0 )
	readttfkerns(ttf,info);
      if ( info->opbd_start!=0 )
	readttfopbd(ttf,info);
    }
    if ( info->gsub_start!=0 )
      readttfgpossub(ttf,info,false);
    else {
      /* We will default the gsub table later... */;
      if ( info->morx_start!=0 || info->mort_start!=0 )
	readttfmort(ttf,info);
    }
    if ( info->pfed_start!=0 )
      pfed_read(ttf,info);
    if ( info->tex_start!=0 )
      tex_read(ttf,info);
    //    setlocale(LC_NUMERIC,oldloc);
    if ( !info->onlystrikes && info->glyphlocations_start!=0 && info->glyph_start!=0 )
      ttfFixupReferences(info);
    /* Can't fix up any postscript references until we create a SplineFont */
    /*  so the check for cff is delayed. Generally there aren't any cff refs */
    /*  anyway */
    return( true );
}

static void SymbolFixup(struct ttfinfo *info) {
    /* convert a two-byte symbol encoding (one using PUA) into expected */
    /*  one-byte encoding. */
    int i, max;
    EncMap *map = info->map;

    max = -1;
    for ( i=map->enccount-1; i>=0; --i ) {
	if ( map->map[i]==-1 )
    continue;
	if ( i>=0xf000 && i<=0xf0ff ) {
	    map->map[i-0xf000] = map->map[i];
	    map->map[i] = -1;
    continue;
	}
	if ( i>max ) max = i;
    }
    map->enccount = max;
}

void AltUniFigure(SplineFont *sf,EncMap *map) {
    int i,gid;

    if ( map->enc!=&custom ) {
	for ( i=0; i<map->enccount; ++i ) if ( (gid = map->map[i])!=-1 ) {
	    int uni = UniFromEnc(i,map->enc);
	    AltUniAdd(sf->glyphs[gid],uni);
	}
    }
}

static void UseGivenEncoding(SplineFont *sf,struct ttfinfo *info) {
    int i;
    RefChar *rf, *prev, *next;

    sf->glyphs = info->chars;
    sf->glyphcnt = sf->glyphmax = info->glyph_cnt;
    for ( i=0; i<sf->glyphcnt; ++i )
	if ( sf->glyphs[i]!=NULL )
	    sf->glyphs[i]->parent = sf;

    /* A CFF font could contain type1 charstrings, or a type2 font could use */
    /*  the depreciated convention that endchar =~ seac */
    if ( info->cff_length!=0 )
	SFInstanciateRefs(sf);

    for ( i=0; i<sf->glyphcnt; ++i ) if ( sf->glyphs[i]!=NULL ) {
	for ( rf = sf->glyphs[i]->layers[ly_fore].refs, prev=NULL; rf!=NULL; rf = next ) {
	    next = rf->next;
	    if ( rf->sc==NULL ) {
		if ( prev==NULL ) sf->glyphs[i]->layers[ly_fore].refs = next;
		else prev->next = next;
		RefCharFree(rf);
	    } else {
		rf->orig_pos = rf->sc->orig_pos;
		rf->unicode_enc = rf->sc->unicodeenc;
		prev = rf;
	    }
	}
    }
    sf->map = info->map;
    sf->uni_interp = info->uni_interp;
    AltUniFigure(sf,sf->map);
}

static char *AxisNameConvert(uint32 tag) {
    char buffer[8];

    if ( tag==CHR('w','g','h','t'))
return( copy("Weight"));
    if ( tag==CHR('w','d','t','h'))
return( copy("Width"));
    if ( tag==CHR('o','p','s','z'))
return( copy("OpticalSize"));
    if ( tag==CHR('s','l','n','t'))
return( copy("Slant"));

    buffer[0] = tag>>24;
    buffer[1] = tag>>16;
    buffer[2] = tag>>8;
    buffer[3] = tag&0xff;
    buffer[4] = 0;
return( copy(buffer ));
}

static struct macname *FindMacName(struct ttfinfo *info, int strid) {
    struct macidname *sid;

    for ( sid=info->macstrids; sid!=NULL; sid=sid->next ) {
	if ( sid->id == strid )
return( sid->head );
    }
return( NULL );
}

static SplineFont *SFFromTuple(SplineFont *basesf,struct variations *v,int tuple,
	MMSet *mm, struct ttfinfo *info) {
    SplineFont *sf;
    int i;
    RefChar *r;

    sf = SplineFontEmpty();
    sf->display_size = basesf->display_size;
    sf->display_antialias = basesf->display_antialias;

    sf->fontname = MMMakeMasterFontname(mm,tuple,&sf->fullname);
    sf->familyname = copy(basesf->familyname);
    sf->weight = copy("All");
    sf->italicangle = basesf->italicangle;
    sf->strokewidth = basesf->strokewidth;
    sf->strokedfont = basesf->strokedfont;
    sf->upos = basesf->upos;
    sf->uwidth = basesf->uwidth;
    sf->ascent = basesf->ascent;
    sf->vertical_origin = basesf->vertical_origin;
    sf->hasvmetrics = basesf->hasvmetrics;
    sf->descent = basesf->descent;
    sf->kerns = v->tuples[tuple].khead;
    sf->vkerns = v->tuples[tuple].vkhead;
    sf->map = basesf->map;
    sf->mm = mm;
    sf->glyphmax = sf->glyphcnt = basesf->glyphcnt;
    sf->glyphs = v->tuples[tuple].chars;
    sf->order2 = true;
    for ( i=0; i<sf->glyphcnt; ++i ) if ( basesf->glyphs[i]!=NULL ) {
	sf->glyphs[i]->orig_pos = i;
	sf->glyphs[i]->parent = sf;
    }
    for ( i=0; i<sf->glyphcnt; ++i ) if ( sf->glyphs[i]!=NULL ) {
	for ( r=sf->glyphs[i]->layers[ly_fore].refs; r!=NULL; r=r->next )
	    SCReinstanciateRefChar(sf->glyphs[i],r);
    }

    sf->ttf_tables = v->tuples[tuple].cvt;

    v->tuples[tuple].chars = NULL;
    v->tuples[tuple].khead = NULL;
    v->tuples[tuple].vkhead = NULL;
    v->tuples[tuple].cvt = NULL;
return( sf );
}

static void MMFillFromVAR(SplineFont *sf, struct ttfinfo *info) {
    MMSet *mm = chunkalloc(sizeof(MMSet));
    struct variations *v = info->variations;
    int i,j;

    sf->mm = mm;
    mm->normal = sf;
    mm->apple = true;
    mm->axis_count = v->axis_count;
    mm->instance_count = v->tuple_count;
    mm->instances = galloc(v->tuple_count*sizeof(SplineFont *));
    mm->positions = galloc(v->tuple_count*v->axis_count*sizeof(real));
    for ( i=0; i<v->tuple_count; ++i ) for ( j=0; j<v->axis_count; ++j )
	mm->positions[i*v->axis_count+j] = v->tuples[i].coords[j];
    mm->defweights = gcalloc(v->tuple_count,sizeof(real));	/* Doesn't apply */
    mm->axismaps = gcalloc(v->axis_count,sizeof(struct axismap));
    for ( i=0; i<v->axis_count; ++i ) {
	mm->axes[i] = AxisNameConvert(v->axes[i].tag);
	mm->axismaps[i].min = v->axes[i].min;
	mm->axismaps[i].def = v->axes[i].def;
	mm->axismaps[i].max = v->axes[i].max;
	if ( v->axes[i].paircount==0 ) {
	    mm->axismaps[i].points = 3;
	    mm->axismaps[i].blends = galloc(3*sizeof(real));
	    mm->axismaps[i].designs = galloc(3*sizeof(real));
	    mm->axismaps[i].blends[0] = -1; mm->axismaps[i].designs[0] = mm->axismaps[i].min;
	    mm->axismaps[i].blends[1] =  0; mm->axismaps[i].designs[1] = mm->axismaps[i].def;
	    mm->axismaps[i].blends[2] =  1; mm->axismaps[i].designs[2] = mm->axismaps[i].max;
	} else {
	    mm->axismaps[i].points = v->axes[i].paircount;
	    mm->axismaps[i].blends = galloc(v->axes[i].paircount*sizeof(real));
	    mm->axismaps[i].designs = galloc(v->axes[i].paircount*sizeof(real));
	    for ( j=0; j<v->axes[i].paircount; ++j ) {
		if ( v->axes[i].mapfrom[j]<=0 ) {
		    mm->axismaps[i].designs[j] = mm->axismaps[i].def +
			    v->axes[i].mapfrom[j]*(mm->axismaps[i].def-mm->axismaps[i].min);
		} else {
		    mm->axismaps[i].designs[j] = mm->axismaps[i].def +
			    v->axes[i].mapfrom[j]*(mm->axismaps[i].max-mm->axismaps[i].def);
		}
		mm->axismaps[i].blends[j] = v->axes[i].mapto[j];
	    }
	}
	mm->axismaps[i].axisnames = MacNameCopy(FindMacName(info, v->axes[i].nameid));
    }
    mm->named_instance_count = v->instance_count;
    mm->named_instances = galloc(v->instance_count*sizeof(struct named_instance));
    for ( i=0; i<v->instance_count; ++i ) {
	mm->named_instances[i].coords = v->instances[i].coords;
	v->instances[i].coords = NULL;
	mm->named_instances[i].names = MacNameCopy(FindMacName(info, v->instances[i].nameid));
    }
    for ( i=0; i<mm->instance_count; ++i )
	mm->instances[i] = SFFromTuple(sf,v,i,mm,info);
    VariationFree(info);
}

static void SFRelativeWinAsDs(SplineFont *sf) {
    if ( !sf->pfminfo.winascent_add || !sf->pfminfo.windescent_add ||
	    !sf->pfminfo.hheadascent_add || !sf->pfminfo.hheaddescent_add ) {
	DBounds b;
	CIDFindBounds(sf,&b);
	if ( !sf->pfminfo.winascent_add ) {
	    sf->pfminfo.winascent_add = true;
	    if ( sf->pfminfo.os2_winascent < sf->ascent/8 ) {
		/* There was a bug for a while which gave us really bad values */
		sf->pfminfo.os2_winascent = 0;
		sf->pfminfo.windescent_add = true;
		sf->pfminfo.os2_windescent = 0;
	    } else
		sf->pfminfo.os2_winascent -= b.maxy;
	}
	if ( !sf->pfminfo.windescent_add ) {
	    sf->pfminfo.windescent_add = true;
	    sf->pfminfo.os2_windescent += b.miny;
	}
	if ( !sf->pfminfo.hheadascent_add ) {
	    sf->pfminfo.hheadascent_add = true;
	    sf->pfminfo.hhead_ascent -= b.maxy;
	}
	if ( !sf->pfminfo.hheaddescent_add ) {
	    sf->pfminfo.hheaddescent_add = true;
	    sf->pfminfo.hhead_descent -= b.miny;
	}
    }
    if ( !sf->pfminfo.typoascent_add ) {
	sf->pfminfo.typoascent_add = true;
	sf->pfminfo.os2_typoascent -= sf->ascent;
    }
    if ( !sf->pfminfo.typodescent_add  ) {
	sf->pfminfo.typodescent_add = true;
	sf->pfminfo.os2_typodescent -= -sf->descent;
    }
}

static void PsuedoEncodeUnencoded(EncMap *map,struct ttfinfo *info) {
    int extras, base;
    int i;

    for ( i=0; i<info->glyph_cnt; ++i )
	if ( info->chars[i]!=NULL )
	    info->chars[i]->ticked = false;
    for ( i=0; i<map->enccount; ++i )
	if ( map->map[i]!=-1 )
	    info->chars[map->map[i]]->ticked = true;
    extras = 0;
    for ( i=0; i<info->glyph_cnt; ++i )
	if ( info->chars[i]!=NULL && !info->chars[i]->ticked )
	    ++extras;
    if ( extras!=0 ) {
	if ( map->enccount<=256 )
	    base = 256;
	else if ( map->enccount<=65536 )
	    base = 65536;
	else if ( map->enccount<=17*65536 )
	    base = 17*65536;
	else
	    base = map->enccount;
	if ( base+extras>map->encmax ) {
	    map->map = grealloc(map->map,(base+extras)*sizeof(int));
	    memset(map->map+map->enccount,-1,(base+extras-map->enccount)*sizeof(int));
	    map->encmax = base+extras;
	}
	map->enccount = base+extras;
	extras = 0;
	for ( i=0; i<info->glyph_cnt; ++i )
	    if ( info->chars[i]!=NULL && !info->chars[i]->ticked )
		map->map[base+extras++] = i;
    }
}

static void MapDoBack(EncMap *map,struct ttfinfo *info) {
    int i;

    if ( map==NULL )		/* CID fonts */
return;
    free(map->backmap);		/* CFF files have this */
    map->backmax = info->glyph_cnt;
    map->backmap = galloc(info->glyph_cnt*sizeof(int));
    memset(map->backmap,-1,info->glyph_cnt*sizeof(int));
    for ( i = map->enccount-1; i>=0; --i )
	if ( map->map[i]>=0 && map->map[i]<info->glyph_cnt )
	    if ( map->backmap[map->map[i]]==-1 )
		map->backmap[map->map[i]] = i;
}

static SplineFont *SFFillFromTTF(struct ttfinfo *info) {
    SplineFont *sf, *_sf;
    int i,k;
    BDFFont *bdf;
    struct table_ordering *ord;
    SplineChar *sc;
    struct ttf_table *last[2], *tab, *next;

    
    sf = SplineFontEmpty();
	sf->units_per_em = info->emsize;
    sf->display_size = -default_fv_font_size;
    sf->display_antialias = default_fv_antialias;
    sf->fontname = info->fontname;
    sf->fullname = info->fullname;
    sf->familyname = info->familyname;
    sf->chosenname = info->chosenname;

    sf->order2 = info->to_order2;
    sf->comments = info->fontcomments;

    sf->creationtime = info->creationtime;
    sf->modificationtime = info->modificationtime;

    sf->design_size = info->design_size;
    sf->design_range_bottom = info->design_range_bottom;
    sf->design_range_top = info->design_range_top;
    sf->fontstyle_id = info->fontstyle_id;
    sf->fontstyle_name = info->fontstyle_name;

    sf->gasp_cnt = info->gasp_cnt;
    sf->gasp = info->gasp;

    sf->texdata = info->texdata;

    sf->mark_class_cnt = info->mark_class_cnt;
    sf->mark_classes = info->mark_classes;
    sf->mark_class_names = info->mark_class_names;

    if ( info->fd!=NULL ) {		/* Special hack for type42 fonts */
	sf->fontname = copy(info->fd->fontname);
	sf->uniqueid = info->fd->uniqueid;
	sf->xuid = XUIDFromFD(info->fd->xuid);
	if ( info->fd->fontinfo!=NULL ) {
	    sf->familyname = utf8_verify_copy(info->fd->fontinfo->familyname);
	    sf->fullname = utf8_verify_copy(info->fd->fontinfo->fullname);
	    sf->copyright = utf8_verify_copy(info->fd->fontinfo->notice);
	    sf->weight = utf8_verify_copy(info->fd->fontinfo->weight);
	    sf->version = utf8_verify_copy(info->fd->fontinfo->version);
	    sf->italicangle = info->fd->fontinfo->italicangle;
	    sf->upos = info->fd->fontinfo->underlineposition*(sf->ascent+sf->descent);
	    sf->uwidth = info->fd->fontinfo->underlinethickness*(sf->ascent+sf->descent);
	}
    }

    if ( sf->fontname==NULL ) {
	sf->fontname = EnforcePostScriptName(sf->fullname);
	if ( sf->fontname==NULL )
	    sf->fontname = EnforcePostScriptName(sf->familyname);
	if ( sf->fontname==NULL ) sf->fontname = EnforcePostScriptName("UntitledTTF");
    }
    if ( sf->fullname==NULL ) sf->fullname = copy( sf->fontname );
    if ( sf->familyname==NULL ) sf->familyname = copy( sf->fontname );
    if ( sf->weight==NULL ) {
	if ( info->weight != NULL )
	    sf->weight = info->weight;
	else if ( info->pfminfo.pfmset )
	    sf->weight = copy( info->pfminfo.weight <= 100 ? "Thin" :
				info->pfminfo.weight <= 200 ? "Extra-Light" :
				info->pfminfo.weight <= 300 ? "Light" :
				info->pfminfo.weight <= 400 ? "Book" :
				info->pfminfo.weight <= 500 ? "Medium" :
				info->pfminfo.weight <= 600 ? "Demi" :
				info->pfminfo.weight <= 700 ? "Bold" :
				info->pfminfo.weight <= 800 ? "Heavy" :
				    "Black" );
	else
	    sf->weight = copy("");
    } else
	free( info->weight );
    if ( sf->copyright==NULL )
	sf->copyright = info->copyright;
    else
	free( info->copyright );
    sf->version = info->version;
    sf->italicangle = info->italicAngle;
    sf->strokewidth = info->strokewidth;
    sf->strokedfont = info->strokedfont;
    sf->upos = info->upos;
    sf->uwidth = info->uwidth;
    sf->ascent = info->ascent;
    sf->vertical_origin = info->vertical_origin;
    if ( info->vhea_start!=0 && info->vmetrics_start!=0 )
	sf->hasvmetrics = true;
    sf->descent = info->descent;
    sf->private = info->private;
    sf->xuid = info->xuid;
    sf->uniqueid = info->uniqueid;
    sf->pfminfo = info->pfminfo;
    sf->os2_version = info->os2_version;
    sf->use_typo_metrics = info->use_typo_metrics;
    sf->weight_width_slope_only = info->weight_width_slope_only;
    sf->head_optimized_for_cleartype = info->optimized_for_cleartype;
    sf->gasp_version = info->gasp_version;
    sf->names = info->names;
    sf->anchor = info->ahead;
    sf->kerns = info->khead;
    sf->vkerns = info->vkhead;
    sf->possub = info->possub;
    sf->sm = info->sm;
    sf->features = info->features;
    sf->gentags = info->gentags;
    sf->script_lang = info->script_lang;
    sf->sli_cnt = info->sli_cnt;

    last[0] = sf->ttf_tables;
    last[1] = NULL;
    for ( tab=info->tabs; tab!=NULL; tab = next ) {
	next = tab->next;
	if ( tab->tag==CHR('f','p','g','m') || tab->tag==CHR('p','r','e','p') ||
		tab->tag==CHR('c','v','t',' ') || tab->tag==CHR('m','a','x','p')) {
	    if ( last[0]==NULL )
		sf->ttf_tables = tab;
	    else
		last[0]->next = tab;
	    last[0] = tab;
	} else {
	    if ( last[1]==NULL )
		sf->ttf_tab_saved = tab;
	    else
		last[1]->next = tab;
	    last[1] = tab;
	}
	tab->next = NULL;
    }

    if ( info->twobytesymbol )
	/* rework ms symbol encodings */
	SymbolFixup(info);
    if ( info->map==NULL && info->subfonts==NULL )		/* Can happen when reading a ttf from a pdf */
	info->map = EncMapFromEncoding(sf,FindOrMakeEncoding("original"));
    if ( info->subfontcnt==0 )
	PsuedoEncodeUnencoded(info->map,info);
    MapDoBack(info->map,info);
    sf->map = info->map;
    sf->cidregistry = info->cidregistry;
    sf->ordering = info->ordering;
    sf->supplement = info->supplement;
    sf->cidversion = info->cidfontversion;

    SFDefaultAscent(sf);

    for ( i=0; i<info->glyph_cnt; ++i ) if ( info->chars[i]!=NULL ) {
	SCOrderAP(info->chars[i]);
    }

    if ( info->subfontcnt == 0 ) {
	UseGivenEncoding(sf,info);
    } else {
	sf->subfontcnt = info->subfontcnt;
	sf->subfonts = info->subfonts;
	free(info->chars);		/* This is the GID->char index, don't need it now */
	for ( i=0; i<sf->subfontcnt; ++i ) {
	    sf->subfonts[i]->cidmaster = sf;
	    sf->subfonts[i]->vertical_origin = sf->vertical_origin;
	    sf->subfonts[i]->hasvmetrics = sf->hasvmetrics;
	}
    }
    TTF_PSDupsDefault(sf);
    if ( info->gsub_start==0 && info->mort_start==0 && info->morx_start==0 ) {
	/* Get default ligature values, etc. */
	k=0;
	do {
	    _sf = k<sf->subfontcnt?sf->subfonts[k]:sf;
	    for ( i=0; i<sf->glyphcnt; ++i ) {
		if ( _sf->glyphs[i]!=NULL )		/* Might be null in ttc files */
		    SCLigDefault(_sf->glyphs[i]);
	    }
	    ++k;
	} while ( k<sf->subfontcnt );
    }

    if ( info->feats[0]!=NULL ) {
	ord = chunkalloc(sizeof(struct table_ordering));
	ord->table_tag = CHR('G','S','U','B');		/* or mort/morx */
	ord->ordered_features = info->feats[0];
	sf->orders = ord;
    }
    if ( info->feats[1]!=NULL ) {
	ord = chunkalloc(sizeof(struct table_ordering));
	ord->table_tag = CHR('G','P','O','S');
	ord->ordered_features = info->feats[1];
	ord->next = sf->orders;
	sf->orders = ord;
    }

    otf_orderlangs(NULL,sf);		/* I thought these had to be ordered, but it seems I was wrong. But I depend on the order, so I enforce it here */

    if ( info->variations!=NULL )
	MMFillFromVAR(sf,info);

    if ( info->cff_length!=0 && !sf->order2 ) {
	/* Clean up the hint masks, We create an initial hintmask whether we */
	/*  need it or not */
	k=0;
	do {
	    _sf = k<sf->subfontcnt?sf->subfonts[k]:sf;
	    for ( i=0; i<sf->glyphcnt; ++i ) {
		if ( (sc = _sf->glyphs[i])!=NULL && !sc->hconflicts && !sc->vconflicts &&
			sc->layers[ly_fore].splines!=NULL ) {
		    chunkfree( sc->layers[ly_fore].splines->first->hintmask,sizeof(HintMask) );
		    sc->layers[ly_fore].splines->first->hintmask = NULL;
		}
	    }
	    ++k;
	} while ( k<sf->subfontcnt );
    }
    SFRelativeWinAsDs(sf);
    free(info->savetab);
    return( sf );
}



 SplineFont *_SFReadTTF(FILE *ttf, int flags,enum openflags openflags, char *filename,struct fontdict *fd, char *choice) {
    struct ttfinfo info;
    int ret;

    memset(&info,'\0',sizeof(struct ttfinfo));
    info.onlystrikes = (flags&ttf_onlystrikes)?1:0;
    info.onlyonestrike = (flags&ttf_onlyonestrike)?1:0;
    info.use_typo_metrics = true;
    info.fd = fd;
    ret = readttf(ttf,&info,filename, choice);
    if ( !ret )
      return( NULL );
    return( SFFillFromTTF(&info));
}

SplineFont *_CFFParse(FILE *temp,int len, char *fontsetname) {
    struct ttfinfo info;

    memset(&info,'\0',sizeof(info));
    info.cff_start = 0;
    info.cff_length = len;
    info.barecff = true;
    if ( !readcffglyphs(temp,&info) )
      return( NULL );
    return( SFFillFromTTF(&info));
}

SplineFont *CFFParse(char *filename) {
    FILE *cff = fopen(filename,"r");
    SplineFont *sf;
    long len;

    if ( cff == NULL )
return( NULL );
    fseek(cff,0,SEEK_END);
    len = ftell(cff);
    fseek(cff,0,SEEK_SET);
    sf = _CFFParse(cff,len,NULL);
    fclose(cff);
return( sf );
}

char **NamesReadCFF(char *filename) {
    FILE *cff = fopen(filename,"rb");
    int32 hdrsize, offsize;
    char **fontnames;

    if ( cff==NULL )
return( NULL );
    if ( getc(cff)!='\1' ) {		/* Major version */
	LogError( _("CFF version mismatch\n") );
	fclose(cff);
return( NULL );
    }
    getc(cff);				/* Minor version */
    hdrsize = getc(cff);
    offsize = getc(cff);
    if ( hdrsize!=4 )
	fseek(cff,hdrsize,SEEK_SET);
    fontnames = readcfffontnames(cff,NULL);
    fclose(cff);
return( fontnames );
}

char **NamesReadTTF(char *filename) {
    FILE *ttf = fopen(filename,"rb");
    int32 version, cnt, *offsets;
    int i,j;
    char **ret = NULL;
    char *temp;

    if ( ttf==NULL )
return( NULL );
    version=getlong(ttf);
    if ( version==CHR('t','t','c','f')) {
	/* TTCF version = */ getlong(ttf);
	cnt = getlong(ttf);
	offsets = galloc(cnt*sizeof(int32));
	for ( i=0; i<cnt; ++i )
	    offsets[i] = getlong(ttf);
	ret = galloc((cnt+1)*sizeof(char *));
	for ( i=j=0; i<cnt; ++i ) {
	    temp = TTFGetFontName(ttf,offsets[i],0);
	    if ( temp!=NULL )
		ret[j++] = temp;
	}
	ret[j] = NULL;
	free(offsets);
    } else {
	temp = TTFGetFontName(ttf,0,0);
	if ( temp!=NULL ) {
	    ret = galloc(2*sizeof(char *));
	    ret[0] = temp;
	    ret[1] = NULL;
	}
    }
    fclose(ttf);
return(ret);
}


void VariationFree(struct ttfinfo *info) {
    int i,j;
    struct variations *variation = info->variations;

    if ( variation==NULL )
return;
    if ( variation->axes!=NULL ) {
	for ( i=0; i<variation->axis_count; ++i ) {
	    free(variation->axes[i].mapfrom);
	    free(variation->axes[i].mapto);
	}
	free(variation->axes);
    }
    if ( variation->instances!=NULL ) {
	for ( i=0; i<variation->instance_count; ++i ) {
	    free(variation->instances[i].coords);
	}
	free(variation->instances);
    }
    if ( variation->tuples!=NULL ) {
	for ( i=0; i<variation->tuple_count; ++i ) {
	    free(variation->tuples[i].coords);
	    if ( variation->tuples[i].chars!=NULL )
		for ( j=0; j<info->glyph_cnt; ++j )
		    SplineCharFree(variation->tuples[i].chars[j]);
	    free(variation->tuples[i].chars);
	    KernClassListFree(variation->tuples[i].khead);
	    KernClassListFree(variation->tuples[i].vkhead);
	}
	free(variation->tuples);
    }
    free(variation);
    info->variations = NULL;
}

static void parsefvar(struct ttfinfo *info, FILE *ttf) {
    int data_off, axis_count, instance_count, cnt;
    int i,j;

    fseek(ttf,info->fvar_start,SEEK_SET);
    if ( getlong(ttf)!=0x00010000 )	/* I only understand version 1 */
return;
    data_off = getushort(ttf);
    cnt = getushort(ttf);
    if ( cnt>2 )
	LogError( _("Hmm, this 'fvar' table has more count/size pairs than I expect\n") );
    else if ( cnt<2 ) {
	LogError( _("Hmm, this 'fvar' table has too few count/size pairs, I shan't parse it\n") );
return;
    }
    axis_count = getushort(ttf);
    if ( axis_count==0 || axis_count>4 ) {
	if ( axis_count==0 )
	    LogError( _("Hmm, this 'fvar' table has no axes, that doesn't make sense.\n") );
	else
	    LogError( _("Hmm, this 'fvar' table has more axes than FontForge can handle.\n") );
return;
    }
    if ( getushort(ttf)!=20 ) {
	LogError( _("Hmm, this 'fvar' table has an unexpected size for an axis, I shan't parse it\n") );
return;
    }
    instance_count = getushort(ttf);
    if ( getushort(ttf)!=4+4*axis_count ) {
	LogError( _("Hmm, this 'fvar' table has an unexpected size for an instance, I shan't parse it\n") );
return;
    }
    if ( data_off+axis_count*20+instance_count*(4+4*axis_count)> info->fvar_len ) {
	LogError( _("Hmm, this 'fvar' table is too short\n") );
return;
    }

    info->variations = gcalloc(1,sizeof(struct variations));
    info->variations->axis_count = axis_count;
    info->variations->instance_count = instance_count;
    info->variations->axes = gcalloc(axis_count,sizeof(struct taxis));
    info->variations->instances = gcalloc(instance_count,sizeof(struct tinstance));
    for ( i=0; i<instance_count; ++i )
	info->variations->instances[i].coords = galloc(axis_count*sizeof(real));

    fseek(ttf,info->fvar_start+data_off,SEEK_SET);
    for ( i=0; i<axis_count; ++i ) {
	struct taxis *a = &info->variations->axes[i];
	a->tag = getlong(ttf);
	a->min = getlong(ttf)/65536.0;
	a->def = getlong(ttf)/65536.0;
	a->max = getlong(ttf)/65536.0;
	/* flags = */ getushort(ttf);
	a->nameid = getushort(ttf);
    }
    for ( i=0; i<instance_count; ++i ) {
	struct tinstance *ti = &info->variations->instances[i];
	ti->nameid = getushort(ttf);
	/* flags = */ getushort(ttf);
	for ( j=0 ; j<axis_count; ++j )
	    ti->coords[j] = getlong(ttf)/65536.0;
    }
}

static void parseavar(struct ttfinfo *info, FILE *ttf) {
    int axis_count, pair_count;
    int i,j;

    if ( info->variations==NULL || info->avar_start==0 || info->avar_len==0 )
return;

    fseek(ttf,info->avar_start,SEEK_SET);
    if ( getlong(ttf)!=0x00010000 ) {	/* I only understand version 1 */
	VariationFree(info);
return;
    }
    axis_count = getlong(ttf);
    if ( axis_count!=info->variations->axis_count ) {
	LogError( _("Hmm, the axis count in the 'avar' table is different from that in the 'fvar' table.\n") );
	VariationFree(info);
return;
    }
    for ( i=0; i<axis_count; ++i ) {
	pair_count = getushort(ttf);
	if ( pair_count!=0 ) {
	    info->variations->axes[i].mapfrom = galloc(pair_count*sizeof(real));
	    info->variations->axes[i].mapto= galloc(pair_count*sizeof(real));
	    for ( j=0; j<pair_count; ++j ) {
		info->variations->axes[i].mapfrom[j] = getushort(ttf)/16384.0;
		info->variations->axes[i].mapto[j] = getushort(ttf)/16384.0;
	    }
	}
    }
    if ( ftell(ttf)-info->avar_start>info->avar_len) {
	LogError( _("Hmm, the the 'avar' table is too long.\n") );
	VariationFree(info);
return;
    }
}

static SplineChar **InfoCopyGlyphs(struct ttfinfo *info) {
    SplineChar **chars = galloc(info->glyph_cnt*sizeof(SplineChar *));
    int i;
    RefChar *r;

    for ( i=0; i<info->glyph_cnt; ++i ) {
	if ( info->chars[i]==NULL )
	    chars[i] = NULL;
	else {
	    chars[i] = SplineCharCopy(info->chars[i],NULL);
	    free(chars[i]->ttf_instrs); chars[i]->ttf_instrs = NULL;
	    chars[i]->ttf_instrs_len = 0;
	    PSTFree(chars[i]->possub); chars[i]->possub = NULL;
	    for ( r=chars[i]->layers[ly_fore].refs; r!=NULL; r=r->next )
		r->sc = NULL;
	    chars[i]->changed = false;
	    chars[i]->ticked = false;
	}
    }

    for ( i=0; i<info->glyph_cnt; ++i )
	ttfFixupRef(chars,i);
return( chars );
}

#define BAD_DELTA	0x10001
static int *readpackeddeltas(FILE *ttf,int n) {
    int *deltas;
    int runcnt, i, j;

    deltas = galloc(n*sizeof(int));

    i = 0;
    while ( i<n ) {
	runcnt = getc(ttf);
	if ( runcnt&0x80 ) {
	    /* runcnt zeros get added */
	    for ( j=0; j<=(runcnt&0x3f) && i<n ; ++j )
		deltas[i++] = 0;
	} else if ( runcnt&0x40 ) {
	    /* runcnt shorts from the stack */
	    for ( j=0 ; j<=(runcnt&0x3f) && i<n ; ++j )
		deltas[i++] = (int16) getushort(ttf);
	} else {
	    /* runcnt signed bytes from the stack */
	    for ( j=0; j<=(runcnt&0x3f) && i<n; ++j )
		deltas[i++] = (int8) getc(ttf);
	}
	if ( j<=(runcnt&0x3f) ) {
	    if ( n>0 )
		deltas[0] = BAD_DELTA;
	}
    }
return( deltas );
}

#define ALL_POINTS	0x10001
#define END_OF_POINTS	0x10000

static int *readpackedpoints(FILE *ttf) {
    int *points;
    int n, runcnt, i, j, first;

    n = getc(ttf);
    if ( n==EOF )
	n = 0;
    if ( n&0x80 )
	n = getc(ttf)|((n&0x7f)<<8);
    points = galloc((n+1)*sizeof(int));
    if ( n==0 )
	points[0] = ALL_POINTS;
    else {
	i = 0;
	while ( i<n ) {
	    runcnt = getc(ttf);
	    if ( runcnt&0x80 ) {
		runcnt = (runcnt&0x7f);
		points[i++] = first = getushort(ttf);
		/* first point not included in runcount */
		for ( j=0; j<runcnt && i<n; ++j )
		    points[i++] = (first += getushort(ttf));
	    } else {
		points[i++] = first = getc(ttf);
		for ( j=0; j<runcnt && i<n; ++j )
		    points[i++] = (first += getc(ttf));
	    }
	}
	points[n] = END_OF_POINTS;
    }
return( points );
}

static int TuplesMatch(struct variations *v, int vtest, int dbase) {
    /* variations for [0,1] make up part of design [1,1], but not the other */
    /* way round */
    int i;

    if ( dbase>=v->tuple_count )
return( false );

    if ( vtest==dbase )
return( true );
    for ( i=0; i<v->axis_count; ++i ) {
	if ( v->tuples[vtest].coords[i]==0 && v->tuples[dbase].coords[i]!=0 )
return( false );
	if ( v->tuples[dbase].coords[i]!=0 &&
		v->tuples[dbase].coords[i]!=v->tuples[vtest].coords[i] )
return( false );
    }
return( true );
}

static int PointCount(SplineChar *sc) {
    int i;
    RefChar *ref;
    SplineSet *ss;
    SplinePoint *sp;

    if ( sc->layers[ly_fore].refs!=NULL )
	for ( i=0, ref=sc->layers[ly_fore].refs; ref!=NULL; ++i, ref=ref->next );
    else {
	for ( i=0, ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
	    for ( sp=ss->first; sp!=NULL ; ) {
		if ( sp->ttfindex!=0xffff && sp->ttfindex!=0xfffe )
		    ++i;
		if ( sp->nextcpindex!=0xffff && sp->nextcpindex!=0xfffe )
		    ++i;
		if ( sp->next==NULL )
	    break;
		sp = sp->next->to;
		if ( sp==ss->first )
	    break;
	    }
	}
    }
return( i );
}

static void SCShiftAllBy(SplineChar *sc,int xd, int yd) {
    /* If they change the left/right side-bearing, I think that means everything */
    /*  should be shifted over */
    SplineSet *ss;
    SplinePoint *sp;
    RefChar *ref;

    if ( xd==0 && yd==0 )
return;

    for ( ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
	for ( sp=ss->first; sp!=NULL ; ) {
	    if ( sp->ttfindex!=0xffff && sp->ttfindex!=0xfffe ) {
		sp->me.x += xd;
		sp->me.y += yd;
	    }
	    if ( sp->nextcpindex!=0xffff && sp->nextcpindex!=0xfffe ) {
		sp->nextcp.x += xd;
		sp->nextcp.y += yd;
		if ( sp->next!=NULL )
		    sp->next->to->prevcp = sp->nextcp;
	    }
	    if ( sp->next==NULL )
	break;
	    sp = sp->next->to;
	    if ( sp == ss->first )
	break;
	}
    }
    for ( ref=sc->layers[ly_fore].refs; ref!=NULL; ref=ref->next ) {
	ref->transform[4] += xd;
	ref->transform[5] += yd;
	SCReinstanciateRefChar(sc,ref);
    }
}

static void VaryGlyph(SplineChar *sc,int *points, int *xdeltas, int *ydeltas,
	int pcnt) {
    /* A character contains either composites or contours */
    int i,j;
    RefChar *ref;
    SplineSet *ss;
    SplinePoint *sp;
    Spline *s, *first;

    if ( points[0]==ALL_POINTS ) {
	if ( sc->layers[ly_fore].refs!=NULL ) {
	    for ( i=0, ref=sc->layers[ly_fore].refs; ref!=NULL; ++i, ref=ref->next ) {
		if ( xdeltas[i]!=0 || ydeltas[i]!=0 ) {
		    ref->transform[4] += xdeltas[i];
		    ref->transform[5] += ydeltas[i];
		    SCReinstanciateRefChar(sc,ref);
		}
	    }
	} else {
	    for ( ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
		for ( sp=ss->first; sp!=NULL ; ) {
		    if ( sp->ttfindex!=0xffff && sp->ttfindex!=0xfffe ) {
			sp->me.x += xdeltas[sp->ttfindex];
			sp->me.y += ydeltas[sp->ttfindex];
		    }
		    if ( sp->nextcpindex!=0xffff && sp->nextcpindex!=0xfffe ) {
			sp->nextcp.x += xdeltas[sp->nextcpindex];
			sp->nextcp.y += ydeltas[sp->nextcpindex];
			if ( sp->next!=NULL )
			    sp->next->to->prevcp = sp->nextcp;
		    }
		    if ( sp->next==NULL )
		break;
		    sp = sp->next->to;
		    if ( sp == ss->first )
		break;
		}
	    }
	}
	SCShiftAllBy(sc,-xdeltas[pcnt-4],0);
	SCShiftAllBy(sc,0,-ydeltas[pcnt-2]);
	sc->width += xdeltas[pcnt-3];
	sc->vwidth += ydeltas[pcnt-1];
    } else {
	j = 0;
	if ( sc->layers[ly_fore].refs!=NULL ) {
	    for ( i=0, ref=sc->layers[ly_fore].refs; ref!=NULL; ++i, ref=ref->next ) {
		if ( points[j]==i ) {
		    if ( xdeltas[j]!=0 || ydeltas[j]!=0 ) {
			ref->transform[4] += xdeltas[j];
			ref->transform[5] += ydeltas[j];
			SCReinstanciateRefChar(sc,ref);
		    }
		    ++j;
		}
	    }
	} else {
	    for ( i=0, ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
		if ( ss->first->prev!=NULL && ss->first->prev->from->nextcpindex==points[j] ) {
		    ss->first->prevcp.x += xdeltas[j];
		    ss->first->prevcp.y += ydeltas[j];
		    ss->first->prev->from->nextcp = ss->first->prevcp;
		    ++j;
		}
		for ( sp=ss->first; sp!=NULL ; ++i ) {
		    if ( sp->ttfindex!=0xffff && sp->ttfindex!=0xfffe )
			++i;
		    if ( sp->ttfindex==points[j] ) {
			sp->me.x += xdeltas[j];
			sp->me.y += ydeltas[j++];
		    }
		    if ( sp->nextcpindex!=0xffff && sp->nextcpindex!=0xfffe )
			++i;
		    if ( sp->nextcpindex==points[j] ) {
			sp->nextcp.x += xdeltas[j];
			sp->nextcp.y += ydeltas[j++];
			if ( sp->next!=NULL )
			    sp->next->to->prevcp = sp->nextcp;
		    } else if ( sp->nonextcp ) {
			sp->nextcp = sp->me;
		    }
		    if ( sp->next==NULL )
		break;
		    sp = sp->next->to;
		    if ( sp == ss->first )
		break;
		}
		first = NULL;
	    }
	}
	if ( points[j]==i )
	    SCShiftAllBy(sc,-xdeltas[j++],0);
	if ( points[j]==i+1 )
	    sc->width += xdeltas[j++];
	if ( points[j]==i+2 )
	    SCShiftAllBy(sc,0,-ydeltas[j++]);
	if ( points[j]==i+3 )
	    sc->vwidth += ydeltas[j++];
    }
    if ( sc->layers[ly_fore].refs==NULL ) {
	for ( ss = sc->layers[ly_fore].splines; ss!=NULL; ss=ss->next ) {
	    for ( sp=ss->first; sp!=NULL ; ++i ) {
		if ( sp->ttfindex==0xffff ) {
		    sp->me.x = ( sp->nextcp.x + sp->prevcp.x )/2;
		    sp->me.y = ( sp->nextcp.y + sp->prevcp.y )/2;
		}
		if ( sp->next==NULL )
	    break;
		sp = sp->next->to;
		if ( sp == ss->first )
	    break;
	    }

	    first = NULL;
	    for ( s=ss->first->next; s!=NULL && s!=first; s=s->to->next ) {
		SplineRefigure(s);
		if ( first==NULL ) first = s;
	    }
	}
    }
}

static void VaryGlyphs(struct ttfinfo *info,int tupleIndex,int gnum,
	int *points, FILE *ttf ) {
    /* one annoying thing about gvar, is that the variations do not describe */
    /*  designs. well variations for [0,1] describes that design, but the */
    /*  design for [1,1] includes the variations [0,1], [1,0], and [1,1] */
    int pcnt, tc;
    int *xdeltas, *ydeltas;
    struct variations *v = info->variations;

    if ( info->chars[gnum]==NULL )	/* Apple doesn't support ttc so this */
return;					/*  can't happen */
    if ( points==NULL ) {
	LogError( _("Mismatched local and shared tuple flags.\n") );
return;
    }

    if ( points[0]==ALL_POINTS )
	pcnt = PointCount(info->chars[gnum])+4;
    else {
	for ( pcnt=0; points[pcnt]!=END_OF_POINTS; ++pcnt );
    }
    xdeltas = readpackeddeltas(ttf,pcnt);
    ydeltas = readpackeddeltas(ttf,pcnt);
    if ( xdeltas[0]!=BAD_DELTA && ydeltas[0]!=BAD_DELTA )
	for ( tc = 0; tc<v->tuple_count; ++tc ) {
	    if ( TuplesMatch(v,tc,tupleIndex))
		VaryGlyph(v->tuples[tc].chars[gnum],points,xdeltas,ydeltas,pcnt);
    } else {
	static int warned = false;
	if ( !warned )
	    LogError( _("Incorrect number of deltas in glyph %d (%s)\n"), gnum,
		    info->chars[gnum]->name!=NULL?info->chars[gnum]->name:"<Nameless>" );
	warned = true;
    }
    free(xdeltas);
    free(ydeltas);
}

static void parsegvar(struct ttfinfo *info, FILE *ttf) {
    /* I'm only going to support a subset of the gvar. Only the global tuples */
    int axiscount, globaltc, gvarflags, gc, i,j,g;
    uint32 tupoff, dataoff, *gvars;
    struct variations *v = info->variations;
    int warned=false;

    fseek(ttf,info->gvar_start,SEEK_SET);
    if ( getlong(ttf)!=0x00010000 ) {	/* I only understand version 1 */
	VariationFree(info);
return;
    }
    axiscount = getushort(ttf);
    if ( axiscount!=info->variations->axis_count ) {
	LogError( _("Hmm, the axis count in the 'gvar' table is different from that in the 'fvar' table.\n") );
	VariationFree(info);
return;
    }
    globaltc = getushort(ttf);
    tupoff = getlong(ttf) + info->gvar_start;
    gc = getushort(ttf);
    gvarflags = getushort(ttf);
    dataoff = getlong(ttf) + info->gvar_start;
    if ( globaltc==0 || globaltc>AppleMmMax ) {
	if ( globaltc==0 )
	    LogError( _("Hmm, no global tuples specified in the 'gvar' table.\n") );
	else
	    LogError( _("Hmm, too many global tuples specified in the 'gvar' table.\n FontForge only supports %d\n"), AppleMmMax );
	VariationFree(info);
return;
    }
    if ( gc>info->glyph_cnt ) {
	LogError( _("Hmm, more glyph variation data specified than there are glyphs in font.\n") );
	VariationFree(info);
return;
    }

    gvars = galloc((gc+1)*sizeof(uint32));
    if ( gvarflags&1 ) {	/* 32 bit data */
	for ( i=0; i<=gc; ++i )
	    gvars[i] = getlong(ttf)+dataoff;
    } else {
	for ( i=0; i<=gc; ++i )
	    gvars[i] = getushort(ttf)*2 +dataoff;	/* Undocumented *2 */
    }

    v->tuple_count = globaltc;
    v->tuples = gcalloc(globaltc,sizeof(struct tuples));
    fseek(ttf,tupoff,SEEK_SET);
    for ( i=0; i<globaltc; ++i ) {
	v->tuples[i].coords = galloc(axiscount*sizeof(float));
	for ( j=0; j<axiscount; ++j )
	    v->tuples[i].coords[j] = ((short) getushort(ttf))/16384.0;
	v->tuples[i].chars = InfoCopyGlyphs(info);
    }

    for ( g=0; g<gc; ++g ) if ( gvars[g]!=gvars[g+1] ) {
	int tc;
	uint32 datoff;
	int *sharedpoints=NULL;
	fseek(ttf,gvars[g],SEEK_SET);
	tc = getushort(ttf);
	datoff = gvars[g]+getushort(ttf);
	if ( tc&0x8000 ) {
	    uint32 here = ftell(ttf);
	    fseek(ttf,datoff,SEEK_SET);
	    sharedpoints = readpackedpoints(ttf);
	    datoff = ftell(ttf);
	    fseek(ttf,here,SEEK_SET);
	}
	for ( i=0; i<(tc&0xfff); ++i ) {
	    int tupleDataSize, tupleIndex;
	    tupleDataSize = getushort(ttf);
	    tupleIndex = getushort(ttf);
	    if ( tupleIndex&0xc000 ) {
		if ( !warned )
		    LogError( _("Warning: Glyph %d contains either private or intermediate tuple data.\n FontForge supports neither.\n"),
			    g);
		warned = true;
		if ( tupleIndex&0x8000 )
		    fseek(ttf,2*axiscount,SEEK_CUR);
		if ( tupleIndex&0x4000 )
		    fseek(ttf,4*axiscount,SEEK_CUR);
	    } else {
		int *localpoints=NULL;
		uint32 here = ftell(ttf);
		fseek(ttf,datoff,SEEK_SET);
		if ( tupleIndex&0x2000 )
		    localpoints = readpackedpoints(ttf);
		VaryGlyphs(info,tupleIndex&0xfff,g,
			(tupleIndex&0x2000)?localpoints:sharedpoints,ttf);
		free(localpoints);
		fseek(ttf,here,SEEK_SET);
	    }
	    datoff += tupleDataSize;
	}
	free(sharedpoints);
    }
    free(gvars);
}

static void AlterEntry(struct ttf_table *cvt, int i, int delta ) {
    int val = memushort(cvt->data,cvt->len,2*i);
    memputshort(cvt->data,2*i,val+delta);
}

static void VaryCvt(struct tuples *tuple,int *points, int *deltas,
	int pcnt, struct ttf_table *orig_cvt) {
    struct ttf_table *cvt;
    int i;

    if ( (cvt = tuple->cvt)==NULL ) {
	cvt = tuple->cvt = chunkalloc(sizeof(struct ttf_table));
	cvt->tag = orig_cvt->tag;
	cvt->len = cvt->maxlen = orig_cvt->len;
	cvt->data = galloc(cvt->len);
	memcpy(cvt->data,orig_cvt->data,cvt->len);
    }
    if ( points[0]==ALL_POINTS ) {
	for ( i=0; i<pcnt; ++i )
	    AlterEntry(cvt,i,deltas[i]);
    } else {
	for ( i=0; i<pcnt; ++i )
	    AlterEntry(cvt,points[i],deltas[i]);
    }
}

static void VaryCvts(struct ttfinfo *info,int tupleIndex, int *points, FILE *ttf,
    struct ttf_table *origcvt ) {
    /* one annoying thing about gvar, is that the variations do not describe */
    /*  designs. well variations for [0,1] describes that design, but the */
    /*  design for [1,1] includes the variations [0,1], [1,0], and [1,1] */
    /* And same is true of cvar */
    int pcnt, tc;
    int *deltas;
    struct variations *v = info->variations;

    if ( points[0]==ALL_POINTS )
	pcnt = origcvt->len/sizeof(uint16);
    else {
	for ( pcnt=0; points[pcnt]!=END_OF_POINTS; ++pcnt );
    }
    deltas = readpackeddeltas(ttf,pcnt);
    if ( deltas[0]!=BAD_DELTA )
	for ( tc = 0; tc<v->tuple_count; ++tc ) {
	    if ( TuplesMatch(v,tc,tupleIndex))
		VaryCvt(&v->tuples[tc],points,deltas,pcnt,origcvt);
    } else {
	static int warned = false;
	if ( !warned )
	    LogError( _("Incorrect number of deltas in cvt\n") );
	warned = true;
    }
    free(deltas);
}

static void parsecvar(struct ttfinfo *info, FILE *ttf) {
    struct ttf_table *cvt;
    int tuplecount;
    uint32 offset;
    int *sharedpoints=NULL;
    int i;
    int warned = false;

    for ( cvt = info->tabs; cvt!=NULL && cvt->tag!=CHR('c','v','t',' '); cvt=cvt->next );
    if ( cvt==NULL )
return;

    fseek(ttf,info->cvar_start,SEEK_SET);
    if ( getlong(ttf)!=0x00010000 ) {	/* I only understand version 1 */
	/* I think I can live without cvt variations... */
	/* So I shan't free the structure */
return;
    }

    tuplecount = getushort(ttf);
    offset = info->cvar_start+getushort(ttf);
    /* The documentation implies there are flags packed into the tuplecount */
    /*  but John Jenkins tells me that shared points don't apply to cvar */
    /*  Might as well parse it just in case */
    if ( tuplecount&0x8000 ) {
	uint32 here = ftell(ttf);
	fseek(ttf,offset,SEEK_SET);
	sharedpoints = readpackedpoints(ttf);
	offset = ftell(ttf);
	fseek(ttf,here,SEEK_SET);
    }
    for ( i=0; i<(tuplecount&0xfff); ++i ) {
	int tupleDataSize, tupleIndex;
	tupleDataSize = getushort(ttf);
	tupleIndex = getushort(ttf);
	/* there is no provision here for a global tuple coordinate section */
	/*  so John says there are no tuple indeces. Just embedded tuples */
	if ( tupleIndex&0x4000 ) {
	    if ( !warned )
		LogError( _("Warning: 'cvar' contains intermediate tuple data.\n FontForge doesn't support this.\n") );
	    warned = true;
	    if ( tupleIndex&0x8000 )
		fseek(ttf,2*info->variations->axis_count,SEEK_CUR);
	    if ( tupleIndex&0x4000 )
		fseek(ttf,4*info->variations->axis_count,SEEK_CUR);
	} else {
	    int *localpoints=NULL;
	    uint32 here;
	    int j,k,ti;
	    ti = tupleIndex&0xfff;
	    if ( tupleIndex&0x8000 ) {
		real *coords = galloc(info->variations->axis_count*sizeof(real));
		for ( j=0; j<info->variations->axis_count; ++j )
		    coords[j] = ((int16) getushort(ttf))/16384.0;
		for ( k=0 ; k<info->variations->tuple_count; ++k ) {
		    for ( j=0; j<info->variations->axis_count; ++j )
			if ( coords[j]!=info->variations->tuples[k].coords[j] )
		    break;
		    if ( j==info->variations->axis_count )
		break;
		}
		ti = -1;
		if ( k!=info->variations->tuple_count )
		    ti = k;
	    }
	    if ( ti!=-1 ) {
		here = ftell(ttf);
		fseek(ttf,offset,SEEK_SET);
		if ( tupleIndex&0x2000 )
		    localpoints = readpackedpoints(ttf);
		VaryCvts(info,ti,(tupleIndex&0x2000)?localpoints:sharedpoints,ttf,cvt);
		free(localpoints);
		fseek(ttf,here,SEEK_SET);
	    }
	}
	offset += tupleDataSize;
    }
    free(sharedpoints);
}

void readttfvariations(struct ttfinfo *info, FILE *ttf) {
    if ( info->gvar_start==0 || info->gvar_len==0 || info->fvar_start==0 || info->fvar_len==0 )
return;

    parsefvar(info,ttf);
    if ( info->variations!=NULL && info->avar_start!=0 )
	parseavar(info,ttf);
    if ( info->variations!=NULL )
	parsegvar(info,ttf);
    if ( info->variations!=NULL && info->cvar_start!=0 && info->cvt_start!=0 )
	parsecvar(info,ttf);
}


SplineFont *SplineFontEmpty(void) {
    time_t now;
    SplineFont *sf;

    sf = gcalloc(1,sizeof(SplineFont));
    sf->pfminfo.fstype = -1;
    sf->top_enc = -1;
    sf->macstyle = -1;
    sf->desired_row_cnt = default_fv_row_count; 
    sf->desired_col_cnt = default_fv_col_count;
    sf->display_antialias = default_fv_antialias;
    sf->display_bbsized = default_fv_bbsized;
    sf->display_size = -default_fv_font_size;
    sf->pfminfo.winascent_add = sf->pfminfo.windescent_add = true;
    sf->pfminfo.hheadascent_add = sf->pfminfo.hheaddescent_add = true;
    sf->pfminfo.typoascent_add = sf->pfminfo.typodescent_add = true;
    if ( TTFFoundry!=NULL )
        strncpy(sf->pfminfo.os2_vendor,TTFFoundry,4);
    else
        memcpy(sf->pfminfo.os2_vendor,"PfEd",4);
    //sf->for_new_glyphs = DefaultNameListForNewFonts();
    time(&now);
    sf->creationtime = sf->modificationtime = now;
return( sf );
}
