// Base class for a filesystem hook module which depends on filename.
// default behaviour filters on extension.
class FSModule
{
public:
	// call this function with true it the file we are looking at has already been processed by another
	// filter. Returns true if this filter wants to process it too.
	virtual bool CanAdd( bool already_added ) { return !already_added; };

	/*	If this module can process this file, this method will return a "virtual"
		file name for the processed file. Else, returns empty string.
		example : fname = "music.flac" => returns "music.flac$.wav" */
	virtual string GetVirtualFilename(const char* fname);

	// returns true if it is this module which generated this virtual filename
	virtual bool IsMyVirtualFilename(const char* fname);

	// returns real filename from virtual filename, or empty string if impossible
	virtual string GetRealFilename( const char* virtuel );

	//	Opens a file and returns a handle to it. NULL if error.
	virtual FileHandle*	Open( const char* path );

	// this is ".flac"
	virtual const char* GetRealExtension() { return ""; }

	// this is "$.wav"
	virtual const char* GetVirtualExtension() { return ""; }
	
	typedef vector<FSModule*> Array;
	
	virtual const char* Identity() { return "FSModule"; }
protected:
	// instanciate handle
	virtual FileHandle* CreateHandle(const char* name, const char* real_name) =0;
	
};

class FSModule_Default : public FSModule
{
public:
	virtual string GetVirtualFilename(const char* fname);
	bool IsMyVirtualFilename(const char* fname);
	string GetRealFilename( const char* virtuel );
	FileHandle* CreateHandle(const char* name, const char* real_name) { return new FileHandle_Default(name,real_name); }
	virtual const char* Identity() { return "FSModule_Default"; }
};

class FSModule_Flac : public FSModule
{
public:
	const char* GetRealExtension() { return ".flac"; }
	const char* GetVirtualExtension() { return ".wav"; }
	FileHandle* CreateHandle(const char* name, const char* real_name) { return new FileHandle_Flac(name,real_name); }
	virtual const char* Identity() { return "FSModule_Flac"; }
};

/*
class FSModule_Ogg : public FSModule
{
public:
	const char* GetRealExtension() { return ".ogg"; }
	const char* GetVirtualExtension() { return ".wav"; }
	FileHandle* CreateHandle(const char* name, const char* real_name) { return new FileHandle_Ogg(name,real_name); }
	virtual const char* Identity() { return "FSModule_Ogg"; }
};

class FSModule_MP3 : public FSModule
{
public:
	const char* GetRealExtension() { return ".mp3"; }
	const char* GetVirtualExtension() { return ".wav"; }
	FileHandle* CreateHandle(const char* name, const char* real_name) { return new FileHandle_MP3(name,real_name); }
	virtual const char* Identity() { return "FSModule_MP3"; }
};
*/
