/*
 * Luola - 2D multiplayer cavern-flying game
 * Copyright (C) 2003 Calle Laakkonen
 *
 * File        : levelfile.c
 * Description : Level loading
 * Author(s)   : Calle Laakkonen
 *
 * Luola is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Luola is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <dirent.h>
#include <string.h>
#include <stdlib.h>
#include <SDL.h>
#include <SDL_image.h>

#include "defines.h"
#include "console.h"
#include "levelfile.h"
#include "fs.h"
#include "game.h"
#include "font.h"

#if HAVE_LIBSDL_GFX
#include <SDL_rotozoom.h>
#endif

#ifdef CLOSEDIR_VOID
/* Fake a return value. */
# define CLOSEDIR(d) (closedir (d), 0)
#else
# define CLOSEDIR(d) closedir (d)
#endif

typedef struct {
  enum {Unknown,Artwork,Collision,Both,Config,Compact} filetype;
  LevelType type;
} LevelFormat;

/* Internally used functions */
static struct LevelFile *newlevel(char *levelname);
static void levels_post_process(struct LevelFile *list);
static LevelFormat getLevelFormat(char *typestr);
static char is_compatable(char *filename);

static char *level_name(char *fname) {
  int len,r;
  char *newname;
  len=strstr(fname,".lev.")-fname;
  if(len<=0) len=strstr(fname,".coll.")-fname;
  if(len<=0) len=strstr(fname,".conf.")-fname;
  if(len<=0) len=strstr(fname,".compact.")-fname;
	if(len<=0) len=strlen(fname);
  newname=malloc(len+1);
  for(r=0;r<len;r++)
    newname[r]=(fname[r]!='_')?fname[r]:' ';
  newname[len]=0;
  return newname;
}

/* Scan the level directory and make a list of levels */
int scan_levels(char user) {
  struct LevelFile *newentry;
  struct dirent *next;
  const char *dirname;
  LevelFormat format;
  char *levelname;
  DIR *reading;

  /* Which directory to read */
  if(user)
    dirname=getfullpath(USERLEVEL_DIRECTORY,"");
  else
    dirname=getfullpath(LEVEL_DIRECTORY,"");
  /* Open the directory */
  reading=opendir(dirname);
  if(!reading) {
    printf("Error ! Cannot open directory \"%s\" !\n(Check fs.c that you have correct directory definitions)\n",dirname);
    return 1;
  }
  /* Loop thru the directory */
  while((next=readdir(reading))!=NULL) {
    if(is_compatable(next->d_name)==0) continue;
    /* Get level type */
    levelname=level_name(next->d_name);
    format=getLevelFormat(next->d_name+strlen(levelname));
    if(format.filetype==Unknown) continue;
    /* Search the list for this level */
    newentry=game_settings.first_level;
    while(newentry) {
      if(strcmp(newentry->name,levelname)==0 && newentry->user==user) break;
      newentry=newentry->next;
    }
    /* If it is not found, add it there */
    if(newentry==NULL) {
      newentry=newlevel(levelname);
      newentry->type=format.type;
      newentry->user=user;
    } else
      free(levelname);
    /* Set the proper filenames */
    switch(format.filetype) {
      case Artwork:
        newentry->filename=strdup(next->d_name);
	break;
      case Collision:
        newentry->collmap=strdup(next->d_name);
	break;
      case Both:
        newentry->filename=strdup(next->d_name);
	newentry->collmap=newentry->filename;
	break;
      case Config:
        newentry->config=strdup(next->d_name);
	break;
      case Compact:
        newentry->filename=strdup(next->d_name);
	newentry->collmap=newentry->filename;
	newentry->compact=1;
	break;
      default: printf("Warning ! Reached default in switch(format.filetype) !\n"); break;
    }
  }
  CLOSEDIR(reading);
  levels_post_process(game_settings.first_level);
  game_settings.levels=game_settings.first_level;
  return 0;
}

/* Check if the file is a levelfile */
/* (By checking the filename */
static char is_compatable(char *filename) {
  char *extension;
  extension=rindex(filename,'.');
  if(extension==NULL) return 0;
  extension++;
  if(strcasecmp(extension,"bmp"))
  if(strcasecmp(extension,"pnm"))
  if(strcasecmp(extension,"xpm"))
  if(strcasecmp(extension,"xcf"))
  if(strcasecmp(extension,"pcx"))
  if(strcasecmp(extension,"gif"))
  if(strcasecmp(extension,"jpg"))
  if(strcasecmp(extension,"tif"))
  if(strcasecmp(extension,"png"))
  if(strcasecmp(extension,"lbm"))
  if(strcasecmp(extension,"lev"))
  if(strcasecmp(extension,"lcmap")) return 0; /* Not a level */
  return 1; /* This will do */
}

/* Get the type of the level file */
static LevelFormat getLevelFormat(char *typestr) {
  LevelFormat type;
  type.filetype=Unknown;
  if(strncasecmp(typestr,".lev.",5)==0) { type.type=LuolaLevel; type.filetype=Artwork; }
  else if(strncasecmp(typestr,".coll.",6)==0) { type.type=LuolaLevel; type.filetype=Collision; }
  else if(strncasecmp(typestr,".conf.",6)==0) { type.type=LuolaLevel; type.filetype=Config; }
  else if(strncasecmp(typestr,".compact.",9)==0) { type.type=LuolaLevel; type.filetype=Compact; }
  else if(strncasecmp(typestr,".8bit.",6)==0) { type.type=Luola8bitLevel; type.filetype=Both; }
  return type;
}

/* Create a new level entry */
/* Returns null if inapropriate */
static struct LevelFile *newlevel(char *levelname) {
  struct LevelFile *newentry=(struct LevelFile*)malloc(sizeof(struct LevelFile));
  newentry->prev=game_settings.last_level;
  newentry->next=NULL;
  newentry->prev=game_settings.last_level;
  newentry->name=levelname;
  newentry->filename=NULL;
  newentry->collmap=NULL;
  newentry->config=NULL;
  newentry->icon=NULL;
  newentry->compact=0;
  newentry->settings=NULL;
  if(game_settings.first_level==NULL) game_settings.first_level=newentry;
  if(game_settings.last_level) game_settings.last_level->next=newentry;
  game_settings.last_level=newentry;
  game_settings.levelcount++;
  return newentry;
}

/* Add some finishing touches to the level list */
static void levels_post_process(struct LevelFile *list) {
  while(list) { /* Load basic level settings */
    list->scalex=1;
    list->scaley=1;
    if(list->config || list->compact) {
      LevelInfo *levelinfo;
      if(list->compact) {
	LDAT *ldat;
	ldat=ldat_open_file(getfullpath(list->user?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,list->filename));
        levelinfo=quickload_levelsettings_rw(ldat_get_item(ldat,"SETTINGS",0),ldat_get_item_length(ldat,"SETTINGS",0));
	ldat_free(ldat);
      } else
        levelinfo=quickload_levelsettings(getfullpath(list->user?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,list->config));
      if(levelinfo->mainblock){
        if(levelinfo->mainblock->artwork) list->filename=levelinfo->mainblock->artwork;
        if(levelinfo->mainblock->collmap) list->collmap=levelinfo->mainblock->collmap;
        if(levelinfo->mainblock->name) list->name=levelinfo->mainblock->name;
        if(levelinfo->mainblock->scalex) list->scalex=levelinfo->mainblock->scalex;
        if(levelinfo->mainblock->scaley) list->scaley=levelinfo->mainblock->scaley;
        list->type=levelinfo->mainblock->type;
      } else {
        printf("Warning ! Level configuration file \"%s\" doesn't contain a mainblock !\n",list->config);
      }
      if(levelinfo->icon) {
        list->icon=levelinfo->icon;
      }
      free(levelinfo->mainblock);
      free(levelinfo);
    }
    list=list->next;
  }
}

/* Loads the level artwork file from a file */
/* The surface returned is in the same format as the screen */
SDL_Surface *load_level_art(struct LevelFile *level) {
  SDL_Surface *art,*tmpart;
  char smoothscale=1;
  if(level->compact==0) { /* Level files are seperate */
    if(level->filename==NULL) {
      printf("Level artwork filename not given !\n");
      exit(1);
    }
    tmpart=load_image((level->user)?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,level->filename,0,0);
  } else { /* Level files are stored in an LDAT file */
    LDAT *lf;
    SDL_RWops *rw;
    lf=ldat_open_file(getfullpath(level->user?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,level->filename));
    if(ldat_find_item(lf,"ARTWORK",0))
      rw=ldat_get_item(lf,"ARTWORK",0);
    else /* If the artwork is not present, get the collisionmap */
      rw=ldat_get_item(lf,"COLLISION",0);
    tmpart=load_image_rw(rw,0,0);
    ldat_free(lf);
  }
  if(level->scalex!=1 || level->scaley!=1) {
    if(level->settings)
      if(level->settings->mainblock) smoothscale=level->settings->mainblock->smooth_scale;
    art=scale_surface(tmpart,level->scalex,level->scaley,smoothscale);
    SDL_FreeSurface(tmpart);
  } else art=tmpart;
  return art;
}

/* Loads the level collisionmap from a file */
/* The surface is returned as it is, that is 8bit */
/* If the image is not 8 bit, NULL is returned */
SDL_Surface *load_level_coll(struct LevelFile *level) {
  SDL_Surface *coll,*tmpcoll;
  if(level->compact==0) { /* Level files are seperate */
    if(level->collmap==NULL) {
      printf("Level collisionmap filenmae not given !\n");
      exit(1);
    }
    tmpcoll=load_image((level->user)?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,level->collmap,0,-1);
  } else { /* Level files are stored in an LDAT file */
    LDAT *lf;
    SDL_RWops *rw;
    lf=ldat_open_file(getfullpath(level->user?USERLEVEL_DIRECTORY:LEVEL_DIRECTORY,level->filename));
    rw=ldat_get_item(lf,"COLLISION",0);
    tmpcoll=load_image_rw(rw,0,-1);
    ldat_free(lf);
  }
  if(level->scalex!=1 || level->scaley!=1) {
    coll=scale_surface(tmpcoll,level->scalex,level->scaley,0);
    SDL_FreeSurface(tmpcoll);
  } else coll=tmpcoll;
  return coll;
}


#ifndef PACKAGE_DATA_DIR	/* Test compile ? */
#define PACKAGE_DATA_DIR "../data"
#endif

/* Display this message when no levels are found */
void no_levels_found(void) {
  SDL_Rect r1,r2;
  SDL_Event event;
  char tmps[256];
  r1.x=10; r1.w=SCREEN_W-20;
  r1.y=10; r1.h=SCREEN_H-20;
  r2.x=r1.x+2; r2.y=r1.y+2;
  r2.w=r1.w-4; r2.h=r1.h-4;
  SDL_FillRect(screen,&r1,SDL_MapRGB(screen->format,255,0,0));
  SDL_FillRect(screen,&r2,SDL_MapRGB(screen->format,0,0,0));
  centered_string(screen,Bigfont,r2.y+10,"No levels found",font_color_red);
  centered_string(screen,Bigfont,r2.y+50,"No levels were found in the data directory",font_color_white);
  sprintf(tmps,"%s/levels/",PACKAGE_DATA_DIR);
  centered_string(screen,Bigfont,r2.y+80,tmps,font_color_white);
  centered_string(screen,Bigfont,r2.y+120,"Please download the level pack from the luola homepage",font_color_white);
  centered_string(screen,Bigfont,r2.y+150,"http://www.saunalahti.fi/~laakkon1/linux/luola/index.php",font_color_white);
  centered_string(screen,Bigfont,r2.y+180,"and unpack it into the level directory",font_color_white);
  centered_string(screen,Bigfont,r2.y+r2.h-50,"Press enter to exit",font_color_red);
  SDL_UpdateRect(screen,r1.x,r1.y,r1.w,r1.h);
  while(1) {
    SDL_WaitEvent(&event);
    if(event.type==SDL_KEYDOWN && event.key.keysym.sym==SDLK_RETURN) break;
    if(event.type==SDL_JOYBUTTONDOWN) break;
  }
  exit(0);
}

