/* This file is part of
 * ======================================================
 *
 *           LyX, The Document Processor
 *
 *           Copyright 1995 Matthias Ettrich
 *           Copyright 1995-2001 The LyX Team.
 *
 * ====================================================== */

#include <config.h>

#include "importer.h"
#include "converter.h"
#include "frontends/LyXView.h"
#include "funcrequest.h"

#include "bufferlist.h"
#include "support/filetools.h"
#include "frontends/Alert.h"
#include "gettext.h"
#include "BufferView.h"

#include "BoostFormat.h"

#include <algorithm>

using std::vector;
using std::find;

extern BufferList bufferlist;
extern void InsertAsciiFile(BufferView *, string const &, bool);


bool Importer::Import(LyXView * lv, string const & filename,
		      string const & format)
{
	string const displaypath = MakeDisplayPath(filename);
	ostringstream s1;
#if USE_BOOST_FORMAT
	s1 << boost::format(_("Importing %1$s...")) % displaypath;
#else
	s1 << _("Importing ") << displaypath << _("...");
#endif
	lv->message(STRCONV(s1.str()));

	string const lyxfile = ChangeExtension(filename, ".lyx");

	string loader_format;
	vector<string> loaders = Loaders();
	if (find(loaders.begin(), loaders.end(), format) == loaders.end()) {
		for (vector<string>::const_iterator it = loaders.begin();
		     it != loaders.end(); ++it) {
			if (converters.isReachable(format, *it)) {
				if (!converters.convert(0, filename, filename,
							format, *it))
					return false;
				loader_format = *it;
				break;
			}
		}
		if (loader_format.empty()) {
#if USE_BOOST_FORMAT
			Alert::alert(_("Cannot import file"),
				     boost::io::str(boost::format(_("No information for importing from %1$s"))
				   % formats.prettyName(format)));
#else
			Alert::alert(_("Cannot import file"),
				     _("No information for importing from ")
				     + formats.prettyName(format));
#endif
			return false;
		}
	} else
		loader_format = format;


	if (loader_format == "lyx") {
		Buffer * buffer = bufferlist.loadLyXFile(lyxfile);
		if (buffer)
			lv->view()->buffer(buffer);
	} else {
		lv->view()->buffer(bufferlist.newFile(lyxfile, string(), true));
		bool as_paragraphs = loader_format == "textparagraph";
		string filename2 = (loader_format == format) ? filename
			: ChangeExtension(filename,
					  formats.extension(loader_format));
		InsertAsciiFile(lv->view().get(), filename2, as_paragraphs);
		lv->dispatch(FuncRequest(LFUN_MARK_OFF));
	}

	// we are done
	lv->message(_("imported."));
	return true;
}


vector<Format const *> const Importer::GetImportableFormats()
{
	vector<string> loaders = Loaders();
	vector<Format const *> result =
		converters.getReachableTo(loaders[0], true);
	for (vector<string>::const_iterator it = loaders.begin() + 1;
	     it != loaders.end(); ++it) {
		vector<Format const *> r =
			converters.getReachableTo(*it, false);
		result.insert(result.end(), r.begin(), r.end());
	}
	return result;
}


vector<string> const Importer::Loaders()
{
	vector<string> v;
	v.push_back("lyx");
	v.push_back("text");
	v.push_back("textparagraph");
	return v;
}
