// -*- C++ -*-
/**
 * \file package.h
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Angus Leeming
 *
 * Full author contact details are available in file CREDITS.
 *
 * A store of the paths to the various different directoies used
 * by LyX. These paths differ markedly from one OS to another,
 * following the local Windows, MacOS X or Posix conventions.
 */
#ifndef LYX_PACHAGE_H
#define LYX_PACHAGE_H

#include "LString.h"

namespace lyx {

class Package;

/** Initialise package() with the command line data.
 *  This data is exactly as it was passed in the argv[] array.
 *
 *  @param command_line_arg0 argv[0], the name of the LyX executable
 *  as passed on the command line.
 *
 *  @param command_line_system_support_dir, the LyX system dir,
 *  as specified on the command line with "-sysdir <path to dir>".
 *
 *  @param command_line_user_support_dir, the LyX user dir,
 *  as specified on the command line with "-userdir <path to dir>".
 */
void init_package(string const & command_line_arg0,
		  string const & command_line_system_support_dir,
		  string const & command_line_user_support_dir);

/** Accessor to the global data.
 *  Asserts that init_package() has been called first.
 */
Package const & package();

class Package {
public:
	/// Default constructor does not lead to the paths being set.
	Package();

	/** Called by init_package, above.
	 *  All paths will be initialized.
	 */
	Package(string const & command_line_arg0,
		string const & command_line_system_support_dir,
		string const & command_line_user_support_dir);

	/** The directory containing the LyX executable.
	 */
	string const & binary_dir() const;

	/** The path to the system-level support files
	 *  we're actually going to use.
	 */
	string const & system_support() const;

	/** The path to the autogenerated support files
	 *  when running in-place.
	 */
	string const & build_support() const;

	/** The path to the user-level support files.
	 */
	string const & user_support() const;

	/** The user_support directory was set explicitly using either
	 *  the -userdir command line switch or
	 *  the LYX_USERDIR_13x environment variable.
	 */
	bool explicit_user_support() const;

	/** The path to the locale directory.
	 */
	string const & locale_dir() const;

	/** The default document directory.
	 *  Can be reset by LyXRC.
	 */
	string & document_dir() const;

	/** The path to the temporary directory.
	 *  (Eg /tmp on *nix.)
	 *  Can be reset by LyXRC.
	 */
	string & temp_dir() const;

	/** Used when setting the user_support directory.
	 *  Used also when expanding "~/" or contracting to "~/". (filetools.C)
	 *  Used by the XForms file dialog.
	 *  Used in emergencyWrite (bufferlist.C) as one possible location
	 *  for the dump.
	 */
	string const & home_dir() const;

private:
	string binary_dir_;
	string system_support_dir_;
	string build_support_dir_;
	string user_support_dir_;
	string locale_dir_;
	mutable string document_dir_;
	mutable string temp_dir_;
	string home_dir_;
	bool explicit_user_support_dir_;
};


inline
Package::Package() {}

inline
string const & Package::binary_dir() const
{
	return binary_dir_;
}

inline
string const & Package::system_support() const
{
	return system_support_dir_;
}

inline
string const & Package::build_support() const
{
	return build_support_dir_;
}

inline
string const & Package::user_support() const
{
	return user_support_dir_;
}

inline
bool Package::explicit_user_support() const
{
	return explicit_user_support_dir_;
}

inline
string const & Package::locale_dir() const
{
	return locale_dir_;
}

inline
string & Package::document_dir() const
{
	return document_dir_;
}

inline
string & Package::temp_dir() const
{
	return temp_dir_;
}

inline
string const & Package::home_dir() const
{
	return home_dir_;
}

} // namespace lyx

#endif // LYX_PACHAGE_H
