/**
 * \file lastfiles.C
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author Lars Gullik Bjnnes
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "lastfiles.h"
#include "debug.h"

#include <boost/filesystem/operations.hpp>

#include <algorithm>
#include <fstream>
#include <iterator>

namespace fs = boost::filesystem;

using std::copy;
using std::endl;
using std::find;
using std::getline;
using std::string;
using std::ifstream;
using std::ofstream;
using std::ostream_iterator;


LastFiles::LastFiles(string const & filename, bool st, unsigned int num)
	: dostat(st)
{
	setNumberOfFiles(num);
	readFile(filename);
}


void LastFiles::setNumberOfFiles(unsigned int no)
{
	if (0 < no && no <= ABSOLUTEMAXLASTFILES)
		num_files = no;
	else {
		lyxerr << "LyX: lastfiles: too many files\n"
			"\tdefault (=" << int(DEFAULTFILES)
		       << ") used." << endl;
		num_files = DEFAULTFILES;
	}
}


void LastFiles::readFile(string const & filename)
{
	// we will not complain if we can't find filename nor will
	// we issue a warning. (Lgb)
	ifstream ifs(filename.c_str());
	string tmp;

	while (getline(ifs, tmp) && files.size() < num_files) {
		if (dostat && !fs::exists(tmp))
				continue;
		files.push_back(tmp);
	}
}


void LastFiles::writeFile(string const & filename) const
{
	ofstream ofs(filename.c_str());
	if (ofs) {
		copy(files.begin(), files.end(),
		     ostream_iterator<string>(ofs, "\n"));
	} else
		lyxerr << "LyX: Warning: unable to save LastFiles: "
		       << filename << endl;
}


void LastFiles::newFile(string const & file)
{
	// If file already exist, delete it and reinsert at front.
	Files::iterator it = find(files.begin(), files.end(), file);
	if (it != files.end())
		files.erase(it);
	files.push_front(file);
	if (files.size() > num_files)
		files.pop_back();
}


string const LastFiles::operator[](unsigned int i) const
{
	if (i < files.size())
		return files[i];
	return string();
}
