// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.IO;
using System.Collections.Generic;

namespace Anculus.Core
{
	public sealed class MemoryConfigurationSection : AbstractConfigurationSection
	{
		private Dictionary<string, MemoryObject> _objects;
		
		public MemoryConfigurationSection (IConfigurationSection parent, string name)
			: base (parent, name)
		{
			_objects = new Dictionary<string, MemoryObject> ();
		}

		public MemoryConfigurationSection ()
			: base ()
		{
			_objects = new Dictionary<string, MemoryObject> ();
		}
		
		public override IConfigurationSection this [string name]
		{
			get { throw new NotSupportedException (); }
		}

		internal Dictionary<string, MemoryObject> Objects
		{
			get { return _objects; }
		}

		public override bool ContainsKey (string name)
		{
			if (name == null)
				throw new ArgumentNullException ("name");

			return _objects.ContainsKey (name);
		}
		
		public override IEnumerable<string> Keys
		{
			get { return _objects.Keys; }
		}

		public override void Remove (string name)
		{
			if (name == null)
				throw new ArgumentNullException ("name");
			
			_objects.Remove (name);
		}

		protected internal override void SetInternal (string name, ConfigurationObjectType type, object value)
		{
			if (name == null)
				throw new ArgumentNullException ("name");

			MemoryObject mem;
			if (_objects.TryGetValue (name, out mem)) {
				mem.ObjectType = type;
				mem.Value = value;
			} else {
				mem = new MemoryObject (type, value);
				_objects.Add (name, mem);
			}
		}

		protected internal override object GetInternal (string name, ConfigurationObjectType type)
		{
			if (name == null)
				throw new ArgumentNullException ("name");

			MemoryObject mem;
			if (_objects.TryGetValue (name, out mem)) {
				if (mem.ObjectType == ConfigurationObjectType.Object || mem.ObjectType == type)
					return mem.Value;
			}
			return null;
		}
		
		public override ConfigurationObjectType GetObjectType (string name)
		{
			if (name == null)
				throw new ArgumentNullException ("name");

			MemoryObject mem;
			if (_objects.TryGetValue (name, out mem))
				return mem.ObjectType;
			return ConfigurationObjectType.Unknown;
		}

		protected internal override int CheckType (string name, ConfigurationObjectType type)
		{
			//TODO: if generic type; check generic params (ret -3 ?)
			MemoryObject mem;
			if (_objects.TryGetValue (name, out mem)) {
				if (mem.ObjectType == type) {
					return 0;
				} else {
					Log.Warn ("Incompatible type {0} (expected {1}).", mem.ObjectType, type);
					return -2;
				}
			}
			return -1;
		}

		protected override IConfigurationSection CreateChildSection (string name)
		{
			return new MemoryConfigurationSection (this, name);
		}
	}
}