// 
// Copyright (c) 2006-2008 Ben Motmans
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// Author(s):
//    Ben Motmans <ben.motmans@gmail.com>
//

using System;
using System.Threading;
using System.Collections.Generic;

using Anculus.Core;

namespace Anculus.Gui
{
	public abstract class AbstractDispatcher : IDispatcher
	{
		protected int threadId;
		protected Queue<IDispatchContainer> queue;
		
		protected object sync = new object ();

		public virtual void Initialize ()
		{
			threadId = Thread.CurrentThread.ManagedThreadId;
			queue = new Queue<IDispatchContainer> ();
		}

		public virtual void Dispatch (VoidDispatchHandler callback)
		{
			if (callback == null)
				throw new ArgumentNullException ("callback");

			VoidDispatchContainer container = new VoidDispatchContainer (callback, false);
			QueueContainer (container);
		}

		public virtual void Dispatch (ObjectDispatchHandler callback, object obj)
		{
			if (callback == null)
				throw new ArgumentNullException ("callback");

			ObjectDispatchContainer container = new ObjectDispatchContainer (callback, obj, false);
			QueueContainer (container);
		}

		public virtual void SyncDispatch (VoidDispatchHandler callback)
		{
			if (callback == null)
				throw new ArgumentNullException ("callback");

			VoidDispatchContainer container = new VoidDispatchContainer (callback, true);
			QueueContainer (container);
		}

		public virtual void SyncDispatch (ObjectDispatchHandler callback, object obj)
		{
			if (callback == null)
				throw new ArgumentNullException ("callback");

			ObjectDispatchContainer container = new ObjectDispatchContainer (callback, obj, true);
			QueueContainer (container);
		}

		public virtual bool IsDispatchRequired ()
		{
			return threadId != Thread.CurrentThread.ManagedThreadId;
		}

		protected virtual void QueueContainer (IDispatchContainer container)
		{
			if (IsDispatchRequired ()) {
				if (container.IsSynchronous) {
					lock (container) {
						lock (sync)
							queue.Enqueue (container);
						ExecuteProcessQueue ();
						Monitor.Wait (container);
					}
				} else {
					lock (sync)
						queue.Enqueue (container);
					ExecuteProcessQueue ();
				}
			} else {
				container.Execute ();
			}
		}
		
		protected abstract void ExecuteProcessQueue ();

		protected virtual void ProcessQueue ()
		{
			int count = GetCount ();
			if (count == 0) {
				return;
			} else {
				IDispatchContainer container = null;
				while (count > 0) {
					lock (sync)
						container = queue.Dequeue ();

					container.Execute ();
					if (container.Exception != null)
						Log.Error (container.Exception, "An error occured inside a dispatched method.");

					count = GetCount ();
				}
			}
		}

		protected virtual int GetCount ()
		{
			int count = 0;
			lock (sync)
				count = queue.Count;
			return count;
		}
	}
}