/*
   Copyright (C) 2003 Commonwealth Scientific and Industrial Research
   Organisation (CSIRO) Australia

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   - Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

   - Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

   - Neither the name of CSIRO Australia nor the names of its
   contributors may be used to endorse or promote products derived from
   this software without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
   PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE ORGANISATION OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/**
 * \file xtag.h
 *
 * XTag: A trivial parser for xml-like tags.
 *
 * API Version: 1.0
 * Author: Conrad Parker, 2003
 */

#ifndef __XTAG_H__
#define __XTAG_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * An opaque object representing an XML element.
 * XTag objects are structured recursively, each containing child elements
 * as other XTag objects.
 */
typedef struct _XTag XTag;

/**
 * Instantiate a new XTag object from a memory buffer of character data.
 * \param s The memory buffer to parse, treated as a C string.
 * \param n The maximum number of bytes to parse.
 * \returns A newly created XTag object
 * \retval NULL The tag could not be parsed by XTag
 */
XTag * xtag_new_parse (const char * s, int n);

/**
 * Retrieve the name of an element.
 * \param xtag An XTag object
 * \returns The element name
 * \retval NULL \a xtag is NULL.
 */
const char * xtag_get_name (XTag * xtag);

/**
 * Retrieve the parsed character data contained by an element.
 * \param xtag An XTag object
 * \returns The parsed character data contained by \a xtag.
 * \retval NULL \a xtag is NULL or contains no parsed character data.
 */
const char * xtag_get_pcdata (XTag * xtag);

/**
 * Retrieve an explicitly named attribute from an XTag
 * \param xtag An XTag object
 * \param attribute The attribute to retrieve
 * \returns The attribute value
 * \retval NULL \a xtag is NULL, or contains no such attribute.
 */
const char * xtag_get_attribute (XTag * xtag, const char * attribute);

/**
 * Retrieve the first child element with an explicit name from an XTag
 * \param xtag An XTag object
 * \param name The name of the child to retrieve, or NULL to retrieve the
 * first child regardless of name.
 * \returns An XTag object representing the desired child, if found.
 * \retval NULL \a xtag is NULL, or no such child exists.
 */
XTag * xtag_first_child (XTag * xtag, const char * name);

/**
 * Retrieve the next explicitly named child element from an XTag.
 * \param xtag An XTag object
 * \param name The name of the child to retrieve, or NULL to retrieve the
 * next child regardless of name.
 * \returns An XTag object representing the desired child, if found.
 * \retval NULL \a xtag is NULL, or no such child exists.
 */
XTag * xtag_next_child (XTag * xtag, const char * name);

/**
 * Free an XTag object.
 * \param xtag An XTag object
 * \returns NULL
 */
XTag * xtag_free (XTag * xtag);

/**
 * Write an XTag into a memory buffer, in XML format.
 * \param buf The memory buffer to write into.
 * \param n The maximum number of bytes to write.
 * \param xtag An XTag object
 * \returns The total length in bytes of the printed XTag. This may be
 * greater than \a n if the given buffer is not large enough to fully
 * print \a xtag.
 * \note The return value of xtag_snprint() conforms to C99-style snprintf()
 * semantics on all systems. xtag_snprint() may be called with \a buf = NULL
 * or \a n = 0 to query the buffer length required to print the given XTag.
 */
int xtag_snprint (char * buf, int n, XTag * xtag);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __XTAG_H__ */
