/*-------------------------------------------------------------------------------

Copyright (c) 2001 CerebraSoft

Module Name:

	broadcast.cpp

Abstract:

	Functions to broadcast performance information to mod backhand server.

License:

   All rights reserved.

   This product includes software developed at The Center for
   Networking and Distributed Systems at The Johns Hopkins University
   for use in the Backhand project (http://www.cnds.jhu.edu/backhand).
     Creator: Theo Schlossnagle 
     Guidance: Yair Amir

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer. 
 
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
 
  3. All advertising materials mentioning features or use of this
     software must display the following acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  4. The names "Backhand Broadcaster" and "NT Backhand Broadcaster" must
     not be used to endorse or promote products derived from this
     software without prior written permission. For written permission,
     please contact www.cerebrasoft.com.
 
  5. Products derived from this software may not be called "Backhand Broadcaster"
     or "NT Backhand Broadcaster" nor may "Backhand Broadcaster" or
     "NT Backhand Broadcaster" appear in their names without prior written
     permission. For written permission, please contact www.cerebrasoft.com.
 
  6. Redistributions of any form whatsoever must retain the following
     acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
  ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE.
---------------------------------------------------------------------------------*/

#include <string>

using namespace std;
#include <winsock2.h>
#include "broadcast.h"
#include "registry.h"
#include "NTServerStats.h"
#include "NTStats.h"
#include "eventLog.h"

/*-------------------------------------------------------------------------------
	Global Defines
---------------------------------------------------------------------------------*/

#define LoadCPU_ITERS 6666		// number of times to loop in loadCPU
#define defArribaThreads 12		// default number of threads to use in calculating arriba
#define appName "NT Backhand Broadcaster"			// event log app name

/*-------------------------------------------------------------------------------
	Global Variables
---------------------------------------------------------------------------------*/

HANDLE broadcastThread;		// handle to broadcast thread
bool keepRunning = false;	// flag to stop broadcast thread
DWORD defDelay = 0;			// time to delay broadcast init & start.
WSAData wsData;				// Holds windows socket implementation data
SOCKET bcstSocket;			// broadcast socket
SOCKET mcstSocket;			// multicast socket
int sockTrue = 1;			// setsockopt requires a pointer to an int valued at 1 for true
int sockFalse = 0;			// setsockopt requires a pointer to an int values at 0 for false;
DWORD unUsed = 0;				// some socket functions require an output that we don't use.

typedef struct {			// simple structure to hold contact information
	char hostname[40];		// hostname we are referred to as
	sockaddr_in contact;	// ip address to use in contacting us
	sockaddr_in sendTo;		// ip address to muticast / broadcast to.
	int ttl;				// ttl to use in multicasting stats
	bool isMulticast;		// true if we should multicast, false otherwise.
	bool skip;				// if true, we do not broadcast this one.
} contactInfo;

contactInfo *transmitTo;	// pointer to an array of contact info.
DWORD numTransmitTo;		// how big is the above array.

/*-------------------------------------------------------------------------------
	Function Definitions
---------------------------------------------------------------------------------*/

DWORD WINAPI loadCPU(LPVOID param);
DWORD WINAPI broadcast(LPVOID param);
int calcArriba();
void transmit(serverstat *stats);
bool setupSockets();
void closeSockets();

/*-------------------------------------------------------------------------------

	startBroadcasting(DWORD delay) - begins broadcasting performance data to network
	delay specifies the time in seconds in which to delay broadcasting.  It is only
	the default delay time, and can be overridden with a registry entry for Delay.

---------------------------------------------------------------------------------*/

bool startBroadcasting(DWORD delay){

	keepRunning = true;
	defDelay = delay;

	broadcastThread = CreateThread(NULL, 0, &broadcast, NULL, 0, NULL);
	if(broadcastThread == NULL){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Attempt to start broadcast thread failed!");
		keepRunning = false;
		closeStats();
		closeSockets();
		return false;	// returns false if thread cannot be created.
	}
	return true;
}

/*-------------------------------------------------------------------------------

	stopBroadcasting - stops broadcasting performance data to network

---------------------------------------------------------------------------------*/

void stopBroadcasting(){

	keepRunning = false;
	if(broadcastThread != NULL)
		WaitForSingleObject(broadcastThread, 10000);
	closeStats();
	closeSockets();
}

/*-------------------------------------------------------------------------------

	broadcast(LPVOID sleepTime) - Broadcast perfomance information onto the network to
	mod backhand.
---------------------------------------------------------------------------------*/

DWORD WINAPI broadcast(LPVOID param){

	serverstat stats;

		// get delay from registry
	int delay = getRegDWORD(HKEY_LOCAL_MACHINE, regRoot, "Delay", defDelay);

		// guarantee delay is not negative.
	if(delay < 0)
		delay = defDelay;

		// delay startup
	Sleep(delay * 1000);

		// get arriba from registry
	int arriba = getRegDWORD(HKEY_LOCAL_MACHINE, regRoot, "Arriba", 0);
	
		// if arriba == 0, registry didn't have it
	if(arriba < 1){
		logAppMsg(appName, EVENTLOG_INFORMATION_TYPE, "Arriba not found in registry.  Calculating Arriba.");
		arriba = calcArriba();	// calculate arriba
		regSetDword(HKEY_LOCAL_MACHINE, regRoot, "Arriba", arriba);	// store in registry
	}

		// if arriba still == 0, error in getting from reg, and could not calculate, exit
	if(arriba < 1){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Failure attempting to calculate Arriba!");
		broadcastThread = NULL;
		return 0;
	}

		// setup stats structure
	stats.arriba = htonl(arriba);	// convert to network order

		// convert to network order
	stats.ncpu = htonl(getRegDWORD(HKEY_LOCAL_MACHINE, regRoot, "numCPU", 1));  

		// ignored stats, set to 0 - no need to convert network order, all bytes 0
	stats.aservers = 0;
	stats.nservers = 0;
	stats.load_hwm = 0;
	stats.mtime = 0;
	stats.numbacked = 0;
	stats.tatime = 0;

	if(setupSockets() == false){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "CANNOT broadcast mod_backhand statistics!");
		broadcastThread = NULL;	
		return 0;
	}

		// initialize NT statistics
	if(initStats() == false){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Failure attempting to initialize statistics system!");
		broadcastThread = NULL;
		return 0;
	}

	logAppMsg(appName, EVENTLOG_INFORMATION_TYPE, "Broadcasting mod_backhand stastics.");

	while(keepRunning){
		if(getStats(&stats)){	// get performance stats.  (returns true if successful)
			htonss(stats);		// convert to network byte order
			transmit(&stats);	// transmit stats.
		}
		Sleep(1000);			// wait 1 second and do it again
	}
	return 0;
}

/*-------------------------------------------------------------------------------
	transmit(serverstat) - transmit data to network.
---------------------------------------------------------------------------------*/

void transmit(serverstat *stats){

	for(DWORD i = 0; i < numTransmitTo; i++){
		if(transmitTo[i].skip == false){

				// set the appropriate hostname and contact info.
			strcpy(stats->hostname, transmitTo[i].hostname);
			memcpy(&stats->contact, &transmitTo[i].contact, sizeof(stats->contact));

				// is it multicast, or broadcast
			if(transmitTo[i].isMulticast){	// multicast stats
					// attempt change TTL for this transmission

				if(WSAIoctl(mcstSocket, SIO_MULTICAST_SCOPE, &transmitTo[i].ttl, sizeof(transmitTo[i].ttl), NULL, 0, &unUsed, NULL, NULL) == 0){
						// successful, transmit data
					sendto(mcstSocket, (const char *) stats, sizeof(serverstat), 0, (LPSOCKADDR) &transmitTo[i].sendTo, sizeof(sockaddr_in));
				}
			}else{	// broadcast stats

				sendto(bcstSocket, (const char *) stats, sizeof(serverstat), 0, (LPSOCKADDR) &transmitTo[i].sendTo, sizeof(sockaddr_in));
			}
		} // closes if(skip == false)
	} // closes for loop
}

/*-------------------------------------------------------------------------------

	calcArriba - Calculate the Arriba for this machine.
	by default the arriba is calculated for 12 threads.  The number of threads
	can be modified by a registry setting.

	returns the arriba for the machine.  Returns a negative number if there was
	an error
---------------------------------------------------------------------------------*/

int calcArriba(){

	time_t start, finish;

		// get the number of threads we should use from the registry
	DWORD numThreads = getRegDWORD(HKEY_LOCAL_MACHINE, regRoot, "ArribaThreads", defArribaThreads);
		// make sure we do at least 1 thread
	if(numThreads < 1)
		numThreads = 1;

		// allocate an array to hold each threads handle
	HANDLE *threads = (HANDLE *) malloc(numThreads * sizeof(HANDLE));

		// we could not allocate memory, return an error
	if(threads == NULL)
		return -1;

		// store start time.
	time(&start);
	
		// create the threads, and let em bog down the cpu(s)
	for(DWORD i = 0; i < numThreads; i++)
		threads[i] = CreateThread(NULL, 0, &loadCPU, NULL, 0, NULL);

		// wait for all the threads to finish
	WaitForMultipleObjects(numThreads, threads, true, INFINITE);
		// get finish time
	time(&finish);

		// calculate total time and multiply by 1000, this is the arriba.
	return (int) (LoadCPU_ITERS * 101 / difftime(finish, start)) * 1000;
}

/*-------------------------------------------------------------------------------

	loadCPU - a long running function that loads up the cpu(s) in the
	system.  We determine how fast a machine is by seeing how fast it can do 
	this calculation for many threads.

	This function copied (and modified slightly) from mod_backhand.
	See Backhand_Broadcast.h for original notice.
---------------------------------------------------------------------------------*/

DWORD WINAPI loadCPU(LPVOID param){

	double z[101] = {0.0};

	for(int i=0; i < LoadCPU_ITERS; i++){
		for(int j=0; j<101; j++){
			z[j] = (!z[j])?j: (z[(i-1)%101]+1.0)*(z[(i+1)%101]+1.0);
			if(z[j] > 100000000.0 || z[j]<0) z[j]=3.0;
		}
	}
	return 0;
}

/*-------------------------------------------------------------------------------

	setupSockets - sets up the sockets for broadcasting stats.

---------------------------------------------------------------------------------*/

bool setupSockets(){

	char *keyName;
	string subKey;
	char *tmp;
	DWORD keyCnt;
	int structLen;


		// initialize winsock DLL
	if(WSAStartup(0x0202, &wsData) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Could not initialize winsock!");
		return false;
	}

	bcstSocket = WSASocket(AF_INET, SOCK_DGRAM, 0, NULL, 0, WSA_FLAG_MULTIPOINT_C_LEAF | WSA_FLAG_MULTIPOINT_D_LEAF);

	if(bcstSocket == INVALID_SOCKET){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Unable to create broadcast socket!");
		closeSockets();
		return false;
	}

		// allow broadcasting on the socket.
	if(setsockopt(bcstSocket, SOL_SOCKET, SO_BROADCAST, (const char *) &sockTrue, sizeof(int)) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Cannot enable socket for broadcasting!");
		closeSockets();
		return false;
	}

		// allow the socket to be re-used.
	if(setsockopt(bcstSocket, SOL_SOCKET, SO_REUSEADDR, (const char *) &sockTrue, sizeof(int)) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Cannot enable broadcast socket re-use!");
		closeSockets();
		return false;
	}


	mcstSocket = WSASocket(AF_INET, SOCK_DGRAM, 0, NULL, 0, WSA_FLAG_MULTIPOINT_C_LEAF | WSA_FLAG_MULTIPOINT_D_LEAF);

	if(mcstSocket == INVALID_SOCKET){

		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Unable to create multicast socket!");
		closeSockets();
		return false;
	}

			// allow broadcasting on the socket.
	if(setsockopt(mcstSocket, SOL_SOCKET, SO_BROADCAST, (const char *) &sockTrue, sizeof(int)) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Cannot enable socket for multicasting!");
		closeSockets();
		return false;
	}

		// allow the socket to be re-used.
	if(setsockopt(mcstSocket, SOL_SOCKET, SO_REUSEADDR, (const char *) &sockTrue, sizeof(int)) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Cannot enable multicast socket re-use!");
		closeSockets();
		return false;
	}
	
		// disable loopback 
    if(WSAIoctl(mcstSocket, SIO_MULTIPOINT_LOOPBACK, &sockFalse, sizeof(int), NULL, 0, &unUsed, NULL, NULL) != 0){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Cannot disable multicast socket loopback!");
		closeSockets();
		return false;
	}

		// determine number of registry subkeys
	keyCnt = regSubKeyCnt(HKEY_LOCAL_MACHINE, regRoot);
	if(keyCnt < 1){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "No broadcast data in registry");
		closeSockets();
		return false;
	}

		// allocate array to hold contact and related information
	transmitTo = (contactInfo *) malloc(keyCnt * sizeof(contactInfo));
	if(transmitTo == NULL){
		logAppMsg(appName, EVENTLOG_ERROR_TYPE, "Unable to allocate memory for transmit array!");
		closeSockets();
		return false;
	}

		// remember our array size
	numTransmitTo = keyCnt;

		// loop through all registry sub keys
	for(int i = 0; i < keyCnt; i++){

			// get subkey name
		keyName = regEnumKeys(HKEY_LOCAL_MACHINE, regRoot, i);
		if(keyName != NULL){

				// create registry subkey path
			subKey.assign(regRoot).append("\\").append(keyName);

				// get registry information
			tmp = getRegString(HKEY_LOCAL_MACHINE, subKey.c_str(), "HostName", NULL, false);
			if(tmp != NULL){
				strncpy(transmitTo[i].hostname, tmp, 39);
				transmitTo[i].hostname[39] = NULL;	// make sure to null terminate string.
				free(tmp);		// don't need hostName anymore, get rid of it.
			}else{
				strncpy(transmitTo[i].hostname, "NT Remote Server", 39);
			}

				// initialize some variables (as required)
			transmitTo[i].contact.sin_family = AF_INET;
			structLen = sizeof(sockaddr_in);
			transmitTo[i].skip = false;

				// get data from registry.  No need to convert to network byte order, WSAStringToAddress already does.
			tmp = getRegString(HKEY_LOCAL_MACHINE, subKey.c_str(), "ContactIP", NULL, false);
			if(tmp != NULL){
				if(WSAStringToAddress(tmp, AF_INET, NULL, (LPSOCKADDR) &transmitTo[i].contact, &structLen) == SOCKET_ERROR){
					logAppMsg(appName, EVENTLOG_ERROR_TYPE, 3, "ContactIP for ", keyName, " incorrect format.  Format should be aaa.bbb.ccc.ddd");
					transmitTo[i].skip = true;
				}
				free(tmp);
			}else{
				logAppMsg(appName, EVENTLOG_ERROR_TYPE, 3, "ContactIP for ", keyName, " not found!");
				transmitTo[i].skip = true;
			}

				// get contact port, and convert to network order.
			transmitTo[i].contact.sin_port = htons(getRegDWORD(HKEY_LOCAL_MACHINE, subKey.c_str(), "ContactPort", 80));

				// initialize some variables (as required)
			transmitTo[i].sendTo.sin_family = AF_INET;
			structLen = sizeof(sockaddr_in);

				// get data from registry.  No need to convert to network byte order, WSAStringToAddress already does.
			tmp = getRegString(HKEY_LOCAL_MACHINE, subKey.c_str(), "SendIP", NULL, false);
			if(tmp != NULL){
				if(WSAStringToAddress(tmp, AF_INET, NULL, (LPSOCKADDR) &transmitTo[i].sendTo, &structLen) == SOCKET_ERROR){
					logAppMsg(appName, EVENTLOG_ERROR_TYPE, 3, "SendIP for ", keyName, " incorrect format.  Format should be aaa.bbb.ccc.ddd");
					transmitTo[i].skip = true;
				}
				free(tmp);

					// determine if we are broadcasting or multicasting.
				DWORD tcm = transmitTo[i].sendTo.sin_addr.s_addr >> 24;

				if(tcm >=224 && tcm <=239){
					transmitTo[i].isMulticast = true;
				}else{
					transmitTo[i].isMulticast = false;
				}
			}else{
				logAppMsg(appName, EVENTLOG_ERROR_TYPE, 3, "SendIP for ", keyName, " not found!");
				transmitTo[i].skip = true;
			}
				// get port & convert to network order
			transmitTo[i].sendTo.sin_port = htons(getRegDWORD(HKEY_LOCAL_MACHINE, subKey.c_str(), "SendPort", 4445));
				// get TTL.  Using locally, no need to convert to network order
			transmitTo[i].ttl = getRegDWORD(HKEY_LOCAL_MACHINE, subKey.c_str(), "SendTTL", 1);

				// done with keyname, get rid of it.
			free(keyName);

		}else {		// keyname was null;
			break;	// exit loop
		}
	}	// ends for loop
	return true;
}

/*-------------------------------------------------------------------------------

	closeSockets() - returns all socket resources to the system.

---------------------------------------------------------------------------------*/

void closeSockets(){
	if(bcstSocket != NULL && bcstSocket != INVALID_SOCKET){
		closesocket(bcstSocket);
		bcstSocket = NULL;
	}
	if(mcstSocket != NULL && mcstSocket != INVALID_SOCKET){
		closesocket(mcstSocket);
		mcstSocket = NULL;
	}
	// DO NOT PERFORM A WSACleanup!!
	/* If this is running inside of IIS calling WSACleanup will cause windows sockets DLL
	   to be unloaded and unavailable for ALL THREADS including IIS threads!  We don't
	   want to take down IIS (with a huge crash) if for some reason we can't broadcast
	   our stats.  If we are not in IIS, eventually the DLL should be unloaded and collected
	   through normal Win32 application cleanup when this app ends.
	*/
}