/*
 * ====================================================================
 * Copyright (c) 1995-1998 Lyonel VINCENT.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by the Apache Group
 *    for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * mod_access_ldap.c: LDAP-based access checking for Apache
 *
 * Lyonel VINCENT (vincent@trotek.ec-lyon.fr)
 * changes by Sam Alcoff (alcoff@survival.net) to compile with
 *   Apache 1.3
 * changes by Rick Perry (perry@ece.vill.edu): bugs fixes and code
 *   cleanups.
 *
 * Version 1.4 (Oct 1998)
 *
 * based on the mod_auth_ldap.c module written by
 * Norman RICHARDS (orb@cs.utexas.edu)
 *
 * =============================================
 *
 * For more information on LDAP, see the LDAPWorld web 
 * page at:  http://www.critical-angle.com/ldapworld/
 *
 * =============================================
 *
 * The goal of this module is to allow access checking 
 * through user information stored in an LDAP directory. 
 *
 * This module works very similarly to other access checking
 * modules.
 * 
 * To use the module, you need to add this file to your 
 * apache src directory.  Add the following line to your
 * Configuration file:
 *
 *     Module ldap_module     mod_ldap.o
 *
 * You will also need to add the appropriate locations
 * for your ldap/lber libraries and includes to 
 * EXTRA_LIBS and EXTRA_INCLUDES.
 *
 * =============================================
 *
 * The following directives can be used in your access.conf
 * file in the same way as you would configure the standard
 * authentication modules.
 *
 * LDAPAuth <flag>
 *
 * LDAPServer <LDAP URL>
 *
 *   This sets the LDAP server to be used.  The ldap url 
 *   is of the form "ldap://yourhost.com/"
 *
 * LDAPBindName <Distinguished Name>
 * LDAPBindPass <password>
 *
 *   If the module needs to authenticate itself, enter the user 
 *   name and password here.
 *
 * The LDAP-specific require directive for <limit> is:
 *
 * require filter <LDAP search filter>
 *
 *   The search filter is in LDAP standard format as specified by
 * RFC1960
 *
 * =============================================
 * example:
 *	LDAPServer	ldap://x500.hp.com/
 *	require	filter	(|(&(ou=ENSD)(l=Grenoble))(ou=HP Labs))
 */

#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"

#include "lber.h"
#include "ldap.h"

#include "ap_vercompat.h"

module ldap_module;

/* --------------------------------- */

typedef struct  {
    char *host;
    int  port;

    char *base;
    char *bindname;
    char *bindpass;
    int ldap_auth;

    char *userid_attr;
    char *userpassword_attr;

    LDAP *ld;
} ldap_config_struct;


void *create_ldap_dir_config(pool *p, char *d) {
    ldap_config_struct *conf;

    conf = (ldap_config_struct *) ap_pcalloc(p, sizeof(ldap_config_struct));

    conf->host       = "localhost";
    conf->port       = LDAP_PORT;
 
    conf->base       = NULL;
    conf->bindname   = NULL;
    conf->bindpass   = NULL;

    conf->userid_attr		= "userid";
    conf->userpassword_attr	= NULL;

    conf->ldap_auth  = 0;

    conf->ld         = NULL;

    return conf;
}

/* --------------------------------- */

const char *set_ldap_auth(cmd_parms *cmd, ldap_config_struct *conf, int arg)
{
  conf->ldap_auth = arg;
  return NULL;
}

const char *set_ldap_base(cmd_parms *cmd, ldap_config_struct *conf, char *base)
{
  conf->base=ap_pstrdup(cmd->pool, base);
  return NULL;
}

const char *set_ldap_useridattr(cmd_parms *cmd, ldap_config_struct *conf, char *attr)
{
  conf->userid_attr=ap_pstrdup(cmd->pool, attr);
  return NULL;
}

const char *set_ldap_passwordattr(cmd_parms *cmd, ldap_config_struct *conf, char *attr)
{
  conf->userpassword_attr=ap_pstrdup(cmd->pool, attr);
  return NULL;
}

const char *set_ldap_server(cmd_parms *cmd, ldap_config_struct *conf, char *url) {
    LDAPURLDesc *ldap_url;

    if (!ldap_is_ldap_url(url)) {
        return "server is not a properly formatted LDAP url";
    }
 
    if (ldap_url_parse(url,&ldap_url)!=0) {
        return "cannot parse LDAP url";
    }
 
    conf->host=ap_pstrdup(cmd->pool, ldap_url->lud_host);
    if (ldap_url->lud_port!=0)  
        conf->port=ldap_url->lud_port; 

    ldap_free_urldesc(ldap_url);

    return NULL;
}

const char *set_ldap_bindname(cmd_parms *cmd, ldap_config_struct *conf, char *name) {
    conf->bindname = ap_pstrdup(cmd->pool,name);  
    return NULL;
}

const char *set_ldap_bindpass(cmd_parms *cmd, ldap_config_struct *conf, char *pass) {
    conf->bindpass = ap_pstrdup(cmd->pool,pass);  
    return NULL;
}

command_rec ldap_auth_cmds[] = {
    { "LDAPAuth",     set_ldap_auth,     NULL, OR_AUTHCFG, FLAG,    "Activate LDAP auth" },
    { "LDAPServer",     set_ldap_server,     NULL, OR_AUTHCFG, TAKE1,    "LDAP URL" },
    { "LDAPBase",     set_ldap_base,     NULL, OR_AUTHCFG, TAKE1,    "LDAP search base" },
    { "LDAPuseridAttr",     set_ldap_useridattr,     NULL, OR_AUTHCFG, TAKE1,    "LDAP user id attribute name" },
    { "LDAPpasswordAttr",     set_ldap_passwordattr,     NULL, OR_AUTHCFG, TAKE1,    "LDAP user password attribute name" },
    { "LDAPBindName",   set_ldap_bindname,   NULL, OR_AUTHCFG, RAW_ARGS, NULL },
    { "LDAPBindPass",   set_ldap_bindpass,   NULL, OR_AUTHCFG, RAW_ARGS, NULL },
    {NULL}
};

/* --------------------------------- */
static LDAP *ldap_open_and_bind (char *host,int port,char *username,char *password) {
    LDAP *ld;
    int res;


    ld=ldap_open(host,port);
    if (!ld)
        return NULL;

    if (username == NULL) {
        res = ldap_simple_bind_s(ld,NULL,NULL);
    } else {
        res = ldap_simple_bind_s(ld,username,password);
    }
    if (res!=LDAP_SUCCESS) {
	ldap_unbind(ld);
        return NULL; 
    }

    return ld;
}

int match_ldap_filter(LDAP *ld, char *dn, char *filter)
{
    LDAPMessage *msg, *entry;
    int res;

    res=ldap_search_s(ld,dn,LDAP_SCOPE_BASE,filter,NULL,0,&msg);

    if ((res!=LDAP_SUCCESS) || !msg)
        return 0;

    entry=ldap_first_entry(ld,msg);
    if (entry==NULL) 
    {
	ldap_msgfree(msg);
        return 0;
    }

    ldap_msgfree(msg);

    return 1;
}

int ldap_authenticate (request_rec *r)
{
    ldap_config_struct *conf;
    conn_rec *c = r->connection;
    const char *sent_pw;
    char *filter;
    LDAPMessage *msg, *entry;
    int res;

    conf = (ldap_config_struct *) 
                   ap_get_module_config(r->per_dir_config, &ldap_module);

    if(!conf->ldap_auth) return DECLINED;
    /* security bug fix here (RP): if sent_pw=="", bind will succeed as anonymous ! */
    if (ap_get_basic_auth_pw (r, &sent_pw) || !strlen(sent_pw)) return AUTH_REQUIRED;

    conf->ld = ldap_open_and_bind(conf->host,
		conf->port,
		conf->bindname,
		conf->bindpass);
    /* we should have an open and bound connection.  if not then error */
    if (conf->ld==NULL) {
	ap_log_reason("ldap bind failed",r->uri,r);
        return SERVER_ERROR;
    }

    filter = ap_pstrcat(r->pool,
		"(",
		conf->userid_attr,
		"=",
		r->connection->user,
		")",
		NULL);

    res=ldap_search_s(conf->ld,
		conf->base,
		LDAP_SCOPE_ONELEVEL,
		filter,
		NULL,
		0,
		&msg);
    if ((res!=LDAP_SUCCESS) || !msg)
    {
      ldap_unbind(conf->ld);
      ap_log_reason(ap_pstrcat(r->pool,
		"can't search user ",
		r->connection->user,
		" in ",
		conf->base,
		NULL),r->uri,r);
      r->connection->user=NULL;
      return AUTH_REQUIRED;
    }

    entry=ldap_first_entry(conf->ld,msg);
    if(entry)
      r->connection->user=ap_pstrdup(r->pool,ldap_get_dn(conf->ld,entry));
    else
    {
      ap_log_reason(ap_pstrcat(r->pool,
		"can't find ",
		r->connection->user,
		" in ",
		conf->base,
		" on server ",
		conf->host,
		" with filter ",
		filter,
		NULL),r->uri,r);
      r->connection->user=NULL;
    }

    ldap_msgfree(msg);

    if(!conf->userpassword_attr)
    {
      ldap_unbind(conf->ld);
      conf->ld = ldap_open_and_bind(conf->host,
  		conf->port,
  		r->connection->user,
  		sent_pw);
  
      if(conf->ld==NULL)
        r->connection->user=NULL;
      else ldap_unbind(conf->ld);
    }
    else
    {
      if(ldap_compare_s(conf->ld,
      		r->connection->user,
      		conf->userpassword_attr,
      		sent_pw)!=LDAP_COMPARE_TRUE)
        r->connection->user=NULL;
      ldap_unbind(conf->ld);
    }

    if(!r->connection->user)
    {
      ap_log_reason(ap_pstrcat(r->pool,
	"authentication failed",
	NULL),r->uri,r);
      return AUTH_REQUIRED;
    }
      else return OK;
}

int ldap_check_auth (request_rec *r) {
    ldap_config_struct *conf;
    array_header *reqs_arr;
    require_line *reqs;
    char *userdn;
    int i;
    int active = 0;

    conf = (ldap_config_struct *) 
                   ap_get_module_config(r->per_dir_config, &ldap_module);
    reqs_arr = ap_requires (r);
    reqs = reqs_arr ? (require_line *)reqs_arr->elts : NULL;

    /* must have require field(s) */
    if (!reqs_arr || !r->connection->user) {
        return DECLINED;
    }

    for(i=0; (i < reqs_arr->nelts) ; i++) {
        char *t, *w;
        if (! (reqs[i].method_mask & (1 << (r->method_number)))) continue;
	
        t = reqs[i].requirement;
        w = ap_getword(r->pool, &t, ' ');

        if (strcmp(w,"filter")==0)
	{
	    active = 1;
            w = ap_getword_conf (r->pool, &t);
            conf->ld = ldap_open_and_bind(conf->host,
			conf->port,
			conf->bindname,
			conf->bindpass);
            /* we should have an open and bound connection. if not then error */
            if (conf->ld==NULL) {
              ap_log_reason("ldap bind failed",r->uri,r);
              return SERVER_ERROR;
            }
            if (match_ldap_filter(conf->ld, r->connection->user, w)) 
	    {
		ldap_unbind(conf->ld);
                return OK;
	    }
            ldap_unbind(conf->ld);
        }
	else
	if (strcmp(w,"valid-user")==0)
	{
            if(!conf->ldap_auth)
             return DECLINED;
            else
            {
	      active = 1;
              conf->ld = ldap_open_and_bind(conf->host,
			  conf->port,
			  conf->bindname,
			  conf->bindpass);
              /* we should have an open and bound connection. */
              /* if not then error */
              if (conf->ld==NULL) {
                ap_log_reason("ldap bind failed",r->uri,r);
                return SERVER_ERROR;
              }
              if (match_ldap_filter(conf->ld,
		  r->connection->user,
		  "(objectClass=*)")) 
	      {
		ap_log_reason(ap_pstrcat(r->pool, "OK user=", r->connection->user, NULL), r->uri,r); 
		ldap_unbind(conf->ld);
                  return OK;
	      }
              ldap_unbind(conf->ld);
            }
	}
    }

    if(!active) return DECLINED;

    ap_log_reason(ap_pstrcat(r->pool,
		"LDAP access denied for ",
		r->connection->user,
		NULL),
		r->uri,r);
    return AUTH_REQUIRED;
}


/* --------------------------------- */

module ldap_module = {
    STANDARD_MODULE_STUFF,
    NULL,				/* initializer */
    create_ldap_dir_config,		/* dir config creater */
    NULL,				/* dir merger - default=override */
    NULL,				/* server config */
    NULL,				/* merge server config */
    ldap_auth_cmds,			/* command table */
    NULL,				/* handlers */
    NULL,				/* filename translation */
    ldap_authenticate,			/* check_user_id */
    ldap_check_auth,			/* check auth */
    NULL,				/* check access */
    NULL,				/* type_checker */
    NULL,				/* pre-run fixups */
    NULL				/* logger */
};

