/*
 * Copyright 2007  Heiko Hund <heikoh@users.sf.net>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * or see the LICENSE file that should have been distributed with this code.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Apache httpd SASL authentication module.
 *
 * Provides SASL username and password verification for HTTP Basic
 * Authentication. Uses Cyrus' libsasl2 backends for this task.
 * Based on mod_authn_file.c from the httpd-2.2.3 distribution.
 */

#include "ap_config.h"
#include "ap_provider.h"
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"
#include "apr_strings.h"

#include "mod_auth.h"

#include <sasl/sasl.h>


/* Forward declaration of the module struct. */
module AP_MODULE_DECLARE_DATA authn_sasl_module;

/* Default appname for use with libsasl. */
static char *default_appname = "http";

/* The per-directory module config. */
typedef struct authn_sasl_cfg_t
{
	char *pwcheck_method;
	char *appname;
	char *realm;
} authn_sasl_cfg;


/* authn_sasl_create_dir_config -
 * Create a default per-directory configuration. Set the default
 * app name and no "pwcheck_method" so libsasl2 uses it's own.
 */
static void *
authn_sasl_create_dir_config(apr_pool_t *p, char *dirspec)
{
	authn_sasl_cfg *cfg = (authn_sasl_cfg *) apr_pcalloc(p, sizeof(authn_sasl_cfg));

	cfg->appname = default_appname;

	return (void *) cfg;
}


/* set_pwcheck_method -
 * Store the user defined pwcheck_method(s) into the config.
 */
static const char *
set_pwcheck_method(cmd_parms *cmd, void *mconfig, const char *arg1, const char *arg2)
{
	authn_sasl_cfg *cfg = (authn_sasl_cfg *) mconfig;

	if (apr_strnatcmp(arg1, "auxprop") != 0 && apr_strnatcmp(arg1, "saslauthd") != 0) {
		return apr_pstrcat(cmd->pool, "Invalid SASL pwcheck method string: ", arg1, NULL);
	}

	cfg->pwcheck_method = apr_pstrdup(cmd->pool, arg1);

	if (arg2) {
		if (apr_strnatcmp(arg2, arg1) == 0 ||
		    (apr_strnatcmp(arg2, "auxprop") != 0 && apr_strnatcmp(arg2, "saslauthd") != 0)) {
			return apr_pstrcat(cmd->pool, "Invalid SASL pwcheck method string: ", arg2, NULL);
		}
		cfg->pwcheck_method = apr_pstrcat(cmd->pool, arg1, " ", arg2, NULL);
	}

	return NULL;
}


/* set_appname -
 * Store the user defined application name into the config.
 */
static const char *
set_appname(cmd_parms *cmd, void *mconfig, const char *arg1)
{
	authn_sasl_cfg *cfg = (authn_sasl_cfg *) mconfig;

	cfg->appname = apr_pstrdup(cmd->pool, arg1);

	return NULL;
}


/* set_realm -
 * Store the user defined user realm into the config.
 */
static const char *
set_realm(cmd_parms *cmd, void *mconfig, const char *arg1)
{
	authn_sasl_cfg *cfg = (authn_sasl_cfg *) mconfig;

	cfg->realm = apr_pstrdup(cmd->pool, arg1);

	return NULL;
}


/* The config directives introduces by this module. */
static const command_rec authn_sasl_cmds[] = {
	AP_INIT_TAKE12("AuthSaslPwcheckMethod", set_pwcheck_method, NULL, OR_AUTHCFG,
	               "Set this to override libsasl's default 'pwcheck_method' used for "
	               "authentication. Valid values are 'auxprop' and 'saslauthd'."),
	AP_INIT_TAKE1("AuthSaslAppname", set_appname, NULL, OR_AUTHCFG,
	              "Set the application name to be used by libsasl during user authentication"),
	AP_INIT_TAKE1("AuthSaslRealm", set_realm, NULL, OR_AUTHCFG,
	              "Set the user realm to be used by libsasl during user authentication"),
	{ NULL }
};


/* authn_sasl_cb_getopt -
 * Callback function libsasl2 uses to get option values.
 * The value for option "pwcheck_method" is set if it was
 * user defined via config directive "AuthSaslPwcheckMethod"
 */
static int
authn_sasl_cb_getopt(void *mconfig, const char *pname, const char *opt, const char **res, unsigned *len)
{
	authn_sasl_cfg *cfg = (authn_sasl_cfg *) mconfig;

	if (cfg->pwcheck_method && apr_strnatcmp(opt, "pwcheck_method") == 0) {
		*res = cfg->pwcheck_method;
		return SASL_OK;
	}

	return SASL_CONTINUE;
}


/* check_password -
 * Authentication backend provider callback function used by
 * mod_auth_basic to verify credentials. Uses functions from
 * libsasl2 to do the job.
 */
static authn_status
check_password(request_rec *r, const char *user, const char *pass)
{
	sasl_conn_t *sasl_conn;
	authn_status result = AUTH_GRANTED;

	authn_sasl_cfg *cfg = ap_get_module_config(r->per_dir_config, &authn_sasl_module);

	sasl_callback_t const cb[] = {
		{ SASL_CB_GETOPT, authn_sasl_cb_getopt, (void *) cfg },
		{ SASL_CB_LIST_END, NULL, NULL }
	};

	if (sasl_server_new(cfg->appname, NULL, cfg->realm, NULL, NULL, cb, 0, &sasl_conn) != SASL_OK ||
	    sasl_checkpass(sasl_conn, user, strlen(user), pass, strlen(pass)) != SASL_OK) {
		ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s", sasl_errdetail(sasl_conn));
		result = AUTH_DENIED;
	}

	sasl_dispose(&sasl_conn);

	return result;
}

static apr_status_t
authn_sasl_child_exit(void *data)
{
	sasl_done();
	return APR_SUCCESS;
}


/* authn_sasl_child_init -
 * Initialize libsasl2 once in every child process and register
 * a cleanup function to finalize it before the process exits.
 */
static void
authn_sasl_child_init(apr_pool_t *p, server_rec *s)
{
	sasl_server_init(NULL, NULL);
	apr_pool_cleanup_register(p, s, NULL, authn_sasl_child_exit);
}


/* authn_sasl_register_hooks -
 * Registers an auth provider to be used with
 * mod_auth_basic and a iit function to init sasl.
 */
static void
authn_sasl_register_hooks(apr_pool_t *p)
{
	static const authn_provider authn_sasl_provider = {
		check_password
	};

	ap_register_provider(p, AUTHN_PROVIDER_GROUP, "sasl", "0", &authn_sasl_provider);
	ap_hook_child_init(authn_sasl_child_init, NULL, NULL, APR_HOOK_MIDDLE);
}


module AP_MODULE_DECLARE_DATA authn_sasl_module = {
	STANDARD20_MODULE_STUFF,
	authn_sasl_create_dir_config,    /* per-directory config creator */
	NULL,                            /* dir config merger */
	NULL,                            /* server config creator */
	NULL,                            /* server config merger */
	authn_sasl_cmds,                 /* command table */
	authn_sasl_register_hooks        /* set up other request processing hooks */
};
