/***************************************************************************
 $RCSfile: ohbci.c,v $
                             -------------------
    cvs         : $Id: ohbci.c,v 1.20 2005/04/22 20:21:38 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

//#define DEBUG_OHBCI_MODULE



#include "ohbci_p.h"
#include "aqhbci_l.h"
#include <gwenhywfar/debug.h>
#include <gwenhywfar/misc.h>
#include <gwenhywfar/padd.h>
#include <gwenhywfar/md.h>
#include <gwenhywfar/waitcallback.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#ifdef OS_WIN32
# define ftruncate chsize
#endif


GWEN_INHERIT(AH_MEDIUM, AH_MEDIUM_OHBCI)


AH_MEDIUM *AH_MediumOHBCI_new(AH_HBCI *hbci,
                              AH_MEDIUMPROVIDER *mp,
                              const char *mediumName){
  AH_MEDIUM *m;
  AH_MEDIUM_OHBCI *mrdh;

  m=AH_MediumRDH_new(hbci,
                     AH_MEDIUM_OHBCI_NAME,
                     mediumName);
  GWEN_NEW_OBJECT(AH_MEDIUM_OHBCI, mrdh);
  mrdh->cryptoTag=AH_MEDIUM_OHBCI_TAG_CRYPT;
  mrdh->keyfile_mode = 0;
  GWEN_INHERIT_SETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI,
                       m, mrdh,
                       AH_MediumOHBCI_FreeData);

  AH_Medium_SetDeviceType(m, AH_MediumDeviceFile);

  /* set implemented functions for Medium */
  AH_Medium_SetMountFn(m, AH_MediumOHBCI_Mount);
  AH_Medium_SetCreateFn(m, AH_MediumOHBCI_Create);
  AH_Medium_SetUnmountFn(m, AH_MediumOHBCI_Unmount);
  AH_Medium_SetGetNextSignSeqFn(m, AH_MediumOHBCI_GetNextSignSeq);
  AH_Medium_SetSetLocalSignSeqFn(m, AH_MediumOHBCI_SetLocalSignSeq);
  AH_Medium_SetSignFn(m, AH_MediumOHBCI_Sign);
  AH_Medium_SetVerifyFn(m, AH_MediumOHBCI_Verify);
  AH_Medium_SetEncryptKeyFn(m, AH_MediumOHBCI_EncryptKey);
  AH_Medium_SetDecryptKeyFn(m, AH_MediumOHBCI_DecryptKey);
  AH_Medium_SetGenerateMsgKeyFn(m, AH_MediumOHBCI_GenerateMsgKey);
  AH_Medium_SetSelectContextFn(m, AH_MediumOHBCI_SelectContext);
  AH_Medium_SetCreateContextFn(m, AH_MediumOHBCI_CreateContext);
  AH_Medium_SetRemoveContextFn(m, AH_MediumOHBCI_RemoveContext);
  AH_Medium_SetReadContextFn(m, AH_MediumOHBCI_ReadContext);
  AH_Medium_SetWriteContextFn(m, AH_MediumOHBCI_WriteContext);

  /* set implemented functions for MediumRDH */
  AH_MediumRDH_SetCreateKeysFn(m, AH_MediumOHBCI_CreateKeys);
  AH_MediumRDH_SetActivateKeysFn(m, AH_MediumOHBCI_ActivateKeys);

  AH_MediumRDH_SetGetLocalPubSignKeyFn(m, AH_MediumOHBCI_GetLocalPubSignKey);
  AH_MediumRDH_SetGetLocalPubCryptKeyFn(m, AH_MediumOHBCI_GetLocalPubCryptKey);

  AH_MediumRDH_SetGetLocalTmpSignKeyFn(m, AH_MediumOHBCI_GetLocalTmpSignKey);
  AH_MediumRDH_SetGetLocalTmpCryptKeyFn(m, AH_MediumOHBCI_GetLocalTmpCryptKey);

  AH_MediumRDH_SetSetPubSignKeyFn(m, AH_MediumOHBCI_SetPubSignKey);
  AH_MediumRDH_SetGetPubSignKeyFn(m, AH_MediumOHBCI_GetPubSignKey);
  AH_MediumRDH_SetSetPubCryptKeyFn(m, AH_MediumOHBCI_SetPubCryptKey);
  AH_MediumRDH_SetGetPubCryptKeyFn(m, AH_MediumOHBCI_GetPubCryptKey);


  return m;
}



void AH_MediumOHBCI_FreeData(void *bp, void *p) {
  AH_MEDIUM_OHBCI *mrdh;

  mrdh=(AH_MEDIUM_OHBCI*) p;
  memset(mrdh->password, 0, sizeof(mrdh->password));
  free(mrdh->bankId);
  free(mrdh->userId);
  free(mrdh->systemId);
  GWEN_CryptKey_free(mrdh->localSignKey);
  GWEN_CryptKey_free(mrdh->localCryptKey);
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  GWEN_FREE_OBJECT(mrdh);
}






GWEN_CRYPTKEY *AH_MediumOHBCI_GetLocalPubSignKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->localSignKey) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localSignKey, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_DB_Group_free(db);
    return 0;
  }

  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    GWEN_DB_Group_free(db);
    return 0;
  }

  GWEN_DB_Group_free(db);
  return key;
}



GWEN_CRYPTKEY *AH_MediumOHBCI_GetLocalPubCryptKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->localCryptKey) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localCryptKey, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_DB_Group_free(db);
    return 0;
  }
  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    GWEN_DB_Group_free(db);
    return 0;
  }
  GWEN_DB_Group_free(db);

  return key;
}



GWEN_CRYPTKEY *AH_MediumOHBCI_GetLocalTmpSignKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->localSignKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localSignKeyBak, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return 0;
  }

  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    return 0;
  }

  return key;
}



GWEN_CRYPTKEY *AH_MediumOHBCI_GetLocalTmpCryptKey(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key;
  GWEN_DB_NODE *db;
  GWEN_ERRORCODE err;
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->localCryptKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  db=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(mrdh->localCryptKeyBak, db, 1);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return 0;
  }
  key=GWEN_CryptKey_FromDb(db);
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key from previous export");
    return 0;
  }

  return key;
}



GWEN_CRYPTKEY *AH_MediumOHBCI_GetPubSignKey(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (mrdh->remoteSignKey)
    return GWEN_CryptKey_dup(mrdh->remoteSignKey);
  else
    return 0;
}



int AH_MediumOHBCI_SetPubSignKey(AH_MEDIUM *m,
                                 const GWEN_CRYPTKEY *key){
  AH_MEDIUM_OHBCI *mrdh;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  assert(key);

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    AH_MediumOHBCI__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    return AH_MediumResultGenericError;
  }

  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=GWEN_CryptKey_dup(key);

  /* write file (just to be sure that the last change in on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return 0;
}



GWEN_CRYPTKEY *AH_MediumOHBCI_GetPubCryptKey(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (mrdh->remoteCryptKey)
    return GWEN_CryptKey_dup(mrdh->remoteCryptKey);
  else
    return 0;
}



int AH_MediumOHBCI_SetPubCryptKey(AH_MEDIUM *m,
                                  const GWEN_CRYPTKEY *key){
  AH_MEDIUM_OHBCI *mrdh;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  assert(key);

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    AH_MediumOHBCI__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    return AH_MediumResultGenericError;
  }

  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=GWEN_CryptKey_dup(key);

  /* write file (just to be sure that the last change in on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return 0;
}



int AH_MediumOHBCI_CreateKeys(AH_MEDIUM *m){
  GWEN_CRYPTKEY *key1;
  GWEN_CRYPTKEY *key2;
  GWEN_ERRORCODE err;
  AH_MEDIUM_OHBCI *mrdh;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->selected) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No context selected");
    return -1;
  }

  /* create and generate local sign key */
  key1=GWEN_CryptKey_Factory("RSA");
  if (!key1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create an RSA key");
    return -1;
  }
  err=GWEN_CryptKey_Generate(key1, AH_DEFAULT_KEYLEN);
  if (!GWEN_Error_IsOk(err)){
    GWEN_CryptKey_free(key1);
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return -1;
  }
  GWEN_CryptKey_SetKeyName(key1, "S");
  GWEN_CryptKey_SetOwner(key1, mrdh->userId);

  /* create and generate local crypt key */
  key2=GWEN_CryptKey_Factory("RSA");
  if (!key2) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create an RSA key");
    GWEN_CryptKey_free(key1);
    return -1;
  }
  err=GWEN_CryptKey_Generate(key2, AH_DEFAULT_KEYLEN);
  if (!GWEN_Error_IsOk(err)){
    GWEN_CryptKey_free(key1);
    GWEN_CryptKey_free(key2);
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    return -1;
  }
  GWEN_CryptKey_SetKeyName(key2, "V");
  GWEN_CryptKey_SetOwner(key2, mrdh->userId);

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    GWEN_CryptKey_free(key1);
    GWEN_CryptKey_free(key2);
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    AH_MediumOHBCI__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    GWEN_CryptKey_free(key1);
    GWEN_CryptKey_free(key2);
    return AH_MediumResultGenericError;
  }

  /* store keys as backup keys (can be activated later) */
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=key1;
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=key2;

  /* write file (just to be sure that the last change in on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  DBG_NOTICE(AQHBCI_LOGDOMAIN, "Created this keys: ");
  GWEN_KeySpec_Dump(GWEN_CryptKey_GetKeySpec(key1), stderr, 2);
  GWEN_KeySpec_Dump(GWEN_CryptKey_GetKeySpec(key2), stderr, 2);
  return 0;
}



int AH_MediumOHBCI_ActivateKeys(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!mrdh->localSignKeyBak ||
      !mrdh->localCryptKeyBak) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No keys created");
    return -1;
  }

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    AH_MediumOHBCI__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    return AH_MediumResultGenericError;
  }

  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=mrdh->localSignKeyBak;
  mrdh->localSignKeyBak=0;
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=mrdh->localCryptKeyBak;
  mrdh->localCryptKeyBak=0;

  if (mrdh->localSignKey)
      AH_Medium_SetLocalSignKeySpec(m,
                                    GWEN_CryptKey_GetKeySpec(mrdh->localSignKey));

  if (mrdh->localCryptKey)
      AH_Medium_SetLocalCryptKeySpec(m,
                                     GWEN_CryptKey_GetKeySpec(mrdh->localCryptKey));

  /* write file (just to be sure that the last change is on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return 0;
}



int AH_MediumOHBCI__OpenFile(AH_MEDIUM *m, int wr){
#ifndef OS_WIN32
  struct flock fl;
#endif
  int fd;
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (wr) {
    /* write file */
    fd=open(AH_Medium_GetMediumName(m),
            O_RDWR|O_CREAT
#ifdef OS_WIN32
            | O_BINARY
#endif
            ,
	    S_IRUSR|S_IWUSR | mrdh->keyfile_mode);
  }
  else {
    /* Remember the access permissions when opening the file */
    struct stat statbuffer;
    if (!stat(AH_Medium_GetMediumName(m), &statbuffer)) {
      /* Save the access mode, but masked by the bit masks for
	 user/group/other permissions */
      mrdh->keyfile_mode = 
	statbuffer.st_mode & (S_IRWXU
#ifndef OS_WIN32
			      | S_IRWXG | S_IRWXO
#endif
			      );
    }

    /* and open the file */
    fd=open(AH_Medium_GetMediumName(m),
            O_RDONLY
#ifdef OS_WIN32
            | O_BINARY
#endif
           );
  }

  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "open(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    return -1;
  }

#ifndef OS_WIN32
  /* lock file for reading or writing */
  memset(&fl, 0, sizeof(fl));
  fl.l_type=wr?F_WRLCK:F_RDLCK;
  fl.l_whence=SEEK_SET;
  fl.l_start=0;
  fl.l_len=0;
  if (fcntl(fd, F_SETLKW, &fl)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "fcntl(%s, F_SETLKW): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    close(fd);
    return -1;
  }
#endif

  return fd;
}



int AH_MediumOHBCI__CloseFile(AH_MEDIUM *m, int fd){
#ifndef OS_WIN32
  struct flock fl;
#endif
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Keyfile \"%s\"not open",
	      AH_Medium_GetMediumName(m));
    return -1;
  }

#ifndef OS_WIN32
  /* unlock file */
  memset(&fl, 0, sizeof(fl));
  fl.l_type=F_UNLCK;
  fl.l_whence=SEEK_SET;
  fl.l_start=0;
  fl.l_len=0;
  if (fcntl(fd, F_SETLK, &fl)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "fcntl(%s, F_SETLK): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    close(fd);
    return -1;
  }
#endif

  if (close(fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "close(%s): %s",
	      AH_Medium_GetMediumName(m), strerror(errno));
    return -1;
  }

  return 0;
}



int AH_MediumOHBCI__DecryptFile(AH_MEDIUM *m,
                                GWEN_BUFFER *fbuf,
                                int trynum){
  AH_MEDIUM_OHBCI *mrdh;
  GWEN_CRYPTKEY *key;
  GWEN_ERRORCODE err;
  char password[64];
  GWEN_BUFFER *rawbuf;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (mrdh->passWordIsSet==0) {
    /* create key from password */
    memset(mrdh->password, 0, sizeof(mrdh->password));

    if (AH_Medium_InputPin(m,
                           password,
                           AH_MEDIUM_OHBCI_PINMINLENGTH,
                           sizeof(password),
                           trynum?AB_BANKING_INPUT_FLAGS_RETRY:0)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
      return AB_ERROR_USER_ABORT;
    }

    if (strlen(password)<AH_MEDIUM_OHBCI_PINMINLENGTH) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "Your program returned a shorter PIN than instructed!");
      return AB_ERROR_BAD_DATA;
    }

    DBG_NOTICE(AQHBCI_LOGDOMAIN, "Checking...");
    if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT) {
      DBG_NOTICE(AQHBCI_LOGDOMAIN, "New OpenHBCI file detected");
      if (GWEN_CryptKey_FromPassword(password,
                                     mrdh->password, sizeof(mrdh->password))) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
        return AB_ERROR_GENERIC;
      }
    }
    else if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT_OLD) {
      DBG_NOTICE(AQHBCI_LOGDOMAIN, "Old OpenHBCI file detected");
      if (GWEN_CryptKey_FromPasswordSSL(password,
                                        mrdh->password,
                                        sizeof(mrdh->password))) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
        return AB_ERROR_GENERIC;
      }
    }
    else {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Unexpected crypto tag %d", mrdh->cryptoTag);
      abort();
    }

    mrdh->passWordIsSet=1;
  }

  key=GWEN_CryptKey_Factory("DES");
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES key");
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=0;
    return AB_ERROR_GENERIC;
  }

  err=GWEN_CryptKey_SetData(key, mrdh->password, sizeof(mrdh->password));
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(key);
    mrdh->passWordIsSet=0;
    return AB_ERROR_GENERIC;
  }

  /* decrypt file */
  DBG_INFO(AQHBCI_LOGDOMAIN, "Decrypting file");
  rawbuf=GWEN_Buffer_new(0, 1024, 0, 1);
  GWEN_Buffer_Rewind(fbuf);
  err=GWEN_CryptKey_Decrypt(key, fbuf, rawbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(rawbuf);
    GWEN_CryptKey_free(key);
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=0;
    return AB_ERROR_GENERIC;
  }

  /* unpadd raw data */
  DBG_INFO(AQHBCI_LOGDOMAIN, "Unpadding file");
  if (GWEN_Padd_UnpaddWithANSIX9_23(rawbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not unpadd keyfile, i.e. wrong PIN");
    GWEN_Buffer_free(rawbuf);
    GWEN_CryptKey_free(key);
    AH_Medium_SetPinStatus(m, password, AB_Banking_PinStatusBad);
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=0;
    return AB_ERROR_BAD_DATA;
  }
  GWEN_CryptKey_free(key);

  /* parse raw data */
  DBG_INFO(AQHBCI_LOGDOMAIN, "Parsing file");
  GWEN_Buffer_Rewind(rawbuf);
  if (AH_MediumOHBCI__Decode(m, rawbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(rawbuf);
    AH_Medium_SetPinStatus(m, password, AB_Banking_PinStatusBad);
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=0;
    return AB_ERROR_BAD_DATA;
  }
  GWEN_Buffer_free(rawbuf);
  AH_Medium_SetPinStatus(m, password, AB_Banking_PinStatusOk);
  return 0;
}



int AH_MediumOHBCI__ReadFile(AH_MEDIUM *m, int fd){
  AH_MEDIUM_OHBCI *mrdh;
  GWEN_BUFFER *rbuf;
  GWEN_BUFFER *fbuf;
  struct stat st;
  unsigned char c;
  OHBCI_TLV *tlv;
  int i;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* some file checks */
  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "stat(%s): %s",
              AH_Medium_GetMediumName(m),
              strerror(errno));
    return -1;
  }
  if (!S_ISREG(st.st_mode)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "\"%s\" is not a regular file",
              AH_Medium_GetMediumName(m));
    return -1;
  }
#ifndef OS_WIN32
  if (st.st_mode & 0007) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "WARNING: Your keyfile \"%s\" is world accessible!\n"
             "Nobody but you should have access to the file. You \n"
	     "should probably change this with \"chmod 600 %s\"",
             AH_Medium_GetMediumName(m),
             AH_Medium_GetMediumName(m));
  }
#endif
  mrdh->mtime=st.st_mtime;
  mrdh->ctime=st.st_ctime;

  if (lseek(fd, 0, SEEK_SET)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "lseek(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    return -1;
  }

  rbuf=GWEN_Buffer_new(0, 1024, 0, 1);
  /* read file into rbuf */
  while(1) {
    char buffer[256];
    int rv;

    rv=read(fd, buffer, sizeof(buffer));
    if (rv==-1) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "read: %s", strerror(errno));
      return -1;
    }
    if (rv==0)
      break;
    GWEN_Buffer_AppendBytes(rbuf, buffer, rv);
  }

  if (GWEN_Buffer_GetUsedBytes(rbuf)<3) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "This seems not to be an OpenHBCI key file");
    GWEN_Buffer_free(rbuf);
    return -1;
  }

  /* check whether this is a known OpenHBCI(2) keyfile */
  GWEN_Buffer_Rewind(rbuf);
  c=*GWEN_Buffer_GetStart(rbuf);
  if (c!=AH_MEDIUM_OHBCI_TAG_CRYPT &&
      c!=AH_MEDIUM_OHBCI_TAG_CRYPT_OLD) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "This seems not to be an OpenHBCI key file");
    GWEN_Buffer_free(rbuf);
    return -1;
  }
  mrdh->cryptoTag=c;

  tlv=OHBCI_TLV_fromBuffer(rbuf, 0);
  if (!tlv) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad file data");
    GWEN_Buffer_free(rbuf);
    return -1;
  }

  fbuf=GWEN_Buffer_new(0, OHBCI_TLV_GetTagLength(tlv), 0, 1);
  GWEN_Buffer_AppendBytes(fbuf,
                          OHBCI_TLV_GetTagData(tlv),
                          OHBCI_TLV_GetTagLength(tlv));
  GWEN_Buffer_Rewind(fbuf);
  GWEN_Buffer_free(rbuf);
  OHBCI_TLV_free(tlv);
  /* now fbuf contains the data from the crypt TLV */


  for (i=0;;i++) {
    int rv;

    if (i>AH_MEDIUM_OHBCI_MAX_PIN_TRY) {
      DBG_ERROR(AQBANKING_LOGDOMAIN,
                "No valid PIN within %d tries, giving up", i);
      AB_Banking_MessageBox(AH_Medium_GetBankingApi(m),
                            AB_BANKING_MSG_FLAGS_TYPE_ERROR |
                            AB_BANKING_MSG_FLAGS_SEVERITY_NORMAL,
                            I18N("Error"),
                            I18N("No valid PIN (tried too often).\n"
				 "Aborting."),
                            I18N("Dismiss"), 0, 0);
      GWEN_Buffer_free(fbuf);
      return AB_ERROR_INVALID;
    }

    rv=AH_MediumOHBCI__DecryptFile(m, fbuf, i);
    if (rv==0)
      break;
    else {
      switch(rv) {
      case AB_ERROR_USER_ABORT:
        DBG_INFO(AQHBCI_LOGDOMAIN, "Aborted by user");
        GWEN_Buffer_free(fbuf);
        return rv;
      case AB_ERROR_BAD_DATA:
        DBG_INFO(AQHBCI_LOGDOMAIN, "Bad pin. Will ask user what to do.");
	if (AB_Banking_MessageBox(AH_Medium_GetBankingApi(m),
				  AB_BANKING_MSG_FLAGS_TYPE_ERROR |
				  AB_BANKING_MSG_FLAGS_SEVERITY_NORMAL,
				  I18N("Wrong PIN"),
				  I18N("You entered a wrong PIN.\n"
				       "Do you want to try again?"),
				  I18N("Try again"), I18N("Abort"), 0) == 1)
	  break;
	else {
	  /* User wanted to abort */
	  GWEN_Buffer_free(fbuf);
	  return AB_ERROR_USER_ABORT;
	}
      default:
        DBG_INFO(AQHBCI_LOGDOMAIN, "Other error, giving up");
        GWEN_Buffer_free(fbuf);
        return rv;
      }
    }

  } /* for */
  GWEN_Buffer_free(fbuf);

  return 0;
}



void AH_MediumOHBCI__DecodeKey(AH_MEDIUM *m,
                               OHBCI_TLV *keyTlv,
                               GWEN_DB_NODE *dbKeys,
                               const char *keyName) {
  GWEN_BUFFER *dbuf;
  const char *p;
  int size;
  GWEN_DB_NODE *node;
  GWEN_TYPE_UINT32 flags;
  const char defaultExpo[3]={0x01, 0x00, 0x01};

  p=OHBCI_TLV_GetTagData(keyTlv);
  size=OHBCI_TLV_GetTagLength(keyTlv);
  if (size<2) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Tag too small to contain any subtag");
    return;
  }
  /* create static buffer */
  dbuf=GWEN_Buffer_new((char*)p, size, size, 0);
  GWEN_Buffer_SubMode(dbuf, GWEN_BUFFER_MODE_DYNAMIC);

  node=GWEN_DB_GetGroup(dbKeys, GWEN_DB_FLAGS_DEFAULT, keyName);
  assert(node);

  /* preset */
  GWEN_DB_SetCharValue(node,
                       GWEN_DB_FLAGS_OVERWRITE_VARS,
                       "type",
                       "RSA");
  GWEN_DB_SetBinValue(node,
                      GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "data/e",
                      defaultExpo,
                      sizeof(defaultExpo));
  flags=0;
  if (AH_Medium_GetFlags(m) & AH_MEDIUM_FLAGS_DISABLE_SMALLER_SIGNATURE)
    flags|=GWEN_CRYPT_FLAG_DISABLE_SMALLER_SIGNATURE;
  GWEN_DB_SetIntValue(node,
                      GWEN_DB_FLAGS_OVERWRITE_VARS,
                      "flags",
                      flags);

  while(GWEN_Buffer_GetBytesLeft(dbuf)) {
    const char *pp;
    char *p;
    OHBCI_TLV *tlv;
    unsigned int l;

    tlv=OHBCI_TLV_fromBuffer(dbuf, 0);
    if (!tlv) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad file (no TLV)");
      return;
    }
    p=0;
    pp=(const char*)OHBCI_TLV_GetTagData(tlv);
    l=OHBCI_TLV_GetTagLength(tlv);
    if (pp && l) {
      p=(char*)malloc(l+1);
      assert(p);
      memmove(p, pp, l);
      p[l]=0;
    }
    switch(OHBCI_TLV_GetTagType(tlv)) {
    case AH_MEDIUM_OHBCI_TAG_KEY_ISPUBLIC:
      /* ignore this tag, since it is buggy in OpenHBCI(2) */
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_ISCRYPT:
      if (strcasecmp(p, "yes")==0)
        GWEN_DB_SetCharValue(node,
                             GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "name",
                             "V");
      else
        GWEN_DB_SetCharValue(node,
                             GWEN_DB_FLAGS_OVERWRITE_VARS,
                             "name",
                             "S");
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_OWNER: {
      GWEN_BUFFER *obuf;
      const char *s;

      /* workaround for a bug in older OpenHBCI versions: here the escape
       * character "?" was falsely included for the owner name */
      obuf=GWEN_Buffer_new(0, 32, 0, 1);
      s=p;
      while(*s) {
        if (*s!='?')
          GWEN_Buffer_AppendByte(obuf, *s);
        s++;
      } /* while */
      GWEN_DB_SetCharValue(node,
                           GWEN_DB_FLAGS_OVERWRITE_VARS,
                           "owner",
                           GWEN_Buffer_GetStart(obuf));
      GWEN_Buffer_free(obuf);
      break;
    }

    case AH_MEDIUM_OHBCI_TAG_KEY_VERSION:
      GWEN_DB_SetIntValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "version",
                          atoi(p));
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_NUMBER:
      GWEN_DB_SetIntValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "number",
                          atoi(p));
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_MODULUS:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/n",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_EXP_OLD:
      DBG_INFO(AQHBCI_LOGDOMAIN, "Ignoring old exponent (%d), keeping default", l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_EXP:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/e",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_N:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/n",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_P:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/p",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_Q:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/q",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_D:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/d",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_DMP1:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/dmp1",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_DMQ1:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/dmq1",
                          p, l);
      break;

    case AH_MEDIUM_OHBCI_TAG_KEY_IQMP:
      GWEN_DB_SetBinValue(node,
                          GWEN_DB_FLAGS_OVERWRITE_VARS,
                          "data/iqmp",
                          p, l);
      break;

    default:
      DBG_WARN(AQHBCI_LOGDOMAIN, "Unknown tag %02x", OHBCI_TLV_GetTagType(tlv));
      break;
    } /* switch */

    OHBCI_TLV_free(tlv);
    free(p);
  } /* while */
  GWEN_Buffer_free(dbuf);
}



int AH_MediumOHBCI__Decode(AH_MEDIUM *m, GWEN_BUFFER *dbuf) {
  AH_MEDIUM_OHBCI *mrdh;
  OHBCI_TLV *tlv;
  int rv;
  GWEN_DB_NODE *dbKeys;
  GWEN_DB_NODE *dbKey;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  tlv=OHBCI_TLV_fromBuffer(dbuf, 0);
  GWEN_Buffer_Rewind(dbuf);
  if (!tlv) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "File doesn't contain a TLV: Either bad pin or bad file");
    return -1;
  }

  if (OHBCI_TLV_GetTagType(tlv)!=AH_MEDIUM_OHBCI_TAG_VERSION_MAJOR) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "File doesn't start with version info.");
    OHBCI_TLV_free(tlv);
    return -1;
  }
  OHBCI_TLV_free(tlv);

  /* now parse it */
  dbKeys=GWEN_DB_Group_new("keys");
  while(GWEN_Buffer_GetBytesLeft(dbuf)) {
    int i;
    const char *pp;
    char *p;
    unsigned int l;

    tlv=OHBCI_TLV_fromBuffer(dbuf, 0);
    if (!tlv) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "File doesn't contain a TLV: Either bad pin or bad file");
      return -1;
    }
    p=0;
    pp=(const char*)OHBCI_TLV_GetTagData(tlv);
    l=OHBCI_TLV_GetTagLength(tlv);
    if (pp && l) {
      p=(char*)malloc(l+1);
      assert(p);
      memmove(p, pp, l);
      p[l]=0;
    }

    switch(OHBCI_TLV_GetTagType(tlv)) {
    case AH_MEDIUM_OHBCI_TAG_VERSION_MAJOR:
      i=atoi(p);
      if (i!=AH_MEDIUM_OHBCI_VMAJOR) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Unsupported keyfile version (%d)", i);
        GWEN_WaitCallback_Log(0,
                              "Basically this file type is supported.\n"
                              "However, the major versions do not match,\n"
                              "so this particular version is not supported");
        OHBCI_TLV_free(tlv);
        return -1;
      }
      break;
    case AH_MEDIUM_OHBCI_TAG_VERSION_MINOR:
      i=atoi(p);
      if (i>AH_MEDIUM_OHBCI_VMINOR) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Keyfile version is higher than mine (%d).\n"
                  "Using it would loose data on the file, refusing to.",
                  i);
        GWEN_WaitCallback_Log(0,
                              "Basically this file type is supported.\n"
                              "However, this file has been created with a "
                              "newer library version.\n"
                              "Using this file with the current version "
                              "would degrade it.\n"
                              "So for safety reasons I refuse to work "
                              "with this file.");
        OHBCI_TLV_free(tlv);
        return -1;
      }
      else if (i<AH_MEDIUM_OHBCI_VMINOR) {
        DBG_INFO(AQHBCI_LOGDOMAIN, "Will update this file upon unmount");
      }
      break;

    case AH_MEDIUM_OHBCI_TAG_SEQ:
      mrdh->localSignSeq=atoi(p);
      break;

    case AH_MEDIUM_OHBCI_TAG_USER_ID:
      free(mrdh->userId);
      mrdh->userId=strdup(p);
      break;

    case AH_MEDIUM_OHBCI_TAG_INST_COUNTRY:
      mrdh->country=atoi(p);
      break;

    case AH_MEDIUM_OHBCI_TAG_INST_CODE:
      free(mrdh->bankId);
      mrdh->bankId=strdup(p);
      break;

    case AH_MEDIUM_OHBCI_TAG_INST_SYSTEMID:
      free(mrdh->systemId);
      mrdh->systemId=strdup(p);
      AH_Medium_SetSecurityId(m, mrdh->systemId, strlen(mrdh->systemId));
      break;

    case AH_MEDIUM_OHBCI_TAG_SERVER_ADDR:
      /* new in 1.4 */
      AH_Medium_SetPeerAddr(m, p);
      break;

    case AH_MEDIUM_OHBCI_TAG_SERVER_PORT:
      /* new in 1.4 */
      AH_Medium_SetPeerPort(m, atoi(p));
      break;

    case AH_MEDIUM_OHBCI_TAG_REMOTE_SEQ:
      /* new in 1.4 */
      mrdh->remoteSignSeq=atoi(p);
      break;

    case AH_MEDIUM_OHBCI_TAG_USER_PUBSIGNKEY:
    case AH_MEDIUM_OHBCI_TAG_USER_PRIVSIGNKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "localSignKey");
      break;

    case AH_MEDIUM_OHBCI_TAG_USER_PUBCRYPTKEY:
    case AH_MEDIUM_OHBCI_TAG_USER_PRIVCRYPTKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "localCryptKey");
      break;

    case AH_MEDIUM_OHBCI_TAG_INST_PUBSIGNKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "remoteSignKey");
      break;

    case AH_MEDIUM_OHBCI_TAG_INST_PUBCRYPTKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "remoteCryptKey");
      break;

    case AH_MEDIUM_OHBCI_TAG_TEMP_PUBSIGNKEY:
    case AH_MEDIUM_OHBCI_TAG_TEMP_PRIVSIGNKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "localSignKeyBak");
      break;

    case AH_MEDIUM_OHBCI_TAG_TEMP_PUBCRYPTKEY:
    case AH_MEDIUM_OHBCI_TAG_TEMP_PRIVCRYPTKEY:
      AH_MediumOHBCI__DecodeKey(m, tlv, dbKeys, "localCryptKeyBak");
      break;
    default:
      DBG_WARN(AQHBCI_LOGDOMAIN, "Unknown tag %02x", OHBCI_TLV_GetTagType(tlv));
      break;
    } /* switch */

    OHBCI_TLV_free(tlv);
    free(p);
  } /* while */


  /* now check for keys */
  rv=0;

  /* local sign key */
  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "localSignKey");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 0);
    mrdh->localSignKey=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->localSignKey) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }

  /* local crypt key */
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "localCryptKey");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 0);
    mrdh->localCryptKey=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->localCryptKey) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }

  /* remote sign key */
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "remoteSignKey");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 1);
    mrdh->remoteSignKey=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->remoteSignKey) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }

  /* remote crypt key */
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "remoteCryptKey");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 1);
    mrdh->remoteCryptKey=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->remoteCryptKey) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }

  /* local sign key bak */
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "localSignKeyBak");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 0);
    mrdh->localSignKeyBak=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->localSignKeyBak) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }

  /* local crypt key bak */
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=0;
  dbKey=GWEN_DB_GetGroup(dbKeys, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                         "localCryptKeyBak");
  if (dbKey) {
    GWEN_DB_SetIntValue(dbKey, GWEN_DB_FLAGS_OVERWRITE_VARS,
                        "data/public", 0);
    mrdh->localCryptKeyBak=GWEN_CryptKey_FromDb(dbKey);
    if (!mrdh->localCryptKeyBak) {
      rv=-1;
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Bad key format");
    }
  }
  GWEN_DB_Group_free(dbKeys);

  return rv;
}



int AH_MediumOHBCI__EncodeKey(const GWEN_CRYPTKEY *key,
                              unsigned int tagType,
                              int wantPublic,
                              int isCrypt,
                              GWEN_BUFFER *dbuf) {
  GWEN_DB_NODE *dbKey;
  GWEN_ERRORCODE err;
  const void *p;
  unsigned int bs;
  const char *s;
  char *pp;
  GWEN_TYPE_UINT32 pos;
  char numbuf[16];

  if (!key) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "No key");
    return 0;
  }
  dbKey=GWEN_DB_Group_new("key");
  err=GWEN_CryptKey_ToDb(key, dbKey, wantPublic);
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_DB_Group_free(dbKey);
    return -1;
  }

  GWEN_Buffer_AppendByte(dbuf, tagType & 0xff);
  /* remember pos to insert size later */
  pos=GWEN_Buffer_GetPos(dbuf);
  GWEN_Buffer_AppendBytes(dbuf, "00", 2);

  /* always write "NO" for "isPublic", since OpenHBCI always writes "NO"
   * due to a bug */
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_ISPUBLIC,
                             "NO",
                             -1,
                             dbuf);
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_ISCRYPT,
                             isCrypt?"YES":"NO",
                             -1,
                             dbuf);
  s=GWEN_CryptKey_GetOwner(key);
  if (s)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_OWNER,
                               s,
                               -1,
                               dbuf);

  snprintf(numbuf, sizeof(numbuf), "%d", GWEN_CryptKey_GetNumber(key));
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_NUMBER,
                             numbuf,
                             -1,
                             dbuf);
  snprintf(numbuf, sizeof(numbuf), "%d", GWEN_CryptKey_GetVersion(key));
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_VERSION,
                             numbuf,
                             -1,
                             dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/e", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_EXP, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/n", 0, 0, 0, &bs);
  if (p && bs) {
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_MODULUS, p, bs, dbuf);
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_N, p, bs, dbuf);
  }
  else {
    DBG_WARN(AQHBCI_LOGDOMAIN, "No modulus !");
  }

  p=GWEN_DB_GetBinValue(dbKey, "data/p", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_P, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/q", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_Q, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/d", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_D, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/dmp1", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_DMP1, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/dmq1", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_DMQ1, p, bs, dbuf);

  p=GWEN_DB_GetBinValue(dbKey, "data/iqmp", 0, 0, 0, &bs);
  if (p && bs)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_KEY_IQMP, p, bs, dbuf);

  GWEN_DB_Group_free(dbKey);
  bs=(GWEN_Buffer_GetPos(dbuf)-pos)-2;
  pp=GWEN_Buffer_GetStart(dbuf)+pos;
  pp[0]=bs & 0xff;
  pp[1]=(bs>>8) & 0xff;

  return 0;
}



int AH_MediumOHBCI_Encode(AH_MEDIUM *m, GWEN_BUFFER *dbuf) {
  AH_MEDIUM_OHBCI *mrdh;
  char numbuf[16];
  const char *p;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* remember pos to insert size later */
  snprintf(numbuf, sizeof(numbuf), "%d", AH_MEDIUM_OHBCI_VMAJOR);
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_VERSION_MAJOR,
                             numbuf, -1, dbuf);

  snprintf(numbuf, sizeof(numbuf), "%d", AH_MEDIUM_OHBCI_VMINOR);
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_VERSION_MINOR,
                             numbuf, -1, dbuf);

  snprintf(numbuf, sizeof(numbuf), "%d", mrdh->localSignSeq);
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_SEQ,
                             numbuf, -1, dbuf);

  if (AH_MediumOHBCI__EncodeKey(mrdh->localSignKey,
                                AH_MEDIUM_OHBCI_TAG_USER_PUBSIGNKEY,
                                1, 0, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localSignKey,
                                AH_MEDIUM_OHBCI_TAG_USER_PRIVSIGNKEY,
                                0, 0, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localCryptKey,
                                AH_MEDIUM_OHBCI_TAG_USER_PUBCRYPTKEY,
                                1, 1, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localCryptKey,
                                AH_MEDIUM_OHBCI_TAG_USER_PRIVCRYPTKEY,
                                0, 1, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localSignKeyBak,
                                AH_MEDIUM_OHBCI_TAG_TEMP_PUBSIGNKEY,
                                1, 0, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localSignKeyBak,
                                AH_MEDIUM_OHBCI_TAG_TEMP_PRIVSIGNKEY,
                                0, 0, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localCryptKeyBak,
                                AH_MEDIUM_OHBCI_TAG_TEMP_PUBCRYPTKEY,
                                1, 1, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->localCryptKeyBak,
                                AH_MEDIUM_OHBCI_TAG_TEMP_PRIVCRYPTKEY,
                                0, 1, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (mrdh->userId)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_USER_ID,
                               mrdh->userId, -1, dbuf);

  if (AH_MediumOHBCI__EncodeKey(mrdh->remoteSignKey,
                                AH_MEDIUM_OHBCI_TAG_INST_PUBSIGNKEY,
                                1, 0, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  if (AH_MediumOHBCI__EncodeKey(mrdh->remoteCryptKey,
                                AH_MEDIUM_OHBCI_TAG_INST_PUBCRYPTKEY,
                                1, 1, dbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key");
    return -1;
  }

  snprintf(numbuf, sizeof(numbuf), "%d", mrdh->country);
  OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_INST_COUNTRY,
                             numbuf, -1, dbuf);

  if (mrdh->bankId)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_INST_CODE,
                               mrdh->bankId, -1, dbuf);

  if (mrdh->systemId)
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_INST_SYSTEMID,
                               mrdh->systemId, -1, dbuf);

  /* new in 1.4 */
  p=AH_Medium_GetPeerAddr(m);
  if (p) {
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_SERVER_ADDR,
                               p, -1, dbuf);
    snprintf(numbuf, sizeof(numbuf), "%d",
             AH_Medium_GetPeerPort(m));
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_SERVER_PORT,
                               numbuf, -1, dbuf);
    snprintf(numbuf, sizeof(numbuf), "%d",
             mrdh->remoteSignSeq);
    OHBCI_TLV_DirectlyToBuffer(AH_MEDIUM_OHBCI_TAG_REMOTE_SEQ,
                               numbuf, -1, dbuf);
  }

  return 0;
}






int AH_MediumOHBCI_Mount(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;
  char password[64];
  int fd;
  int rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* create key from password */
  password[0]=0;
  mrdh->passWordIsSet=0;
  mrdh->cryptoTag=AH_MEDIUM_OHBCI_TAG_CRYPT;
  memset(mrdh->password, 0, sizeof(mrdh->password));

  /* overwrite password as soon as possible */
  memset(password, 0, sizeof(password));
  mrdh->passWordIsSet=0;

  /* open file */
  fd=AH_MediumOHBCI__OpenFile(m, 0);
  if (fd==-1) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not open keyfile for reading");
    return -1;
  }

  /* read file */
  rv=AH_MediumOHBCI__ReadFile(m, fd);
  if (rv) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reading keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return rv;
  }

  /* close file */
  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return -1;
  }

  return 0;
}



int AH_MediumOHBCI__ReloadIfNeeded(AH_MEDIUM *m, int fd){
  AH_MEDIUM_OHBCI *mrdh;
  struct stat st;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "stat(%s): %s",
              AH_Medium_GetMediumName(m),
              strerror(errno));
    return -1;
  }
  if (mrdh->mtime!=st.st_mtime ||
      mrdh->ctime!=st.st_ctime) {
    int rv;

    /* file has chenged, reload it */
    DBG_NOTICE(AQHBCI_LOGDOMAIN, "Keyfile changed externally, reloading it");
    /* read file */
    rv=AH_MediumOHBCI__ReadFile(m, fd);
    if (rv) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Error reading keyfile");
      return rv;
    }
  }
  else {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Keyfile unchanged, not reloading");
  }
  return 0;
}



int AH_MediumOHBCI_Create(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;
  struct stat st;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_GetMediumName(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium name given");
    return -1;
  }

  if (stat(AH_Medium_GetMediumName(m), &st)) {
    if (errno!=ENOENT) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "stat(%s): %s",
                AH_Medium_GetMediumName(m),
                strerror(errno));
      return -1;
    }
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
              "Keyfile \"%s\" already exists, will not create it",
              AH_Medium_GetMediumName(m));
    return -1;
  }


  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* write file (just to be sure that the last change is on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return 0;
}



int AH_MediumOHBCI__WriteFile(AH_MEDIUM *m, int fd){
  AH_MEDIUM_OHBCI *mrdh;
  int rv;
  GWEN_BUFFER *fbuf;
  GWEN_BUFFER *rawbuf;
  GWEN_CRYPTKEY *key;
  GWEN_ERRORCODE err;
  struct stat st;
  char *p;
  unsigned int bs;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_GetMediumName(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium name given");
    return -1;
  }

  /* create raw data */
  rawbuf=GWEN_Buffer_new(0, 1024, 0, 1);
  if (AH_MediumOHBCI_Encode(m, rawbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not encode key file");
    return -1;
  }

#ifdef DEBUG_OHBCI_MODULE
  if (1) {
    FILE *f;

    f=fopen("encoded.medium", "w+b");
    if (f) {
      if (1!=fwrite(GWEN_Buffer_GetStart(rawbuf),
                    GWEN_Buffer_GetUsedBytes(rawbuf),
                    1, f)) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not save encoded file.");
      }
      if (fclose(f)) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close encoded file.");
      }
    }
    else {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open encoded file.");
    }
  }
#endif


  /* create key from password */
  if (!mrdh->passWordIsSet) {
    char password[64];

    password[0]=0;
    if (AH_Medium_InputPin(m,
                           password,
                           AH_MEDIUM_OHBCI_PINMINLENGTH,
                           sizeof(password),
                           AB_BANKING_INPUT_FLAGS_CONFIRM)) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
      GWEN_Buffer_free(rawbuf);
      return -1;
    }
    if (strlen(password)<AH_MEDIUM_OHBCI_PINMINLENGTH) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
      GWEN_Buffer_free(rawbuf);
      return -1;
    }

    if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT) {
      if (GWEN_CryptKey_FromPassword(password,
                                     mrdh->password,
                                     sizeof(mrdh->password))) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
        return -1;
      }
    }
    else if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT_OLD) {
      if (GWEN_CryptKey_FromPasswordSSL(password,
                                        mrdh->password,
                                        sizeof(mrdh->password))) {
        DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
        return -1;
      }
    }
    else {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Unexpected crypto tag %d", mrdh->cryptoTag);
      abort();
    }

    /* overwrite password as soon as possible */
    AH_Medium_SetPinStatus(m, password, AB_Banking_PinStatusOk);
    memset(password, 0, sizeof(password));
    mrdh->passWordIsSet=1;
  } /* if password is not set */

  key=GWEN_CryptKey_Factory("DES");
  if (!key) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES key");
    GWEN_Buffer_free(rawbuf);
    return -1;
  }
  err=GWEN_CryptKey_SetData(key, mrdh->password, sizeof(mrdh->password));
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(key);
    GWEN_Buffer_free(rawbuf);
    return -1;
  }

  /* padd raw data */
  if (GWEN_Padd_PaddWithANSIX9_23(rawbuf)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not padd keyfile");
    GWEN_CryptKey_free(key);
    GWEN_Buffer_free(rawbuf);
    return -1;
  }

  /* encrypt file */
  fbuf=GWEN_Buffer_new(0, 256, 0, 1);
  GWEN_Buffer_ReserveBytes(fbuf, 3);

  err=GWEN_CryptKey_Encrypt(key, rawbuf, fbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(fbuf);
    GWEN_CryptKey_free(key);
    GWEN_Buffer_free(rawbuf);
    return -1;
  }
  GWEN_Buffer_free(rawbuf);
  GWEN_CryptKey_free(key);

  GWEN_Buffer_Rewind(fbuf);
  bs=GWEN_Buffer_GetUsedBytes(fbuf);
  GWEN_Buffer_InsertBytes(fbuf, "000", 3);
  p=GWEN_Buffer_GetStart(fbuf);
  p[0]=(unsigned char)(mrdh->cryptoTag);
  p[1]=(unsigned char)(bs & 0xff);
  p[2]=(unsigned char)((bs>>8) & 0xff);

  if (ftruncate(fd, 0)==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "ftruncate(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }

  while (1) {
    rv=write(fd,
             GWEN_Buffer_GetPosPointer(fbuf),
             GWEN_Buffer_GetBytesLeft(fbuf));
    if (rv==-1) {
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "write(%s): %s",
                AH_Medium_GetMediumName(m),
		strerror(errno));
      GWEN_Buffer_free(fbuf);
      return -1;
    }
    else if (rv==0)
      break;
    GWEN_Buffer_IncrementPos(fbuf, rv);
  } /* while */

  /* get times */
  if (fstat(fd, &st)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN,
	      "stat(%s): %s",
	      AH_Medium_GetMediumName(m),
	      strerror(errno));
    GWEN_Buffer_free(fbuf);
    return -1;
  }

#ifndef OS_WIN32
  if (st.st_mode & 0007) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "WARNING: Your keyfile \"%s\" is world accessible!\n"
             "Nobody but you should have access to the file. You \n"
	     "should probably change this with \"chmod 600 %s\"",
             AH_Medium_GetMediumName(m),
             AH_Medium_GetMediumName(m));
  }
#endif
  mrdh->mtime=st.st_mtime;
  mrdh->ctime=st.st_ctime;

  GWEN_Buffer_free(fbuf);

  return 0;
}



int AH_MediumOHBCI_Unmount(AH_MEDIUM *m, int force){
  AH_MEDIUM_OHBCI *mrdh;
  int rv;
  unsigned int i;
  int fd;
  const char *s;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* sample vars */
  s=AH_Medium_GetSecurityIdPtr(m);
  free(mrdh->systemId);
  if (s) mrdh->systemId=strdup(s);
  else mrdh->systemId=0;

  /* open file */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not open keyfile for writing");
    return -1;
  }

  /* write file */
  rv=AH_MediumOHBCI__WriteFile(m, fd);
  if (rv) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return rv;
  }

  /* close file */
  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return -1;
  }

  /* overwrite password */
  mrdh->passWordIsSet=0;
  for (i=0; i<sizeof(mrdh->password); i++)
    mrdh->password[i]=0;

  /* free all keys */
  GWEN_CryptKey_free(mrdh->localSignKey);
  mrdh->localSignKey=0;
  GWEN_CryptKey_free(mrdh->localCryptKey);
  mrdh->localCryptKey=0;
  GWEN_CryptKey_free(mrdh->remoteSignKey);
  mrdh->remoteSignKey=0;
  GWEN_CryptKey_free(mrdh->remoteCryptKey);
  mrdh->remoteCryptKey=0;
  GWEN_CryptKey_free(mrdh->localSignKeyBak);
  mrdh->localSignKeyBak=0;
  GWEN_CryptKey_free(mrdh->localCryptKeyBak);
  mrdh->localCryptKeyBak=0;

  AH_Medium_SetLocalSignKeySpec(m, 0);
  AH_Medium_SetLocalCryptKeySpec(m, 0);
  AH_Medium_SetRemoteSignKeySpec(m, 0);
  AH_Medium_SetRemoteCryptKeySpec(m, 0);

  mrdh->mtime=0;
  mrdh->ctime=0;

  mrdh->selected=0;
  return 0;
}



AH_MEDIUM_RESULT AH_MediumOHBCI_Sign(AH_MEDIUM *m,
                                     GWEN_BUFFER *msgbuf,
                                     GWEN_BUFFER *signbuf){
  AH_MEDIUM_OHBCI *mrdh;
  GWEN_BUFFER *hashbuf;
  GWEN_ERRORCODE err;
  char buffer[20];
  unsigned int bsize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    AH_MediumOHBCI__CloseFile(m, fd);
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    return AH_MediumResultGenericError;
  }

  if (mrdh->localSignKey==0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No local sign key");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultNoKey;
  }

  /* hash data */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hash data");
  bsize=sizeof(buffer);
  if (GWEN_MD_Hash("RMD160",
                   GWEN_Buffer_GetStart(msgbuf),
                   GWEN_Buffer_GetUsedBytes(msgbuf),
                   buffer,
                   &bsize)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error hashing message");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing done");

  hashbuf=GWEN_Buffer_new(0, bsize, 0, 1);
  GWEN_Buffer_AppendBytes(hashbuf, buffer, bsize);

  /* padd */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Padding hash using ISO 9796");
  if (GWEN_Padd_PaddWithISO9796(hashbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(hashbuf);
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  /* sign hash */
  GWEN_Buffer_Rewind(hashbuf);
  err=GWEN_CryptKey_Sign(mrdh->localSignKey,
                         hashbuf,
                         signbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_Buffer_free(hashbuf);
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  GWEN_Buffer_free(hashbuf);

  /* increment signature counter */
  mrdh->localSignSeq++;

  /* write file (just to be sure that the last change in on disc) */
  if (AH_MediumOHBCI__WriteFile(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Signing done");
  return AH_MediumResultOk;
}



int AH_MediumOHBCI_GetNextSignSeq(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  return mrdh->localSignSeq;
}



int AH_MediumOHBCI_SetLocalSignSeq(AH_MEDIUM *m, int i) {
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (i<mrdh->localSignSeq) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "New localSignSeq is lesser than current one, "
             "this is not a good idea");
  }
  mrdh->localSignSeq=i;
  return 0;
}



AH_MEDIUM_RESULT AH_MediumOHBCI_Verify(AH_MEDIUM *m,
                                       GWEN_BUFFER *msgbuf,
                                       GWEN_BUFFER *signbuf,
                                       int signseq){
  AH_MEDIUM_OHBCI *mrdh;
  GWEN_BUFFER *hashbuf;
  GWEN_ERRORCODE err;
  char buffer[20];
  unsigned int bsize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* open and lock file (open for writing) */
  fd=AH_MediumOHBCI__OpenFile(m, 1);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  if (mrdh->remoteSignKey==0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No remote sign key");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultNoKey;
  }

  /* check signature counter */
  if (signseq) {
    if (mrdh->remoteSignSeq>=signseq){
      DBG_ERROR(AQHBCI_LOGDOMAIN,
                "Double use of signature detected (%d>=%d)",
                mrdh->remoteSignSeq, signseq);
      AH_MediumOHBCI__CloseFile(m, fd);
      return AH_MediumResultSignSeq;
    }
    mrdh->remoteSignSeq=signseq;

    /* write file (just to be sure that the last change is on disc) */
    if (AH_MediumOHBCI__WriteFile(m, fd)) {
      DBG_INFO(AQHBCI_LOGDOMAIN, "Error writing file");
      AH_MediumOHBCI__CloseFile(m, fd);
      return AH_MediumResultGenericError;
    }
  }

  /* hash data */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing data");
  bsize=sizeof(buffer);
  if (GWEN_MD_Hash("RMD160",
                   GWEN_Buffer_GetStart(msgbuf),
                   GWEN_Buffer_GetUsedBytes(msgbuf),
                   buffer,
                   &bsize)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Error hashing message");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Hashing done");

  hashbuf=GWEN_Buffer_new(0, bsize, 0, 1);
  GWEN_Buffer_AppendBytes(hashbuf, buffer, bsize);

  /* padd */
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Padding hash using ISO 9796");
  if (GWEN_Padd_PaddWithISO9796(hashbuf)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "here");
    GWEN_Buffer_free(hashbuf);
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  /* verify hash */
  GWEN_Buffer_Rewind(hashbuf);
  GWEN_Buffer_Rewind(signbuf);
  err=GWEN_CryptKey_Verify(mrdh->remoteSignKey,
                           hashbuf,
                           signbuf);
  if (!GWEN_Error_IsOk(err)) {
    DBG_INFO_ERR(AQHBCI_LOGDOMAIN, err);
    DBG_WARN(AQHBCI_LOGDOMAIN, "Invalid signature");
    GWEN_Buffer_free(hashbuf);
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultInvalidSignature;
  }
  DBG_INFO(AQHBCI_LOGDOMAIN, "Signature is valid");

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  GWEN_Buffer_free(hashbuf);
  DBG_DEBUG(AQHBCI_LOGDOMAIN, "Verification done");
  return AH_MediumResultOk;
}



AH_MEDIUM_RESULT AH_MediumOHBCI_EncryptKey(AH_MEDIUM *m,
                                                GWEN_BUFFER *srckey,
                                                GWEN_BUFFER *enckey){
  AH_MEDIUM_OHBCI *mrdh;
  int fd;
  AH_MEDIUM_RESULT rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumOHBCI__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  rv=AH_MediumRDH_EncryptKey(m, mrdh->remoteCryptKey, srckey, enckey);

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return rv;
}



AH_MEDIUM_RESULT AH_MediumOHBCI_DecryptKey(AH_MEDIUM *m,
                                           GWEN_BUFFER *srckey,
                                           GWEN_BUFFER *deckey){
  AH_MEDIUM_OHBCI *mrdh;
  int fd;
  AH_MEDIUM_RESULT rv;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumOHBCI__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return AH_MediumResultGenericError;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return AH_MediumResultGenericError;
  }

  rv=AH_MediumRDH_DecryptKey(m, mrdh->localCryptKey, srckey, deckey);

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return AH_MediumResultGenericError;
  }

  return rv;
}



GWEN_BUFFER *AH_MediumOHBCI_GenerateMsgKey(AH_MEDIUM *m){
  AH_MEDIUM_OHBCI *mrdh;
  GWEN_ERRORCODE err;
  GWEN_CRYPTKEY *sessionKey;
  GWEN_BUFFER *kbuf;
  unsigned char skbuffer[16];
  unsigned int sksize;
  int fd;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  /* open and lock file (open for reading) */
  fd=AH_MediumOHBCI__OpenFile(m, 0);
  if (fd==-1) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not open/lock keyfile");
    return 0;
  }

  /* reload keyfile if necessary */
  if (AH_MediumOHBCI__ReloadIfNeeded(m, fd)) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "Error reloading keyfile");
    AH_MediumOHBCI__CloseFile(m, fd);
    return 0;
  }

  /* generate session key */
  sessionKey=GWEN_CryptKey_Factory("DES");
  if (!sessionKey) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create DES session key");
    AH_MediumOHBCI__CloseFile(m, fd);
    return 0;
  }
  err=GWEN_CryptKey_Generate(sessionKey, 0);
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(sessionKey);
    AH_MediumOHBCI__CloseFile(m, fd);
    return 0;
  }

  sksize=sizeof(skbuffer);
  err=GWEN_CryptKey_GetData(sessionKey,
                            skbuffer,
                            &sksize);
  if (!GWEN_Error_IsOk(err)) {
    DBG_ERROR_ERR(AQHBCI_LOGDOMAIN, err);
    GWEN_CryptKey_free(sessionKey);
    AH_MediumOHBCI__CloseFile(m, fd);
    return 0;
  }
  GWEN_CryptKey_free(sessionKey);

  if (AH_MediumOHBCI__CloseFile(m, fd)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not close keyfile");
    return 0;
  }

  kbuf=GWEN_Buffer_new(0, sksize, 0, 1);
  GWEN_Buffer_AppendBytes(kbuf, skbuffer, sksize);
  return kbuf;
}



int AH_MediumOHBCI_ChangePin(AH_MEDIUM *m){
  char password[64];
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }

  /* create key from password */
  memset(password, 0, sizeof(password));
  if (AH_Medium_InputPin(m,
                         password,
                         AH_MEDIUM_OHBCI_PINMINLENGTH,
                         sizeof(password),
                         AB_BANKING_INPUT_FLAGS_CONFIRM)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not get PIN");
    return -1;
  }
  if (strlen(password)<AH_MEDIUM_OHBCI_PINMINLENGTH) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Your program returned a shorter PIN than instructed!");
    return -1;
  }

  if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT) {
    if (GWEN_CryptKey_FromPassword(password,
                                   mrdh->password,
                                   sizeof(mrdh->password))) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
      return -1;
    }
  }
  else if (mrdh->cryptoTag==AH_MEDIUM_OHBCI_TAG_CRYPT_OLD) {
    if (GWEN_CryptKey_FromPasswordSSL(password,
                                      mrdh->password,
                                      sizeof(mrdh->password))) {
      DBG_ERROR(AQHBCI_LOGDOMAIN, "Could not create key data from password");
      return -1;
    }
  }
  else {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Unexpected crypto tag %d", mrdh->cryptoTag);
    abort();
  }
  /* overwrite password as soon as possible */
  AH_Medium_SetPinStatus(m, password, AB_Banking_PinStatusOk);
  memset(password, 0, sizeof(password));
  mrdh->passWordIsSet=1;

  return 0;
}



int AH_MediumOHBCI_SelectContext(AH_MEDIUM *m, int idx) {
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  if (mrdh->localSignKey)
    AH_Medium_SetLocalSignKeySpec(m,
                                  GWEN_CryptKey_GetKeySpec(mrdh->localSignKey));

  if (mrdh->localCryptKey)
    AH_Medium_SetLocalCryptKeySpec(m,
                                   GWEN_CryptKey_GetKeySpec(mrdh->localCryptKey));

  if (mrdh->remoteSignKey)
    AH_Medium_SetRemoteSignKeySpec(m,
                                   GWEN_CryptKey_GetKeySpec(mrdh->remoteSignKey));

  if (mrdh->remoteCryptKey)
    AH_Medium_SetRemoteCryptKeySpec(m,
                                    GWEN_CryptKey_GetKeySpec(mrdh->remoteCryptKey));

  mrdh->selected=1;
  return 0;
}



int AH_MediumOHBCI_CreateContext(AH_MEDIUM *m,
                                 int country,
                                 const char *bankId,
                                 const char *userId){
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  assert(bankId);
  assert(userId);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }

  if (mrdh->selected ||
      mrdh->country ||
      mrdh->bankId ||
      mrdh->userId) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium already contains a user");
    return -1;
  }

  mrdh->country=country;
  free(mrdh->bankId);
  mrdh->bankId=strdup(bankId);
  free(mrdh->userId);
  mrdh->userId=strdup(userId);

  return 0;
}



int AH_MediumOHBCI_RemoveContext(AH_MEDIUM *m, int idx) {
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  mrdh->country=0;
  free(mrdh->bankId); mrdh->bankId=0;
  free(mrdh->userId); mrdh->userId=0;

  mrdh->selected=0;
  return 0;
}



int AH_MediumOHBCI_ReadContext(AH_MEDIUM *m,
                               int idx,
                               int *country,
                               GWEN_BUFFER *bankId,
                               GWEN_BUFFER *userId,
                               GWEN_BUFFER *server,
                               int *port){
  AH_MEDIUM_OHBCI *mrdh;
  const char *s;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx>0 || mrdh->country==0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }
  if (country)
    *country=mrdh->country;
  if (bankId && mrdh->bankId)
    GWEN_Buffer_AppendString(bankId, mrdh->bankId);
  if (userId && mrdh->userId)
    GWEN_Buffer_AppendString(userId, mrdh->userId);
  s=AH_Medium_GetPeerAddr(m);
  if (server && s)
    GWEN_Buffer_AppendString(server, s);
  if (port)
    *port=AH_Medium_GetPeerPort(m);
  return 0;
}




int AH_MediumOHBCI_WriteContext(AH_MEDIUM *m,
                                int idx,
                                int country,
                                const char *bankId,
                                const char *userId,
                                const char *server,
                                int port){
  AH_MEDIUM_OHBCI *mrdh;

  assert(m);
  mrdh=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_OHBCI, m);
  assert(mrdh);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mrdh->selected=0;
    return -1;
  }

  if (idx>0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }

  if (country)
    mrdh->country=country;
  if (bankId) {
    free(mrdh->bankId);
    mrdh->bankId=strdup(bankId);
  }

  if (userId) {
    free(mrdh->userId);
    mrdh->userId=strdup(userId);
  }

  if (server)
    AH_Medium_SetPeerAddr(m, server);

  if (port)
    AH_Medium_SetPeerPort(m, port);
  return 0;
}














AH_MEDIUMPROVIDER *ohbci_MediumProvider_new(AH_HBCI *hbci){
  AH_MEDIUMPROVIDER *mp;

  mp=AH_MediumProvider_new(hbci, AH_MEDIUM_OHBCI_NAME);
  AH_MediumProvider_SetDeviceType(mp, AH_MediumDeviceFile);
  AH_MediumProvider_SetFactoryFn(mp, AH_MediumOHBCIProvider_Factory);
  AH_MediumProvider_SetCheckFn(mp, AH_MediumOHBCIProvider_Check);

  return mp;
}



AH_MEDIUM *AH_MediumOHBCIProvider_Factory(AH_MEDIUMPROVIDER *mp,
                                          AH_HBCI *hbci,
                                          const char *mediumName) {
  return AH_MediumOHBCI_new(hbci,
                            mp,
                            mediumName);
}



AH_MEDIUM_CHECKRESULT AH_MediumOHBCIProvider_Check(AH_MEDIUMPROVIDER *mp,
                                                   AH_HBCI *hbci,
                                                   GWEN_BUFFER *mediumName) {
  FILE *f;
  const char *p;
  unsigned char buffer[3];
  int rv;

  if (GWEN_Buffer_GetUsedBytes(mediumName)==0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Empty name");
    return AH_MediumCheckResultWrongName;
  }

  p=GWEN_Buffer_GetStart(mediumName);
  if (access(p, F_OK)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "File does not exist");
    GWEN_WaitCallback_Log(0, "File does not exist");
    return AH_MediumCheckResultWrongName;
  }

  if (access(p, R_OK | W_OK)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "File exists but I have no writes on it");
    GWEN_WaitCallback_Log(0, "File exists but I have no writes on it");
    return AH_MediumCheckResultGenericError;
  }

  f=fopen(p, "rb");
  if (!f) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "File exists, I have all rights but still can't open it");
    GWEN_WaitCallback_Log(0, "File exists, I have all rights but "
                          "still can't open it");
    return AH_MediumCheckResultGenericError;
  }

  rv=fread(buffer, 3, 1, f);
  fclose(f);
  if (rv<0) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "This seems not to be an OpenHBCI keyfile");
    GWEN_WaitCallback_Log(1, "This seems not to be an OpenHBCI keyfile");
    return AH_MediumCheckResultWrongMedium;
  }

  if (rv!=1) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "This seems not to be an OpenHBCI keyfile (bad size)");
    GWEN_WaitCallback_Log(1, "This seems not to be an OpenHBCI keyfile "
                          "(bad size)");
    return AH_MediumCheckResultWrongMedium;
  }

  if (buffer[0]!=AH_MEDIUM_OHBCI_TAG_CRYPT &&
      buffer[0]!=AH_MEDIUM_OHBCI_TAG_CRYPT_OLD) {
    DBG_INFO(AQHBCI_LOGDOMAIN, "This seems not to be an OpenHBCI keyfile (bad data)");
    GWEN_WaitCallback_Log(1, "This seems not to be an OpenHBCI keyfile "
                          "(bad data)");
    return AH_MediumCheckResultWrongMedium;
  }

  /* do nothing for now */
  return AH_MediumCheckResultOk;
}












