/***************************************************************************
 $RCSfile: pintan.c,v $
                             -------------------
    cvs         : $Id: pintan.c,v 1.4 2005/03/28 02:35:15 aquamaniac Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "pintan_p.h"
#include "aqhbci_l.h"

#include <gwenhywfar/debug.h>
#include <gwenhywfar/misc.h>
#include <gwenhywfar/padd.h>
#include <gwenhywfar/md.h>
#include <gwenhywfar/waitcallback.h>

#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#ifdef OS_WIN32
# define ftruncate chsize
#endif


GWEN_INHERIT(AH_MEDIUM, AH_MEDIUM_PINTAN);



AH_MEDIUM *AH_Medium_PinTan_new(AH_HBCI *hbci, const char *mediumName) {
  AH_MEDIUM *m;
  AH_MEDIUM_PINTAN *mpt;

  m=AH_MediumPINTAN_new(hbci,
                        AH_MEDIUM_PINTAN_NAME,
                        mediumName);
  GWEN_NEW_OBJECT(AH_MEDIUM_PINTAN, mpt);
  GWEN_INHERIT_SETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN,
                       m, mpt,
                       AH_Medium_PinTan_FreeData);
  AH_Medium_SetDeviceType(m, AH_MediumDevicePseudo);

  /* set implemented functions for Medium */
  AH_Medium_SetChangePinFn(m, AH_Medium_PinTan_ChangePin);
  AH_Medium_SetMountFn(m, AH_Medium_PinTan_Mount);
  AH_Medium_SetCreateFn(m, AH_Medium_PinTan_Create);
  AH_Medium_SetUnmountFn(m, AH_Medium_PinTan_Unmount);
  AH_Medium_SetSignFn(m, AH_Medium_PinTan_Sign);
  AH_Medium_SetGetNextSignSeqFn(m, AH_Medium_PinTan_GetNextSignSeq);
  AH_Medium_SetSetLocalSignSeqFn(m, AH_Medium_PinTan_SetLocalSignSeq);
  AH_Medium_SetVerifyFn(m, AH_Medium_PinTan_Verify);
  AH_Medium_SetEncryptKeyFn(m, AH_Medium_PinTan_EncryptKey);
  AH_Medium_SetDecryptKeyFn(m, AH_Medium_PinTan_DecryptKey);
  AH_Medium_SetGenerateMsgKeyFn(m, AH_Medium_PinTan_GenerateMsgKey);
  AH_Medium_SetSelectContextFn(m, AH_Medium_PinTan_SelectContext);
  AH_Medium_SetCreateContextFn(m, AH_Medium_PinTan_CreateContext);
  AH_Medium_SetRemoveContextFn(m, AH_Medium_PinTan_RemoveContext);
  AH_Medium_SetReadContextFn(m, AH_Medium_PinTan_ReadContext);
  AH_Medium_SetWriteContextFn(m, AH_Medium_PinTan_WriteContext);
  AH_Medium_SetToDbFn(m, AH_Medium_PinTan_toDb);
  AH_Medium_SetFromDbFn(m, AH_Medium_PinTan_fromDb);

  /* set implemented functions for MediumPINTAN */
  AH_MediumPINTAN_SetGetPinFn(m, AH_Medium_PinTan_GetPin);
  AH_MediumPINTAN_SetGetTanFn(m, AH_Medium_PinTan_GetTan);
  AH_MediumPINTAN_SetSetTanStatusFn(m, AH_Medium_PinTan_SetTanStatus);

  return m;
}



void AH_Medium_PinTan_FreeData(void *bp, void *p){
  AH_MEDIUM_PINTAN *mpt;

  mpt=(AH_MEDIUM_PINTAN*)p;

  free(mpt->bankId);
  free(mpt->userId);
  free(mpt->systemId);
  GWEN_FREE_OBJECT(mpt);
}



int AH_Medium_PinTan_Mount(AH_MEDIUM *m){
  AH_MEDIUM_PINTAN *mpt;
  GWEN_KEYSPEC *ks;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  /* setup default keyspecs */
  ks=GWEN_KeySpec_new();
  GWEN_KeySpec_SetKeyType(ks, "RSA");
  GWEN_KeySpec_SetKeyName(ks, "S");
  GWEN_KeySpec_SetNumber(ks, 1);
  GWEN_KeySpec_SetVersion(ks, 1);

  AH_Medium_SetLocalSignKeySpec(m, ks);
  AH_Medium_SetRemoteSignKeySpec(m, ks);

  GWEN_KeySpec_SetKeyName(ks, "V");
  AH_Medium_SetLocalCryptKeySpec(m, ks);
  AH_Medium_SetRemoteCryptKeySpec(m, ks);
  GWEN_KeySpec_free(ks);

  return 0;
}



int AH_Medium_PinTan_Create(AH_MEDIUM *m){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_GetMediumName(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "No medium name given");
    return -1;
  }

  return 0;
}



int AH_Medium_PinTan_Unmount(AH_MEDIUM *m, int force){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  /* free all keys */
  AH_Medium_SetLocalSignKeySpec(m, 0);
  AH_Medium_SetLocalCryptKeySpec(m, 0);
  AH_Medium_SetRemoteSignKeySpec(m, 0);
  AH_Medium_SetRemoteCryptKeySpec(m, 0);

  mpt->selected=0;
  return 0;
}



AH_MEDIUM_RESULT AH_Medium_PinTan_Sign(AH_MEDIUM *m,
                                          GWEN_BUFFER *msgbuf,
                                          GWEN_BUFFER *signbuf){
  return AH_MediumResultNotSupported;
}



int AH_Medium_PinTan_GetNextSignSeq(AH_MEDIUM *m){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  return mpt->localSignSeq;
}



int AH_Medium_PinTan_SetLocalSignSeq(AH_MEDIUM *m, int i) {
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (i<mpt->localSignSeq) {
    DBG_WARN(AQHBCI_LOGDOMAIN,
             "New localSignSeq is lesser than current one, "
             "this is not a good idea");
  }
  mpt->localSignSeq=i;
  return 0;
}



AH_MEDIUM_RESULT AH_Medium_PinTan_Verify(AH_MEDIUM *m,
                                        GWEN_BUFFER *msgbuf,
					GWEN_BUFFER *signbuf,
					int signseq){
  return AH_MediumResultNotSupported;
}



AH_MEDIUM_RESULT AH_Medium_PinTan_EncryptKey(AH_MEDIUM *m,
					    GWEN_BUFFER *srckey,
					    GWEN_BUFFER *encKey){
  return AH_MediumResultNotSupported;
}



AH_MEDIUM_RESULT AH_Medium_PinTan_DecryptKey(AH_MEDIUM *m,
					    GWEN_BUFFER *srckey,
					    GWEN_BUFFER *deckey){
  return AH_MediumResultNotSupported;
}



int AH_Medium_PinTan_ChangePin(AH_MEDIUM *m){
  return AB_ERROR_NOT_SUPPORTED;
}



int AH_Medium_PinTan_SelectContext(AH_MEDIUM *m, int idx) {
  AH_MEDIUM_PINTAN *mpt;
  GWEN_KEYSPEC *ks;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mpt->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  mpt->selected=1;

  ks=GWEN_KeySpec_new();
  GWEN_KeySpec_SetKeyType(ks, "RSA");
  GWEN_KeySpec_SetKeyName(ks, "S");
  GWEN_KeySpec_SetOwner(ks, mpt->userId);
  GWEN_KeySpec_SetNumber(ks, 1);
  GWEN_KeySpec_SetVersion(ks, 1);

  AH_Medium_SetLocalSignKeySpec(m, ks);
  AH_Medium_SetRemoteSignKeySpec(m, ks);

  GWEN_KeySpec_SetKeyName(ks, "V");
  AH_Medium_SetLocalCryptKeySpec(m, ks);
  AH_Medium_SetRemoteCryptKeySpec(m, ks);
  GWEN_KeySpec_free(ks);
  return 0;
}



int AH_Medium_PinTan_CreateContext(AH_MEDIUM *m,
                                   int country,
                                   const char *bankId,
                                   const char *userId){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  assert(bankId);
  assert(userId);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    return -1;
  }

  if (mpt->selected ||
      mpt->country ||
      mpt->bankId ||
      mpt->userId) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium already contains a user");
  }

  mpt->country=country;
  free(mpt->bankId);
  mpt->bankId=strdup(bankId);
  free(mpt->userId);
  mpt->userId=strdup(userId);

  return 0;
}



int AH_Medium_PinTan_RemoveContext(AH_MEDIUM *m, int idx) {
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mpt->selected=0;
    return -1;
  }

  if (idx!=0) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Invalid index %d (only 0 is accepted)", idx);
    return -1;
  }

  mpt->country=0;
  free(mpt->bankId); mpt->bankId=0;
  free(mpt->userId); mpt->userId=0;

  mpt->selected=0;
  return 0;
}



int AH_Medium_PinTan_ReadContext(AH_MEDIUM *m,
                                 int idx,
                                 int *country,
                                 GWEN_BUFFER *bankId,
                                 GWEN_BUFFER *userId,
                                 GWEN_BUFFER *server,
                                 int *port){
  AH_MEDIUM_PINTAN *mpt;
  const char *s;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mpt->selected=0;
    return -1;
  }

  if (idx>0 || mpt->country==0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }
  if (country)
    *country=mpt->country;
  if (bankId && mpt->bankId)
    GWEN_Buffer_AppendString(bankId, mpt->bankId);
  if (userId && mpt->userId)
    GWEN_Buffer_AppendString(userId, mpt->userId);

  s=AH_Medium_GetPeerAddr(m);
  if (server && s)
    GWEN_Buffer_AppendString(server, s);
  if (port)
    *port=AH_Medium_GetPeerPort(m);
  return 0;
}




int AH_Medium_PinTan_WriteContext(AH_MEDIUM *m,
                                  int idx,
                                  int country,
                                  const char *bankId,
                                  const char *userId,
                                  const char *server,
                                  int port){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mpt->selected=0;
    return -1;
  }

  if (idx>0) {
    DBG_DEBUG(AQHBCI_LOGDOMAIN, "Context %d not found", idx);
    return -1;
  }

  if (country)
    mpt->country=country;
  if (bankId) {
    free(mpt->bankId);
    mpt->bankId=strdup(bankId);
  }

  if (userId) {
    free(mpt->userId);
    mpt->userId=strdup(userId);
  }

  if (server)
    AH_Medium_SetPeerAddr(m, server);

  if (port)
    AH_Medium_SetPeerPort(m, port);
  return 0;
}



GWEN_BUFFER *AH_Medium_PinTan_GenerateMsgKey(AH_MEDIUM *m){
  DBG_ERROR(AQHBCI_LOGDOMAIN,
            "AH_Medium_PinTan_GenerateMsgKey: Function not supported.");
  return 0;
}



int AH_Medium_PinTan_GetPin(AH_MEDIUM *m,
			    char *pwbuffer,
			    int minLen,
			    int maxLen,
			    int crea){
  AH_MEDIUM_PINTAN *mpt;
  int rv;
  GWEN_TYPE_UINT32 flags = AB_BANKING_INPUT_FLAGS_NUMERIC;
  const char *name;
  char buffer[512];

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (!AH_Medium_IsMounted(m)) {
    DBG_ERROR(AQHBCI_LOGDOMAIN, "Medium is not mounted");
    mpt->selected=0;
    return -1;
  }

  flags=(crea?AB_BANKING_INPUT_FLAGS_CONFIRM:0);

  assert(m);

  buffer[0]=0;
  buffer[sizeof(buffer)-1]=0;

  if (flags & AB_BANKING_INPUT_FLAGS_CONFIRM) {
    snprintf(buffer, sizeof(buffer)-1,
	     I18N("Please enter a new pin for \n"
		  "%s\n"
		  "The pin must be at least %d characters long."
		  "<html>"
		  "Please enter a new pin for <i>%s</i>. "
		  "The pin must be at least %d characters long."
		  "</html>"),
	     AH_Medium_GetDescriptiveName(m),
	     minLen,
	     AH_Medium_GetDescriptiveName(m),
	     minLen);
  }
  else {
    snprintf(buffer, sizeof(buffer)-1,
	     I18N("Please enter the pin for \n"
		  "%s\n"
		  "<html>"
		  "Please enter the pin for <i>%s</i>."
		  "</html>"),
	     AH_Medium_GetDescriptiveName(m),
	     AH_Medium_GetDescriptiveName(m));
  }

  name=AH_Medium_GetMediumName(m);
  if (name) {
    GWEN_BUFFER *nbuf;

    nbuf=GWEN_Buffer_new(0, 256 ,0 ,1);
    GWEN_Buffer_AppendString(nbuf, "PASSWORD::");
    GWEN_Buffer_AppendString(nbuf, name);
    rv=AB_Banking_GetPin(AH_HBCI_GetBankingApi(AH_Medium_GetHBCI(m)),
                         flags,
                         GWEN_Buffer_GetStart(nbuf),
			 I18N("Enter HBCI Pin"),
			 buffer,
			 pwbuffer,
                         minLen,
                         maxLen);
    GWEN_Buffer_free(nbuf);
  }
  else {
    rv=AB_Banking_InputBox(AH_HBCI_GetBankingApi(AH_Medium_GetHBCI(m)),
                           flags,
			   I18N("Enter HBCI Pin"),
                           buffer,
			   pwbuffer,
                           minLen,
                           maxLen);
  }
  return rv;
}



int AH_Medium_PinTan_GetTan(AH_MEDIUM *m,
                               char *buffer,
                               int minSize,
                               int maxSize){
  /* for now simply ask the user */
  return AH_Medium_InputTan(m,
                            buffer,
                            minSize, maxSize);
}



int AH_Medium_PinTan_SetTanStatus(AH_MEDIUM *m,
                                  const char *buffer,
                                  AH_MEDIUMPINTAN_TANSTATUS tst){
  /* for now simply forward this to the application */
  return AH_Medium_SetTanStatus(m, buffer, tst);
}



int AH_Medium_PinTan_toDb(const AH_MEDIUM *m, GWEN_DB_NODE *db){
  AH_MEDIUM_PINTAN *mpt;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  if (mpt->bankId)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
                         "bankId", mpt->bankId);
  if (mpt->userId)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
			 "userId", mpt->userId);
  if (mpt->systemId)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
			 "systemId", mpt->systemId);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
		      "country", mpt->country);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
		      "localSignSeq", mpt->localSignSeq);
  GWEN_DB_SetIntValue(db, GWEN_DB_FLAGS_OVERWRITE_VARS,
		      "remoteSignSeq", mpt->remoteSignSeq);

  return 0;
}



int AH_Medium_PinTan_fromDb(AH_MEDIUM *m, GWEN_DB_NODE *db){
  AH_MEDIUM_PINTAN *mpt;
  const char *s;

  assert(m);
  mpt=GWEN_INHERIT_GETDATA(AH_MEDIUM, AH_MEDIUM_PINTAN, m);
  assert(mpt);

  free(mpt->bankId);
  s=GWEN_DB_GetCharValue(db, "bankId", 0, 0);
  if (s) mpt->bankId=strdup(s);
  else mpt->bankId=0;

  free(mpt->userId);
  s=GWEN_DB_GetCharValue(db, "userId", 0, 0);
  if (s) mpt->userId=strdup(s);
  else mpt->userId=0;

  free(mpt->systemId);
  s=GWEN_DB_GetCharValue(db, "systemId", 0, 0);
  if (s) mpt->systemId=strdup(s);
  else mpt->systemId=0;

  mpt->country=GWEN_DB_GetIntValue(db, "country", 0, 280);
  mpt->localSignSeq=GWEN_DB_GetIntValue(db, "localSignSeq", 0, 1);
  mpt->remoteSignSeq=GWEN_DB_GetIntValue(db, "remoteSignSeq", 0, 0);

  return 0;
}















AH_MEDIUMPROVIDER *pintan_MediumProvider_new(AH_HBCI *hbci){
  AH_MEDIUMPROVIDER *mp;

  mp=AH_MediumProvider_new(hbci, AH_MEDIUM_PINTAN_NAME);
  AH_MediumProvider_SetDeviceType(mp, AH_MediumDevicePseudo);
  AH_MediumProvider_SetFactoryFn(mp, AH_Medium_PinTanProvider_Factory);
  AH_MediumProvider_SetCheckFn(mp, AH_Medium_PinTanProvider_Check);
  return mp;
}



AH_MEDIUM *AH_Medium_PinTanProvider_Factory(AH_MEDIUMPROVIDER *mp,
					   AH_HBCI *hbci,
					   const char *mediumName) {
  return AH_Medium_PinTan_new(hbci, mediumName);
}



AH_MEDIUM_CHECKRESULT
AH_Medium_PinTanProvider_Check(AH_MEDIUMPROVIDER *mp,
			      AH_HBCI *hbci,
			      GWEN_BUFFER *mediumName) {
  return AH_MediumCheckResultWrongMedium;
}


















