// -*- c++ -*-
//------------------------------------------------------------------------------
//                          Reactor.cpp
//------------------------------------------------------------------------------
//  Copyright (C) 1997-2002,2005  Vladislav Grinchenko
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Library General Public
//  License as published by the Free Software Foundation; either
//  version 2 of the License, or (at your option) any later version.
//----------------------------------------------------------------------------- 
//  Created: 05/25/1999
//----------------------------------------------------------------------------- 
#include <iostream>
#include <sstream>
#include <string>

#include "assa/Reactor.h"

using namespace ASSA;

Reactor::
Reactor () : 
	m_noFiles (1024), m_maxfd (0), m_active (true),
	m_readSet ((EventHandler**) NULL), 
	m_writeSet ((EventHandler**) NULL), 
	m_exceptSet ((EventHandler**) NULL)
{
    trace_with_mask("Reactor::Reactor",REACTTRACE);
	
    struct rlimit rlim;
    rlim.rlim_max = 0;

    if ( getrlimit (RLIMIT_NOFILE, &rlim) == 0 ) {
		m_noFiles = rlim.rlim_cur;
    }

    m_readSet = new EventHandler* [m_noFiles];
    m_writeSet = new EventHandler* [m_noFiles];
    m_exceptSet = new EventHandler* [m_noFiles];

    for (int i = 0; i < m_noFiles; i++) {
		m_readSet[i] = NULL;
		m_writeSet[i] = NULL;
		m_exceptSet[i] = NULL;
    }
}

Reactor::
~Reactor()
{	
    trace_with_mask("Reactor::~Reactor",REACTTRACE);

    delete [] m_readSet;
    delete [] m_writeSet;
    delete [] m_exceptSet;
}

TimerId
Reactor::
registerTimerHandler (EventHandler* eh_, 
					  const TimeVal& timeout_,
					  const std::string& name_)
{
    trace_with_mask( "Reactor::registerTimerHandler",REACTTRACE);
    Assure_return (eh_);

    TimeVal now (TimeVal::gettimeofday());
    TimeVal t (now + timeout_);

    DL((REACT,"TIMEOUT_EVENT: (%d,%d)\n",  timeout_.sec(),timeout_.msec()));
    DL((REACT,"Time now...........: %s\n", now.fmtString().c_str() ));
    DL((REACT,"Scheduled to expire: %s\n", t.fmtString().c_str() ));

    TimerId tid =  m_tqueue.insert (eh_, t, timeout_, name_);

    DL((REACT,"---Modified Timer Queue----\n"));
    m_tqueue.dump();
    DL((REACT,"---------------------------\n"));

    return (tid);
}

bool 
Reactor::
registerIOHandler (EventHandler* eh_, int fd_, EventType et_)
{
    trace_with_mask("Reactor::registerHandler(I/O)",REACTTRACE);

    std::ostringstream msg;
    Assure_return (eh_ && !isSignalEvent (et_) && !isTimeoutEvent (et_));

    if ( isReadEvent (et_) ) {

		if ( !m_waitSet.m_rset.setFd (fd_) ) {
			DL((ERROR,"readset: fd %d out of range\n", fd_));
			return (false);
		}
		m_readSet[fd_] = eh_;
		msg << "READ_EVENT";
    }

    if ( isWriteEvent (et_) ) {

		if ( !m_waitSet.m_wset.setFd (fd_) ) {
			DL((ERROR,"writeset: fd %d out of range\n", fd_));
			return (false);
		}
		m_writeSet[fd_] = eh_;
		msg << " WRITE_EVENT";
    }
    if ( isExceptEvent (et_) ) {
		if ( !m_waitSet.m_eset.setFd (fd_) ) {
			DL((ERROR,"exceptset: fd %d out of range\n", fd_));
			return (false);
		}
		m_exceptSet[fd_] = eh_;
		msg << " EXCEPT_EVENT";
    }
    msg << std::ends;
    DL((REACT,"Registered EventHandler 0x%x FD (%d) for event(s) %s\n", 
		(u_long)eh_, fd_, msg.str ().c_str () ));

    if ( m_maxfd < fd_+1 ) {
		m_maxfd = fd_+1;
		DL((REACT,"maxfd+1 adjusted to %d\n",m_maxfd));
    }
    DL((REACT,"Modified waitSet:\n"));
    m_waitSet.dump ();

    return (true);
}

bool 
Reactor::
removeTimerHandler (TimerId tid_)
{
    trace_with_mask("Reactor::removeTimer",REACTTRACE);
    bool ret;

    if ((ret = m_tqueue.remove (tid_))) {
		DL((REACT,"---Modified Timer Queue----\n"));
		m_tqueue.dump();
		DL((REACT,"---------------------------\n"));
    }
    else {
		EL((ERROR,"Timer tid 0x%x wasn't found!\n", (u_long)tid_ ));
    }
    return (ret);
}

bool 
Reactor::
removeHandler (EventHandler* eh_, EventType et_)
{
    trace_with_mask("Reactor::removeHandler(eh_,et_)",REACTTRACE);

	if (eh_ == NULL) {
		return false;
	}
    bool ret = false;
    register int fd;

    if (isTimeoutEvent (et_)) {
		ret = m_tqueue.remove (eh_);
    }

    if (isReadEvent (et_) || isWriteEvent (et_) || isExceptEvent (et_)) {
		for (fd = 0; fd < m_maxfd; fd++) {
			if (m_readSet[fd] == eh_ ||  m_writeSet[fd] == eh_ ||
				m_exceptSet[fd] == eh_) {
				ret = removeIOHandler (fd);
			}
		}
    }
    return (ret);
}

bool
Reactor::
removeIOHandler (int fd_)
{
    trace_with_mask("Reactor::removeIOHandler",REACTTRACE);

    Assure_return (fd_ >= 0 && fd_ < m_noFiles);

    DL((REACT,"Removing Handler fd = %d\n",fd_));

    EventHandler* eh = NULL;

    if      ( m_readSet[fd_] )   eh = m_readSet[fd_];
    else if ( m_writeSet[fd_] )  eh = m_writeSet[fd_];
    else if ( m_exceptSet[fd_] ) eh = m_exceptSet[fd_];

    if (eh) {
		DL((REACT,"Found EvtHandler 0x%x\n",(u_long)eh));
		eh->handle_close (fd_);
    }

    m_readSet[fd_] = NULL;
    m_writeSet[fd_] = NULL;
    m_exceptSet[fd_] = NULL;

    m_waitSet.m_rset.clear (fd_);
    m_waitSet.m_wset.clear (fd_);
    m_waitSet.m_eset.clear (fd_);

    m_readySet.m_rset.clear (fd_);
    m_readySet.m_wset.clear (fd_);
    m_readySet.m_eset.clear (fd_);

    if ( m_maxfd == fd_+1 ) {
		while ( m_maxfd > 0                    &&
				m_readSet  [m_maxfd-1] == NULL &&
				m_writeSet [m_maxfd-1] == NULL &&
				m_exceptSet[m_maxfd-1] == NULL )
			{
				m_maxfd--;
			}
    }
    DL((REACT,"maxfd+1 adjusted to %d\n", m_maxfd));
    DL((REACT,"Modifies waitSet:\n"));
    m_waitSet.dump ();

    return (true);
}

bool
Reactor::
checkFDs (void)
{
    trace_with_mask("Reactor::checkFDs",REACTTRACE);
	
    bool num_removed = false;
    FdSet mask;
    timeval poll = { 0, 0 };

    for (int fd = 0; fd < m_noFiles; fd++) {
		if ( m_readSet[fd] != NULL ) {
			mask.setFd (fd);
			if ( ::select (fd+1, &mask, NULL, NULL, &poll) < 0 ) {
				removeIOHandler (fd);
				num_removed = true;
				DL((REACT,"Detected BAD FD: %d\n", fd ));
			}
			mask.clear (fd);
		}
    }
    return (num_removed);
}

bool
Reactor::
handleError (void)
{
    trace_with_mask("Reactor::handleError",REACTTRACE);

    /*---  If commanded to stop, do so --*/
    if ( !m_active ) {
		DL((REACT,"Received cmd to stop Reactor\n"));
		return (false);
    }

    /*---
      TODO: If select(2) returns before time expires, with
      a descriptor ready or with EINTR, timeval is not
      going to be updated with number of seconds remaining.
      This is true for all systems except Linux, which will
      do so. Therefore, to restart correctly in case of
      EINTR, we ought to take time measurement before and
      after select, and try to select() for remaining time.
	
      For now, we restart with the initial timing value.
      ---*/
    /*---
      BSD kernel never restarts select(2). SVR4 will restart if
      the SA_RESTART flag is specified when the signal handler
      for the signal delivered is installed. This means taht for
      portability, we must handle signal interrupts.
      ---*/

    if ( errno == EINTR ) {
		EL((REACT,"EINTR: interrupted select(2)\n"));
		/*
		  If I was sitting in select(2) and received SIGTERM,
		  the signal handler would have set m_active to 'false',
		  and this function would have returned 'false' as above.
		  For any other non-critical signals (USR1,...),
		  we retry select.
		*/
		return (true);
    }
    /*
      EBADF - bad file number. One of the file descriptors does
      not reference an open file to open(), close(), ioctl().
      This can happen if user closed fd and forgot to remove
      handler from Reactor.
    */
    if ( errno == EBADF ) {
		DL((REACT,"EBADF: bad file descriptor\n"));
		return (checkFDs ());
    }
    /*
      Any other error from select
    */
    EL((ERROR,"select(3) error\n"));
    return (false);
}

int
Reactor::
isAnyReady (void)
{
    trace_with_mask("Reactor::isAnyReady",REACTTRACE);

    int n = m_readySet.m_rset.numSet () +
		m_readySet.m_wset.numSet () +
		m_readySet.m_eset.numSet ();

    if ( n > 0 ) {
		DL((REACT,"m_readySet: %d FDs are ready for processing\n", n));
		m_readySet.dump ();
    }
    return (n);
}

void 
Reactor::
calculateTimeout (TimeVal*& howlong_, TimeVal* maxwait_)
{
    trace_with_mask("Reactor::calculateTimeout",REACTTRACE);

    TimeVal now;
    TimeVal tv;

    if (m_tqueue.isEmpty () ) {
		howlong_ = maxwait_;
		goto done;
    }
    now = TimeVal::gettimeofday ();
    tv = m_tqueue.top ();
	
    if (tv < now) {
		/*--- 
		  It took too long to get here (fraction of a millisecond), 
		  and top timer had already expired. In this case,
		  perform non-blocking select in order to drain the timer queue.
		  ---*/
		*howlong_ = 0;
    }
    else {	
		DL((REACT,"--------- Timer Queue ----------\n"));
		m_tqueue.dump();
		DL((REACT,"--------------------------------\n"));

		if (maxwait_ == NULL || *maxwait_ == TimeVal::zeroTime ()) {
			*howlong_ = tv - now;
		}
		else {
			*howlong_ = (*maxwait_+now) < tv ? *maxwait_ : tv-now;
		}
    }

 done:
    if (howlong_ != NULL) {
		DL((REACT,"delay (%f)\n", double (*howlong_) ));
    }
    else {
		DL((REACT,"delay (forever)\n"));
    }
}

void
Reactor::
waitForEvents (void)
{
    while ( m_active ) {
		waitForEvents ((TimeVal*) NULL);
    }
}

/*******************************************************************************
   =====================================================================
   | select() | errno |      Events         | Behavior                 |
   |===================================================================|
   |    < 0   | EINTR | Interrup by signal  | Retry                    |
   +----------+-------+---------------------+--------------------------+
   |    < 0   | EBADF | Bad file descriptor | Remove bad fds and retry |
   |          |       |                     | and retry                |
   +----------+-------+---------------------+--------------------------+
   |    < 0   | others| Some other error    | Fall through             |
   +----------+-------+---------------------+--------------------------+
   |   == 0   |   0   | Timed out           | Fall through             |
   +----------+-------+---------------------+--------------------------+
   |    > 0   |   0   | Got some work to do | Fall through             |
   +-------------------------------------------------------------------+ 
*******************************************************************************/
void
Reactor::
waitForEvents (TimeVal* tv_)
{
    trace_with_mask("Reactor::waitForEvents",REACTTRACE);

    TimerCountdown traceTime (tv_);
    DL((REACT,"======================================\n"));

    /*--- Expire all stale Timers ---*/
    m_tqueue.expire (TimeVal::gettimeofday ());

	/* Test to see if Reactor has been deactivated as a result
	 * of processing done by any TimerHandlers.
	 */
	if (!m_active) {
		return;
	}

    int nReady;
    TimeVal  delay;
    TimeVal* dlp = &delay;

    /*---
      In case if not all data are processed by the EventHandler,
      and EventHandler stated so in its callback's return value
      to dispatcher (), it will be called again. This way 
      underlying file/socket stream can efficiently utilize its
      buffering mechaninsm.
      ---*/
    if ((nReady = isAnyReady ())) {
		DL((REACT,"isAnyReady returned: %d\n",nReady));
		dispatch (nReady);
		return;
    }

    DL((REACT,"=== m_waitSet ===\n"));
    m_waitSet.dump ();

    do {
		m_readySet.reset ();
		m_readySet = m_waitSet;
		calculateTimeout (dlp, tv_);

		nReady = ::select (m_maxfd, 
						   &m_readySet.m_rset,
						   &m_readySet.m_wset, 
						   &m_readySet.m_eset, 
						   dlp);
		DL((REACT,"::select() returned: %d\n",nReady));
    } 
	while (nReady < 0 && handleError ());

    dispatch (nReady);
}

void 
Reactor::
dispatchHandler (FdSet& mask_, EventHandler** fdSet_, EH_IO_Callback callback_)
{
    trace_with_mask("Reactor::dispatchHandler",REACTTRACE);

    register int fd;
    register int ret;

	/*---
	  This spot needs re-thinking. When you have several high data-rate
	  connections sending data at the same time, the one that had
	  connected first would get lower FD number and would get data
	  transfer preference over everybody else who has connected later on.
	  ---*/

    for (fd = 0; m_active && fd < m_maxfd; fd++) {
		if (mask_.isSet (fd) && fdSet_[fd] != NULL) {

			DL((REACT,"Data detected on connection %s (FD=%d)\n",
				fdSet_[fd]->get_id ().c_str (), fd));

			if ((ret = (fdSet_[fd]->*callback_) (fd)) == -1) {
				removeIOHandler (fd);
			}
			else if (ret > 0) {
				DL((REACT,"More data (%d bytes) are pending on FD=%d\n",
					ret,fd));
				//return;   <-- would starve other connections
			}
			else {
				DL((REACT,"All data are consumed from FD=%d\n", fd));
				mask_.clear (fd);
			}
		}
    }
}

bool
Reactor::
dispatch (int ready_)
{
    /*---
      Many UNIX systems will count a particular file descriptor in the 
      ready_ only ONCE, even if it was flagged by ::select(2) in, say, 
      both read and write masks.
      ---*/
    trace_with_mask("Reactor::dispatch", REACTTRACE);

    m_tqueue.expire (TimeVal::gettimeofday ());

    if ( ready_ < 0 ) {
		EL((ERROR,"::select(3) error\n"));
		return (false);
    }
    if ( ready_ == 0 ) {
		return (true);
    }
    DL((REACT,"Dispatching %d FDs\n",ready_));

    /*--- Writes first ---*/
    dispatchHandler (m_readySet.m_wset, 
					 m_writeSet, 
					 &EventHandler::handle_write);

    /*--- Exceptions next ---*/
    dispatchHandler (m_readySet.m_eset, 
					 m_exceptSet, 
					 &EventHandler::handle_except);

    /*--- Finally, the Reads ---*/
    dispatchHandler (m_readySet.m_rset, 
					 m_readSet, 
					 &EventHandler::handle_read);
    return (true);
}

void 
Reactor::
stopReactor (void) 
{ 
    trace_with_mask("Reactor::stopReactor", REACTTRACE);

    m_active = false; 
    register int i;
	
    for (i = 0; i < m_maxfd; i++) {
		removeIOHandler (i);
    }
}


