// -*- c++ -*-
// Generated by assa-genesis
//------------------------------------------------------------------------------
// $Id: pipe_test.cpp,v 1.10 2005/12/07 03:12:55 vlg Exp $
//------------------------------------------------------------------------------
//                            PipeTest.cpp
//------------------------------------------------------------------------------
//
// Author : Vladislav Grinchenko
// Date   : Sat Jul 20  2002
//------------------------------------------------------------------------------
static const char help_msg[]=
"                                                                            \n"
" NAME:                                                                      \n"
"   PipeTest - test for Pipe class.                                          \n"
"                                                                            \n"
" DESCRIPTION:                                                               \n"
"                                                                            \n"
"  Test for class Pipe. pipe_test uses 'smoker' as data reader/writer.       \n"
"                                                                            \n"
"  Test 1 spans off 'smoker' in writing mode. 'smoker' writes 10             \n"
"  messages to its standard output and exits. Test 1 reads 10 messages       \n"
"  and closes connection.                                                    \n"
"                                                                            \n"
"  Test 2 spans off 'smoker' in a writing mode setup to write endless        \n"
"  stream of messages to its standard output with delay of two seconds.      \n"
"  Test 2 starts a timer and keeps reading messages from the pipe. When      \n"
"  timer expires, Test 2 kills 'smoker' and tries to detect if 'smoker'      \n"
"  has been killed simulating call to system() with Pipe.                    \n"
"                                                                            \n"
"  Both processes generate log files, pipe_test.log and smoker{1,2}.log,     \n"
"  respectively.                                                             \n"
"                                                                            \n"
"        pipe_test         smoker                                            \n"
"     --------------     ----------                                          \n"
"           |                |                                               \n"
"           |      pipe      |                     < Test 1 starts           \n"
"           |--------------->| start t=1 sec                                 \n"
"           |                |   |                                           \n"
"           |      msg       |   v                                           \n"
"           |<---------------|   =                                           \n"
"           |                | start t=1 sec                                 \n"
"           |                |   |                                           \n"
"           |      msg       |   v                                           \n"
"           |<---------------|   =                                           \n"
"           |                |                                               \n"
"                  ...                                                       \n"
"           |                |   |                                           \n"
"           |      msg       |   v                                           \n"
"           |<---------------|   = Last (10th) msg sent out                  \n"
"           |                |                                               \n"
"           |      close     |                                               \n"
"           |<-X-------------|                     < Test 1 ends             \n"
"                                                                            \n"
"                  ...                                                       \n"
"           |                |                                               \n"
"           |      pipe      |                     < Test 2 starts           \n"
" start t=5 |--------------->| start t=2 secs                                \n"
"    |      |                |   |                                           \n"
"    |      |      msg       |   v                                           \n"
"    |      |<---------------|   =                                           \n"
"    |      |                | start t=2 secs                                \n"
"    |      |                |   |                                           \n"
"    |      |      msg       |   v                                           \n"
"    |      |<---------------|   =                                           \n"
"    v      |                |                                               \n"
"    =      |     kill       |                                               \n"
"           |------------X-->|                                               \n"
"           |                |                                               \n"
"           | Try to lock ~/.smoker2.pid                                     \n"
"           | to test if smoker has been killed                              \n"
"           |----+           |                                               \n"
"           |    |           |                                               \n"
"           |    |           |                     < Test 2 ends             \n"
"           |<---+           |                                               \n"
"           |                |                                               \n"
"                                                                            \n"
" USAGE:                                                                     \n"
"                                                                            \n"
"   shell>  pipe_test [OPTIONS]                                              \n"
"                                                                            \n"
" OPTIONS:                                                                   \n"
"                                                                            \n"
"     --build-dir STRING  - Directory where executables are located.         \n"
" -D, --log-file NAME     - Write debug to NAME file                         \n"
" -d, --log-stdout        - Write debug to standard output                   \n"
" -z, --log-size NUM      - Maximum size debug file can reach (dfl: is 10Mb) \n"
" -m, --mask MASK         - Mask (default: ALL = 0x7fffffff)                 \n"
" -h, --help              - Print this messag                                \n"
" -v, --version           - Print version number                            \n";
/******************************************************************************/

#include <cstdio>
#include <iostream>
#include <string>
using std::string;
using namespace std;

#include "assa/GenServer.h"
#include "assa/Singleton.h"
#include "assa/TimeVal.h"
#include "assa/Pipe.h"
#include "assa/AutoPtr.h"
#include "assa/IPv4Socket.h"
#include "assa/CommonUtils.h"
#include "assa/PidFileLock.h"
using namespace ASSA;

static const bool STILL_RUNNING=true;

/*******************************************************************************
 Class PipeTest
*******************************************************************************/
class PipeTest :
    public GenServer,
    public Singleton<PipeTest>
{
public:
    PipeTest ();

    virtual void init_service ();
    virtual void process_events ();

	virtual int handle_read (int fd_);
	virtual int handle_timeout (TimerId);
	virtual int handle_close (int fd_);

	string get_build_dir () const { return m_build_dir; }


private:
	void setup_test_1 ();
	int run_test_1 ();

	void setup_test_2 ();
	int run_test_2 ();

private:
	enum state_t { start, test1, test2 };

	Pipe* m_pipe;
	FILE* m_pinstream;

	int m_line_count;
	int m_max_count_expected;
	state_t m_state;
	string m_build_dir;
};

// Useful defines

#define PIPETEST PipeTest::get_instance()
#define REACTOR  PipeTest::get_instance()->get_reactor()

// Static declarations mandated by Singleton class
ASSA_DECL_SINGLETON(PipeTest);

/*******************************************************************************
 Member functions
*******************************************************************************/
PipeTest::
PipeTest () : 
	m_pipe (NULL), m_pinstream (NULL),
	m_line_count (0), m_max_count_expected (0),
	m_state (start)
{
	trace("PipeTest::PipeTest");

	set_id ("PT");
	add_opt (0, "build-dir", &m_build_dir);

    // ---Configuration---
    rm_opt ('f', "config-file"  );
    rm_opt ('n', "instance"     );
	rm_opt ('p', "port"         );

    // ---Process bookkeeping---
    rm_opt ('b', "daemon"       );
    rm_opt ('l', "pidfile"      );
    rm_opt ('L', "ommit-pidfile");

    /*---
     * Disable all debugging
     *---*/
    // m_mask = 0x0;

	m_log_size = 1024*1024*10;
	m_mask = 0x7fffffff;
	m_log_file = "pipe_test.log";
}

void
PipeTest::
init_service ()
{
	trace("PipeTest::init_service");
    Log::disable_timestamp ();

	if (m_build_dir.length () == 0) {
		m_build_dir = ASSA::Utils::get_cwd_name ();
	}
	DL((APP,"build-dir = \"%s\"\n", m_build_dir.c_str ()));

	std::cout << "= Running pipe_test Test =" << endl;
	DL((APP,"Test reading from pipe\n"));
	setup_test_1 ();
}

int
PipeTest::
handle_read (int fd_)
{
	trace("PipeTest::handle_read");
	int ret;

	DL((APP,"fd_ = %d\n", fd_));

	switch (m_state) {
	case test1: 
		ret = run_test_1 ();
		break;
	case test2:
		ret = run_test_2 ();
		break;
	};
	return ret;
}

void
PipeTest::
setup_test_1 ()
{
	trace("PipeTest::setup_test_11");
	
	string cmd (get_build_dir ());
	cmd += "/smoker --write=10 --delay=1 --mask=0x7fffffff "
		"--log-file=smoker_t1.log --pidfile=~/.smoker1.pid ";

	AutoPtr<Pipe> pipe (new Pipe);
	
	if (pipe->open (cmd.c_str (), "r") == NULL) {
		DL((APP,"Reading from pipe failed\n"));
		set_exit_value (1);
		return;
	}
	/* Tie the I/O standard stream to iostream */
	m_pinstream = pipe->fp ();
	Assure_exit (REACTOR->registerIOHandler (this, pipe->fd (), READ_EVENT));
	m_pipe = pipe.release ();
	m_max_count_expected = 10;
	m_state = test1;
}

int
PipeTest::
run_test_1 ()
{
	trace("PipeTest::run_test_1");

	char buf [128];
	char* ret = NULL;

	/* fgets() is a blocking call. We keep reading until we
	 * exhaust the pipe and the close it.
	 */
	while ((ret = fgets (buf, 128, m_pinstream)) != NULL) {
		DL((APP,"gcount = %d, line %d: \"%s\"\n", 
			strlen (buf), m_line_count++, buf));
	}
	DL((APP,"Reached EOF on read() from pipe.\n"));
	return -1;					// close the connection
}

void
PipeTest::
setup_test_2 ()
{
	trace("PipeTest::setup_test_11");

	::unlink ("smoker2.log");

	string cmd (get_build_dir ());
	cmd += "/smoker --write=-1 --delay=2 --mask=0x7fffffff "
		"--log-file=smoker_t2.log --pidfile=~/.smoker2.pid";

	AutoPtr<Pipe> pipe (new Pipe);
	if (pipe->open (cmd.c_str (), "r") == NULL) {
		DL((APP,"Reading from pipe failed\n"));
		set_exit_value (1);
		return;
	}
	m_pinstream = pipe->fp ();
	Assure_exit (REACTOR->registerIOHandler (this, pipe->fd (), READ_EVENT));
	TimeVal tv (5.0);			// 5 seconds delay
	REACTOR->registerTimerHandler (this, tv, "5 secs"); 
	m_pipe = pipe.release ();
	m_max_count_expected = -1;
	m_state = test2;
}

int
PipeTest::
run_test_2 ()
{
	trace("PipeTest::run_test_2");

	char buf [128];
	char* ret = NULL;

	/* Read a line at a time */
	if ((ret = fgets (buf, 128, m_pinstream)) == NULL) {
		DL((APP,"Received unexpected EOF on read() from pipe\n"));
		return -1;
	}

	DL((APP,"gcount = %d, line %d: \"%s\"\n", 
		strlen (buf), m_line_count++, buf));

	/* Following the Reactor's rules, what we should return here
	 * is the number of bytes left in the file descriptor buffer.
	 * However, we know that 'smoker' keeps filling the pipe
	 * and we won't starve.
	 */
	return 0;
}

/** At the end of the secod test, we kill smoker(2), then wait for
	2 seconds and then test to see if its pid file still exists
	and, if so, whether we can lock it. 
	If file exists and we fail to lock it - it means smoker(2) is
	still running and our 'kill' failed. If we can lock it, then
	it is an indication that our 'kill' caused smoker(2) to core dump
	and it didn't have a chance for a clean exit.
*/
int 
PipeTest::
handle_timeout (TimerId /* tid_ */)
{
	trace("PipeTest::handle_timeout");

	switch (m_state) {
	case test1:
		setup_test_2 ();
		break;

	case test2:
		{
			m_pipe->kill ();	// knock out pipe forecfully
			::sleep (2);		// give smoker a chance to exit.
			struct stat file_stat;
			if (stat ("~/.smoker2.pid", &file_stat) < 0 ||
				!S_ISREG (file_stat.st_mode)) 
			{
				DL ((APP,"smoker2 stopped. Passed test 2.\n"));
				cout << "Passed test 2\n";
			}
			else {
				ASSA::PidFileLock pflock;
				if (pflock.lock ("~/.smoker2.pid") == STILL_RUNNING) {
					DL ((APP,"Test 2 failed: was able to lock smoker(2) PID"
						 "- still running.\n"));
					cout << "Test 2 failed\n";
					cout << "error: smoker(2) is still running.\n";
				}
				else {
					DL ((APP,"Test 2 failed: smoker(2) core dumped\n"));
					cout << "Test 2 failed\n";
					cout << "error: smoker(2) core dumped.\n";
				}
			}
			cout << flush;
			REACTOR->removeHandler (this);
		}
		break;

	default:
		DL((APP,"Unexpected state %d\n", m_state));
	};

	return 0;
}

int
PipeTest::
handle_close (int fd_)
{
	trace("PipeTest::handle_close");
	TimerId tid;

	if (m_pipe) {
		delete m_pipe;
		m_pipe = NULL;
		m_pinstream = NULL;
	}

	switch (m_state) {
	case test1:
		if (m_line_count == m_max_count_expected) {
			DL((APP,"Passed test 1\n"));
			cout << "Passed test 1\n";
		}
		else {
			DL((APP,"Test 1 failed\n"));
			cout << "Test 1 failed\n";
		}
		tid = REACTOR->registerTimerHandler (this, TimeVal (0.1), "Inter-test");
		break;

	case test2:
		PIPETEST->stop_service ();
		break;
	};

	return 0;
}

void
PipeTest::
process_events ()
{
	trace("PipeTest::process_events");

	REACTOR->waitForEvents ();
	REACTOR->stopReactor ();
}

int
main (int argc, char* argv[])
{
    static const char release[] = "$Revision: 1.10 $";
    int patch_level = 0;

    PIPETEST->set_version (release, patch_level);
    PIPETEST->set_author  ("Vladislav Grinchenko");
	PIPETEST->set_flags (GenServer::RMLOG);

    PIPETEST->init (&argc, argv, help_msg);
 
    PIPETEST->init_service ();
    PIPETEST->process_events ();

    return PIPETEST->get_exit_value ();
}

