------------------------------------------------------------------------------
--                              Ada Web Server                              --
--                                                                          --
--                         Copyright (C) 2000-2006                          --
--                                 AdaCore                                  --
--                                                                          --
--  This library is free software; you can redistribute it and/or modify    --
--  it under the terms of the GNU General Public License as published by    --
--  the Free Software Foundation; either version 2 of the License, or (at   --
--  your option) any later version.                                         --
--                                                                          --
--  This library is distributed in the hope that it will be useful, but     --
--  WITHOUT ANY WARRANTY; without even the implied warranty of              --
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       --
--  General Public License for more details.                                --
--                                                                          --
--  You should have received a copy of the GNU General Public License       --
--  along with this library; if not, write to the Free Software Foundation, --
--  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.          --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Calendar;
with Ada.Unchecked_Deallocation;

with AI302.Containers;

with AWS.Messages;
with AWS.MIME;
with AWS.Utils;

with GNAT.Calendar.Time_IO;
with System;

package body AWS.Server.Push is

   use AWS.Net;

   function To_Holder
     (Socket      : in Net.Socket_Type'Class;
      Environment : in Client_Environment;
      Kind        : in Mode;
      Groups      : in Group_Set)
      return Client_Holder;

   procedure Free (Holder : in out Client_Holder);

   function To_Stream (Socket : in Net.Socket_Type'Class) return Stream_Access
     renames AWS.Net.Stream_IO.Stream;

   New_Line : constant String := ASCII.CR & ASCII.LF;
   --  HTTP new line.

   Boundary : constant String := "--AWS.Push.Boundary_"
     & GNAT.Calendar.Time_IO.Image (Ada.Calendar.Clock, "%s")
     & New_Line;
   --  This is the multi-part boundary string used by AWS push server.

   -----------
   -- Count --
   -----------

   function Count (Server : in Object) return Natural is
   begin
      return Server.Count;
   end Count;

   ----------
   -- Free --
   ----------

   procedure Free (Holder : in out Client_Holder) is
      procedure Free
        is new Ada.Unchecked_Deallocation (Group_Set, Groups_Access);
   begin
      Net.Stream_IO.Free (Holder.Stream);
      Free (Holder.Groups);
   end Free;

   -------------
   -- Is_Open --
   -------------

   function Is_Open (Server : in Object) return Boolean is
   begin
      return Server.Is_Open;
   end Is_Open;

   ------------
   -- Object --
   ------------

   protected body Object is

      ---------------
      -- Send_Data --
      ---------------

      procedure Send_Data
        (Holder       : in Client_Holder;
         Data         : in Client_Output_Type;
         Content_Type : in String);
      --  Send Data to a client identified by Holder.

      -----------
      -- Count --
      -----------

      function Count return Natural is
      begin
         return Natural (Table.Length (Container));
      end Count;

      -------------
      -- Is_Open --
      -------------

      function Is_Open return Boolean is
      begin
         return Open;
      end Is_Open;

      --------------
      -- Register --
      --------------

      procedure Register
        (Client_Id       : in     Client_Key;
         Holder          : in out Client_Holder;
         Close_Duplicate : in     Boolean)
      is
         Cursor  : Table.Cursor;
         Success : Boolean;
      begin
         if not Open then
            Free (Holder);
            raise Closed;
         end if;

         Table.Insert (Container, Client_Id, Holder, Cursor, Success);

         if not Success then
            if Close_Duplicate then
               Unregister (Client_Id, Close_Socket => True);
               Table.Insert (Container, Client_Id, Holder, Cursor, Success);
               pragma Assert (Success);
            else
               Free (Holder);
               raise Duplicate_Client_Id;
            end if;
         end if;

         if Holder.Groups /= null then
            for J in Holder.Groups'Range loop
               declare
                  use Group_Maps;
                  Name : constant String := To_String (Holder.Groups (J));

                  C : constant Group_Maps.Cursor := Find (Groups, Name);
                  G : Map_Access;

                  Dummy_B : Boolean;
                  Dummy_C : Group_Maps.Cursor;
                  Dummy_0 : Table.Cursor;
               begin
                  if Has_Element (C) then
                     G := Element (C);
                  else
                     G := new Table.Map;
                     Insert (Groups, Name, G, Dummy_C, Dummy_B);
                     pragma Assert (Dummy_B);
                  end if;

                  Table.Insert (G.all, Client_Id, Holder, Dummy_0, Dummy_B);
                  pragma Assert (Dummy_B);
               end;
            end loop;
         end if;

         begin
            String'Write
              (Holder.Stream,
               "HTTP/1.1 200 OK" & New_Line
                 & "Server: AWS (Ada Web Server) v"
                 & Version & New_Line
                 & Messages.Connection ("Close") & New_Line);

            if Holder.Kind = Chunked then
               String'Write
                 (Holder.Stream,
                  Messages.Transfer_Encoding ("chunked")
                    & New_Line & New_Line);

            elsif Holder.Kind = Multipart then
               String'Write
                 (Holder.Stream,
                  Messages.Content_Type
                    (MIME.Multipart_X_Mixed_Replace, Boundary)
                    & New_Line);

            else
               String'Write (Holder.Stream, New_Line);
            end if;

            Net.Stream_IO.Flush (Holder.Stream);

            Socket_Taken (True);
         exception
            when others =>
               Unregister (Client_Id, Close_Socket => False);
               raise;
         end;
      end Register;

      procedure Register
        (Client_Id         : in     Client_Key;
         Holder            : in out Client_Holder;
         Init_Data         : in     Client_Output_Type;
         Init_Content_Type : in     String;
         Close_Duplicate   : in     Boolean) is
      begin
         Register (Client_Id, Holder, Close_Duplicate);

         begin
            Send_Data (Holder, Init_Data, Init_Content_Type);
         exception
            when others =>
               Unregister (Client_Id, Close_Socket => False);
               Socket_Taken (False);
               raise;
         end;
      end Register;

      -------------
      -- Restart --
      -------------

      procedure Restart is
      begin
         Open := True;
      end Restart;

      ----------
      -- Send --
      ----------

      procedure Send
        (Data         : in     Client_Output_Type;
         Group_Id     : in     String;
         Content_Type : in     String;
         Unregistered : in out Table.Map)
      is
         Cursor : Table.Cursor;
      begin
         if Group_Id = "" then
            Cursor := Table.First (Container);
         else
            declare
               use Group_Maps;
               C : constant Group_Maps.Cursor := Find (Groups, Group_Id);
            begin
               if not Has_Element (C) then
                  return;
               end if;

               Cursor := Table.First (Element (C).all);
            end;
         end if;

         while Table.Has_Element (Cursor) loop
            declare
               Holder : constant Client_Holder := Table.Element (Cursor);
            begin
               declare
                  Success : Boolean;
               begin
                  Send_Data (Holder, Data, Content_Type);

                  Table.Next (Cursor);
               exception
                  when Net.Socket_Error =>
                     declare
                        C   : Table.Cursor;
                        Key : constant Client_Key := Table.Key (Cursor);
                     begin
                        Table.Insert (Unregistered, Key, Holder, C, Success);

                        --  We have to move cursor to the next position before
                        --  delete element from current position.

                        Table.Next (Cursor);

                        Unregister (Key, True);
                     end;
               end;
            end;
         end loop;
      end Send;

      ---------------
      -- Send_Data --
      ---------------

      procedure Send_Data
        (Holder       : in Client_Holder;
         Data         : in Client_Output_Type;
         Content_Type : in String)
      is
         Data_To_Send : constant Stream_Output_Type
           := To_Stream_Output (Data, Holder.Environment);

      begin
         if Holder.Kind = Multipart then
            String'Write
              (Holder.Stream,
               Boundary
                 & Messages.Content_Type (Content_Type) & New_Line & New_Line);

         elsif Holder.Kind = Chunked then
            String'Write
              (Holder.Stream,
               Utils.Hex (Data_To_Send'Size / System.Storage_Unit) & New_Line);
         end if;

         Stream_Output_Type'Write (Holder.Stream, Data_To_Send);

         if Holder.Kind = Multipart then
            String'Write (Holder.Stream, New_Line & New_Line);

         elsif Holder.Kind = Chunked then
            String'Write (Holder.Stream, New_Line);
         end if;

         Net.Stream_IO.Flush (Holder.Stream);
      end Send_Data;

      -------------
      -- Send_To --
      -------------

      procedure Send_To
        (Client_Id    : in Client_Key;
         Data         : in Client_Output_Type;
         Content_Type : in String)
      is
         Cursor : Table.Cursor;
      begin
         Cursor := Table.Find (Container, Client_Id);

         if Table.Has_Element (Cursor) then
            Send_Data (Table.Element (Cursor), Data, Content_Type);
         else
            Ada.Exceptions.Raise_Exception
              (Client_Gone'Identity, "No such client id.");
         end if;

      exception
         when E : Net.Socket_Error =>
            Unregister (Client_Id, True);

            Ada.Exceptions.Raise_Exception
              (Client_Gone'Identity, Ada.Exceptions.Exception_Message (E));
      end Send_To;

      --------------
      -- Shutdown --
      --------------

      procedure Shutdown (Close_Sockets : in Boolean) is
      begin
         Open := False;
         Unregister_Clients (Close_Sockets => Close_Sockets);
      end Shutdown;

      procedure Shutdown
        (Final_Data         : in Client_Output_Type;
         Final_Content_Type : in String)
      is
         Gone : Table.Map;
      begin
         Send (Final_Data, "", Final_Content_Type, Gone);
         Table.Clear (Gone);
         Shutdown (Close_Sockets => True);
      end Shutdown;

      -----------------------
      -- Shutdown_If_Empty --
      -----------------------

      procedure Shutdown_If_Empty (Open : out Boolean) is
         use type AI302.Containers.Count_Type;
      begin
         if Table.Length (Container) = 0 then
            Object.Open := False;
         end if;
         Shutdown_If_Empty.Open := Object.Open;
      end Shutdown_If_Empty;

      ----------------
      -- Unregister --
      ----------------

      procedure Unregister
        (Client_Id    : in Client_Key;
         Close_Socket : in Boolean)
      is
         Cursor : Table.Cursor;
         Value  : Client_Holder;
      begin
         Cursor := Table.Find (Container, Client_Id);

         if Table.Has_Element (Cursor) then
            Value := Table.Element (Cursor);

            if Value.Groups /= null then
               for J in Value.Groups'Range loop
                  Table.Delete
                    (Group_Maps.Element
                       (Groups, To_String (Value.Groups (J))).all,
                     Client_Id);
               end loop;
            end if;

            if Close_Socket then
               Net.Stream_IO.Shutdown (Value.Stream);
            end if;

            Free (Value);

            Table.Delete (Container, Cursor);
         end if;
      end Unregister;

      ------------------------
      -- Unregister_Clients --
      ------------------------

      procedure Unregister_Clients (Close_Sockets : in Boolean) is
         Cursor : Table.Cursor;
      begin
         loop
            Cursor := Table.First (Container);

            exit when not Table.Has_Element (Cursor);

            Unregister (Table.Key (Cursor), Close_Sockets);
         end loop;
      end Unregister_Clients;

   end Object;

   --------------
   -- Register --
   --------------

   procedure Register
     (Server            : in out Object;
      Client_Id         : in     Client_Key;
      Socket            : in     Net.Socket_Type'Class;
      Environment       : in     Client_Environment;
      Init_Data         : in     Client_Output_Type;
      Init_Content_Type : in     String := "";
      Kind              : in     Mode := Plain;
      Close_Duplicate   : in     Boolean := False;
      Groups            : in     Group_Set          := Empty_Group)
   is
      Holder : Client_Holder := To_Holder (Socket, Environment, Kind, Groups);
   begin
      Server.Register
        (Client_Id,
         Holder,
         Init_Data,
         Init_Content_Type,
         Close_Duplicate);
   end Register;

   procedure Register
     (Server          : in out Object;
      Client_Id       : in     Client_Key;
      Socket          : in     Net.Socket_Type'Class;
      Environment     : in     Client_Environment;
      Kind            : in     Mode               := Plain;
      Close_Duplicate : in     Boolean            := False;
      Groups          : in     Group_Set          := Empty_Group)
   is
      Holder : Client_Holder := To_Holder (Socket, Environment, Kind, Groups);
   begin
      Server.Register (Client_Id, Holder, Close_Duplicate);
   end Register;

   -------------
   -- Restart --
   -------------

   procedure Restart (Server : in out Object) is
   begin
      Server.Restart;
   end Restart;

   ----------
   -- Send --
   ----------

   procedure Send
     (Server       : in out Object;
      Data         : in     Client_Output_Type;
      Group_Id     : in     String             := "";
      Content_Type : in     String             := "")
   is
      Gone : Table.Map;
   begin
      Server.Send (Data, Group_Id, Content_Type, Gone);
      Table.Clear (Gone);
   end Send;

   ------------
   -- Send_G --
   ------------

   procedure Send_G
     (Server       : in out Object;
      Data         : in     Client_Output_Type;
      Group_Id     : in     String             := "";
      Content_Type : in     String             := "")
   is
      Cursor : Table.Cursor;
      Gone   : Table.Map;
   begin
      Server.Send (Data, Group_Id, Content_Type, Gone);

      Cursor := Table.First (Gone);

      while Table.Has_Element (Cursor) loop
         Client_Gone (Table.Key (Cursor));
         Table.Next (Cursor);
      end loop;

      Table.Clear (Gone);
   end Send_G;

   -------------
   -- Send_To --
   -------------

   procedure Send_To
     (Server       : in out Object;
      Client_Id    : in     Client_Key;
      Data         : in     Client_Output_Type;
      Content_Type : in     String             := "") is
   begin
      Server.Send_To (Client_Id, Data, Content_Type);
   end Send_To;

   --------------
   -- Shutdown --
   --------------

   procedure Shutdown
     (Server        : in out Object;
      Close_Sockets : in     Boolean := True) is
   begin
      Server.Shutdown (Close_Sockets => Close_Sockets);
   end Shutdown;

   procedure Shutdown
     (Server             : in out Object;
      Final_Data         : in     Client_Output_Type;
      Final_Content_Type : in     String             := "") is
   begin
      Server.Shutdown (Final_Data, Final_Content_Type);
   end Shutdown;

   -----------------------
   -- Shutdown_If_Empty --
   -----------------------

   procedure Shutdown_If_Empty (Server : in out Object; Open : out Boolean) is
   begin
      Server.Shutdown_If_Empty (Open);
   end Shutdown_If_Empty;

   ---------------
   -- To_Holder --
   ---------------

   function To_Holder
     (Socket      : in Net.Socket_Type'Class;
      Environment : in Client_Environment;
      Kind        : in Mode;
      Groups      : in Group_Set)
      return Client_Holder
   is
      Groups_Ptr : Groups_Access;
   begin
      if Groups /= Empty_Group then
         Groups_Ptr := new Group_Set'(Groups);
      end if;

      return (Kind        => Kind,
              Environment => Environment,
              Stream      => To_Stream (Socket),
              Groups      => Groups_Ptr);
   end To_Holder;

   ----------------
   -- Unregister --
   ----------------

   procedure Unregister
     (Server       : in out Object;
      Client_Id    : in     Client_Key;
      Close_Socket : in     Boolean    := True) is
   begin
      Server.Unregister (Client_Id, Close_Socket);
   end Unregister;

   ------------------------
   -- Unregister_Clients --
   ------------------------

   procedure Unregister_Clients
     (Server        : in out Object;
      Close_Sockets : in     Boolean := True) is
   begin
      Server.Unregister_Clients (Close_Sockets => Close_Sockets);
   end Unregister_Clients;

end AWS.Server.Push;
