/*============================================================================
 * Trap floating-point exceptions on some systems
 *============================================================================*/

/*
  This file is part of the "Base Functions and Types" library, intended to
  simplify and enhance portability, memory and I/O use for scientific codes.

  Copyright (C) 2004  EDF

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*-----------------------------------------------------------------------------*/

#include "bft_config_defs.h"

/*
 * Standard C library headers
 */

#if defined(__linux) && defined(__GNUC__)
#if defined(__i386) || defined(__x86_64__) || defined(__amd64__)
#define BFT_USE_LINUX_I386_FPU_CONTROL 1
#include <fpu_control.h>
#endif
#endif

/*
 * Optional library and BFT headers
 */

#include "bft_fp_trap.h"

/*-----------------------------------------------------------------------------*/

#ifdef __cplusplus
extern "C" {
#if 0
} /* Fake brace to force back Emacs auto-indentation back to column 0 */
#endif
#endif /* __cplusplus */

/*-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Local type definitions
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Local function prototypes
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
 * Local static variable definitions
 *-----------------------------------------------------------------------------*/

#if defined(BFT_USE_LINUX_I386_FPU_CONTROL)
static int           _bft_fpu_cw_set = 0;   /* Indicates if behavior modified */
static fpu_control_t _bft_fpu_cw_old;       /* Old FPU control word */
#endif

/*-----------------------------------------------------------------------------
 * Local function definitions
 *-----------------------------------------------------------------------------*/

/*============================================================================
 * Public function definitions
 *============================================================================*/

/*!
 * \brief Enable trapping of floating point exceptions if possible.
 *
 * This function is currently implemented only on X86 and X86-64
 * type architectures running Linux, with BFT compiled with GCC,
 * and does nothing in other cases.
 *
 * On some system variants, some system calls have been seen to throw
 * floating-point exceptions in unexpected cases. For example:
 *
 * - Using the Electric Fence malloc() library under SuSE Linux 8.0,
 *   certain functions such as mktime() or atof() would throw a
 *   floating-point exception under the GDB debugger, but not on a simple
 *   (single-threaded) program run. This problem disappeared on
 *   SuSE Linux 8.1, and has not been encountered since.
 *
 * - Reading a file with libxml2 2.2.6 under Debian Linux 2.2 (and
 *   possibly other variants) throws floating-point exceptions.
 */

void
bft_fp_trap_set(void)
{
#if defined(BFT_USE_LINUX_I386_FPU_CONTROL)

  fpu_control_t cw =
    _FPU_DEFAULT & ~(_FPU_MASK_IM | _FPU_MASK_ZM | _FPU_MASK_OM);

  if (_bft_fpu_cw_set == 0) {
    _FPU_GETCW(_bft_fpu_cw_old);
    _bft_fpu_cw_set = 1;
  }
  _FPU_SETCW(cw);

#endif
}

/*!
 * \brief Revert to initial enable trapping behavior of floating point
 *  exceptions if possible.
 *
 * This function is currently implemented only on X86 and X86-64
 * type architectures running Linux, with BFT compiled with GCC,
 * and does nothing in other cases.
 */

void
bft_fp_trap_unset(void)
{
#if defined(BFT_USE_LINUX_I386_FPU_CONTROL)

  if (_bft_fpu_cw_set == 1) {
    fpu_control_t cw = _bft_fpu_cw_old;
    _FPU_SETCW(cw);
  }

#endif
}

/*-----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif /* __cplusplus */

#undef BFT_USE_LINUX_I386_FPU_CONTROL

