/**
 * Copyright (c) 2011-2015 libbitcoin developers (see AUTHORS)
 *
 * This file is part of libbitcoin.
 *
 * libbitcoin is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License with
 * additional permissions to the one published by the Free Software
 * Foundation, either version 3 of the License, or (at your option)
 * any later version. For more information see LICENSE.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef LIBBITCOIN_TEST_SCRIPT_NUMBER_HPP
#define LIBBITCOIN_TEST_SCRIPT_NUMBER_HPP

#include <cstdint>
#include <limits>
#include <bitcoin/bitcoin.hpp>

// (-2147483647 - 1)  and (-9223372036854775807 - 1) prevent warnings.

// DO NOT TEST WITH MACRO VALUES BELOW THAT VARY BY PLATFORM,
// SUCH AS LONG_MIN, LONG_MAX.
const int64_t script_number_values[] =
{
    0,
    1,
    CHAR_MIN,
    CHAR_MAX,
    UCHAR_MAX,
    SHRT_MIN,
    SHRT_MAX,
    USHRT_MAX,
    INT_MIN,
    INT_MAX,
    UINT_MAX,
    LLONG_MIN,
    LLONG_MAX
};

const int64_t script_number_offsets[] =
{
    1,
    0x79,
    0x80,
    0x81,
    0xFF,
    0x7FFF,
    0x8000,
    0xFFFF,
    0x10000
};

BC_CONSTEXPR size_t script_number_values_count =
sizeof(script_number_values) / sizeof(script_number_values[0]);

BC_CONSTEXPR size_t script_number_offsets_count =
sizeof(script_number_offsets) / sizeof(script_number_offsets[0]);

struct script_number_buffer
{
    int32_t number;
    bc::data_chunk bytes;
};

struct script_number_subtract
{
    script_number_buffer forward;
    script_number_buffer reverse;
};

struct script_number_compare
{
    bool eq;
    bool ne;
    bool lt;
    bool gt;
    bool le;
    bool ge;
};

const script_number_buffer script_number_adds[13][9][12] =
{
    {
        {
            { 0, {} },
            { 0, {} },
            { 1, { 0x01, } },
            { -1, { 0x81, } },
            { 2, { 0x02, } },
            { 0, {} },
            { 0, {} },
            { -2, { 0x82, } },
            { 2, { 0x02, } },
            { 0, {} },
            { 0, {} },
            { -2, { 0x82, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 121, { 0x79, } },
            { -121, { 0xf9, } },
            { 242, { 0xf2, 0x00, } },
            { 0, {} },
            { 0, {} },
            { -242, { 0xf2, 0x80, } },
            { 242, { 0xf2, 0x00, } },
            { 0, {} },
            { 0, {} },
            { -242, { 0xf2, 0x80, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 128, { 0x80, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { 256, { 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -256, { 0x00, 0x81, } },
            { 256, { 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -256, { 0x00, 0x81, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 129, { 0x81, 0x00, } },
            { -129, { 0x81, 0x80, } },
            { 258, { 0x02, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -258, { 0x02, 0x81, } },
            { 258, { 0x02, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -258, { 0x02, 0x81, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 255, { 0xff, 0x00, } },
            { -255, { 0xff, 0x80, } },
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -510, { 0xfe, 0x81, } },
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -510, { 0xfe, 0x81, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 32767, { 0xff, 0x7f, } },
            { -32767, { 0xff, 0xff, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 0, {} },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 0, {} },
            { -65534, { 0xfe, 0xff, 0x80, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -65536, { 0x00, 0x00, 0x81, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -131070, { 0xfe, 0xff, 0x81, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 0, {} },
            { -131070, { 0xfe, 0xff, 0x81, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 65536, { 0x00, 0x00, 0x01, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 131072, { 0x00, 0x00, 0x02, } },
            { 0, {} },
            { 0, {} },
            { -131072, { 0x00, 0x00, 0x82, } },
            { 131072, { 0x00, 0x00, 0x02, } },
            { 0, {} },
            { 0, {} },
            { -131072, { 0x00, 0x00, 0x82, } },
        },
    },
    {
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 2, { 0x02, } },
            { 0, {} },
            { 3, { 0x03, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -1, { 0x81, } },
            { 4, { 0x04, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { 0, {} },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 122, { 0x7a, } },
            { -120, { 0xf8, } },
            { 243, { 0xf3, 0x00, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -241, { 0xf1, 0x80, } },
            { 244, { 0xf4, 0x00, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -240, { 0xf0, 0x80, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 129, { 0x81, 0x00, } },
            { -127, { 0xff, } },
            { 257, { 0x01, 0x01, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -255, { 0xff, 0x80, } },
            { 258, { 0x02, 0x01, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -254, { 0xfe, 0x80, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 130, { 0x82, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { 259, { 0x03, 0x01, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -257, { 0x01, 0x81, } },
            { 260, { 0x04, 0x01, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -256, { 0x00, 0x81, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 256, { 0x00, 0x01, } },
            { -254, { 0xfe, 0x80, } },
            { 511, { 0xff, 0x01, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -509, { 0xfd, 0x81, } },
            { 512, { 0x00, 0x02, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -508, { 0xfc, 0x81, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32766, { 0xfe, 0xff, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -65533, { 0xfd, 0xff, 0x80, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -65532, { 0xfc, 0xff, 0x80, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 32769, { 0x01, 0x80, 0x00, } },
            { -32767, { 0xff, 0xff, } },
            { 65537, { 0x01, 0x00, 0x01, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 65538, { 0x02, 0x00, 0x01, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 65536, { 0x00, 0x00, 0x01, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { 131071, { 0xff, 0xff, 0x01, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -131069, { 0xfd, 0xff, 0x81, } },
            { 131072, { 0x00, 0x00, 0x02, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -131068, { 0xfc, 0xff, 0x81, } },
        },
        {
            { 2, { 0x02, } },
            { 0, {} },
            { 65537, { 0x01, 0x00, 0x01, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 131073, { 0x01, 0x00, 0x02, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -131071, { 0xff, 0xff, 0x81, } },
            { 131074, { 0x02, 0x00, 0x02, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -131070, { 0xfe, 0xff, 0x81, } },
        },
    },
    {
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { -127, { 0xff, } },
            { -129, { 0x81, 0x80, } },
            { -126, { 0xfe, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -130, { 0x82, 0x80, } },
            { -254, { 0xfe, 0x80, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -258, { 0x02, 0x81, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { -7, { 0x87, } },
            { -249, { 0xf9, 0x80, } },
            { 114, { 0x72, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -370, { 0x72, 0x81, } },
            { -14, { 0x8e, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -498, { 0xf2, 0x81, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 0, {} },
            { -256, { 0x00, 0x81, } },
            { 128, { 0x80, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -384, { 0x80, 0x81, } },
            { 0, {} },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -512, { 0x00, 0x82, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 1, { 0x01, } },
            { -257, { 0x01, 0x81, } },
            { 130, { 0x82, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -386, { 0x82, 0x81, } },
            { 2, { 0x02, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -514, { 0x02, 0x82, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 127, { 0x7f, } },
            { -383, { 0x7f, 0x81, } },
            { 382, { 0x7e, 0x01, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -638, { 0x7e, 0x82, } },
            { 254, { 0xfe, 0x00, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -766, { 0xfe, 0x82, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 32639, { 0x7f, 0x7f, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { 65406, { 0x7e, 0xff, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -65662, { 0x7e, 0x00, 0x81, } },
            { 65278, { 0xfe, 0xfe, 0x00, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 32640, { 0x80, 0x7f, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { 65408, { 0x80, 0xff, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { 65280, { 0x00, 0xff, 0x00, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -65792, { 0x00, 0x01, 0x81, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 65407, { 0x7f, 0xff, 0x00, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { 130942, { 0x7e, 0xff, 0x01, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -131198, { 0x7e, 0x00, 0x82, } },
            { 130814, { 0xfe, 0xfe, 0x01, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -131326, { 0xfe, 0x00, 0x82, } },
        },
        {
            { -256, { 0x00, 0x81, } },
            { 0, {} },
            { 65408, { 0x80, 0xff, 0x00, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { 130944, { 0x80, 0xff, 0x01, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -131200, { 0x80, 0x00, 0x82, } },
            { 130816, { 0x00, 0xff, 0x01, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -131328, { 0x00, 0x01, 0x82, } },
        },
    },
    {
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 128, { 0x80, 0x00, } },
            { 126, { 0x7e, } },
            { 129, { 0x81, 0x00, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { 125, { 0x7d, } },
            { 256, { 0x00, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { 252, { 0xfc, 0x00, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 248, { 0xf8, 0x00, } },
            { 6, { 0x06, } },
            { 369, { 0x71, 0x01, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -115, { 0xf3, } },
            { 496, { 0xf0, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { 12, { 0x0c, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 255, { 0xff, 0x00, } },
            { -1, { 0x81, } },
            { 383, { 0x7f, 0x01, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -129, { 0x81, 0x80, } },
            { 510, { 0xfe, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -2, { 0x82, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 256, { 0x00, 0x01, } },
            { -2, { 0x82, } },
            { 385, { 0x81, 0x01, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -131, { 0x83, 0x80, } },
            { 512, { 0x00, 0x02, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -4, { 0x84, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 382, { 0x7e, 0x01, } },
            { -128, { 0x80, 0x80, } },
            { 637, { 0x7d, 0x02, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -383, { 0x7f, 0x81, } },
            { 764, { 0xfc, 0x02, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -256, { 0x00, 0x81, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 32894, { 0x7e, 0x80, 0x00, } },
            { -32640, { 0x80, 0xff, } },
            { 65661, { 0x7d, 0x00, 0x01, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { 65788, { 0xfc, 0x00, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -65280, { 0x00, 0xff, 0x80, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 32895, { 0x7f, 0x80, 0x00, } },
            { -32641, { 0x81, 0xff, } },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -65409, { 0x81, 0xff, 0x80, } },
            { 65790, { 0xfe, 0x00, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -65282, { 0x02, 0xff, 0x80, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 65662, { 0x7e, 0x00, 0x01, } },
            { -65408, { 0x80, 0xff, 0x80, } },
            { 131197, { 0x7d, 0x00, 0x02, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -130943, { 0x7f, 0xff, 0x81, } },
            { 131324, { 0xfc, 0x00, 0x02, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -130816, { 0x00, 0xff, 0x81, } },
        },
        {
            { 254, { 0xfe, 0x00, } },
            { 0, {} },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { -65409, { 0x81, 0xff, 0x80, } },
            { 131199, { 0x7f, 0x00, 0x02, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -130945, { 0x81, 0xff, 0x81, } },
            { 131326, { 0xfe, 0x00, 0x02, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -130818, { 0x02, 0xff, 0x81, } },
        },
    },
    {
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 256, { 0x00, 0x01, } },
            { 254, { 0xfe, 0x00, } },
            { 257, { 0x01, 0x01, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { 253, { 0xfd, 0x00, } },
            { 512, { 0x00, 0x02, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 508, { 0xfc, 0x01, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 376, { 0x78, 0x01, } },
            { 134, { 0x86, 0x00, } },
            { 497, { 0xf1, 0x01, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { 13, { 0x0d, } },
            { 752, { 0xf0, 0x02, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 268, { 0x0c, 0x01, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 383, { 0x7f, 0x01, } },
            { 127, { 0x7f, } },
            { 511, { 0xff, 0x01, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -1, { 0x81, } },
            { 766, { 0xfe, 0x02, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 254, { 0xfe, 0x00, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 384, { 0x80, 0x01, } },
            { 126, { 0x7e, } },
            { 513, { 0x01, 0x02, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -3, { 0x83, } },
            { 768, { 0x00, 0x03, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 252, { 0xfc, 0x00, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 765, { 0xfd, 0x02, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -255, { 0xff, 0x80, } },
            { 1020, { 0xfc, 0x03, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 33022, { 0xfe, 0x80, 0x00, } },
            { -32512, { 0x00, 0xff, } },
            { 65789, { 0xfd, 0x00, 0x01, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -65279, { 0xff, 0xfe, 0x80, } },
            { 66044, { 0xfc, 0x01, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { -65024, { 0x00, 0xfe, 0x80, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 33023, { 0xff, 0x80, 0x00, } },
            { -32513, { 0x01, 0xff, } },
            { 65791, { 0xff, 0x00, 0x01, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -65281, { 0x01, 0xff, 0x80, } },
            { 66046, { 0xfe, 0x01, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { -65026, { 0x02, 0xfe, 0x80, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 65790, { 0xfe, 0x00, 0x01, } },
            { -65280, { 0x00, 0xff, 0x80, } },
            { 131325, { 0xfd, 0x00, 0x02, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -130815, { 0xff, 0xfe, 0x81, } },
            { 131580, { 0xfc, 0x01, 0x02, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { -130560, { 0x00, 0xfe, 0x81, } },
        },
        {
            { 510, { 0xfe, 0x01, } },
            { 0, {} },
            { 65791, { 0xff, 0x00, 0x01, } },
            { -65281, { 0x01, 0xff, 0x80, } },
            { 131327, { 0xff, 0x00, 0x02, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -130817, { 0x01, 0xff, 0x81, } },
            { 131582, { 0xfe, 0x01, 0x02, } },
            { 510, { 0xfe, 0x01, } },
            { 510, { 0xfe, 0x01, } },
            { -130562, { 0x02, 0xfe, 0x81, } },
        },
    },
    {
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -32767, { 0xff, 0xff, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { -32766, { 0xfe, 0xff, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32770, { 0x02, 0x80, 0x80, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65538, { 0x02, 0x00, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -32647, { 0x87, 0xff, } },
            { -32889, { 0x79, 0x80, 0x80, } },
            { -32526, { 0x0e, 0xff, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -33010, { 0xf2, 0x80, 0x80, } },
            { -65294, { 0x0e, 0xff, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65778, { 0xf2, 0x00, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -32640, { 0x80, 0xff, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { -32512, { 0x00, 0xff, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -33024, { 0x00, 0x81, 0x80, } },
            { -65280, { 0x00, 0xff, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65792, { 0x00, 0x01, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -32639, { 0x7f, 0xff, } },
            { -32897, { 0x81, 0x80, 0x80, } },
            { -32510, { 0xfe, 0xfe, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -33026, { 0x02, 0x81, 0x80, } },
            { -65278, { 0xfe, 0xfe, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65794, { 0x02, 0x01, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -32513, { 0x01, 0xff, } },
            { -33023, { 0xff, 0x80, 0x80, } },
            { -32258, { 0x02, 0xfe, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -33278, { 0xfe, 0x81, 0x80, } },
            { -65026, { 0x02, 0xfe, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -66046, { 0xfe, 0x01, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -1, { 0x81, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 32766, { 0xfe, 0x7f, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -2, { 0x82, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -131070, { 0xfe, 0xff, 0x81, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { 0, {} },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -98304, { 0x00, 0x80, 0x81, } },
            { 0, {} },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -131072, { 0x00, 0x00, 0x82, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { 32767, { 0xff, 0x7f, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { 98302, { 0xfe, 0x7f, 0x01, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -163838, { 0xfe, 0x7f, 0x82, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -196606, { 0xfe, 0xff, 0x82, } },
        },
        {
            { -65536, { 0x00, 0x00, 0x81, } },
            { 0, {} },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -98304, { 0x00, 0x80, 0x81, } },
            { 98304, { 0x00, 0x80, 0x01, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -163840, { 0x00, 0x80, 0x82, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -196608, { 0x00, 0x00, 0x83, } },
        },
    },
    {
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32766, { 0xfe, 0x7f, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32765, { 0xfd, 0x7f, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65532, { 0xfc, 0xff, 0x00, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 32888, { 0x78, 0x80, 0x00, } },
            { 32646, { 0x86, 0x7f, } },
            { 33009, { 0xf1, 0x80, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32525, { 0x0d, 0x7f, } },
            { 65776, { 0xf0, 0x00, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65292, { 0x0c, 0xff, 0x00, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 32895, { 0x7f, 0x80, 0x00, } },
            { 32639, { 0x7f, 0x7f, } },
            { 33023, { 0xff, 0x80, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32511, { 0xff, 0x7e, } },
            { 65790, { 0xfe, 0x00, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65278, { 0xfe, 0xfe, 0x00, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32638, { 0x7e, 0x7f, } },
            { 33025, { 0x01, 0x81, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32509, { 0xfd, 0x7e, } },
            { 65792, { 0x00, 0x01, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65276, { 0xfc, 0xfe, 0x00, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 33022, { 0xfe, 0x80, 0x00, } },
            { 32512, { 0x00, 0x7f, } },
            { 33277, { 0xfd, 0x81, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32257, { 0x01, 0x7e, } },
            { 66044, { 0xfc, 0x01, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65024, { 0x00, 0xfe, 0x00, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 98301, { 0xfd, 0x7f, 0x01, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -32767, { 0xff, 0xff, } },
            { 131068, { 0xfc, 0xff, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -1, { 0x81, } },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { -2, { 0x82, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 98302, { 0xfe, 0x7f, 0x01, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 163837, { 0xfd, 0x7f, 0x02, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { 196604, { 0xfc, 0xff, 0x02, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { -65536, { 0x00, 0x00, 0x81, } },
        },
        {
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 0, {} },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { 163839, { 0xff, 0x7f, 0x02, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -98305, { 0x01, 0x80, 0x81, } },
            { 196606, { 0xfe, 0xff, 0x02, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { -65538, { 0x02, 0x00, 0x81, } },
        },
    },
    {
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65537, { 0x01, 0x00, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65533, { 0xfd, 0xff, 0x00, } },
            { 131072, { 0x00, 0x00, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131068, { 0xfc, 0xff, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 65656, { 0x78, 0x00, 0x01, } },
            { 65414, { 0x86, 0xff, 0x00, } },
            { 65777, { 0xf1, 0x00, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65293, { 0x0d, 0xff, 0x00, } },
            { 131312, { 0xf0, 0x00, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 130828, { 0x0c, 0xff, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { 65407, { 0x7f, 0xff, 0x00, } },
            { 65791, { 0xff, 0x00, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65279, { 0xff, 0xfe, 0x00, } },
            { 131326, { 0xfe, 0x00, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 130814, { 0xfe, 0xfe, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 65664, { 0x80, 0x00, 0x01, } },
            { 65406, { 0x7e, 0xff, 0x00, } },
            { 65793, { 0x01, 0x01, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65277, { 0xfd, 0xfe, 0x00, } },
            { 131328, { 0x00, 0x01, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 130812, { 0xfc, 0xfe, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 65790, { 0xfe, 0x00, 0x01, } },
            { 65280, { 0x00, 0xff, 0x00, } },
            { 66045, { 0xfd, 0x01, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65025, { 0x01, 0xfe, 0x00, } },
            { 131580, { 0xfc, 0x01, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 130560, { 0x00, 0xfe, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 98302, { 0xfe, 0x7f, 0x01, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 131069, { 0xfd, 0xff, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 1, { 0x01, } },
            { 196604, { 0xfc, 0xff, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 65536, { 0x00, 0x00, 0x01, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { 32767, { 0xff, 0x7f, } },
            { 131071, { 0xff, 0xff, 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -1, { 0x81, } },
            { 196606, { 0xfe, 0xff, 0x02, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 196605, { 0xfd, 0xff, 0x02, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 262140, { 0xfc, 0xff, 0x03, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
        },
        {
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 0, {} },
            { 131071, { 0xff, 0xff, 0x01, } },
            { -1, { 0x81, } },
            { 196607, { 0xff, 0xff, 0x02, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -65537, { 0x01, 0x00, 0x81, } },
            { 262142, { 0xfe, 0xff, 0x03, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { 131070, { 0xfe, 0xff, 0x01, } },
            { -2, { 0x82, } },
        },
    },
    {
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x02, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x02, 0x00, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147483527, { 0x87, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x79, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483406, { 0x0e, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xf2, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x0e, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xf2, 0x00, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147483520, { 0x80, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483392, { 0x00, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x01, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x01, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x81, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483390, { 0xfe, 0xfe, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x02, 0x01, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xfe, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x02, 0x01, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147483393, { 0x01, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483138, { 0x02, 0xfe, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x01, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x02, 0xfe, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0x01, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147450881, { 0x01, 0x80, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } },
            { -2147418114, { 0x02, 0x00, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x02, 0x00, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147450880, { 0x00, 0x80, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0x00, 0x80, 0x80, } },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x01, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x01, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147418113, { 0x01, 0x00, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } },
            { -2147352578, { 0x02, 0x00, 0xfe, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x01, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x02, 0x00, 0xfe, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x01, 0x00, 0x81, } },
        },
        {
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { 0, {} },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x01, 0x80, 0x80, } },
            { -2147352576, { 0x00, 0x00, 0xfe, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x02, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xfe, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x02, 0x00, 0x81, } },
        },
    },
    {
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483646, { 0xfe, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483645, { 0xfd, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfc, 0xff, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0x78, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483526, { 0x86, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xf1, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483405, { 0x0d, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xf0, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x0c, 0xff, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0x7f, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483391, { 0xff, 0xfe, 0xff, 0x7f, } },
            { 2147483647, { 0xfe, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xfe, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483518, { 0x7e, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x01, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483389, { 0xfd, 0xfe, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x01, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfc, 0xfe, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483392, { 0x00, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xfd, 0x01, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483137, { 0x01, 0xfe, 0xff, 0x7f, } },
            { 2147483647, { 0xfc, 0x01, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x00, 0xfe, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0x7f, 0x00, 0x80, 0x00, } },
            { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } },
            { 2147483647, { 0xfd, 0xff, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0xfc, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } },
            { 2147450879, { 0xff, 0x7f, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147418111, { 0xff, 0xff, 0xfe, 0x7f, } },
            { 2147483647, { 0xfe, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xfe, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0xff, 0x00, 0x80, 0x00, } },
            { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0xfd, 0xff, 0x01, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147352577, { 0x01, 0x00, 0xfe, 0x7f, } },
            { 2147483647, { 0xfc, 0xff, 0x01, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0xfe, 0xff, 0x00, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
            { 2147418111, { 0xff, 0xff, 0xfe, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0x01, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147352575, { 0xff, 0xff, 0xfd, 0x7f, } },
            { 2147483647, { 0xfe, 0xff, 0x01, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0xfd, 0xff, 0x00, } },
        },
    },
    {
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfd, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfc, 0xff, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0x78, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x86, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xf1, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x0d, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xf0, 0x00, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0x0c, 0xff, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0x7f, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xfe, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0x00, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xfe, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0x80, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x7e, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x01, 0x01, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfd, 0xfe, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x00, 0x01, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfc, 0xfe, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0x00, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfd, 0x01, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x01, 0xfe, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfc, 0x01, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0x00, 0xfe, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0x7f, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfd, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfc, 0xff, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0x7f, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0x00, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xfe, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xfd, 0xff, 0x01, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0x01, 0x00, 0xfe, 0xff, 0x00, } },
            { 2147483647, { 0xfc, 0xff, 0x01, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0x00, 0x00, 0xfe, 0xff, 0x01, } },
        },
        {
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0x00, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0x01, 0x00, 0x01, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xfd, 0xff, 0x00, } },
            { 2147483647, { 0xfe, 0xff, 0x01, 0x00, 0x02, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } },
            { 2147483647, { 0xfe, 0xff, 0xfd, 0xff, 0x01, } },
        },
    },
    {
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { (-2147483647 - 1), { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
        },
    },
    {
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x0d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x03, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x03, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
        },
    },
};

const script_number_buffer script_number_negates[13][9][12] =
{
    {
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -121, { 0xf9, } },
            { -121, { 0xf9, } },
            { 121, { 0x79, } },
            { 121, { 0x79, } },
            { -121, { 0xf9, } },
            { -121, { 0xf9, } },
            { 121, { 0x79, } },
            { 121, { 0x79, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -129, { 0x81, 0x80, } },
            { -129, { 0x81, 0x80, } },
            { 129, { 0x81, 0x00, } },
            { 129, { 0x81, 0x00, } },
            { -129, { 0x81, 0x80, } },
            { -129, { 0x81, 0x80, } },
            { 129, { 0x81, 0x00, } },
            { 129, { 0x81, 0x00, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { 255, { 0xff, 0x00, } },
            { 255, { 0xff, 0x00, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65536, { 0x00, 0x00, 0x01, } },
        },
    },
    {
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
            { 0, {} },
            { -2, { 0x82, } },
            { -2, { 0x82, } },
            { 0, {} },
            { 0, {} },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -122, { 0xfa, } },
            { -122, { 0xfa, } },
            { 120, { 0x78, } },
            { 120, { 0x78, } },
            { -122, { 0xfa, } },
            { -122, { 0xfa, } },
            { 120, { 0x78, } },
            { 120, { 0x78, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -129, { 0x81, 0x80, } },
            { -129, { 0x81, 0x80, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { -129, { 0x81, 0x80, } },
            { -129, { 0x81, 0x80, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -130, { 0x82, 0x80, } },
            { -130, { 0x82, 0x80, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -130, { 0x82, 0x80, } },
            { -130, { 0x82, 0x80, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { 254, { 0xfe, 0x00, } },
            { 254, { 0xfe, 0x00, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 32766, { 0xfe, 0x7f, } },
            { 32766, { 0xfe, 0x7f, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 32766, { 0xfe, 0x7f, } },
            { 32766, { 0xfe, 0x7f, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { -32769, { 0x01, 0x80, 0x80, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
            { 65534, { 0xfe, 0xff, 0x00, } },
        },
        {
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { -65537, { 0x01, 0x00, 0x81, } },
            { -65537, { 0x01, 0x00, 0x81, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { -65537, { 0x01, 0x00, 0x81, } },
            { -65537, { 0x01, 0x00, 0x81, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
        },
    },
    {
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { 129, { 0x81, 0x00, } },
            { 129, { 0x81, 0x00, } },
            { 127, { 0x7f, } },
            { 127, { 0x7f, } },
            { 129, { 0x81, 0x00, } },
            { 129, { 0x81, 0x00, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 7, { 0x07, } },
            { 7, { 0x07, } },
            { 249, { 0xf9, 0x00, } },
            { 249, { 0xf9, 0x00, } },
            { 7, { 0x07, } },
            { 7, { 0x07, } },
            { 249, { 0xf9, 0x00, } },
            { 249, { 0xf9, 0x00, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 0, {} },
            { 0, {} },
            { 256, { 0x00, 0x01, } },
            { 256, { 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { 256, { 0x00, 0x01, } },
            { 256, { 0x00, 0x01, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { 257, { 0x01, 0x01, } },
            { 257, { 0x01, 0x01, } },
            { -1, { 0x81, } },
            { -1, { 0x81, } },
            { 257, { 0x01, 0x01, } },
            { 257, { 0x01, 0x01, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { 383, { 0x7f, 0x01, } },
            { 383, { 0x7f, 0x01, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { 383, { 0x7f, 0x01, } },
            { 383, { 0x7f, 0x01, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -32639, { 0x7f, 0xff, } },
            { -32639, { 0x7f, 0xff, } },
            { 32895, { 0x7f, 0x80, 0x00, } },
            { 32895, { 0x7f, 0x80, 0x00, } },
            { -32639, { 0x7f, 0xff, } },
            { -32639, { 0x7f, 0xff, } },
            { 32895, { 0x7f, 0x80, 0x00, } },
            { 32895, { 0x7f, 0x80, 0x00, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -32640, { 0x80, 0xff, } },
            { -32640, { 0x80, 0xff, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { -32640, { 0x80, 0xff, } },
            { -32640, { 0x80, 0xff, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32896, { 0x80, 0x80, 0x00, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { 65663, { 0x7f, 0x00, 0x01, } },
            { 65663, { 0x7f, 0x00, 0x01, } },
        },
        {
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -65408, { 0x80, 0xff, 0x80, } },
            { -65408, { 0x80, 0xff, 0x80, } },
            { 65664, { 0x80, 0x00, 0x01, } },
            { 65664, { 0x80, 0x00, 0x01, } },
            { -65408, { 0x80, 0xff, 0x80, } },
            { -65408, { 0x80, 0xff, 0x80, } },
            { 65664, { 0x80, 0x00, 0x01, } },
            { 65664, { 0x80, 0x00, 0x01, } },
        },
    },
    {
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -126, { 0xfe, } },
            { -126, { 0xfe, } },
            { -128, { 0x80, 0x80, } },
            { -128, { 0x80, 0x80, } },
            { -126, { 0xfe, } },
            { -126, { 0xfe, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -248, { 0xf8, 0x80, } },
            { -248, { 0xf8, 0x80, } },
            { -6, { 0x86, } },
            { -6, { 0x86, } },
            { -248, { 0xf8, 0x80, } },
            { -248, { 0xf8, 0x80, } },
            { -6, { 0x86, } },
            { -6, { 0x86, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { 2, { 0x02, } },
            { 2, { 0x02, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -382, { 0x7e, 0x81, } },
            { -382, { 0x7e, 0x81, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
            { -382, { 0x7e, 0x81, } },
            { -382, { 0x7e, 0x81, } },
            { 128, { 0x80, 0x00, } },
            { 128, { 0x80, 0x00, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -32894, { 0x7e, 0x80, 0x80, } },
            { -32894, { 0x7e, 0x80, 0x80, } },
            { 32640, { 0x80, 0x7f, } },
            { 32640, { 0x80, 0x7f, } },
            { -32894, { 0x7e, 0x80, 0x80, } },
            { -32894, { 0x7e, 0x80, 0x80, } },
            { 32640, { 0x80, 0x7f, } },
            { 32640, { 0x80, 0x7f, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { 32641, { 0x81, 0x7f, } },
            { 32641, { 0x81, 0x7f, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { 32641, { 0x81, 0x7f, } },
            { 32641, { 0x81, 0x7f, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -65662, { 0x7e, 0x00, 0x81, } },
            { -65662, { 0x7e, 0x00, 0x81, } },
            { 65408, { 0x80, 0xff, 0x00, } },
            { 65408, { 0x80, 0xff, 0x00, } },
            { -65662, { 0x7e, 0x00, 0x81, } },
            { -65662, { 0x7e, 0x00, 0x81, } },
            { 65408, { 0x80, 0xff, 0x00, } },
            { 65408, { 0x80, 0xff, 0x00, } },
        },
        {
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { 65409, { 0x81, 0xff, 0x00, } },
            { 65409, { 0x81, 0xff, 0x00, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { 65409, { 0x81, 0xff, 0x00, } },
            { 65409, { 0x81, 0xff, 0x00, } },
        },
    },
    {
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -254, { 0xfe, 0x80, } },
            { -254, { 0xfe, 0x80, } },
            { -256, { 0x00, 0x81, } },
            { -256, { 0x00, 0x81, } },
            { -254, { 0xfe, 0x80, } },
            { -254, { 0xfe, 0x80, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -376, { 0x78, 0x81, } },
            { -376, { 0x78, 0x81, } },
            { -134, { 0x86, 0x80, } },
            { -134, { 0x86, 0x80, } },
            { -376, { 0x78, 0x81, } },
            { -376, { 0x78, 0x81, } },
            { -134, { 0x86, 0x80, } },
            { -134, { 0x86, 0x80, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -383, { 0x7f, 0x81, } },
            { -383, { 0x7f, 0x81, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
            { -383, { 0x7f, 0x81, } },
            { -383, { 0x7f, 0x81, } },
            { -127, { 0xff, } },
            { -127, { 0xff, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -384, { 0x80, 0x81, } },
            { -384, { 0x80, 0x81, } },
            { -126, { 0xfe, } },
            { -126, { 0xfe, } },
            { -384, { 0x80, 0x81, } },
            { -384, { 0x80, 0x81, } },
            { -126, { 0xfe, } },
            { -126, { 0xfe, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -510, { 0xfe, 0x81, } },
            { -510, { 0xfe, 0x81, } },
            { 0, {} },
            { 0, {} },
            { -510, { 0xfe, 0x81, } },
            { -510, { 0xfe, 0x81, } },
            { 0, {} },
            { 0, {} },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { 32512, { 0x00, 0x7f, } },
            { 32512, { 0x00, 0x7f, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { 32512, { 0x00, 0x7f, } },
            { 32512, { 0x00, 0x7f, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -33023, { 0xff, 0x80, 0x80, } },
            { -33023, { 0xff, 0x80, 0x80, } },
            { 32513, { 0x01, 0x7f, } },
            { 32513, { 0x01, 0x7f, } },
            { -33023, { 0xff, 0x80, 0x80, } },
            { -33023, { 0xff, 0x80, 0x80, } },
            { 32513, { 0x01, 0x7f, } },
            { 32513, { 0x01, 0x7f, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { 65280, { 0x00, 0xff, 0x00, } },
            { 65280, { 0x00, 0xff, 0x00, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { 65280, { 0x00, 0xff, 0x00, } },
            { 65280, { 0x00, 0xff, 0x00, } },
        },
        {
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -255, { 0xff, 0x80, } },
            { -65791, { 0xff, 0x00, 0x81, } },
            { -65791, { 0xff, 0x00, 0x81, } },
            { 65281, { 0x01, 0xff, 0x00, } },
            { 65281, { 0x01, 0xff, 0x00, } },
            { -65791, { 0xff, 0x00, 0x81, } },
            { -65791, { 0xff, 0x00, 0x81, } },
            { 65281, { 0x01, 0xff, 0x00, } },
            { 65281, { 0x01, 0xff, 0x00, } },
        },
    },
    {
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32767, { 0xff, 0x7f, } },
            { 32767, { 0xff, 0x7f, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32769, { 0x01, 0x80, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32647, { 0x87, 0x7f, } },
            { 32647, { 0x87, 0x7f, } },
            { 32889, { 0x79, 0x80, 0x00, } },
            { 32889, { 0x79, 0x80, 0x00, } },
            { 32647, { 0x87, 0x7f, } },
            { 32647, { 0x87, 0x7f, } },
            { 32889, { 0x79, 0x80, 0x00, } },
            { 32889, { 0x79, 0x80, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32640, { 0x80, 0x7f, } },
            { 32640, { 0x80, 0x7f, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32640, { 0x80, 0x7f, } },
            { 32640, { 0x80, 0x7f, } },
            { 32896, { 0x80, 0x80, 0x00, } },
            { 32896, { 0x80, 0x80, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32639, { 0x7f, 0x7f, } },
            { 32639, { 0x7f, 0x7f, } },
            { 32897, { 0x81, 0x80, 0x00, } },
            { 32897, { 0x81, 0x80, 0x00, } },
            { 32639, { 0x7f, 0x7f, } },
            { 32639, { 0x7f, 0x7f, } },
            { 32897, { 0x81, 0x80, 0x00, } },
            { 32897, { 0x81, 0x80, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32513, { 0x01, 0x7f, } },
            { 32513, { 0x01, 0x7f, } },
            { 33023, { 0xff, 0x80, 0x00, } },
            { 33023, { 0xff, 0x80, 0x00, } },
            { 32513, { 0x01, 0x7f, } },
            { 32513, { 0x01, 0x7f, } },
            { 33023, { 0xff, 0x80, 0x00, } },
            { 33023, { 0xff, 0x80, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { 65535, { 0xff, 0xff, 0x00, } },
            { 65535, { 0xff, 0xff, 0x00, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 0, {} },
            { 0, {} },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 0, {} },
            { 0, {} },
            { 65536, { 0x00, 0x00, 0x01, } },
            { 65536, { 0x00, 0x00, 0x01, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { 98303, { 0xff, 0x7f, 0x01, } },
            { 98303, { 0xff, 0x7f, 0x01, } },
        },
        {
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 98304, { 0x00, 0x80, 0x01, } },
            { 98304, { 0x00, 0x80, 0x01, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { 98304, { 0x00, 0x80, 0x01, } },
            { 98304, { 0x00, 0x80, 0x01, } },
        },
    },
    {
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32766, { 0xfe, 0xff, } },
            { -32766, { 0xfe, 0xff, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32766, { 0xfe, 0xff, } },
            { -32766, { 0xfe, 0xff, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32888, { 0x78, 0x80, 0x80, } },
            { -32888, { 0x78, 0x80, 0x80, } },
            { -32646, { 0x86, 0xff, } },
            { -32646, { 0x86, 0xff, } },
            { -32888, { 0x78, 0x80, 0x80, } },
            { -32888, { 0x78, 0x80, 0x80, } },
            { -32646, { 0x86, 0xff, } },
            { -32646, { 0x86, 0xff, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32639, { 0x7f, 0xff, } },
            { -32639, { 0x7f, 0xff, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32895, { 0x7f, 0x80, 0x80, } },
            { -32639, { 0x7f, 0xff, } },
            { -32639, { 0x7f, 0xff, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { -32638, { 0x7e, 0xff, } },
            { -32638, { 0x7e, 0xff, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { -32896, { 0x80, 0x80, 0x80, } },
            { -32638, { 0x7e, 0xff, } },
            { -32638, { 0x7e, 0xff, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -32512, { 0x00, 0xff, } },
            { -32512, { 0x00, 0xff, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -33022, { 0xfe, 0x80, 0x80, } },
            { -32512, { 0x00, 0xff, } },
            { -32512, { 0x00, 0xff, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { 0, {} },
            { 0, {} },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { 0, {} },
            { 0, {} },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { 32768, { 0x00, 0x80, 0x00, } },
            { 32768, { 0x00, 0x80, 0x00, } },
        },
        {
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { 32769, { 0x01, 0x80, 0x00, } },
            { 32769, { 0x01, 0x80, 0x00, } },
        },
    },
    {
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65536, { 0x00, 0x00, 0x81, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
            { -65534, { 0xfe, 0xff, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65656, { 0x78, 0x00, 0x81, } },
            { -65656, { 0x78, 0x00, 0x81, } },
            { -65414, { 0x86, 0xff, 0x80, } },
            { -65414, { 0x86, 0xff, 0x80, } },
            { -65656, { 0x78, 0x00, 0x81, } },
            { -65656, { 0x78, 0x00, 0x81, } },
            { -65414, { 0x86, 0xff, 0x80, } },
            { -65414, { 0x86, 0xff, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65663, { 0x7f, 0x00, 0x81, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
            { -65407, { 0x7f, 0xff, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { -65406, { 0x7e, 0xff, 0x80, } },
            { -65406, { 0x7e, 0xff, 0x80, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { -65664, { 0x80, 0x00, 0x81, } },
            { -65406, { 0x7e, 0xff, 0x80, } },
            { -65406, { 0x7e, 0xff, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65280, { 0x00, 0xff, 0x80, } },
            { -65280, { 0x00, 0xff, 0x80, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65790, { 0xfe, 0x00, 0x81, } },
            { -65280, { 0x00, 0xff, 0x80, } },
            { -65280, { 0x00, 0xff, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -98302, { 0xfe, 0x7f, 0x81, } },
            { -32768, { 0x00, 0x80, 0x80, } },
            { -32768, { 0x00, 0x80, 0x80, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -98303, { 0xff, 0x7f, 0x81, } },
            { -32767, { 0xff, 0xff, } },
            { -32767, { 0xff, 0xff, } },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -131070, { 0xfe, 0xff, 0x81, } },
            { -131070, { 0xfe, 0xff, 0x81, } },
            { 0, {} },
            { 0, {} },
            { -131070, { 0xfe, 0xff, 0x81, } },
            { -131070, { 0xfe, 0xff, 0x81, } },
            { 0, {} },
            { 0, {} },
        },
        {
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -65535, { 0xff, 0xff, 0x80, } },
            { -131071, { 0xff, 0xff, 0x81, } },
            { -131071, { 0xff, 0xff, 0x81, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
            { -131071, { 0xff, 0xff, 0x81, } },
            { -131071, { 0xff, 0xff, 0x81, } },
            { 1, { 0x01, } },
            { 1, { 0x01, } },
        },
    },
    {
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483527, { 0x87, 0xff, 0xff, 0x7f, } },
            { 2147483527, { 0x87, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x79, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x79, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483527, { 0x87, 0xff, 0xff, 0x7f, } },
            { 2147483527, { 0x87, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x79, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x79, 0x00, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483520, { 0x80, 0xff, 0xff, 0x7f, } },
            { 2147483520, { 0x80, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483520, { 0x80, 0xff, 0xff, 0x7f, } },
            { 2147483520, { 0x80, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } },
            { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x81, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x81, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } },
            { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x81, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x81, 0x00, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483393, { 0x01, 0xff, 0xff, 0x7f, } },
            { 2147483393, { 0x01, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483393, { 0x01, 0xff, 0xff, 0x7f, } },
            { 2147483393, { 0x01, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147450881, { 0x01, 0x80, 0xff, 0x7f, } },
            { 2147450881, { 0x01, 0x80, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } },
            { 2147450881, { 0x01, 0x80, 0xff, 0x7f, } },
            { 2147450881, { 0x01, 0x80, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } },
            { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x80, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x80, 0x00, 0x80, 0x00, } },
            { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } },
            { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x80, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x80, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } },
            { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
            { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } },
            { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } },
        },
        {
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } },
            { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } },
            { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } },
            { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } },
            { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } },
            { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } },
        },
    },
    {
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } },
            { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } },
            { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483526, { 0x86, 0xff, 0xff, 0xff, } },
            { -2147483526, { 0x86, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483526, { 0x86, 0xff, 0xff, 0xff, } },
            { -2147483526, { 0x86, 0xff, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } },
            { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } },
            { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483518, { 0x7e, 0xff, 0xff, 0xff, } },
            { -2147483518, { 0x7e, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483518, { 0x7e, 0xff, 0xff, 0xff, } },
            { -2147483518, { 0x7e, 0xff, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483392, { 0x00, 0xff, 0xff, 0xff, } },
            { -2147483392, { 0x00, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x80, 0x80, } },
            { -2147483392, { 0x00, 0xff, 0xff, 0xff, } },
            { -2147483392, { 0x00, 0xff, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x80, 0x80, } },
            { -2147450880, { 0x00, 0x80, 0xff, 0xff, } },
            { -2147450880, { 0x00, 0x80, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x80, 0x80, } },
            { -2147450880, { 0x00, 0x80, 0xff, 0xff, } },
            { -2147450880, { 0x00, 0x80, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } },
            { -2147450879, { 0xff, 0x7f, 0xff, 0xff, } },
            { -2147450879, { 0xff, 0x7f, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } },
            { -2147450879, { 0xff, 0x7f, 0xff, 0xff, } },
            { -2147450879, { 0xff, 0x7f, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
            { -2147418112, { 0x00, 0x00, 0xff, 0xff, } },
        },
        {
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { -2147483647, { 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } },
            { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } },
            { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } },
            { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } },
            { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } },
        },
    },
    {
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } },
        },
    },
    {
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 0, {} },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
    },
    {
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 0, {} },
            { 0, {} },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x88, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x88, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x88, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x88, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x81, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x81, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x81, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x81, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
        {
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
            { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } },
        },
    },
};

const script_number_subtract script_number_subtracts[13][9][12] =
{
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -121, { 0xf9, } }, { 121, { 0x79, } }, },
            { { 121, { 0x79, } }, { -121, { 0xf9, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 128, { 0x80, 0x00, } }, { -128, { 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -129, { 0x81, 0x80, } }, { 129, { 0x81, 0x00, } }, },
            { { 129, { 0x81, 0x00, } }, { -129, { 0x81, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -255, { 0xff, 0x80, } }, { 255, { 0xff, 0x00, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -32767, { 0xff, 0xff, } }, { 32767, { 0xff, 0x7f, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -65535, { 0xff, 0xff, 0x80, } }, { 65535, { 0xff, 0xff, 0x00, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 3, { 0x03, } }, { -3, { 0x83, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -120, { 0xf8, } }, { 120, { 0x78, } }, },
            { { 122, { 0x7a, } }, { -122, { 0xfa, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 243, { 0xf3, 0x00, } }, { -243, { 0xf3, 0x80, } }, },
            { { -241, { 0xf1, 0x80, } }, { 241, { 0xf1, 0x00, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -127, { 0xff, } }, { 127, { 0x7f, } }, },
            { { 129, { 0x81, 0x00, } }, { -129, { 0x81, 0x80, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 257, { 0x01, 0x01, } }, { -257, { 0x01, 0x81, } }, },
            { { -255, { 0xff, 0x80, } }, { 255, { 0xff, 0x00, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 130, { 0x82, 0x00, } }, { -130, { 0x82, 0x80, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 259, { 0x03, 0x01, } }, { -259, { 0x03, 0x81, } }, },
            { { -257, { 0x01, 0x81, } }, { 257, { 0x01, 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -254, { 0xfe, 0x80, } }, { 254, { 0xfe, 0x00, } }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 511, { 0xff, 0x01, } }, { -511, { 0xff, 0x81, } }, },
            { { -509, { 0xfd, 0x81, } }, { 509, { 0xfd, 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -32766, { 0xfe, 0xff, } }, { 32766, { 0xfe, 0x7f, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { -65533, { 0xfd, 0xff, 0x80, } }, { 65533, { 0xfd, 0xff, 0x00, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -32767, { 0xff, 0xff, } }, { 32767, { 0xff, 0x7f, } }, },
            { { 32769, { 0x01, 0x80, 0x00, } }, { -32769, { 0x01, 0x80, 0x80, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 65537, { 0x01, 0x00, 0x01, } }, { -65537, { 0x01, 0x00, 0x81, } }, },
            { { -65535, { 0xff, 0xff, 0x80, } }, { 65535, { 0xff, 0xff, 0x00, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 131071, { 0xff, 0xff, 0x01, } }, { -131071, { 0xff, 0xff, 0x81, } }, },
            { { -131069, { 0xfd, 0xff, 0x81, } }, { 131069, { 0xfd, 0xff, 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -65535, { 0xff, 0xff, 0x80, } }, { 65535, { 0xff, 0xff, 0x00, } }, },
            { { 65537, { 0x01, 0x00, 0x01, } }, { -65537, { 0x01, 0x00, 0x81, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 131073, { 0x01, 0x00, 0x02, } }, { -131073, { 0x01, 0x00, 0x82, } }, },
            { { -131071, { 0xff, 0xff, 0x81, } }, { 131071, { 0xff, 0xff, 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -129, { 0x81, 0x80, } }, { 129, { 0x81, 0x00, } }, },
            { { -127, { 0xff, } }, { 127, { 0x7f, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { -126, { 0xfe, } }, { 126, { 0x7e, } }, },
            { { -130, { 0x82, 0x80, } }, { 130, { 0x82, 0x00, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -249, { 0xf9, 0x80, } }, { 249, { 0xf9, 0x00, } }, },
            { { -7, { 0x87, } }, { 7, { 0x07, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 114, { 0x72, } }, { -114, { 0xf2, } }, },
            { { -370, { 0x72, 0x81, } }, { 370, { 0x72, 0x01, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 128, { 0x80, 0x00, } }, { -128, { 0x80, 0x80, } }, },
            { { -384, { 0x80, 0x81, } }, { 384, { 0x80, 0x01, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -257, { 0x01, 0x81, } }, { 257, { 0x01, 0x01, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 130, { 0x82, 0x00, } }, { -130, { 0x82, 0x80, } }, },
            { { -386, { 0x82, 0x81, } }, { 386, { 0x82, 0x01, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -383, { 0x7f, 0x81, } }, { 383, { 0x7f, 0x01, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 382, { 0x7e, 0x01, } }, { -382, { 0x7e, 0x81, } }, },
            { { -638, { 0x7e, 0x82, } }, { 638, { 0x7e, 0x02, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -32895, { 0x7f, 0x80, 0x80, } }, { 32895, { 0x7f, 0x80, 0x00, } }, },
            { { 32639, { 0x7f, 0x7f, } }, { -32639, { 0x7f, 0xff, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 65406, { 0x7e, 0xff, 0x00, } }, { -65406, { 0x7e, 0xff, 0x80, } }, },
            { { -65662, { 0x7e, 0x00, 0x81, } }, { 65662, { 0x7e, 0x00, 0x01, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -32896, { 0x80, 0x80, 0x80, } }, { 32896, { 0x80, 0x80, 0x00, } }, },
            { { 32640, { 0x80, 0x7f, } }, { -32640, { 0x80, 0xff, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 65408, { 0x80, 0xff, 0x00, } }, { -65408, { 0x80, 0xff, 0x80, } }, },
            { { -65664, { 0x80, 0x00, 0x81, } }, { 65664, { 0x80, 0x00, 0x01, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -65663, { 0x7f, 0x00, 0x81, } }, { 65663, { 0x7f, 0x00, 0x01, } }, },
            { { 65407, { 0x7f, 0xff, 0x00, } }, { -65407, { 0x7f, 0xff, 0x80, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 130942, { 0x7e, 0xff, 0x01, } }, { -130942, { 0x7e, 0xff, 0x81, } }, },
            { { -131198, { 0x7e, 0x00, 0x82, } }, { 131198, { 0x7e, 0x00, 0x02, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { -65664, { 0x80, 0x00, 0x81, } }, { 65664, { 0x80, 0x00, 0x01, } }, },
            { { 65408, { 0x80, 0xff, 0x00, } }, { -65408, { 0x80, 0xff, 0x80, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 130944, { 0x80, 0xff, 0x01, } }, { -130944, { 0x80, 0xff, 0x81, } }, },
            { { -131200, { 0x80, 0x00, 0x82, } }, { 131200, { 0x80, 0x00, 0x02, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { 126, { 0x7e, } }, { -126, { 0xfe, } }, },
            { { 128, { 0x80, 0x00, } }, { -128, { 0x80, 0x80, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 129, { 0x81, 0x00, } }, { -129, { 0x81, 0x80, } }, },
            { { 125, { 0x7d, } }, { -125, { 0xfd, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { 6, { 0x06, } }, { -6, { 0x86, } }, },
            { { 248, { 0xf8, 0x00, } }, { -248, { 0xf8, 0x80, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 369, { 0x71, 0x01, } }, { -369, { 0x71, 0x81, } }, },
            { { -115, { 0xf3, } }, { 115, { 0x73, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 383, { 0x7f, 0x01, } }, { -383, { 0x7f, 0x81, } }, },
            { { -129, { 0x81, 0x80, } }, { 129, { 0x81, 0x00, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 385, { 0x81, 0x01, } }, { -385, { 0x81, 0x81, } }, },
            { { -131, { 0x83, 0x80, } }, { 131, { 0x83, 0x00, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -128, { 0x80, 0x80, } }, { 128, { 0x80, 0x00, } }, },
            { { 382, { 0x7e, 0x01, } }, { -382, { 0x7e, 0x81, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 637, { 0x7d, 0x02, } }, { -637, { 0x7d, 0x82, } }, },
            { { -383, { 0x7f, 0x81, } }, { 383, { 0x7f, 0x01, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -32640, { 0x80, 0xff, } }, { 32640, { 0x80, 0x7f, } }, },
            { { 32894, { 0x7e, 0x80, 0x00, } }, { -32894, { 0x7e, 0x80, 0x80, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 65661, { 0x7d, 0x00, 0x01, } }, { -65661, { 0x7d, 0x00, 0x81, } }, },
            { { -65407, { 0x7f, 0xff, 0x80, } }, { 65407, { 0x7f, 0xff, 0x00, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -32641, { 0x81, 0xff, } }, { 32641, { 0x81, 0x7f, } }, },
            { { 32895, { 0x7f, 0x80, 0x00, } }, { -32895, { 0x7f, 0x80, 0x80, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 65663, { 0x7f, 0x00, 0x01, } }, { -65663, { 0x7f, 0x00, 0x81, } }, },
            { { -65409, { 0x81, 0xff, 0x80, } }, { 65409, { 0x81, 0xff, 0x00, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -65408, { 0x80, 0xff, 0x80, } }, { 65408, { 0x80, 0xff, 0x00, } }, },
            { { 65662, { 0x7e, 0x00, 0x01, } }, { -65662, { 0x7e, 0x00, 0x81, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 131197, { 0x7d, 0x00, 0x02, } }, { -131197, { 0x7d, 0x00, 0x82, } }, },
            { { -130943, { 0x7f, 0xff, 0x81, } }, { 130943, { 0x7f, 0xff, 0x01, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { -65409, { 0x81, 0xff, 0x80, } }, { 65409, { 0x81, 0xff, 0x00, } }, },
            { { 65663, { 0x7f, 0x00, 0x01, } }, { -65663, { 0x7f, 0x00, 0x81, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 131199, { 0x7f, 0x00, 0x02, } }, { -131199, { 0x7f, 0x00, 0x82, } }, },
            { { -130945, { 0x81, 0xff, 0x81, } }, { 130945, { 0x81, 0xff, 0x01, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 254, { 0xfe, 0x00, } }, { -254, { 0xfe, 0x80, } }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 257, { 0x01, 0x01, } }, { -257, { 0x01, 0x81, } }, },
            { { 253, { 0xfd, 0x00, } }, { -253, { 0xfd, 0x80, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 134, { 0x86, 0x00, } }, { -134, { 0x86, 0x80, } }, },
            { { 376, { 0x78, 0x01, } }, { -376, { 0x78, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 497, { 0xf1, 0x01, } }, { -497, { 0xf1, 0x81, } }, },
            { { 13, { 0x0d, } }, { -13, { 0x8d, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 127, { 0x7f, } }, { -127, { 0xff, } }, },
            { { 383, { 0x7f, 0x01, } }, { -383, { 0x7f, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 511, { 0xff, 0x01, } }, { -511, { 0xff, 0x81, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 126, { 0x7e, } }, { -126, { 0xfe, } }, },
            { { 384, { 0x80, 0x01, } }, { -384, { 0x80, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 513, { 0x01, 0x02, } }, { -513, { 0x01, 0x82, } }, },
            { { -3, { 0x83, } }, { 3, { 0x03, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 765, { 0xfd, 0x02, } }, { -765, { 0xfd, 0x82, } }, },
            { { -255, { 0xff, 0x80, } }, { 255, { 0xff, 0x00, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -32512, { 0x00, 0xff, } }, { 32512, { 0x00, 0x7f, } }, },
            { { 33022, { 0xfe, 0x80, 0x00, } }, { -33022, { 0xfe, 0x80, 0x80, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 65789, { 0xfd, 0x00, 0x01, } }, { -65789, { 0xfd, 0x00, 0x81, } }, },
            { { -65279, { 0xff, 0xfe, 0x80, } }, { 65279, { 0xff, 0xfe, 0x00, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -32513, { 0x01, 0xff, } }, { 32513, { 0x01, 0x7f, } }, },
            { { 33023, { 0xff, 0x80, 0x00, } }, { -33023, { 0xff, 0x80, 0x80, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 65791, { 0xff, 0x00, 0x01, } }, { -65791, { 0xff, 0x00, 0x81, } }, },
            { { -65281, { 0x01, 0xff, 0x80, } }, { 65281, { 0x01, 0xff, 0x00, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -65280, { 0x00, 0xff, 0x80, } }, { 65280, { 0x00, 0xff, 0x00, } }, },
            { { 65790, { 0xfe, 0x00, 0x01, } }, { -65790, { 0xfe, 0x00, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 131325, { 0xfd, 0x00, 0x02, } }, { -131325, { 0xfd, 0x00, 0x82, } }, },
            { { -130815, { 0xff, 0xfe, 0x81, } }, { 130815, { 0xff, 0xfe, 0x01, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -65281, { 0x01, 0xff, 0x80, } }, { 65281, { 0x01, 0xff, 0x00, } }, },
            { { 65791, { 0xff, 0x00, 0x01, } }, { -65791, { 0xff, 0x00, 0x81, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 131327, { 0xff, 0x00, 0x02, } }, { -131327, { 0xff, 0x00, 0x82, } }, },
            { { -130817, { 0x01, 0xff, 0x81, } }, { 130817, { 0x01, 0xff, 0x01, } }, },
            { { 255, { 0xff, 0x00, } }, { -255, { 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -32769, { 0x01, 0x80, 0x80, } }, { 32769, { 0x01, 0x80, 0x00, } }, },
            { { -32767, { 0xff, 0xff, } }, { 32767, { 0xff, 0x7f, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { -32766, { 0xfe, 0xff, } }, { 32766, { 0xfe, 0x7f, } }, },
            { { -32770, { 0x02, 0x80, 0x80, } }, { 32770, { 0x02, 0x80, 0x00, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -32889, { 0x79, 0x80, 0x80, } }, { 32889, { 0x79, 0x80, 0x00, } }, },
            { { -32647, { 0x87, 0xff, } }, { 32647, { 0x87, 0x7f, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { -32526, { 0x0e, 0xff, } }, { 32526, { 0x0e, 0x7f, } }, },
            { { -33010, { 0xf2, 0x80, 0x80, } }, { 33010, { 0xf2, 0x80, 0x00, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -32896, { 0x80, 0x80, 0x80, } }, { 32896, { 0x80, 0x80, 0x00, } }, },
            { { -32640, { 0x80, 0xff, } }, { 32640, { 0x80, 0x7f, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { -32512, { 0x00, 0xff, } }, { 32512, { 0x00, 0x7f, } }, },
            { { -33024, { 0x00, 0x81, 0x80, } }, { 33024, { 0x00, 0x81, 0x00, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -32897, { 0x81, 0x80, 0x80, } }, { 32897, { 0x81, 0x80, 0x00, } }, },
            { { -32639, { 0x7f, 0xff, } }, { 32639, { 0x7f, 0x7f, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { -32510, { 0xfe, 0xfe, } }, { 32510, { 0xfe, 0x7e, } }, },
            { { -33026, { 0x02, 0x81, 0x80, } }, { 33026, { 0x02, 0x81, 0x00, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -33023, { 0xff, 0x80, 0x80, } }, { 33023, { 0xff, 0x80, 0x00, } }, },
            { { -32513, { 0x01, 0xff, } }, { 32513, { 0x01, 0x7f, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { -32258, { 0x02, 0xfe, } }, { 32258, { 0x02, 0x7e, } }, },
            { { -33278, { 0xfe, 0x81, 0x80, } }, { 33278, { 0xfe, 0x81, 0x00, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -65535, { 0xff, 0xff, 0x80, } }, { 65535, { 0xff, 0xff, 0x00, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 32766, { 0xfe, 0x7f, } }, { -32766, { 0xfe, 0xff, } }, },
            { { -98302, { 0xfe, 0x7f, 0x81, } }, { 98302, { 0xfe, 0x7f, 0x01, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { -98304, { 0x00, 0x80, 0x81, } }, { 98304, { 0x00, 0x80, 0x01, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -98303, { 0xff, 0x7f, 0x81, } }, { 98303, { 0xff, 0x7f, 0x01, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 98302, { 0xfe, 0x7f, 0x01, } }, { -98302, { 0xfe, 0x7f, 0x81, } }, },
            { { -163838, { 0xfe, 0x7f, 0x82, } }, { 163838, { 0xfe, 0x7f, 0x02, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { -98304, { 0x00, 0x80, 0x81, } }, { 98304, { 0x00, 0x80, 0x01, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 98304, { 0x00, 0x80, 0x01, } }, { -98304, { 0x00, 0x80, 0x81, } }, },
            { { -163840, { 0x00, 0x80, 0x82, } }, { 163840, { 0x00, 0x80, 0x02, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32766, { 0xfe, 0x7f, } }, { -32766, { 0xfe, 0xff, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 32769, { 0x01, 0x80, 0x00, } }, { -32769, { 0x01, 0x80, 0x80, } }, },
            { { 32765, { 0xfd, 0x7f, } }, { -32765, { 0xfd, 0xff, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32646, { 0x86, 0x7f, } }, { -32646, { 0x86, 0xff, } }, },
            { { 32888, { 0x78, 0x80, 0x00, } }, { -32888, { 0x78, 0x80, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 33009, { 0xf1, 0x80, 0x00, } }, { -33009, { 0xf1, 0x80, 0x80, } }, },
            { { 32525, { 0x0d, 0x7f, } }, { -32525, { 0x0d, 0xff, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32639, { 0x7f, 0x7f, } }, { -32639, { 0x7f, 0xff, } }, },
            { { 32895, { 0x7f, 0x80, 0x00, } }, { -32895, { 0x7f, 0x80, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 33023, { 0xff, 0x80, 0x00, } }, { -33023, { 0xff, 0x80, 0x80, } }, },
            { { 32511, { 0xff, 0x7e, } }, { -32511, { 0xff, 0xfe, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32638, { 0x7e, 0x7f, } }, { -32638, { 0x7e, 0xff, } }, },
            { { 32896, { 0x80, 0x80, 0x00, } }, { -32896, { 0x80, 0x80, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 33025, { 0x01, 0x81, 0x00, } }, { -33025, { 0x01, 0x81, 0x80, } }, },
            { { 32509, { 0xfd, 0x7e, } }, { -32509, { 0xfd, 0xfe, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32512, { 0x00, 0x7f, } }, { -32512, { 0x00, 0xff, } }, },
            { { 33022, { 0xfe, 0x80, 0x00, } }, { -33022, { 0xfe, 0x80, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 33277, { 0xfd, 0x81, 0x00, } }, { -33277, { 0xfd, 0x81, 0x80, } }, },
            { { 32257, { 0x01, 0x7e, } }, { -32257, { 0x01, 0xfe, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 98301, { 0xfd, 0x7f, 0x01, } }, { -98301, { 0xfd, 0x7f, 0x81, } }, },
            { { -32767, { 0xff, 0xff, } }, { 32767, { 0xff, 0x7f, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 98303, { 0xff, 0x7f, 0x01, } }, { -98303, { 0xff, 0x7f, 0x81, } }, },
            { { -32769, { 0x01, 0x80, 0x80, } }, { 32769, { 0x01, 0x80, 0x00, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -32768, { 0x00, 0x80, 0x80, } }, { 32768, { 0x00, 0x80, 0x00, } }, },
            { { 98302, { 0xfe, 0x7f, 0x01, } }, { -98302, { 0xfe, 0x7f, 0x81, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 163837, { 0xfd, 0x7f, 0x02, } }, { -163837, { 0xfd, 0x7f, 0x82, } }, },
            { { -98303, { 0xff, 0x7f, 0x81, } }, { 98303, { 0xff, 0x7f, 0x01, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -32769, { 0x01, 0x80, 0x80, } }, { 32769, { 0x01, 0x80, 0x00, } }, },
            { { 98303, { 0xff, 0x7f, 0x01, } }, { -98303, { 0xff, 0x7f, 0x81, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 163839, { 0xff, 0x7f, 0x02, } }, { -163839, { 0xff, 0x7f, 0x82, } }, },
            { { -98305, { 0x01, 0x80, 0x81, } }, { 98305, { 0x01, 0x80, 0x01, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 65537, { 0x01, 0x00, 0x01, } }, { -65537, { 0x01, 0x00, 0x81, } }, },
            { { 65533, { 0xfd, 0xff, 0x00, } }, { -65533, { 0xfd, 0xff, 0x80, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65414, { 0x86, 0xff, 0x00, } }, { -65414, { 0x86, 0xff, 0x80, } }, },
            { { 65656, { 0x78, 0x00, 0x01, } }, { -65656, { 0x78, 0x00, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 65777, { 0xf1, 0x00, 0x01, } }, { -65777, { 0xf1, 0x00, 0x81, } }, },
            { { 65293, { 0x0d, 0xff, 0x00, } }, { -65293, { 0x0d, 0xff, 0x80, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65407, { 0x7f, 0xff, 0x00, } }, { -65407, { 0x7f, 0xff, 0x80, } }, },
            { { 65663, { 0x7f, 0x00, 0x01, } }, { -65663, { 0x7f, 0x00, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 65791, { 0xff, 0x00, 0x01, } }, { -65791, { 0xff, 0x00, 0x81, } }, },
            { { 65279, { 0xff, 0xfe, 0x00, } }, { -65279, { 0xff, 0xfe, 0x80, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65406, { 0x7e, 0xff, 0x00, } }, { -65406, { 0x7e, 0xff, 0x80, } }, },
            { { 65664, { 0x80, 0x00, 0x01, } }, { -65664, { 0x80, 0x00, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 65793, { 0x01, 0x01, 0x01, } }, { -65793, { 0x01, 0x01, 0x81, } }, },
            { { 65277, { 0xfd, 0xfe, 0x00, } }, { -65277, { 0xfd, 0xfe, 0x80, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65280, { 0x00, 0xff, 0x00, } }, { -65280, { 0x00, 0xff, 0x80, } }, },
            { { 65790, { 0xfe, 0x00, 0x01, } }, { -65790, { 0xfe, 0x00, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 66045, { 0xfd, 0x01, 0x01, } }, { -66045, { 0xfd, 0x01, 0x81, } }, },
            { { 65025, { 0x01, 0xfe, 0x00, } }, { -65025, { 0x01, 0xfe, 0x80, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 32768, { 0x00, 0x80, 0x00, } }, { -32768, { 0x00, 0x80, 0x80, } }, },
            { { 98302, { 0xfe, 0x7f, 0x01, } }, { -98302, { 0xfe, 0x7f, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 131069, { 0xfd, 0xff, 0x01, } }, { -131069, { 0xfd, 0xff, 0x81, } }, },
            { { 1, { 0x01, } }, { -1, { 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 32767, { 0xff, 0x7f, } }, { -32767, { 0xff, 0xff, } }, },
            { { 98303, { 0xff, 0x7f, 0x01, } }, { -98303, { 0xff, 0x7f, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 131071, { 0xff, 0xff, 0x01, } }, { -131071, { 0xff, 0xff, 0x81, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 196605, { 0xfd, 0xff, 0x02, } }, { -196605, { 0xfd, 0xff, 0x82, } }, },
            { { -65535, { 0xff, 0xff, 0x80, } }, { 65535, { 0xff, 0xff, 0x00, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -1, { 0x81, } }, { 1, { 0x01, } }, },
            { { 131071, { 0xff, 0xff, 0x01, } }, { -131071, { 0xff, 0xff, 0x81, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 196607, { 0xff, 0xff, 0x02, } }, { -196607, { 0xff, 0xff, 0x82, } }, },
            { { -65537, { 0x01, 0x00, 0x81, } }, { 65537, { 0x01, 0x00, 0x01, } }, },
            { { 65535, { 0xff, 0xff, 0x00, } }, { -65535, { 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x01, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } }, { 2147483646, { 0xfe, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x02, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x02, 0x00, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x79, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x79, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483527, { 0x87, 0xff, 0xff, 0xff, } }, { 2147483527, { 0x87, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483406, { 0x0e, 0xff, 0xff, 0xff, } }, { 2147483406, { 0x0e, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0xf2, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xf2, 0x00, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483520, { 0x80, 0xff, 0xff, 0xff, } }, { 2147483520, { 0x80, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483392, { 0x00, 0xff, 0xff, 0xff, } }, { 2147483392, { 0x00, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x01, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x01, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x81, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x81, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } }, { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483390, { 0xfe, 0xfe, 0xff, 0xff, } }, { 2147483390, { 0xfe, 0xfe, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x02, 0x01, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x02, 0x01, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0xff, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483393, { 0x01, 0xff, 0xff, 0xff, } }, { 2147483393, { 0x01, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147483138, { 0x02, 0xfe, 0xff, 0xff, } }, { 2147483138, { 0x02, 0xfe, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0xfe, 0x01, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xfe, 0x01, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } }, },
            { { -2147450881, { 0x01, 0x80, 0xff, 0xff, } }, { 2147450881, { 0x01, 0x80, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147418114, { 0x02, 0x00, 0xff, 0xff, } }, { 2147418114, { 0x02, 0x00, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xfe, 0xff, 0x00, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x00, 0x80, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x80, 0x00, 0x80, 0x00, } }, },
            { { -2147450880, { 0x00, 0x80, 0xff, 0xff, } }, { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147418112, { 0x00, 0x00, 0xff, 0xff, } }, { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x01, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } }, { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } }, },
            { { -2147418113, { 0x01, 0x00, 0xff, 0xff, } }, { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147352578, { 0x02, 0x00, 0xfe, 0xff, } }, { 2147352578, { 0x02, 0x00, 0xfe, 0x7f, } }, },
            { { (-2147483647 - 1), { 0xfe, 0xff, 0x01, 0x80, 0x80, } }, { 2147483647, { 0xfe, 0xff, 0x01, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x01, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x01, 0x80, 0x00, } }, },
            { { -2147418112, { 0x00, 0x00, 0xff, 0xff, } }, { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { -2147352576, { 0x00, 0x00, 0xfe, 0xff, } }, { 2147352576, { 0x00, 0x00, 0xfe, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x02, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x02, 0x80, 0x00, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483646, { 0xfe, 0xff, 0xff, 0x7f, } }, { -2147483646, { 0xfe, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x00, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x01, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x01, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483645, { 0xfd, 0xff, 0xff, 0x7f, } }, { -2147483645, { 0xfd, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483526, { 0x86, 0xff, 0xff, 0x7f, } }, { -2147483526, { 0x86, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x78, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xf1, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xf1, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483405, { 0x0d, 0xff, 0xff, 0x7f, } }, { -2147483405, { 0x0d, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483519, { 0x7f, 0xff, 0xff, 0x7f, } }, { -2147483519, { 0x7f, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x7f, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xff, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483391, { 0xff, 0xfe, 0xff, 0x7f, } }, { -2147483391, { 0xff, 0xfe, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483518, { 0x7e, 0xff, 0xff, 0x7f, } }, { -2147483518, { 0x7e, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x80, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0x01, 0x01, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0x01, 0x01, 0x00, 0x80, 0x80, } }, },
            { { 2147483389, { 0xfd, 0xfe, 0xff, 0x7f, } }, { -2147483389, { 0xfd, 0xfe, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483392, { 0x00, 0xff, 0xff, 0x7f, } }, { -2147483392, { 0x00, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfe, 0x00, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfd, 0x01, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfd, 0x01, 0x00, 0x80, 0x80, } }, },
            { { 2147483137, { 0x01, 0xfe, 0xff, 0x7f, } }, { -2147483137, { 0x01, 0xfe, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147450880, { 0x00, 0x80, 0xff, 0x7f, } }, { -2147450880, { 0x00, 0x80, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfe, 0x7f, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfd, 0xff, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0x00, 0x80, 0x80, } }, },
            { { 2147418113, { 0x01, 0x00, 0xff, 0x7f, } }, { -2147418113, { 0x01, 0x00, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147450879, { 0xff, 0x7f, 0xff, 0x7f, } }, { -2147450879, { 0xff, 0x7f, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0x7f, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } }, },
            { { 2147418111, { 0xff, 0xff, 0xfe, 0x7f, } }, { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147418112, { 0x00, 0x00, 0xff, 0x7f, } }, { -2147418112, { 0x00, 0x00, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfe, 0xff, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xfd, 0xff, 0x01, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0x01, 0x80, 0x80, } }, },
            { { 2147352577, { 0x01, 0x00, 0xfe, 0x7f, } }, { -2147352577, { 0x01, 0x00, 0xfe, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147418111, { 0xff, 0xff, 0xfe, 0x7f, } }, { -2147418111, { 0xff, 0xff, 0xfe, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0x00, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x80, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0x01, 0x80, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x01, 0x80, 0x80, } }, },
            { { 2147352575, { 0xff, 0xff, 0xfd, 0x7f, } }, { -2147352575, { 0xff, 0xff, 0xfd, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0x7f, } }, { -2147483647, { 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x00, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x01, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x01, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xfd, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 2, { 0x02, } }, { -2, { 0x82, } }, },
            { { -2, { 0x82, } }, { 2, { 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x86, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x78, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x78, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xf1, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xf1, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0x0d, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x0d, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 242, { 0xf2, 0x00, } }, { -242, { 0xf2, 0x80, } }, },
            { { -242, { 0xf2, 0x80, } }, { 242, { 0xf2, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x7f, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x7f, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xff, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xfe, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xfe, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 256, { 0x00, 0x01, } }, { -256, { 0x00, 0x81, } }, },
            { { -256, { 0x00, 0x81, } }, { 256, { 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x7e, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x80, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x80, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0x01, 0x01, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0x01, 0x01, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xfd, 0xfe, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xfd, 0xfe, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 258, { 0x02, 0x01, } }, { -258, { 0x02, 0x81, } }, },
            { { -258, { 0x02, 0x81, } }, { 258, { 0x02, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfe, 0x00, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0x00, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfd, 0x01, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfd, 0x01, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0x01, 0xfe, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x01, 0xfe, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 510, { 0xfe, 0x01, } }, { -510, { 0xfe, 0x81, } }, },
            { { -510, { 0xfe, 0x81, } }, { 510, { 0xfe, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfe, 0x7f, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0x7f, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfd, 0xff, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65534, { 0xfe, 0xff, 0x00, } }, { -65534, { 0xfe, 0xff, 0x80, } }, },
            { { -65534, { 0xfe, 0xff, 0x80, } }, { 65534, { 0xfe, 0xff, 0x00, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0xff, 0x7f, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0x7f, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xff, 0x7f, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 65536, { 0x00, 0x00, 0x01, } }, { -65536, { 0x00, 0x00, 0x81, } }, },
            { { -65536, { 0x00, 0x00, 0x81, } }, { 65536, { 0x00, 0x00, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfe, 0xff, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xfd, 0xff, 0x01, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0x01, 0x00, 0x81, } }, },
            { { 2147483647, { 0x01, 0x00, 0xfe, 0xff, 0x00, } }, { (-2147483647 - 1), { 0x01, 0x00, 0xfe, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131070, { 0xfe, 0xff, 0x01, } }, { -131070, { 0xfe, 0xff, 0x81, } }, },
            { { -131070, { 0xfe, 0xff, 0x81, } }, { 131070, { 0xfe, 0xff, 0x01, } }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0x01, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0x00, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x00, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0x01, 0x00, 0x01, } }, { (-2147483647 - 1), { 0xff, 0xff, 0x01, 0x00, 0x81, } }, },
            { { 2147483647, { 0xff, 0xff, 0xfd, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfd, 0xff, 0x80, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0x00, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 131072, { 0x00, 0x00, 0x02, } }, { -131072, { 0x00, 0x00, 0x82, } }, },
            { { -131072, { 0x00, 0x00, 0x82, } }, { 131072, { 0x00, 0x00, 0x02, } }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x87, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x0e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xfe, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x02, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x02, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x02, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
    {
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { (-2147483647 - 1), { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x80, } }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x86, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x0d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x0d, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x7e, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xfd, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x03, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x03, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x01, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x01, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0x80, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x03, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0x7f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x03, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x03, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
        {
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 2147483647, { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { (-2147483647 - 1), { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, } }, { 2147483647, { 0x01, 0x00, 0xfe, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, },
            { { 2147483647, { 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xfd, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 2147483647, { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, } }, { (-2147483647 - 1), { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, } }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
            { { 0, {} }, { 0, {} }, },
        },
    },
};

const script_number_compare script_number_compares[13][9][12] =
{
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
    {
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
        {
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
            { 0, 1, 1, 0, 1, 0 },
            { 0, 1, 0, 1, 0, 1 },
            { 1, 0, 0, 0, 1, 1 },
        },
    },
};

#endif