/*
 * Test statistics on mail folders
 *
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <test-utils.h>
#include <buffy/Exception.h>
#include <buffy/MailFolder.h>
#include <buffy/MaildirMailFolder.h>
#include <buffy/MailboxMailFolder.h>
#include <sys/stat.h>
#include <sys/types.h>

namespace tut {
	struct folder_test {
		folder_test() {
			//fprintf(stderr, "Start\n");
			//unlink("test.tdb");
		}
		~folder_test() {
			//fprintf(stderr, "End\n");
			//unlink("test.tdb");
		}
	};
	typedef test_group<folder_test> tg;
	typedef tg::object to;
	tg folder_tg ("Folder");

	template<> template<>
	void to::test<1> ()
	{
		try {
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders("mbox");

			gen_ensure(folders.size() == 2);

			MailFolder empty = folders[0].name() == "empty.mbox" ? folders[0] : folders[1];
			MailFolder test = folders[0].name() == "empty.mbox" ? folders[1] : folders[0];

			gen_ensure(empty.path() == "mbox/empty.mbox");
			gen_ensure(empty.name() == "empty.mbox");
			gen_ensure(empty.type() == "Mailbox");
			gen_ensure(test.path() == "mbox/test.mbox");
			gen_ensure(test.name() == "test.mbox");
			gen_ensure(test.type() == "Mailbox");

			empty.updateStatistics();

			gen_ensure(empty.getMsgTotal() == 0);
			gen_ensure(empty.getMsgUnread() == 0);
			gen_ensure(empty.getMsgNew() == 0);
			gen_ensure(empty.getMsgFlagged() == 0);
			gen_ensure(empty.changed() == false);

			test.updateStatistics();

			gen_ensure(test.getMsgTotal() == 3);
			gen_ensure(test.getMsgUnread() == 0);
			gen_ensure(test.getMsgNew() == 0);
			gen_ensure(test.getMsgFlagged() == 1);
			gen_ensure(test.changed() == false);

		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
			throw;
		}
	}

	template<> template<>
	void to::test<2> ()
	{
		try {
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders("mboxgz");

			gen_ensure(folders.size() == 2);

			MailFolder empty = folders[0].name() == "empty.mbox.gz" ? folders[0] : folders[1];
			MailFolder test = folders[0].name() == "empty.mbox.gz" ? folders[1] : folders[0];

			gen_ensure(empty.path() == "mboxgz/empty.mbox.gz");
			gen_ensure(empty.name() == "empty.mbox.gz");
			gen_ensure(empty.type() == "Mailbox");
			gen_ensure(test.path() == "mboxgz/test.mbox.gz");
			gen_ensure(test.name() == "test.mbox.gz");
			gen_ensure(test.type() == "Mailbox");

			empty.updateStatistics();

			gen_ensure(empty.getMsgTotal() == 0);
			gen_ensure(empty.getMsgUnread() == 0);
			gen_ensure(empty.getMsgNew() == 0);
			gen_ensure(empty.getMsgFlagged() == 0);
			gen_ensure(empty.changed() == false);

			test.updateStatistics();

			gen_ensure(test.getMsgTotal() == 3);
			gen_ensure(test.getMsgUnread() == 0);
			gen_ensure(test.getMsgNew() == 0);
			gen_ensure(test.getMsgFlagged() == 1);
			gen_ensure(test.changed() == false);

		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
			throw;
		}
	}

	template<> template<>
	void to::test<3> ()
	{
		try {
			// An empty database should return empty sets, but not fail
			std::vector<MailFolder> folders = MailFolder::enumerateFolders("maildir");

			gen_ensure(folders.size() == 2);

			MailFolder empty = folders[0].name() == "empty" ? folders[0] : folders[1];
			MailFolder test = folders[0].name() == "empty" ? folders[1] : folders[0];

			gen_ensure(empty.path() == "maildir/empty");
			gen_ensure(empty.name() == "empty");
			gen_ensure(empty.type() == "Maildir");
			gen_ensure(test.path() == "maildir/test");
			gen_ensure(test.name() == "test");
			gen_ensure(test.type() == "Maildir");

			empty.updateStatistics();

			gen_ensure(empty.getMsgTotal() == 0);
			gen_ensure(empty.getMsgUnread() == 0);
			gen_ensure(empty.getMsgNew() == 0);
			gen_ensure(empty.getMsgFlagged() == 0);
			gen_ensure(empty.changed() == false);

			test.updateStatistics();

			gen_ensure(test.getMsgTotal() == 3);
			gen_ensure(test.getMsgUnread() == 0);
			gen_ensure(test.getMsgNew() == 0);
			gen_ensure(test.getMsgFlagged() == 1);
			gen_ensure(test.changed() == false);

		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
			throw;
		}
	}

	template<> template<>
	void to::test<4> ()
	{
		// Test that it is ok to remove a mail folder when its MailFolder
		// exists
		try {
			// Create an empty mailbox
			FILE* out = fopen("empty.mbox", "wt");
			if (!out)
				throw SystemException(errno, "Creating file empty.mbox");
			fclose(out);

			MailFolder mf(new MailboxMailFolder("empty.mbox"));

			mf.updateStatistics();

			gen_ensure(mf.getMsgTotal() == 0);
			gen_ensure(mf.getMsgUnread() == 0);
			gen_ensure(mf.getMsgNew() == 0);
			gen_ensure(mf.getMsgFlagged() == 0);

			if (unlink("empty.mbox") == -1)
				throw SystemException(errno, "Deleting file empty.mbox");

			gen_ensure(mf.changed() == true);

			mf.updateStatistics();

			gen_ensure(mf.getMsgTotal() == 0);
			gen_ensure(mf.getMsgUnread() == 0);
			gen_ensure(mf.getMsgNew() == 0);
			gen_ensure(mf.getMsgFlagged() == 0);
		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
			throw;
		}
	}

	template<> template<>
	void to::test<5> ()
	{
		// Test that it is ok to remove a mail folder when its MailFolder
		// exists
		try {
			// Create an empty maildir
			if (mkdir("empty", 0755) == -1)
				throw SystemException(errno, "Creating directory empty");
			if (mkdir("empty/cur", 0755) == -1)
				throw SystemException(errno, "Creating directory empty/cur");
			if (mkdir("empty/new", 0755) == -1)
				throw SystemException(errno, "Creating directory empty/new");

			MailFolder mf(new MaildirMailFolder("empty"));

			mf.updateStatistics();

			gen_ensure(mf.getMsgTotal() == 0);
			gen_ensure(mf.getMsgUnread() == 0);
			gen_ensure(mf.getMsgNew() == 0);
			gen_ensure(mf.getMsgFlagged() == 0);

			if (rmdir("empty/cur") == -1)
				throw SystemException(errno, "Deleting directory empty/cur");
			if (rmdir("empty/new") == -1)
				throw SystemException(errno, "Deleting directory empty/new");
			if (rmdir("empty") == -1)
				throw SystemException(errno, "Deleting directory empty");

			gen_ensure(mf.changed() == true);

			mf.updateStatistics();

			gen_ensure(mf.getMsgTotal() == 0);
			gen_ensure(mf.getMsgUnread() == 0);
			gen_ensure(mf.getMsgNew() == 0);
			gen_ensure(mf.getMsgFlagged() == 0);
		} catch (Exception& e) {
			fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
			throw;
		}
	}
}

// vim:set ts=4 sw=4:
