#include "config.h"

#include "Config.h"
//#include "Environment.h"

#include <sys/types.h>  // getpwuid, stat, mkdir, getuid
#include <sys/stat.h>   // stat, mkdir
#include <pwd.h>        // getpwuid
#include <unistd.h>     // stat, getuid

#include <errno.h>

using namespace std;
using namespace buffy;

static string escape(const std::string& str)
{
	string res;
	for (string::const_iterator s = str.begin();
			s != str.end(); s++)
		switch (*s)
		{
			case '/': res += "%47;"; break;
			case '[': res += "%91;"; break;
			case ']': res += "%93;"; break;
			case '%': res += "%%"; break;
			default: res += *s; break;
		}
	return res;
}

static string unescape(const std::string& str)
{
	string res;
	for (string::const_iterator s = str.begin();
			s != str.end(); ++s)
		if (*s == '%')
		{
			++s;
			if (s == str.end() || *s == '%')
				res += '%';
			else
			{
				string num;
				for ( ; s != str.end() && *s != ';'; s++)
					num += *s;
				res += (char)atoi(num.c_str());
			}
		} else
			res += *s;
	return res;
}

ConfigItem::ConfigItem(Config& config, const std::string& path)
	: m_config(config), m_path(path)
{
}

xmlpp::Element* ConfigItem::node()
{
	return m_config.obtainElement(m_path);
}

xmlpp::Element* ConfigItem::nodeIfExists()
{
	return m_config.getElement(m_path);
}

bool ConfigItemBool::get()
{
	xmlpp::Element* n = nodeIfExists();
	if (n == 0)
		return m_def;
	xmlpp::TextNode* tn = n->get_child_text();
	if (!tn)
	{
		n->set_child_text(m_def ? "true" : "false");
		return m_def;
	} else {
		string val = tn->get_content();
		if (val == "true")
			return true;
		return false;
	}
}

void ConfigItemBool::set(bool val)
{
	node()->set_child_text(val ? "true" : "false");
}

int ConfigItemInt::get()
{
	xmlpp::Element* n = nodeIfExists();
	if (n == 0)
		return m_def;
	xmlpp::TextNode* tn = n->get_child_text();
	if (!tn)
	{
		n->set_child_text(stringf::fmt(m_def));
		return m_def;
	} else {
		string val = tn->get_content();
		return atoi(val.c_str());
	}
}

void ConfigItemInt::set(int val)
{
	node()->set_child_text(stringf::fmt(val));
}

std::string ConfigItemString::get()
{
	xmlpp::Element* n = nodeIfExists();
	if (n == 0)
		return m_def;
	xmlpp::TextNode* tn = n->get_child_text();
	if (!tn)
	{
		n->set_child_text(m_def);
		return m_def;
	} else {
		return tn->get_content();
	}
}

void ConfigItemString::set(const std::string& val)
{
	node()->set_child_text(val);
}

std::vector<std::string> ConfigItemLocations::computeDefault()
{
	struct passwd* udata = getpwuid(getuid());
	vector<string> deflt;
	deflt.push_back(string("/var/mail/") + udata->pw_name);
	deflt.push_back(string(udata->pw_dir) + "/Maildir");
	deflt.push_back(string(udata->pw_dir) + "/Mail");
	deflt.push_back(string(udata->pw_dir) + "/mail");
	return deflt;
}

std::vector<std::string> ConfigItemLocations::get()
{
	xmlpp::Element* n = nodeIfExists();
	if (n == 0)
		return computeDefault();
	xmlpp::Node::NodeList nl = n->get_children("location");
	if (nl.empty())
	{
		vector<string> def = computeDefault();
		for (vector<string>::const_iterator i = def.begin();
				i != def.end(); i++)
			n->add_child("location")->set_child_text(*i);
		return def;
	} else {
		vector<string> res;
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
				i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
				res.push_back(e->get_child_text()->get_content());
		return res;
	}
}

void ConfigItemLocations::set(const std::vector<std::string>& val)
{
	xmlpp::Element* n = node();
	xmlpp::Node::NodeList nl = n->get_children("location");

	// First remove all location nodes
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		n->remove_child(*i);

	// Then add the new ones
	for (vector<string>::const_iterator i = val.begin();
			i != val.end(); i++)
		n->add_child("location")->set_child_text(*i);
}

std::vector<MailProgram> ConfigItemMailPrograms::computeDefault()
{
	std::vector<MailProgram> deflt;
	deflt.push_back(MailProgram("Mutt", "/usr/bin/x-terminal-emulator -e /usr/bin/mutt -f %p", true));
	deflt.push_back(MailProgram("Other", "/usr/bin/sample-mail-editor --folder %p"));
	return deflt;
}

static xmlpp::Element* get_first_el(xmlpp::Element* parent, const std::string& name) throw ()
{
	xmlpp::Node::NodeList nl = parent->get_children(name);
	if (nl.empty())
		return 0;
	else
		return dynamic_cast<xmlpp::Element*>(*nl.begin());
}

std::vector<MailProgram> ConfigItemMailPrograms::get()
{
	xmlpp::Element* n = nodeIfExists();
	if (n == 0)
		return computeDefault();
	xmlpp::Node::NodeList nl = n->get_children("mail");
	if (nl.empty())
	{
		vector<MailProgram> def = computeDefault();
		for (vector<MailProgram>::const_iterator i = def.begin();
				i != def.end(); i++)
		{
			xmlpp::Element* item = n->add_child("mail");
			if (i->selected())
				item->set_attribute("selected", "true");
			xmlpp::Element* val = item->add_child("name");
			val->set_child_text(i->name());
			val = item->add_child("command");
			val->set_child_text(i->command());
		}
		return def;
	} else {
		vector<MailProgram> res;
		for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
				i != nl.end(); i++)
			if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			{
				bool sel = false;
				if (xmlpp::Attribute* at = e->get_attribute("selected"))
					if (at->get_value() == "true")
						sel = true;

				string name;
				if (xmlpp::Element* el = get_first_el(e, "name"))
					name = el->get_child_text()->get_content();

				string command;
				if (xmlpp::Element* el = get_first_el(e, "command"))
					command = el->get_child_text()->get_content();
				
				res.push_back(MailProgram(name, command, sel));
			}
		return res;
	}
}

void ConfigItemMailPrograms::set(const std::vector<MailProgram>& val)
{
	xmlpp::Element* n = node();
	xmlpp::Node::NodeList nl = n->get_children("mail");

	// First remove all mail nodes
	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		n->remove_child(*i);

	// Then add the new ones
	for (vector<MailProgram>::const_iterator i = val.begin();
			i != val.end(); i++)
	{
		xmlpp::Element* item = n->add_child("mail");
		if (i->selected())
			item->set_attribute("selected", "true");
		xmlpp::Element* val = item->add_child("name");
		val->set_child_text(i->name());
		val = item->add_child("command");
		val->set_child_text(i->command());
	}
}

MailProgram ConfigItemMailPrograms::selected()
{
	vector<MailProgram> progs = get();
	for (vector<MailProgram>::iterator i = progs.begin();
			i != progs.end(); i++)
		if (i->selected())
			return *i;
	return MailProgram();
}

ConfigItemString ConfigItemDictionary::get(const std::string& key, const std::string& def)
{
	return ConfigItemString(m_config, m_path + "/" + key, def);
}


void Config::loadConfiguration(const std::string& rcfile)
	throw (SystemException, ConsistencyCheckException)
{
	struct stat rcfile_stat;
	if (stat(rcfile.c_str(), &rcfile_stat) == -1)
	{
//		feedback("Creating new configuration file %.*s.\n", PFSTR(rcfile));
		
		// Create the configuration from scratch
		doc_conf = new xmlpp::Document();
		doc_conf->create_root_node("buffy");
		//_root = doc_conf->create_root_node("buffy");
	} else {
		if (S_ISDIR(rcfile_stat.st_mode))
			throw ConsistencyCheckException(rcfile + " already exists and is a directory");
		
		if (access(rcfile.c_str(), R_OK) == -1)
			throw ConsistencyCheckException(rcfile + " already exists and is not readable");

//		feedback("Reading configuration from %.*s.\n", PFSTR(rcfile));
	
		// Parse the existing config file
		try {
			// Enable when we'll have a DTD
			//_xmlparser.set_validate();
			_xmlparser.set_substitute_entities(); // Resolve/unescape text automatically
			_xmlparser.parse_file(rcfile);
			if (_xmlparser)
				doc_conf = _xmlparser.get_document();
			else
				throw ConsistencyCheckException("Parser did not parse " + rcfile);
		} catch (const std::exception& ex) {
			throw ConsistencyCheckException(string(ex.what()) + " when parsing configuration file " + rcfile);
		}
	}

	m_el_root = doc_conf->get_root_node();
}

Config::Config() throw (SystemException, ConsistencyCheckException) 
	: doc_conf(0), m_el_root(0), m_el_general(0),
	  m_el_view(0), m_el_applications(0), m_el_folders(0)
{
	struct passwd* udata = getpwuid(getuid());
	rcfile = udata->pw_dir;
	rcfile += "/.buffy";

	loadConfiguration(rcfile);
}

Config::Config(const std::string& fname) throw (SystemException, ConsistencyCheckException) 
	: doc_conf(0), m_el_root(0), m_el_general(0),
	  m_el_view(0), m_el_applications(0), m_el_folders(0)
{
	loadConfiguration(fname);
}

Config::~Config() throw ()
{
	delete m_el_root;
	delete m_el_general;
	delete m_el_view;
	delete m_el_applications;
	delete m_el_folders;
}

xmlpp::Element* Config::getElement(const std::string& path)
{
	return getElement(m_el_root, path);
}

xmlpp::Element* Config::getElement(xmlpp::Element* father, const std::string& path)
{
	size_t pos = path.find('/');
	if (pos == string::npos)
	{
		if (path[path.size() - 1] == ']')
		{
			size_t npos = path.find('[');
			if (npos == string::npos)
				return 0;
			string attr = unescape(path.substr(npos+1, path.size() - npos - 2));
			string npath = path.substr(0, npos);

			xmlpp::Node::NodeList nl = father->get_children(npath);
			for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
					i != nl.end(); i++)
				if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
					if (xmlpp::Attribute* at = e->get_attribute("name"))
						if (at->get_value() == attr)
							return e;
			return 0;
		} else {
			xmlpp::Node::NodeList nl = father->get_children(path);
			if (nl.empty())
				return 0;
			return dynamic_cast<xmlpp::Element*>(*nl.begin());
		}
	}
	else
	{
		xmlpp::Element* n = getElement(father, path.substr(0, pos));
		return n != 0 ? getElement(n, path.substr(pos+1)) : 0;
	}
}

xmlpp::Element* Config::obtainElement(const std::string& path)
{
	return obtainElement(m_el_root, path);
}

xmlpp::Element* Config::obtainElement(xmlpp::Element* father, const std::string& path)
{
	size_t pos = path.find('/');
	if (pos == string::npos)
	{
		if (path[path.size() - 1] == ']')
		{
			size_t npos = path.find('[');
			if (npos == string::npos)
				return 0;
			string attr = unescape(path.substr(npos+1, path.size() - npos - 2));
			string npath = path.substr(0, npos);

			xmlpp::Node::NodeList nl = father->get_children(npath);
			for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
					i != nl.end(); i++)
				if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
					if (xmlpp::Attribute* at = e->get_attribute("name"))
						if (at->get_value() == attr)
							return e;

			xmlpp::Element* app = father->add_child(npath);
			app->set_attribute("name", attr);
			return app;
		} else {
			xmlpp::Node::NodeList nl = father->get_children(path);
			if (nl.empty())
				return father->add_child(path);
			return dynamic_cast<xmlpp::Element*>(*nl.begin());
		}
	}
	else
	{
		xmlpp::Element* n = obtainElement(father, path.substr(0, pos));
		return n != 0 ? obtainElement(n, path.substr(pos+1)) : 0;
	}
}

#if 0
ConfigItem& Config::el_general() throw ()
{
	if (!m_el_general)
		m_el_general = new ConfigItem(*m_el_root, "general");
	return *m_el_general;
}

ConfigItem& Config::el_view() throw ()
{
	if (!m_el_view)
		m_el_view = new ConfigItem(el_general(), "view");
	return *m_el_view;
}

ConfigItem& Config::el_applications() throw ()
{
	if (!m_el_applications)
		m_el_applications = new ConfigItem(*m_el_root, "applications");
	return *m_el_applications;
}

ConfigItem& Config::el_folders() throw ()
{
	if (!m_el_folders)
		m_el_folders = new ConfigItem(*m_el_root, "folders");
	return *m_el_folders;
}
#endif

ConfigItemBool Config::view_empty() throw ()
{
	return ConfigItemBool(*this, "general/view/empty", true);
}

ConfigItemBool Config::view_read() throw ()
{
	return ConfigItemBool(*this, "general/view/read", true);
}

ConfigItemBool Config::view_important() throw ()
{
	return ConfigItemBool(*this, "general/view/important", true);
}

ConfigItemInt Config::update_interval() throw ()
{
	return ConfigItemInt(*this, "general/interval", 30);
}

ConfigItemLocations Config::folder_locations() throw ()
{
	return ConfigItemLocations(*this, "general/locations");
}

ConfigItemMailPrograms Config::mail_programs() throw ()
{
	return ConfigItemMailPrograms(*this, "general/programs");
}

xmlpp::Element* Config::obtainNamedElement(xmlpp::Element* father, const std::string& nodeName, const std::string& name) throw ()
{
	xmlpp::Node::NodeList nl = father->get_children(nodeName);

	for (xmlpp::Node::NodeList::const_iterator i = nl.begin();
			i != nl.end(); i++)
		if (xmlpp::Element* e = dynamic_cast<xmlpp::Element*>(*i))
			if (xmlpp::Attribute* at = e->get_attribute("name"))
				if (at->get_value() == name)
					return e;

	xmlpp::Element* app = father->add_child(nodeName);
	app->set_attribute("name", name);
	return app;
}


ConfigItemDictionary Config::application(const std::string& name) throw ()
{
	return ConfigItemDictionary(*this, "applications/app[" + escape(name) + "]");
}

ConfigItemDictionary Config::folder(const std::string& folder) throw ()
{
	return ConfigItemDictionary(*this, "folders/folder[" + escape(folder) + "]");
}


void Config::save() throw (ConsistencyCheckException)
{
	save(rcfile);
}

void Config::save(const std::string& file) throw (ConsistencyCheckException)
{
	try {
		if (doc_conf)
		{
//			feedback("Saving configuration to %.*s.\n", PFSTR(rcfile));
			doc_conf->write_to_file_formatted(file);
		}// else
//			feedback("No configuration present: ignoring save request.\n");
	} catch (const std::exception& ex) {
		throw ConsistencyCheckException(string(ex.what()) + " when saving configuration file " + rcfile);
	}
}

//#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_buffy;

struct buffy_config_shar {
};
TESTGRP(buffy_config);

template<> template<>
void to::test<1>()
{
	ensure_equals(unescape(escape("foo")), "foo");
	ensure_equals(unescape(escape("f%oo")), "f%oo");
	ensure_equals(unescape(escape("f/oo")), "f/oo");
	ensure_equals(unescape(escape("f[o]o")), "f[o]o");
	ensure_equals(unescape(escape("f[o]o%")), "f[o]o%");
	ensure_equals(unescape(escape("f[o]o/")), "f[o]o/");
}

template<> template<>
void to::test<2>()
{
	Config conf("test.conf");
	ensure(conf.view_empty().get() == true);
	conf.view_empty().set(false);
	ensure(conf.view_empty().get() == false);

	ensure_equals(conf.application("foo").get("bar", "baz").get(), "baz");
	conf.application("foo").get("bar", "baz").set("cippo");
	ensure_equals(conf.application("foo").get("bar", "baz").get(), "cippo");

	ensure_equals(conf.application("foo/bar").get("bar", "baz").get(), "baz");
	conf.application("foo/bar").get("bar", "baz").set("cippo1");
	ensure_equals(conf.application("foo/bar").get("bar", "baz").get(), "cippo1");

	ensure_equals(conf.application("foo]").get("bar", "baz").get(), "baz");
	conf.application("foo]").get("bar", "baz").set("cippo2");
	ensure_equals(conf.application("foo]").get("bar", "baz").get(), "cippo2");
}

}

//#endif


// vim:set ts=4 sw=4:
