#ifndef BUFFY_CONFIG_H
#define BUFFY_CONFIG_H

#include <buffy/Exception.h>
#include <buffy/MailProgram.h>
#include <libxml++/libxml++.h>

namespace buffy {

class Config;

/**
 * Encapsulates a generic item in the configuration.  It is the father of a
 * hierarchy, and it cannot be used to access the node value.
 */
class ConfigItem
{
protected:
	Config& m_config;
	std::string m_path;

	ConfigItem(Config& config, const std::string& path);

	/* Return the node pointed by this ConfigItem, creating it if it does not exist */
	xmlpp::Element* node();

	/* Return the node pointed by this ConfigItem, or 0 if it does not exist */
	xmlpp::Element* nodeIfExists();

public:
	virtual ~ConfigItem() {};

	/**
	 * Check if this node contains a child with the given name
	bool has(const std::string& nodeName);
	 */

	friend class Config;
};

class ConfigItemBool : public ConfigItem
{
private:
	bool m_def;

protected:
	ConfigItemBool(Config& config, const std::string& path, bool def)
		: ConfigItem(config, path), m_def(def) {}

public:
	bool get();
	void set(bool val);

	friend class Config;
};

class ConfigItemInt : public ConfigItem
{
private:
	int m_def;

protected:
	ConfigItemInt(Config& config, const std::string& path, int def)
		: ConfigItem(config, path), m_def(def) {}

public:
	int get();
	void set(int val);

	friend class Config;
};

class ConfigItemString : public ConfigItem
{
private:
	std::string m_def;
		
protected:
	ConfigItemString(Config& config, const std::string& path, const std::string& def)
		: ConfigItem(config, path), m_def(def) {}

public:
	std::string get();
	void set(const std::string& val);

	friend class Config;
	friend class ConfigItemDictionary;
};

class ConfigItemLocations : public ConfigItem
{
protected:
	ConfigItemLocations(Config& config, const std::string& path)
		: ConfigItem(config, path) {}

	std::vector<std::string> computeDefault();

public:
	std::vector<std::string> get();
	void set(const std::vector<std::string>& val);

	friend class Config;
};

class ConfigItemMailPrograms : public ConfigItem
{
protected:
	ConfigItemMailPrograms(Config& config, const std::string& path)
		: ConfigItem(config, path) {}

	std::vector<MailProgram> computeDefault();

public:
	std::vector<MailProgram> get();
	void set(const std::vector<MailProgram>& val);

	MailProgram selected();

	friend class Config;
};

class ConfigItemDictionary : public ConfigItem
{
protected:
	ConfigItemDictionary(Config& config, const std::string& path)
		: ConfigItem(config, path) {}

public:
	ConfigItemString get(const std::string& key, const std::string& def);

	friend class Config;
};


class Config
{
protected:
	xmlpp::DomParser _xmlparser;

	// Cached elements
	xmlpp::Document* doc_conf;

	xmlpp::Element* m_el_root;
	ConfigItem* m_el_general;
	ConfigItem* m_el_view;
	ConfigItem* m_el_applications;
	ConfigItem* m_el_folders;

	// State directory
	std::string rcfile;

#if 0
	ConfigItem& el_general() throw ();
	ConfigItem& el_view() throw ();
	ConfigItem& el_applications() throw ();
	ConfigItem& el_folders() throw ();
#endif

	void loadConfiguration(const std::string& fname) throw (SystemException, ConsistencyCheckException);

	xmlpp::Element* getElement(const std::string& path);
	xmlpp::Element* getElement(xmlpp::Element* father, const std::string& path);
	xmlpp::Element* obtainElement(const std::string& path);
	xmlpp::Element* obtainElement(xmlpp::Element* father, const std::string& path);

	xmlpp::Element* obtainNamedElement(xmlpp::Element* father, const std::string& nodeName, const std::string& name) throw ();

public:
	Config() throw (SystemException, ConsistencyCheckException);
	Config(const std::string& fname) throw (SystemException, ConsistencyCheckException);
	~Config() throw ();

	ConfigItemBool view_empty() throw ();
	ConfigItemBool view_read() throw ();
	ConfigItemBool view_important() throw ();
	ConfigItemInt update_interval() throw ();
	ConfigItemLocations folder_locations() throw ();
	ConfigItemMailPrograms mail_programs() throw ();
	ConfigItemDictionary application(const std::string& name) throw ();
	ConfigItemDictionary folder(const MailFolder& folder) throw ()
	{
		return this->folder(folder.path());
	}
	ConfigItemDictionary folder(const std::string& folder) throw ();

	/*
	bool get_view_read(const MailFolder& mf) throw ();
	bool get_view_empty(const MailFolder& mf) throw ();
	bool get_view_important(const MailFolder& mf) throw ();

	void set_view_read(const MailFolder& mf, bool val) throw ();
	void set_view_empty(const MailFolder& mf, bool val) throw ();
	void set_view_important(const MailFolder& mf, bool val) throw ();
	*/

	void save() throw (ConsistencyCheckException);
	void save(const std::string& file) throw (ConsistencyCheckException);

	friend class ConfigItem;
};

}

// vim:set ts=4 sw=4:
#endif
