/***************************************************************************
  $RCSfile: ctcard.cpp,v $
                             -------------------
    cvs         : $Id: ctcard.cpp,v 1.35 2003/05/13 20:24:13 aquamaniac Exp $
    begin       : Fri Dec 06 2002
    copyright   : (C) 2002 by Martin Preuss
    email       : martin@libchipcard.de


 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_CHIPCARD_DLL
#  define CHIPCARD_API __declspec (dllexport)
# else /* Not BUILDING_CHIPCARD_DLL */
#  define CHIPCARD_API __declspec (dllimport)
# endif /* Not BUILDING_CHIPCARD_DLL */
#else
# define CHIPCARD_API
#endif


#include "ctcard.h"
#include "ctmisc.h"
#include "libchipcard.h"
#include "ctinitializer.h"
#ifdef HAVE_TIME_H
#include <time.h>
#endif
#include <stdio.h>

#include <engine/chameleon/error.h>
#include <engine/chameleon/debug.h>

#define CTCARD_BOUNDARY 249
#define CTCARD_READSIZE 255






int CTCard::_calculateMemorySize(const string &atr) {
  int i1, i2;
  int j1, j2;
  int memsize;

  // set memory size
  if (atr.length()>=2) {
    i1=(atr[1]>>3) & 0x07; // count of elements
    i2=atr[1] & 0x07;      // size of element

    // check element size
    if (i2==0)
      j2=0;
    else if (i2==1)
      j2=1;
    else
      j2=1<<i2;

    // check element number
    if (i1==0)
      j1=0;
    else
      j1=64<<i1;

    // calculate memory size
    if (j1 && j2)
      memsize=j1*j2/8;
    else
      memsize=0;
  }
  else
    memsize=0;

  return memsize;
}


CTCard::CallBackResult CTCard::callback(bool first){
  return CallBackContinue;
}



CTCard::CTCard(const CTCard &c):CTCardBase(c),_cardTypes("CTCard")
{
}


CTCard::CTCard(const CTReaderContext &rc):CTCardBase(rc),_cardTypes("CTCard")
{
}


CTCard::~CTCard(){
}


CTError CTCard::openCard(){
  CTError err;

  err=open();
  if (!err.isOk())
    return CTError("CTCard::openCard", err);

  _memsize=_calculateMemorySize(atr());
  return CTError();
}


CTError CTCard::closeCard(bool force){
  CTError err;

  err=close();
  if (!err.isOk())
    return CTError("CTCard::closeCard", err);

  return CTError();
}


bool CTCard::isProcessorCard() const {
  return (readerStatus()&CHIPCARD_STATUS_PROCESSOR);
}


CTError CTCard::_locateCommandReader(const string &command,
				     const string &reader,
				     string &result) {
  CTError err;
  int pos;
  int tpos;
  string types;
  string ttype;
  string cmd;
  int rv;

  types=cardTypes();
  pos=types.length()-1;

  while(pos>0) {
    tpos=pos;
    ttype.erase();
    while(pos>=0) {
      if (types[pos]==',') {
	if (pos+1<(int)(types.length()))
	  ttype=types.substr(pos+1, tpos-pos);
	if (pos>=0)
	  pos--;
	break;
      }
      pos--;
    }
    if (pos<0)
      ttype=types.substr(0,tpos+1);

    CTMisc::removeBlanks(ttype);

    if (ttype.empty())
      return CTError("CTCard::_locateCommandreader()",
		     k_CTERROR_INVALID,0,0,
		     "Command not found");

    cmd=reader+"/"+ttype+"/"+command;
    rv=ChipCard_ExistsCommand(cmd.c_str());
    if (rv==CHIPCARD_SUCCESS) {
      result=cmd;
      return CTError();
    }
  } // while

  return CTError("CTCard::_locateCommandReader()",
		 k_CTERROR_INVALID,0,0,
		 "Command not found");
}


CTError CTCard::_locateCommand(const string &command,
			       string &result) {
  CTError err;

  // try current reader type first
  err=_locateCommandReader(command,
			   readerDescription().type,
			   result);
  if (err.isOk())
    return CTError();

  // then try "ALL"
  err=_locateCommandReader(command, "all", result);
  if (!err.isOk())
    return CTError("CTCard::_locateCommand",err);
  return CTError();
}



CTError CTCard::execCommand(const string &command,
			    string &cmdCache,
			    string &response,
			    const string &param1,
			    const string &param2,
			    const string &param3,
			    const string &param4,
			    const string &param5){
  CTError err;
  string apdu;

  err=makeAPDU(command, cmdCache, apdu,
	       param1, param2, param3, param4, param5);
  if (!err.isOk()) {
    DBG_ERROR("do command failed: %s", err.errorString().c_str());
    return CTError("CTCard::execCommand", err);
  }

  err=sendAPDU(apdu, response);
  if (!err.isOk()) {
    DBG_NOTICE("Error sending command: %s", err.errorString().c_str());
  }

  return CTError("CTCard::execCommand", err);
}


CTError CTCard::execCommand(CTCommand &cmd) {
  CTError err;
  string response;

  err=sendAPDU(cmd.toString(), response);
  if (err.code()==0) {
    cmd.setSw1(err.subcode1());
    cmd.setSw2(err.subcode2());
  }
  else {
    cmd.setSw1(0);
    cmd.setSw2(0);
  }

  if (!err.isOk())
    return CTError("CTCard::execCommand", err);

  cmd.setData(response);
  return CTError("CTCard::execCommand", err);
}


CTError CTCard::makeAPDU(const string &command,
			 string &cmdCache,
			 string &apdu,
			 const string &param1,
			 const string &param2,
			 const string &param3,
			 const string &param4,
			 const string &param5){
  char buffer[300];
  int bufferlen;
  int params;
  CTError error;
  int err;

  params=5;
  if (param5.empty())
    params--;
  if (param4.empty())
    params--;
  if (param3.empty())
    params--;
  if (param2.empty())
    params--;
  if (param1.empty())
    params--;

  if (cmdCache.empty()) {
    error=_locateCommand(command, cmdCache);
    if (!error.isOk()) {
      DBG_ERROR("Command \"%s\" not found", command.c_str());
      return CTError("CTCard::makeAPDU",error);
    }
  }

  bufferlen=sizeof(buffer);
  switch(params) {
  case 0:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  0);
    break;

  case 1:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  1,
			  param1.c_str());
    break;

  case 2:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  2,
			  param1.c_str(),
			  param2.c_str());
    break;

  case 3:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  3,
			  param1.c_str(),
			  param2.c_str(),
			  param3.c_str());
    break;

  case 4:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  4,
			  param1.c_str(),
			  param2.c_str(),
			  param3.c_str(),
			  param4.c_str());
    break;

  case 5:
  default:
    err=ChipCard_MakeAPDU(buffer, &bufferlen,
			  cmdCache.c_str(),
			  5,
			  param1.c_str(),
			  param2.c_str(),
			  param3.c_str(),
			  param4.c_str(),
			  param5.c_str());
    break;
  } // switch

  if (err!=CHIPCARD_SUCCESS)
    return CTError("CTCard::makeAPDU()",
		   k_CTERROR_API, err,0,
		   "Error building command",
		   command);
  apdu.assign(buffer, bufferlen);
  return CTError();
}


CTError CTCard::selectFile(string &fcp, unsigned short fid){
  return execCommand("select_file_w",
		     _cmdSelectFile,
		     fcp,
		     CTMisc::num2string(fid));
}


CTError CTCard::selectFile(string &fcp, const string &fid){
  return execCommand("select_file_id",
		     _cmdSelectFileId,
		     fcp,
		     CTMisc::bin2hex(fid));
}


CTError CTCard::readBinaryRaw(string &data,
			      unsigned short offset,
			      unsigned char size){
  return execCommand("read_binary",
		     _cmdReadBinary,
		     data,
		     CTMisc::num2string(offset),
		     CTMisc::num2string(size));
}


CTError CTCard::updateBinaryRaw(const string &data,
				unsigned short offset){
  string response;
  CTError err;

  err=execCommand("update_binary",
		  _cmdUpdateBinary,
		  response,
		  CTMisc::num2string(offset),
		  CTMisc::bin2hex(data));
  return err;
}


CTError CTCard::updateBinary(const string &data,
			     unsigned int pos){
  CTError err;
  unsigned int p;
  string tmp;
  int i,j;

  /* Well, the Towitoko driver seems to have problems writing
   * across 32 bytes boundaries. So we have to check for those
   * boundaries ourselves. At least that driver is able to communicate
   * with I2C memory cards, unlike KOBIL drivers ;-)
   */
  p=0;
  while(p<data.length()) {
    // calculate next boundary at 32 bytes
    //j=((pos+p)|CTCARD_BOUNDARY)+1;
    j=(((pos+p)/CTCARD_BOUNDARY)+1)*CTCARD_BOUNDARY;
    // calculate number of bytes till next 32 byte boundary
    i=j-(pos+p);
    // check number of bytes against end of data
    if (p+i>=data.length())
      i=data.length()-p;
    // now cut out THAT number of bytes
    tmp=data.substr(p,i);
    err=updateBinaryRaw(tmp,pos+p);
    if (!err.isOk())
      return err;
    p+=i;
  }
  return CTError();
}


CTError CTCard::readBinary(string &d,
			   unsigned int pos,
			   unsigned int s){
  CTError err;
  string tmp;
  unsigned int i;

  d.erase();
  while (d.length()<s) {
    tmp.erase();
    i=s-d.length();
    if (i>CTCARD_READSIZE)
      i=CTCARD_READSIZE;
    err=readBinaryRaw(tmp,pos,i);
    d+=tmp;
    pos+=tmp.length();
    if (!err.isOk()) {
      if (!err.isOk(0x6b))
	return err;
      DBG_VERBOUS("LIBCHIPCARD: Stopped reading.");
      return CTError();
    } // if error
    if (tmp.empty()){
      return CTError();
    }
  } // while
  return CTError();
}


string CTCard::cardType(){
  return "CTCard";
}


string CTCard::cardTypes(){
  return _cardTypes;
}


CTError CTCard::reopenCard(){
  DBG_WARN("LIBCHIPCARD: The method \"CTCard::reopenCard()\"\n"
	   "has been called. Since THIS class does nothing "
	   "on this method\n"
	   "I guess you have an error in your program.\n");
  return CTError("CTCard::reopenCard()",
		 k_CTERROR_INVALID,0,0,
		 "CTCard::reopenCard() should be overloaded.");
}


